/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


#include "../../config.h"
#include "../eam_types.h"
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <limits.h>
#include <math.h>
#include <malloc.h>

#include "heap.h"
#include "homogeneous.h"
#include "set.h"

#undef DEBUG

/* The header is the first part of a page: in the same memalign()'d block there
   are the GC bits, the allocated bits and the payload (also pointed by the
   gc_bits, allocated_bits and payload fields of the header, for efficiency) */
struct homogeneous_page_header{ /* Size must be a multiple of a word size */
  homogeneous_page_t previous;
  homogeneous_page_t next;    
  size_t object_size; /* in words */
  integer_t max_objects_no;
  integer_t objects_no;
  word_t    head_of_free_list; /* or NULL if the free list is empty */
  unsigned_integer_t* gc_bits;
  unsigned_integer_t* allocated_bits;
  word_t*   payload; /* pointer to the first word of the payload */
};

/* This works since homogeneous_page_header has a size which is multiple of a word */
#define HEADER_SIZE (sizeof(struct homogeneous_page_header) / sizeof(word_t)) /* in words */

#define PAGE_SIZE_IN_CHARS (1 << PAGE_OFFSET_WIDTH)
#define PAGE_SIZE (PAGE_SIZE_IN_CHARS / sizeof(word_t)) /* in words */

/* Returns the number of objects which fit in a page:
   OBJECT_SIZE is in words */
#define OBJECTS_IN_A_PAGE(OBJECT_SIZE) \
  (((int) \
    (((float)PAGE_SIZE - (float)HEADER_SIZE) / \
     ((float)OBJECT_SIZE + 2.0 / (float)WORD_BIT))) - 1)

/* Returns the size in words of the GC bit-array for a page of objects
   with size OBJECT_SIZE (in words) */
#define GC_BITS_SIZE(OBJECT_SIZE) \
  ((int)ceil(((float)OBJECTS_IN_A_PAGE(OBJECT_SIZE)) / (float)WORD_BIT))

void clear_allocated_bits_in_homogeneous_page(homogeneous_page_t p);

/* Defined in gc.c: */
extern set_t set_of_homogeneous_pages;

homogeneous_page_t create_homogeneous_page(size_t object_size){
  homogeneous_page_t r;
  int i;
  
#ifdef DEBUG
  fprintf(stderr, "Creating an homogeneous page for objects of %i words (%i objects)...\n", object_size,
	  OBJECTS_IN_A_PAGE(object_size));
#endif
  /* Allocate storage (header + payload) */
  r = (homogeneous_page_t) memalign(PAGE_SIZE_IN_CHARS, PAGE_SIZE_IN_CHARS);
  
  /* Fail if memalign() failed: */
  if(r == NULL)
    fatal("could not allocate homogeneous page");
  
  /* Initialize fields: */
  r->previous = NULL;
  r->next = NULL;
  r->object_size = object_size;
  r->max_objects_no = OBJECTS_IN_A_PAGE(object_size);
  r->objects_no = 0;
  r->gc_bits = (unsigned_integer_t*)((char*)r + sizeof(struct homogeneous_page_header));
  r->allocated_bits =
    (unsigned_integer_t*)((char*)r + 
			  sizeof(struct homogeneous_page_header) +
			  GC_BITS_SIZE(object_size) * sizeof(word_t));
  r->payload = ((word_t*)((char*)r +
			  sizeof(struct homogeneous_page_header))) +
                          2 * GC_BITS_SIZE(object_size);
  
  /* Create the free list: */
  r->head_of_free_list = &(r->payload[0]);
  for(i = 0; i < r->max_objects_no; i++)
    r->payload[object_size * i] = &(r->payload[object_size * (i + 1)]);
  r->payload[object_size * (r->max_objects_no - 1)] = NULL;

  /* Start with allocated bits and GC bits cleared: */
  clear_allocated_bits_in_homogeneous_page(r);
  clear_gc_bits_in_homogeneous_page(r);

  /* Insert into the set of homogeneous pages: */
  insert_pointer_into_set(r, set_of_homogeneous_pages);

  /* Increment total heap size: */
  total_heap_size += r->object_size * r->max_objects_no;
  
  /* Return: */
  return r;
}

/* Free the storage for the homogeneous page p. This does nothing on systems
   which can not free blocks allocated with memalign(). */
void destroy_homogeneous_page(homogeneous_page_t p){
#ifdef CAN_FREE_MEMALIGNED_BLOCKS
  /* Remove the page from the set of homogeneous pages: */
  remove_pointer_from_set(p, set_of_homogeneous_pages);
  
  /* Decrement total heap size: */
  total_heap_size -= p->object_size * p->max_objects_no;
  
  /* Decrement the global counter of allocated words: */
  allocated_words_since_last_gc -= p->object_size * p->max_objects_no;
  
  /* Free storage: */
  free(p); /* free()'ing memalign()'ed blocks is only safe on GNU systems! */
#endif
}

integer_t homogeneous_page_to_object_size(homogeneous_page_t p){
  return p->object_size;
}

integer_t object_to_index_in_homogeneous_page(word_t object,
					      homogeneous_page_t p);

/* Returns NULL on failure: */
word_t allocate_from_homogeneous_page(homogeneous_page_t p){
  word_t r;
  integer_t index_of_r;
  
  /* Fail if the page is full: */
  if(is_homogeneous_page_full(p))
    return NULL;
  
  /* Take the head of the free list for returning it, and advance
     head_of_free_list: */
  r = p->head_of_free_list;
  p->head_of_free_list = *((word_t*)(p->head_of_free_list));

  /* Set the allocated bit for r: */
  index_of_r = object_to_index_in_homogeneous_page(r, p);
  p->allocated_bits[index_of_r / WORD_BIT] |=
    ((unsigned_integer_t)1u << (index_of_r % WORD_BIT));

  /* Increment the per-page counter of allocated objects: */
  p->objects_no++;

  /* Increment the global counter of allocated words: */
  allocated_words_since_last_gc += p->object_size;

  /* Return the selected object: */
  return r;
}

/* object must belong to p */
void free_from_homogeneous_page(word_t object, homogeneous_page_t p){
  integer_t index_of_object;

  /* Attach object to the free-list of p: */
  ((word_t*)object)[0] = p->head_of_free_list;
  p->head_of_free_list = object;
  
  /* Unset the allocated bit for r: */
  index_of_object = object_to_index_in_homogeneous_page(object, p);
  p->allocated_bits[index_of_object / WORD_BIT] &=
    ~ ((unsigned_integer_t)1u << (index_of_object % WORD_BIT));
  
  /* Decrement the per-page counter of allocated objects: */
  p->objects_no--;

  /* No need to touch allocated_words_since_last_gc; it will be zeroed
     at the end of the sweeping phase. */
}

/* Clear all GC bits of p, which must be an existing homogeneous page. */
void clear_gc_bits_in_homogeneous_page(homogeneous_page_t p){
  /* Clear GC bits, word by word: */
  unsigned_integer_t* pointer_to_word_of_gc_bits = p->gc_bits;
  while((word_t)pointer_to_word_of_gc_bits < (word_t)(p->allocated_bits)){
    /* Clear the current word: */
    *pointer_to_word_of_gc_bits = (unsigned_integer_t)0u;
    /* Go to the next word */
    pointer_to_word_of_gc_bits++;
  }
}

void clear_allocated_bits_in_homogeneous_page(homogeneous_page_t p){
  /* Clear allocated bits, word by word: */
  unsigned_integer_t* pointer_to_word_of_allocated_bits = p->allocated_bits;

  while((word_t)pointer_to_word_of_allocated_bits < (word_t)(p->payload)){
    /* Clear the current word: */
    *pointer_to_word_of_allocated_bits = (unsigned_integer_t)0u;
    /* Go to the next word */
    pointer_to_word_of_allocated_bits++;
  }
}

/* Return the index of object in the payload of p, i.e. the index of its
   GC bit, starting from 0: */
integer_t object_to_index_in_homogeneous_page(word_t object, homogeneous_page_t p){
  return (integer_t)
    (((word_t*)object - (word_t*)(p->payload)) / (p->object_size));
}

/* The index must be in the correct range: */
word_t index_to_object_in_homogeneous_page(integer_t index, homogeneous_page_t p){
  return (word_t)(((word_t*)p->payload) + index * (p->object_size));
}


/* object must belong to p. Returns the GC bit as it was before: */
integer_t mark_in_homogeneous_page(word_t object, homogeneous_page_t p){
  /* Find the right GC bit: */
  const integer_t index_of_object = object_to_index_in_homogeneous_page(object, p);

  unsigned_integer_t* word_of_gc_bits = &(p->gc_bits[index_of_object / WORD_BIT]);
  const unsigned_integer_t bitmask = 
    (unsigned_integer_t)1u << (index_of_object % WORD_BIT);

  /* If the object was already marked then return 1 and do nothing: */
  if(*word_of_gc_bits & bitmask)
    return 1;
  
  /* The object was not marked. Mark it, setting its GC bit: */
  *word_of_gc_bits |= bitmask;
  
  /* If we arrived here then the object was not previously marked: */
  return 0;
}

/* object must belong to p */
void unmark_in_homogeneous_page(word_t object, homogeneous_page_t p){
  /* Find the right GC bit: */
  const integer_t index_of_object = object_to_index_in_homogeneous_page(object, p);
  
  /* Set the bit: */
  p->gc_bits[index_of_object / WORD_BIT] &= 
    ~ ((unsigned_integer_t)1u << (index_of_object % WORD_BIT));
}

/* object must belong to p */
integer_t is_marked_in_homogeneous_page(word_t object, homogeneous_page_t p){
  /* Find the right GC bit: */
  const integer_t index_of_object = object_to_index_in_homogeneous_page(object, p);

  /* Return nonzero if the bit is set: */
  return p->gc_bits[index_of_object / WORD_BIT] 
       & ((unsigned_integer_t)1u << (index_of_object % WORD_BIT));
}

/* The index must be in the right range */
integer_t is_index_marked_in_homogeneous_page(integer_t index,
					      homogeneous_page_t p){
  /* Return nonzero if the bit is set: */
  return p->gc_bits[index / WORD_BIT] & 
    ((unsigned_integer_t)1u << (index % WORD_BIT));
}

/* The index must be in the right range */
integer_t is_index_allocated_in_homogeneous_page(integer_t index,
						 homogeneous_page_t p){
  /* Return nonzero if the bit is set: */
  return p->allocated_bits[index / WORD_BIT] & 
    ((unsigned_integer_t)1u << (index % WORD_BIT));
}

void dump_free_list(homogeneous_page_t p){
  word_t temp = p->head_of_free_list;
  fprintf(stderr,"Free list: [ ");
  while (temp != NULL){
    fprintf(stderr, "%i ", object_to_index_in_homogeneous_page(temp, p));
    temp = *((word_t*)temp); /* Advance in the free list */
  }
  fprintf(stderr,"]\n");
}

#define FROM_0_TO(n) (((integer_t)((double)rand()/(double)RAND_MAX * (double)(n))))
#define FROM_TO(a, b) (a + FROM_0_TO(b - a + 1))

/* Returns the number of freed words. Does *not* clear GC bits. */
int sweep_homogeneous_page(homogeneous_page_t p){
  integer_t i;
  integer_t freed_words = 0;
  word_t temp = p->head_of_free_list;
  
  /* Check each object: free it if it is allocated but not marked: */
  for(i = 0; i < p->max_objects_no; i++)
    if(is_index_allocated_in_homogeneous_page(i, p) &&
       ! is_index_marked_in_homogeneous_page(i, p)){
      free_from_homogeneous_page(index_to_object_in_homogeneous_page(i, p), p);
      freed_words += p->object_size;
    } /* if */

#ifdef DEBUG
  fprintf(stderr, "Freed %i words from the homogeneous page %p\n",
	  freed_words, p);
#endif
  return freed_words;
}

int is_homogeneous_page_full(homogeneous_page_t p){
  return p->head_of_free_list == NULL;
}

int is_homogeneous_page_empty(homogeneous_page_t p){
  return p->objects_no == 0;
}

/* This is only a computation on pointers; it does *not* check whether
   object actually belongs to a homogeneous page. */
homogeneous_page_t object_to_hypothetical_homogeneous_page(word_t object){
  unsigned_integer_t page_as_number;

  /* Set to zero the PAGE_OFFSET_WIDTH least-significant bits of
     object: */
  page_as_number = (unsigned_integer_t)object;
  page_as_number >>= PAGE_OFFSET_WIDTH;
  page_as_number <<= PAGE_OFFSET_WIDTH;

  /* Return the number re-cast to a pointer: */
  return (homogeneous_page_t)page_as_number;
}

/* Returns NULL if the object does not belong to an homogeneous page,
   else return its page: */
homogeneous_page_t object_to_actual_homogeneous_page(word_t object){
  unsigned_integer_t page_as_number;
  unsigned_integer_t offset_as_number;
  integer_t index_of_object;
  
  /* Set to zero the PAGE_OFFSET_WIDTH least-significant bits of
     object: */
  page_as_number = (unsigned_integer_t)
    object_to_hypothetical_homogeneous_page(object);
  
  /* If the hypothetical page is not an actual homogeneous page then fail: */
  if(! does_pointer_belong_to_set((word_t)page_as_number, 
				  set_of_homogeneous_pages))
    return NULL;
  /* If the hypothetical address is not in the payload then fail: */
  if(object < (word_t)((homogeneous_page_t)page_as_number)->payload)
    return NULL;

  /* If the hypothetical address is an internal pointer than fail: */
  /*
  if(((char*)object - (char*)(((homogeneous_page_t)page_as_number)->payload)) %
     (sizeof(word_t) * ((homogeneous_page_t)page_as_number)->object_size) != 0)
    {fprintf(stderr,"WARNING: %p is an internal (false) pointer\n", object);
     return NULL;}
  */
  /* If the object is not allocated then fail: */
  index_of_object =
    object_to_index_in_homogeneous_page(object,
					(homogeneous_page_t)page_as_number);

  if(! ((((homogeneous_page_t)page_as_number)->
	     allocated_bits[index_of_object / WORD_BIT]
	 & ((unsigned_integer_t)1u << (index_of_object % WORD_BIT)))))
    return NULL;
  
  /* If we arrived here then object points to an allocated object of a
     homogeneous page: */
  return (homogeneous_page_t) page_as_number;
}

////////////////////////////

homogeneous_page_t insert_homogeneous_page_into_list(homogeneous_page_t p, homogeneous_page_t list){
  /* Do nothing if p is NULL: */
  if(p == NULL)
    return list;

  /* Update the links: */
  p->next = list;
  p->previous = NULL;
  if(list != NULL)
    list->previous = p;

  /* Return the new list which starts with p: */
  return p;
}

/* Destructively updates list, returning the new version of it.
   The address of the removed page is copied into the location pointed by
   removed: */
homogeneous_page_t remove_pointed_homogeneous_page_from_list(homogeneous_page_t list,
							     homogeneous_page_t* removed){
  homogeneous_page_t former_previous;
  homogeneous_page_t former_next;

  /* Just return if the list is null: */
  if(list == NULL){
    *removed = NULL;
    return NULL;
  }

  /* Update pointers: */
  former_previous = list->previous;
  former_next = list->next;
  if(former_previous != NULL)
    former_previous->next = former_next;
  if(former_next != NULL)
    former_next->previous = former_previous;

  /* Update *removed with the removed page and return the updated list: */
  list->previous = list->next = NULL;
  *removed = list;
  if(former_previous != NULL)
    return former_previous;
  else
    return former_next;
}

homogeneous_page_t first_element_of_list_of_homogeneous_pages(homogeneous_page_t list){
  /* Return NULL if the list is empty */
  if(list == NULL)
    return NULL;

  /* Find beginning: */
  while(list->previous != NULL)
    list = list->previous;

  /* Return the first element: */
  return list;
}

homogeneous_page_t homogeneous_page_to_previous(homogeneous_page_t p){
  return p->previous;
}

homogeneous_page_t homogeneous_page_to_next(homogeneous_page_t p){
  return p->next;
}

void dump_list_of_homogeneous_pages(homogeneous_page_t list){
  homogeneous_page_t p;

  if(list == NULL){
    printf("[ ]\n");
    return;
  }

  /* Find beginning: */
  p = first_element_of_list_of_homogeneous_pages(list);

  /* Output list: */
  printf("List of homogeneous pages: [ ");
  while(p != NULL){
    if(p == list)
      printf("<*>");
    printf("%p ", p);
    p = p->next;
  }
  printf("]\n");
}
