/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


#include "eam_types.h"

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>

#include "bytecode.h"
#include "../commontext.h"
#include "../common/command_line.h"
#include "../common/search_path.h"
#include "../common/string_map.h"
#include "../common/malloc.h"
#include "eamld.h"
#include "eamld_fix_references.h"
#include "prefixes.h"
#include "primitive_exceptions.h"

int verbose = 0;

int modules_no;
int modules_size;
struct epsilon_module **modules;

int sorted_modules_no = 0;
struct epsilon_module **sorted_modules;

struct epsilon_executable linked_executable;
struct string_map *global_to_index;
struct string_map *exception_to_index;

search_path_t modules_search_path;
search_path_t libraries_search_path;

/* This is needed for the depth-first visit used in the implementation
   of the topological sorting: */
typedef enum color
{
  white,			/* not visited */
  gray,				/* being visited */
  black				/* visited */
}
color_t;
color_t *modules_colors;

/* This maps names of modules into index of modules array: */
struct string_map *modules_map;

int
module_name_to_index (char *name)
{
  if(is_string_map_defined_on(modules_map, name))
    return *((int *) access_string_map (modules_map, name));
  else{
    fprintf(stderr, "About the module %s:\n", name);
    fatal("Needed module not supplied");
  }
}

void
initialize_structures ()
{
  modules_size = 16;
  modules = (struct epsilon_module **) xmalloc (modules_size *
						sizeof (struct epsilon_module
							*));
  modules_no = 0;

  modules_map = create_string_map (sizeof (int) /* index of modules */ );
}

void
make_place_for_a_module ()
{
  /* Increment modules_no and resize modules if needed: */
  if (modules_no++ == modules_size)
    {
      modules_size *= 2;
      modules = (struct epsilon_module **) xrealloc (modules,
						     modules_size *
						     sizeof (struct
							     epsilon_module
							     *));
      if (modules == NULL)
	fatal ("Memory exhausted");
    }
}

void
add_module (struct epsilon_module *module)
{
  int index_of_new_module = modules_no;

  if (verbose)
    fprintf (stderr, "    Loading module %s...\n", module->name);
  make_place_for_a_module ();
  modules[index_of_new_module] = module;
  if (is_string_map_defined_on (modules_map, module->name))
    {
      fprintf (stderr, "About the module %s:\n", module->name);
      fatal ("Module entered more than once");
    }
  else
    insert_into_string_map (modules_map, module->name, &index_of_new_module);
}

void
load_module_file (char *filename)
{
  struct epsilon_module *module;
  char* full_pathname = search_for_file_in_search_path(filename,
						       modules_search_path);
  
  if(full_pathname == NULL){
    fprintf(stderr, "About %s:\n", filename);
    fatal("Could not find file");
  }
  if (verbose){
    fprintf (stderr, "  Loading module file \"%s\"...\n", filename);
    fprintf (stderr, "  Found as \"%s\"...\n", full_pathname);
  }
  module = read_epsilon_module_file (full_pathname);
  if (module != NULL)
    add_module (module);
  else{
    fprintf (stderr, "About %s:\n", full_pathname);
    fatal ("Could not read module");
  }
  //fprintf(stderr,">>>%s<<<\n", module->name);
}

void
load_library_file (char *filename)
{
  struct epsilon_library *library;
  char* full_pathname = search_for_file_in_search_path(filename,
						       libraries_search_path);
  int i;

  if(full_pathname == NULL){
    fprintf(stderr, "About %s:\n", filename);
    fatal("Could not find file");
  }
  if (verbose){
    fprintf (stderr, "  Loading library file \"%s\"...\n", filename); 
    fprintf (stderr, "  Found as \"%s\"...\n", full_pathname);
  }
  library = read_epsilon_library_file (full_pathname);
  if (library == NULL)
    {
      fprintf (stderr, "About %s:\n", full_pathname);
      fatal ("Could not read library");
    }
  for (i = 0; i < library->modules_no; i++)
    add_module (library->modules[i]);
}

void
load_file (char *filename)
{
  if (verbose)
    fprintf (stderr, "Loading file %s...\n", filename);
  if (has_extension (filename, "eamo"))	/* Module */
    load_module_file (filename);
  else if (has_extension (filename, "eama"))	/* Library */
    load_library_file (filename);
  else
    {
      fprintf (stderr, "File name %s has unknown extension.\n", filename);
      command_line_fatal
	("All input file names must have extension eamo or eama");
    }
}

void
read_modules (int number_of_files, char **filenames)
{
  int i;

  /* Initialize module structures: */
  initialize_structures ();

  /* Load each file, be it a module or a libraray: */
  for (i = 0; i < number_of_files; i++)
    load_file (filenames[i]);
}

void
output_library ()
{
  struct epsilon_library *library;
  int i;

  if (verbose)
    fprintf (stderr, "Writing library file \"%s\"...\n",
	     long_option_to_value ("output"));
  library =
    (struct epsilon_library *) xmalloc (sizeof (struct epsilon_library));
  library->pragmas = 0;		/* Not used yet */
  library->modules_no = modules_no;
  library->modules =
    (struct epsilon_module **) xmalloc (sizeof (struct epsilon_module *) * modules_no);
  for (i = 0; i < modules_no; i++){
    library->modules[i] = modules[i];
  }

  if (write_epsilon_library_file (library,
				  long_option_to_value ("output")) != 0)
    {
      fprintf (stderr, "About the output file %s:\n",
	       long_option_to_value ("output"));
      fatal ("could not write output file");
    }
}

void
show_unused_modules ()
{
  int i;

  if (sorted_modules_no == modules_no)
    {				/* no unused modules */
      fprintf (stderr, "  All supplied modules are needed.\n");
      return;
    }

  fprintf (stderr, "  These supplied modules are not needed: ");
  for (i = 0; i < modules_no; i++)
    if (modules_colors[i] == white)
      fprintf (stderr, "%s ", modules[i]->name);
  fprintf (stderr, "\n");
}

void
depth_first_visit (int index)
{
  int i;

  if (verbose)
    fprintf (stderr, "DEBUG: depth_first_visit(%s)\n", modules[index]->name);
  modules_colors[index] = gray;	/* We've just discovered index */
  for (i = 0; i < modules[index]->needed_modules_no; i++)
    {
      int index_of_neighbor =
	module_name_to_index (modules[index]->needed_modules[i]);
      color_t color_of_i = modules_colors[index_of_neighbor];
      if (color_of_i == gray)
	{			/* We have discovered a cycle: fail */
	  fprintf (stderr, "About module %s:\n",
		   modules[index_of_neighbor]->name);
	  fatal ("Cyclic dependency");
	}
      else if (color_of_i == white)
	depth_first_visit (index_of_neighbor);
      /* else color of i is black, and we do nothing. */
    }
  modules_colors[index] = black;	/* We've done with index */
  /* Insert modules[index] into the sorted modules array: */
  sorted_modules[sorted_modules_no++] = modules[index];
}

void
sort_modules ()
{
  int i;
  if (verbose)
    fprintf (stderr, "Sorting modules...\n");

  sorted_modules = xmalloc (sizeof (struct epsilon_module *) * modules_no);
  modules_colors = xmalloc (sizeof (color_t) * modules_no);
  for (i = 0; i < modules_no; i++)
    modules_colors[i] = white;	/* no modules are visited yet */
  depth_first_visit (module_name_to_index (long_option_to_value ("main")));

  if (verbose)
    {
      fprintf (stderr, "  Modules will be linked in this order: ");
      for (i = 0; i < sorted_modules_no; i++)
	fprintf (stderr, "%s ", sorted_modules[i]->name);
      fprintf (stderr, "\n");
    }
  if (verbose)
    show_unused_modules ();
}

void
set_registers_no_for_executable ()
{
  int i;
  linked_executable.word_registers_no = 0;
  linked_executable.wide_integer_registers_no = 0;
  linked_executable.float_registers_no = 0;
  linked_executable.wide_float_registers_no = 0;
  linked_executable.wide_wide_float_registers_no = 0;

  /* The number of register of the linked executable is the maximum 
     of the number of registers of each needed module, for each type
     of register: */
  for (i = 0; i < sorted_modules_no; i++)
    {
      if (linked_executable.word_registers_no <
	  sorted_modules[i]->word_registers_no)
	linked_executable.word_registers_no =
	  sorted_modules[i]->word_registers_no;
      if (linked_executable.wide_integer_registers_no <
	  sorted_modules[i]->wide_integer_registers_no)
	linked_executable.wide_integer_registers_no =
	  sorted_modules[i]->wide_integer_registers_no;
      if (linked_executable.float_registers_no <
	  sorted_modules[i]->float_registers_no)
	linked_executable.float_registers_no =
	  sorted_modules[i]->float_registers_no;
      if (linked_executable.wide_float_registers_no <
	  sorted_modules[i]->wide_float_registers_no)
	linked_executable.wide_float_registers_no =
	  sorted_modules[i]->wide_float_registers_no;
      if (linked_executable.wide_wide_float_registers_no <
	  sorted_modules[i]->wide_wide_float_registers_no)
	linked_executable.wide_wide_float_registers_no =
	  sorted_modules[i]->wide_wide_float_registers_no;
    }
}

/* Just concatenate the instructions of every needed module; resolving of
   instructions addresses and identifier will be made later: */
void
set_instructions_for_executable ()
{
  int i, instructions_no_until_here = 0;
  /* Compute linked_executable.instructions_no, as the sum of the
     instructions_no of all needed modules: */
  linked_executable.instructions_no = 0;
  for (i = 0; i < sorted_modules_no; i++)
    linked_executable.instructions_no += sorted_modules[i]->instructions_no;

  /* Allocate linked_executable.instructions and fill it with a copy of
     the instructions of every needed module (the last two instructions are
     added by add_default_exception_handler(): they are "hlt 0" and "dfhn"): */
  linked_executable.instructions = (instruction_t *)
    malloc (sizeof (instruction_t) * (linked_executable.instructions_no + 2));
  for (i = 0; i < sorted_modules_no; i++)
    {
      memcpy (linked_executable.instructions + instructions_no_until_here,
	      sorted_modules[i]->instructions,
	      sizeof (instruction_t) * sorted_modules[i]->instructions_no);
      instructions_no_until_here += sorted_modules[i]->instructions_no;
    }
}

void
set_string_constants_for_executable ()
{
  /* String constants' length can exceed the maximum size allowed in a
     string_map. We can simply append every set of strings, and also their
     addresses will be easier to resolve: */
  /* 9 oct 2003: length of string_map keys has no limit anymore. This can
     be updated, if it's worth the trouble. */
  int i, string_constants_no_until_here = 0;

  /* Compute linked_executable.string_constants_no, as the sum of the
     string_constants_no of all needed modules: */
  linked_executable.string_constants_no = 0;
  for (i = 0; i < sorted_modules_no; i++)
    linked_executable.string_constants_no +=
      sorted_modules[i]->string_constants_no;

  /* Just allocate and copy the buffer of pointers, appending the buffer of
     every module: */
  linked_executable.string_constants = (char **)
    malloc (sizeof (char *) * linked_executable.string_constants_no);
  for (i = 0; i < sorted_modules_no; i++)
    {
      memcpy (linked_executable.string_constants +
	      string_constants_no_until_here,
	      sorted_modules[i]->string_constants,
	      sizeof (char *) * sorted_modules[i]->string_constants_no);
      string_constants_no_until_here +=
	sorted_modules[i]->string_constants_no;
    }
}

/* Builds a string_map mapping globals to unique index, relative to the 
   generated executable: */
void
set_globals_for_executable ()
{
  int i, j;
  integer_t globals_no = (integer_t)0;

  linked_executable.globals_no = 0;
  global_to_index = create_string_map (sizeof (integer_t) /* index */ );
  for (i = 0; i < sorted_modules_no; i++)
    for (j = 0; j < sorted_modules[i]->globals_no; j++)
      if (!is_string_map_defined_on (global_to_index,
				     sorted_modules[i]->globals[j]))
	{
	  insert_into_string_map (global_to_index,
				  sorted_modules[i]->globals[j], /* name */
				  &globals_no);/* index */
	  globals_no++;
	  linked_executable.globals_no++;
	}
}

/* Builds a string_map mapping exceptions to unique index, relative to the 
   generated executable: */
void
set_exceptions_for_executable ()
{
  int i, j;
  integer_t exceptions_no = (integer_t)0;
  integer_t exceptions_current_size = 16; /* linked_executable.exceptions grows */

  /* Create linked_executable.exceptions: */
  linked_executable.exceptions = (char**) xmalloc(sizeof(char*) *
						  exceptions_current_size);
  linked_executable.exceptions_no = 0; /* there are no exceptions yet */

  /* Create the hash table mapping exceptions names into encodings: */
  exception_to_index = create_string_map (sizeof (integer_t) /* index */ );

  /* Add primitive exceptions (these ones have fixed consecutive encodings
     starting at 0, and they must be inserted first): */
  for(i = 0; i < primitive_exceptions_no; i++){
    insert_into_string_map (exception_to_index,
			    (char*)primitive_exceptions_names[i], /* name */
			    &exceptions_no);/* index */
    exceptions_no++;
    linked_executable.exceptions_no++;
    /* Copy the name in exceptions, making place if needed: */
    if(exceptions_no > exceptions_current_size){
      exceptions_current_size *= 2;
      linked_executable.exceptions = (char**)
	xrealloc(linked_executable.exceptions,
		 exceptions_current_size * sizeof(char*));
    }
    linked_executable.exceptions[linked_executable.exceptions_no - 1] =
      (char*)primitive_exceptions_names[i];
  }
  // To do: fix code repetition
  
  /* Add exceptions from modules: */
  for (i = 0; i < sorted_modules_no; i++)
    for (j = 0; j < sorted_modules[i]->exceptions_no; j++)
      if (!is_string_map_defined_on (exception_to_index,
				     sorted_modules[i]->exceptions[j]))
	{
	  insert_into_string_map (exception_to_index,
				  sorted_modules[i]->exceptions[j], /* name */
				  &exceptions_no);/* index */
	  exceptions_no++;
	  linked_executable.exceptions_no++;
	  /* Copy the name in exceptions, making place if needed: */
	  if(exceptions_no > exceptions_current_size){
	    exceptions_current_size *= 2;
	    linked_executable.exceptions = (char**)
	      xrealloc(linked_executable.exceptions,
		       exceptions_current_size * sizeof(char*));
	  }
	  linked_executable.exceptions[linked_executable.exceptions_no - 1] =
	    sorted_modules[i]->exceptions[j];
	}
}

/////////////////////////////////////////////////////////////////
void
set_c_libraries_for_executable (){
  /* Simply append every sequence of strings; their addresses relative to the
     executable will be easy to resolve: */
  int i, c_libraries_no_until_here = 0;

  /* Compute linked_executable.c_libraries_no, as the sum of the
     c_libraries_no of all needed modules: */
  linked_executable.c_libraries_no = 0;
  for (i = 0; i < sorted_modules_no; i++)
    linked_executable.c_libraries_no +=
      sorted_modules[i]->c_libraries_no;

  /* Just allocate and copy the buffer of pointers, appending the buffer of
     every module. Pointed data (the characters) are not touched. */
  linked_executable.c_libraries = (char **)
    xmalloc (sizeof (char *) * linked_executable.c_libraries_no);
  for (i = 0; i < sorted_modules_no; i++)
    {
      memcpy (linked_executable.c_libraries +
	      c_libraries_no_until_here,
	      sorted_modules[i]->c_libraries,
	      sizeof (char *) * sorted_modules[i]->c_libraries_no);
      c_libraries_no_until_here +=
	sorted_modules[i]->c_libraries_no;
    }
}

void
set_c_symbols_for_executable (){
  /* Simply append every sequence of <string, integer>s; their addresses relative
     to the executable will be easy to resolve: */
  int i, c_symbols_no_until_here = 0;

  /* Compute linked_executable.c_symbols_no, as the sum of the
     c_symbols_no of all needed modules: */
  linked_executable.c_symbols_no = 0;
  for (i = 0; i < sorted_modules_no; i++)
    linked_executable.c_symbols_no +=
      sorted_modules[i]->c_symbols_no;

  /* Just allocate and copy the buffer of c_symbol_t's, appending the buffer of
     every module. The data pointed within c_symbol_t's (the characters) are not
     touched. */
  linked_executable.c_symbols = (c_symbol_t *)
    xmalloc (sizeof (c_symbol_t) * linked_executable.c_symbols_no);
  for (i = 0; i < sorted_modules_no; i++)
    {
      memcpy (linked_executable.c_symbols +
	      c_symbols_no_until_here,
	      sorted_modules[i]->c_symbols,
	      sizeof (c_symbol_t) * sorted_modules[i]->c_symbols_no);
      c_symbols_no_until_here +=
	sorted_modules[i]->c_symbols_no;
    }
}
/////////////////////////////////////////////////////////////////////

void
link_sorted_modules ()
{
  int i;
  if(verbose)
    fprintf(stderr, "  Setting header...\n");
  strcpy (linked_executable.bytecode_version, BYTECODE_VERSION);
  linked_executable.pragmas = 0;	/* Not used yet */
  if(verbose)
    fprintf(stderr, "  Setting registers' no...\n");
  set_registers_no_for_executable ();
  if(verbose)
    fprintf(stderr, "  Copying instructions...\n");
  set_instructions_for_executable ();
  if(verbose)
    fprintf(stderr, "  Copying string constants...\n");
  set_string_constants_for_executable ();
  if(verbose)
    fprintf(stderr, "  Copying globals...\n");
  set_globals_for_executable ();
  if(verbose)
    fprintf(stderr, "  Copying exceptions...\n");
  set_exceptions_for_executable ();
  if(verbose)
    fprintf(stderr, "  Copying C libraries...\n");
  set_c_libraries_for_executable ();
  if(verbose)
    fprintf(stderr, "  Copying C symbols...\n");
  set_c_symbols_for_executable ();

  if(verbose)
    fprintf(stderr, "  Fixing foreing references...\n");
  fix_foreign_references ();
}

void
output_executable ()
{
  /* Sort modules according their depencencies, filtering away the ones
     which are not needed: */
  sort_modules ();

  if(verbose)
    fprintf(stderr, "Linking ...\n");
  /* Link the needed modules together, in the order of sorted_modules: */
  link_sorted_modules ();

  /* Fix linked_executable.instructions_no, adding the two trailing fake instructions: */
  add_default_exception_handler();

  /* Write the output file: */
  if(verbose)
    fprintf(stderr, "Writing the output file \"%s\" ...\n", long_option_to_value ("output"));
  if (write_epsilon_executable_file (&linked_executable,
				     long_option_to_value ("output")) != 0)
    fatal ("Could not write output file");
  else if (verbose)
    fprintf (stderr, "%s was written.\n", long_option_to_value ("output"));
}

void
check_command_line ()
{
  /* --library conflicts with --main: */
  if (long_option_to_value ("library") == 0)
    {				/* --no-library */
      if (!strcmp ("", long_option_to_value ("main")))	/* --no-main */
	command_line_fatal
	  ("You must either use --main=XXXX, -mXXXX or --library, -l\n");
    }
  else
    {				/* --library */
      if (strcmp ("", long_option_to_value ("main")))	/* --main */
	command_line_fatal
	  ("You can't use --main=XXXX, -mXXXX and --library, -l together\n");
    }
  /* --output is mandatory: */
  if (!strcmp (long_option_to_value ("output"), ""))
    command_line_fatal ("The option --output=XXXX, -oXXXX is mandatory");

  /* The extension of the output file must not conflict with --library: */
  if (long_option_to_value ("library"))
    {
      if (!has_extension (long_option_to_value ("output"), "eama"))
	command_line_fatal
	  ("The parameter of --output, -o must have extension eama");
    }
  else
    {
      if (!has_extension (long_option_to_value ("output"), "eamx"))
	command_line_fatal
	  ("The parameter of --output, -o must have extension eamx");
    }
}

void build_search_paths(){
  /* Initialize search paths: */
  modules_search_path = create_search_path();
  libraries_search_path = create_search_path();
  
  /* Add search paths from the command line: */
  append_path_to_search_path(
    long_option_to_value("modules-path"),
    modules_search_path);
  append_path_to_search_path(
    long_option_to_value("libraries-path"),
    libraries_search_path);
  
  /* Add default search paths: */
  append_path_to_search_path(
    ".",
    modules_search_path);
  append_path_to_search_path(
    EPSILON_LIBRARIES_PATH ":.",
    //"/home/luca/progetti/epsilon/epsilon-working/epsilon/library:.",
    libraries_search_path);
}

int
main (int argc, char **argv)
{
  int strlen_of_argv_1;
  FILE *f;

  /* Define command line behavior: */
  set_program_name ("eamld (" PACKAGE_NAME ")");
  set_general_help_message
    ("Links the eAM objects FILE1 ... FILEn into an eAM exexcutable \n"
     "XXXX.eamx or an eAM library XXXX.eama. The output file name can be\n"
     "chosen with the --output=OUTPUT, -oOUTPUT option.\n"
     "Each of FILE1 ... FILEn can be either an eAM module or an eAM library.\n"
     "The option --output=XXXX, -oXXXX is mandatory.\n"
     "One of the options --main=XXXX, -mXXXX and --library, -l is mandatory.");
  set_synopsis_string ("eamld OPTIONS FILE1 ... FILEn");
  set_version_string (VERSION_STRING);
  set_copyright_string (COPYRIGHT_STRING);
  set_license_message (LICENSE_STRING);
  set_bug_reporting_message (BUG_REPORTING_MESSAGE);
  
  add_toggle_option ("verbose", 'v', 0,
		     "Be more verbose while processing");
  add_toggle_option ("library", 'l', 0,
		     "Genearate a library instead of a program");
  add_option_with_nonoptional_parameter ("main", 'm', "",
					 "Say what module is the main one");
  add_option_with_nonoptional_parameter ("output", 'o', "",
					 "Define XXXX to be the output file name");
  add_option_with_nonoptional_parameter ("libraries-path", 'L', "",
					 "Prepend XXXX to the directories to search");
  add_option_with_nonoptional_parameter ("modules-path", 'M', "",
					 "Prepend XXXX to the directories to search");
  
  /* Parse command line and check consistency: */
  parse_command_line (&argc, argv);
  check_command_line ();

  /* Build search paths according to command-line parameters: */
  build_search_paths();

  if (long_option_to_value ("verbose"))	/* Remember that we should be verbose */
    verbose = 1;

  /* Read modules from the filenames in the command line (libraries
     are automatically decomponed into modules): */
  read_modules (argc - 1, argv + 1);
  
  /* Output the library or the executable, as requested by the user: */
  if (long_option_to_value ("library"))
    output_library ();
  else
    output_executable ();
  
  return EXIT_SUCCESS;
}
