/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002, 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

#include "../common/string_map.h"
#include "types.h"
#include "term.h"
#include "epsilon.h"
#include "environment.h"

type_t type_error, void_type, integer_type, float_type, boolean_type,
  string_type, character_type, c_type;
type_t tuple_terminator_type;

constructor_t void_c, integer_c, float_c, boolean_c, string_c, character_c, c_c;
constructor_t tuple_terminator_c;
constructor_t io_c, list_c, pair_c, function_c, array_c, promise_c,
  postfix_operator_c, infix_operator_left_c, infix_operator_right_c;

constructor_t abstract_c, concrete_c;

//////////////////////////////////////////////////////////////////////////////////////////////////////////
#define INITIAL_EXCEPTIONS_TABLE_SIZE  64

int exceptions_table_next_position = 0;
int exceptions_table_current_size = INITIAL_EXCEPTIONS_TABLE_SIZE;

struct exceptions_table_element
{
  char name[IDENTIFIER_LENGTH + 1];
  type_t parameter;
}
 *exceptions_table;

void
create_exceptions_table ()
{
  exceptions_table_current_size = INITIAL_EXCEPTIONS_TABLE_SIZE;
  exceptions_table_next_position = 0;
  exceptions_table = (struct exceptions_table_element *)
    malloc (sizeof (struct exceptions_table_element) *
	    exceptions_table_current_size);
}

void
destroy_exceptions_table ()
{
  free (exceptions_table);
}

int
is_exception_defined (char *maybe_qualified_name)
{
  char *qualified_name;
  int i;

  if (is_qualified (maybe_qualified_name))
    qualified_name = maybe_qualified_name;
  else
    qualified_name = qualify_with_this_module (maybe_qualified_name);

  for (i = 0; i < exceptions_table_next_position; i++)
    if (!strcmp (exceptions_table[i].name, qualified_name))
      return 1;
  return 0;
}

/* Returns 0 on success, -1 on 'already exists' */
int
create_exception (char *unqualified_name, type_t parameter)
{
  char *qualified_name = qualify_with_this_module (unqualified_name);

  if (is_exception_defined (qualified_name))
    return -1;

  if (exceptions_table_next_position == exceptions_table_current_size)
    {
      exceptions_table_current_size *= 2;
      exceptions_table = (struct exceptions_table_element *)
	realloc (exceptions_table,
		 exceptions_table_current_size *
		 sizeof (struct exceptions_table_element));
    }
  strcpy (exceptions_table[exceptions_table_next_position].name,
	  qualified_name);
  exceptions_table[exceptions_table_next_position].parameter = parameter;
  exceptions_table_next_position++;
  bind_exception_name (qualified_name);

  return 0;
}

/* Lookup an exception

   returns:
   0 on success
   -1 for 'not found'
*/
int
lookup_exception (char *maybe_qualified_name, type_t * parameter)
{
  char *qualified_name = qualify_exception (maybe_qualified_name);
  int i;
  for (i = 0; i < exceptions_table_next_position; i++)
    if (!strcmp (exceptions_table[i].name, qualified_name))
      {
	*parameter = freshen_type (exceptions_table[i].parameter);
	return 0;		/* ok, the entry was found */
      }

  return -1;			/* not found */
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////

/* type_t's and constructor_t's are encoded as numbers into strings: */
struct string_map *abstract_type_name_to_abstract_type_constructor_map;
struct string_map *abstract_type_constructor_to_abstract_type_name_map;
struct string_map *abstract_type_name_to_abstract_type_definition_map;
struct string_map *abstract_type_name_to_abstract_type_parameters_map;

#define NUMBER_AS_STRING_LENGTH 10

#define CONCRETE_TYPES_INITIAL_SIZE 1
int next_concrete_type = 0;
int concrete_types_size = CONCRETE_TYPES_INITIAL_SIZE;

struct concrete_types_element
{
  char name[IDENTIFIER_LENGTH + 1];
  type_t formal_parameters;
  constructor_t constructor;	/* in the sense of a _term_ constructor */
  int first_case;
  int cases_no;
}
 *concrete_types;

#define CONCRETE_TYPES_CONSTRUCTORS_INITIAL_SIZE 1
int next_concrete_type_constructor = 0;
int concrete_types_constructors_size =
  CONCRETE_TYPES_CONSTRUCTORS_INITIAL_SIZE;

struct concrete_types_constructors_element
{
  char name[IDENTIFIER_LENGTH + 1];
  int concrete_type;		/* index in the table */
  int encoding;
  type_t case_definition;
}
 *concrete_types_constructors;

void
dump_concrete_types ()
{
  int i;

  fprintf (stderr, "%20s| Parameters\n", "Name");
  fprintf (stderr,
	   "--------------------+----------------------------------\n");
  for (i = 0; i < next_concrete_type; i++)
    {
      fprintf (stderr, "%20s|", concrete_types[i].name);
      output_type (stderr, concrete_types[i].formal_parameters);
      fprintf (stderr, "\n");
    }
  fprintf (stderr, "\n");

  fprintf (stderr, "%20s|%10s|%10s| Definition\n", "Case name", "Type",
	   "Encoding");
  fprintf (stderr,
	   "--------------------+----------+----------+----------------------------------\n");
  for (i = 0; i < next_concrete_type_constructor; i++)
    {
      fprintf (stderr, "%20s|%10i|%10i| ",
	       concrete_types_constructors[i].name,
	       concrete_types_constructors[i].concrete_type,
	       concrete_types_constructors[i].encoding);
      output_type (stderr, concrete_types_constructors[i].case_definition);
      fprintf (stderr, "\n");
    }
  fprintf (stderr, "\n");
}

#define INITIAL_OUTPUT_PROCEDURES_STACK_SIZE 1

int output_procedures_current_size;
int output_procedures_next_position;
int output_procedures_index = 1;
struct output_procedures_stack_element
{
  type_t type;
  int index;
}
 *output_procedures_stack;

void create_output_procedures_stack ();
void destroy_output_procedures_stack ();

void
initialize_types_table ()
{
  init_terms ();

  type_error = error;

  void_c = constructor ("void", 0);
  integer_c = constructor ("integer", 0);
  float_c = constructor ("float", 0);
  boolean_c = constructor ("boolean", 0);
  string_c = constructor ("string", 0);
  character_c = constructor ("character", 0);
  c_c = constructor("c_type", 0);

  io_c = constructor ("io", 1);
  list_c = constructor ("list", 1);
  pair_c = constructor ("tuple", 2);
  function_c = constructor ("function", 2);
  array_c = constructor ("array", 1);
  promise_c = constructor ("promise", 1);
  postfix_operator_c = constructor ("postfix", 2);
  infix_operator_left_c = constructor ("infix_left", 3);
  infix_operator_right_c = constructor ("infix_right", 3);

  tuple_terminator_c = constructor ("tuple_terminator", 0);

  abstract_c = constructor ("abstract", 1);
  concrete_c = constructor ("concrete", 1);

  void_type = term (void_c);
  integer_type = term (integer_c);
  float_type = term (float_c);
  boolean_type = term (boolean_c);
  string_type = term (string_c);
  character_type = term (character_c);
  c_type = term(c_c);

  tuple_terminator_type = term (tuple_terminator_c);

  abstract_type_name_to_abstract_type_constructor_map =
    create_string_map (/* for encoding a number */
		       sizeof (char) * (NUMBER_AS_STRING_LENGTH + 1));
  abstract_type_constructor_to_abstract_type_name_map =
    create_string_map (sizeof(char) * (IDENTIFIER_LENGTH + 1));
  abstract_type_name_to_abstract_type_definition_map =
    create_string_map (/* for encoding a number */
		       sizeof (char) * (NUMBER_AS_STRING_LENGTH + 1));
  abstract_type_name_to_abstract_type_parameters_map =
    create_string_map (/* for encoding a number */
		       sizeof (char) * (NUMBER_AS_STRING_LENGTH + 1));
  concrete_types = (struct concrete_types_element *)
    malloc (sizeof (struct concrete_types_element) *
	    CONCRETE_TYPES_INITIAL_SIZE);
  concrete_types_constructors =
    (struct concrete_types_constructors_element *)
    malloc (sizeof (struct concrete_types_constructors_element) *
	    CONCRETE_TYPES_CONSTRUCTORS_INITIAL_SIZE);

  create_exceptions_table ();
  create_output_procedures_stack ();
}

void
destroy_types_table ()
{
  //dump_concrete_types();
  //dump_global_environment();

  destroy_string_map (abstract_type_name_to_abstract_type_constructor_map);
  destroy_string_map (abstract_type_constructor_to_abstract_type_name_map);
  destroy_string_map (abstract_type_name_to_abstract_type_definition_map);
  destroy_string_map (abstract_type_name_to_abstract_type_parameters_map);

  destroy_exceptions_table ();

  free (concrete_types);
  free (concrete_types_constructors);
  free_terms ();

  destroy_output_procedures_stack ();
}

type_t
unify_types (type_t a, type_t b)
{
  return unify (a, b);
}

int
is_error_type (type_t a)
{
  return is_error (a);
}

int
do_types_match (type_t a, type_t b)
{
  return !is_error (unify (a, b));
}

type_t
new_io_type (type_t x)
{
  return term (io_c, x);
}

type_t
new_list_type (type_t x)
{
  return term (list_c, x);
}

type_t
new_pair_type (type_t x, type_t y)
{
  return term (pair_c, x, y);
}

type_t
new_function_type (type_t x, type_t y)
{
  return term (function_c, x, y);
}

type_t
new_array_type (type_t x)
{
  return term (array_c, x);
}

type_t
new_promise_type (type_t x)
{
  return term (promise_c, x);
}

type_t
new_postfix_operator_type (type_t from, type_t to)
{
  return term (postfix_operator_c, from, to);
}

type_t
new_infix_operator_left_type (type_t from_1, type_t from_2, type_t to)
{
  return term (infix_operator_left_c, from_1, from_2, to);
}

type_t
new_infix_operator_right_type (type_t from_1, type_t from_2, type_t to)
{
  return term (infix_operator_right_c, from_1, from_2, to);
}

type_t
new_generic_type ()
{
  return vterm (new_variable ("t"));
}

type_t
new_tuple_type (type_t x, type_t tuple)
{
  return term (pair_c, x, tuple);
}

type_t
new_abstract_type (type_t x)
{
  return term (abstract_c, x);
}

type_t
new_concrete_type (type_t x)
{
  return term (concrete_c, x);
}

type_t
generic_type (char *name)
{
  return vterm (variable (name));
}

int
is_a_void_type (type_t a)
{
  return is_constructed_with (a, void_c);
}

int
is_an_integer_type (type_t a)
{
  return is_constructed_with (a, integer_c);
}

int
is_a_float_type (type_t a)
{
  return is_constructed_with (a, float_c);
}

int
is_a_boolean_type (type_t a)
{
  return is_constructed_with (a, boolean_c);
}

int
is_a_string_type (type_t a)
{
  return is_constructed_with (a, string_c);
}

int
is_a_character_type (type_t a)
{
  return is_constructed_with (a, character_c);
}

int
is_a_c_type (type_t a)
{
  return is_constructed_with (a, c_c);
}

int
is_a_list_type (type_t a)
{
  return is_constructed_with (a, list_c);
}

int
is_an_io_type (type_t a)
{
  return is_constructed_with (a, io_c);
}

int
is_a_pair_type (type_t a)
{
  return is_constructed_with (a, pair_c);
}

int
is_a_tuple_terminator_type (type_t a)
{
  return is_constructed_with (a, tuple_terminator_c);
}

int
is_a_function_type (type_t a)
{
  return is_constructed_with (a, function_c);
}

int
is_an_array_type (type_t a)
{
  return is_constructed_with (a, array_c);
}

int
is_a_promise_type (type_t a)
{
  return is_constructed_with (a, promise_c);
}

int
is_a_postfix_operator_type (type_t a)
{
  return is_constructed_with (a, postfix_operator_c);
}

int
is_an_infix_operator_left_type (type_t a)
{
  return is_constructed_with (a, infix_operator_left_c);
}

int
is_an_infix_operator_right_type (type_t a)
{
  return is_constructed_with (a, infix_operator_right_c);
}

int
is_an_abstract_type (type_t a)
{
  return is_constructed_with (a, abstract_c);
}

int
is_a_concrete_type (type_t a)
{
  return is_constructed_with (a, concrete_c);
}

int
is_a_generic_type (type_t a)
{
  return is_variable (a);
}

type_t
element_type (type_t t, int n)
{
  return subterm (t, n);
}

/* Create a new tuple type with length n+1: the first n-1
   elements are generic, the n-th is assigned a given type
   and the last element is a generic (which also embodies the
   terminator). */
type_t
new_tuple_type_from_selector (int n, type_t t)
{
  type_t x;
  int i;

  if (n != 1)
    x = new_pair_type (t, new_generic_type ());
  else				/* The smallest possibile tuple has 2 elements, not 1. */
    x =
      new_pair_type (t,
		     new_pair_type (new_generic_type (),
				    new_generic_type ()));

  for (i = 0; i < n - 1; i++)
    x = new_pair_type (new_generic_type (), x);
  return x;
}

type_t
append_to_tuple_type (type_t tuple, type_t t)
{
  if (is_a_tuple_terminator_type (tuple))
    return new_pair_type (t, tuple_terminator_type);
  else
    return new_pair_type (element_type (tuple, 0),
			  append_to_tuple_type (element_type (tuple, 1), t));
}

int
arity_of_tuple_type (type_t t)
{
  int i, r = 0;
  for (i = t; !is_a_tuple_terminator_type (i); i = element_type (i, 1))
    r++;

  return r;
}

/* This is O(dimension). n starts at 1. */
type_t
element_of_tuple_type (type_t t, int n)
{
  int i;

  for (i = 1; i < n; i++)
    t = element_type (t, 1);

  return element_type (t, 0);
}

int
is_type_more_general_than_or_equal_to (type_t x, type_t y)
{
  return is_term_more_general_than_or_equal_to (x, y);
}

int
are_types_equal (type_t x, type_t y)
{
  return are_terms_equal (x, y);
}

int
are_types_equal_excluding_variables (type_t x, type_t y)
{
  return are_terms_equal_excluding_variables (x, y);
}

type_t
normalize_type (type_t t)
{
  return normalize_term (t);
}

type_t
freshen_type (type_t t)
{
  return freshen_term (t);
}

void
make_type_unify_with_type (type_t t1, type_t t2)
{
  make_term_unify_with_term (t1, t2);
}

/* ************************************************ abstract types ************************************************** */

constructor_t
abstract_type_name_to_abstract_type_constructor (char *maybe_qualified_name)
{
  char *qualified_name = qualify_abstract_type_name (maybe_qualified_name);
  constructor_t r;
  char *r_as_string;

  r_as_string =
    access_string_map (abstract_type_name_to_abstract_type_constructor_map,
		       qualified_name);
  if (r_as_string == NULL)
    {
      fprintf (stderr, "FATAL: types.c 2df0-92\n");
      exit (EXIT_FAILURE);
    }
  sscanf (r_as_string, "%i", &r);
  return r;
}

void
update_abstract_type_name_to_abstract_type_constructor (char
							*maybe_qualified_name,
							constructor_t c)
{
  char *qualified_name = qualify_abstract_type_name (maybe_qualified_name);
  char c_as_string[NUMBER_AS_STRING_LENGTH + 1];

  sprintf (c_as_string, "%i", c);
  insert_into_string_map (abstract_type_name_to_abstract_type_constructor_map,
			  qualified_name, c_as_string);
}

char *
abstract_type_constructor_to_abstract_type_name (constructor_t constructor)
{
  char c_as_string[NUMBER_AS_STRING_LENGTH + 1];
  char *r;
  sprintf (c_as_string, "%i", constructor);

  r =
    access_string_map (abstract_type_constructor_to_abstract_type_name_map,
		       c_as_string);
  if (r == NULL)
    {
      fprintf (stderr, "FATAL: types.c kl;lm;a\n");
      exit (EXIT_FAILURE);
    }

  return r;
}

void
update_abstract_type_constructor_to_abstract_type_name (constructor_t
							constructor,
							char
							*maybe_qualified_name)
{
  char *qualified_name = qualify_abstract_type_name (maybe_qualified_name);
  char c_as_string[NUMBER_AS_STRING_LENGTH + 1];

  sprintf (c_as_string, "%i", constructor);
  insert_into_string_map (abstract_type_constructor_to_abstract_type_name_map,
			  c_as_string, qualified_name);
}

type_t
abstract_type_name_to_abstract_type_definition (char *maybe_qualified_name)
{
  char *qualified_name = qualify_abstract_type_name (maybe_qualified_name);
  type_t r;
  char *r_as_string;

  r_as_string =
    access_string_map (abstract_type_name_to_abstract_type_definition_map,
		       qualified_name);
  if (r_as_string == NULL)
    {
      fprintf (stderr, "FATAL: types.c 2asdfadsf\n");
      exit (EXIT_FAILURE);
    }
  sscanf (r_as_string, "%i", &r);
  return r;
}

void
update_abstract_type_name_to_abstract_type_definition (char
						       *maybe_qualified_name,
						       type_t definition)
{
  char *qualified_name = qualify_abstract_type_name (maybe_qualified_name);
  char d_as_string[NUMBER_AS_STRING_LENGTH + 1];

  sprintf (d_as_string, "%i", definition);
  insert_into_string_map (abstract_type_name_to_abstract_type_definition_map,
			  qualified_name, d_as_string);
}

type_t
abstract_type_name_to_abstract_type_parameters (char *maybe_qualified_name)
{
  char *qualified_name = qualify_abstract_type_name (maybe_qualified_name);
  type_t r;
  char *r_as_string;

  r_as_string =
    access_string_map (abstract_type_name_to_abstract_type_parameters_map,
		       qualified_name);
  if (r_as_string == NULL)
    {
      fprintf (stderr, "FATAL: types.c kljljkh\n");
      exit (EXIT_FAILURE);
    }
  sscanf (r_as_string, "%i", &r);
  return r;
}

void
update_abstract_type_name_to_abstract_type_parameters (char
						       *maybe_qualified_name,
						       type_t parameters)
{
  char *qualified_name = qualify_abstract_type_name (maybe_qualified_name);
  char p_as_string[NUMBER_AS_STRING_LENGTH + 1];

  sprintf (p_as_string, "%i", parameters);
  remove_from_string_map (abstract_type_name_to_abstract_type_parameters_map, qualified_name);	/* formal parameters may change from declaration to definition */
  insert_into_string_map (abstract_type_name_to_abstract_type_parameters_map,
			  qualified_name, p_as_string);
}

int
is_abstract_type_defined (char *maybe_qualified_name)
{
  char *qualified_name = qualify_abstract_type_name (maybe_qualified_name);
  return
    access_string_map (abstract_type_name_to_abstract_type_definition_map,
		       qualified_name) != NULL;
}

int
is_abstract_type_declared (char *maybe_qualified_name)
{
  char *qualified_name = qualify_abstract_type_name (maybe_qualified_name);
  return
    access_string_map (abstract_type_name_to_abstract_type_constructor_map,
		       qualified_name) != NULL;
}

/* returns:
   0 on success 
   -1 for 'already exists'
*/
int
create_undefined_abstract_type (char *maybe_qualified_name,
				type_t formal_parameters
				/* or tuple_terminator_type */ )
{
  char *qualified_name = qualify_with_this_module (maybe_qualified_name);
  constructor_t c = constructor (qualified_name, 1);

  /* Refresh type variables in formal parameters: */
  formal_parameters = freshen_type (formal_parameters);

  if (is_abstract_type_defined (qualified_name))
    return -1;

  update_abstract_type_constructor_to_abstract_type_name (c, qualified_name);
  update_abstract_type_name_to_abstract_type_constructor (qualified_name, c);
  update_abstract_type_name_to_abstract_type_parameters (qualified_name,
							 formal_parameters);

  bind_abstract_type_name (qualified_name);

  return 0;
}


/* instance is abstract(XXXX(X1...X2...Xn))
   formal parameters are     T1...T2...Tn
   definition is           t(T1...T2...Tn)

   returns                 t(X1...X2...Xn)

  the type must be defined: */
type_t
instantiate_defined_abstract_type (type_t instance)
{
  char *qualified_name =
    abstract_type_constructor_to_abstract_type_name (term_to_constructor
						     (element_type
						      (instance, 0)));
  type_t definition =
    abstract_type_name_to_abstract_type_definition (qualified_name);
  type_t formal_parameters =
    abstract_type_name_to_abstract_type_parameters (qualified_name);
  type_t actual_parameters = element_type (element_type (instance, 0), 0);

  type_t actual_parameters_to_instantiated_type;

  if (!is_abstract_type_defined (qualified_name))
    {
      fprintf (stderr,
	       "instantiate_defined_abstract_type(): FATAL: abstract type is not defined");
      exit (EXIT_FAILURE);
    }

  actual_parameters_to_instantiated_type =
    unify_types (new_function_type (formal_parameters,
				    definition),
		 new_function_type (actual_parameters, new_generic_type ()));

  if (is_error (actual_parameters_to_instantiated_type))
    return type_error;
  else
    return element_type (actual_parameters_to_instantiated_type, 1);
}

/* maybe_qualified_name must be defined: */
void
instantiate_all_references_to_abstract_type (char *maybe_qualified_name)
{
  char *qualified_name = qualify_with_this_module (maybe_qualified_name);
  type_t definition =
    abstract_type_name_to_abstract_type_definition (qualified_name);
  constructor_t constructor_for_abstract_type =
    abstract_type_name_to_abstract_type_constructor (qualified_name);
  type_t i;

  //fprintf(stderr,"Sostituisco le istanze di %s ",qualified_name);
  //fprintf(stderr,"con ");
  //output_type(stderr,definition);
  //fprintf(stderr,"\n");

  for (i = 0; i <= latest_term (); i++)
    {
      if (is_constructed_with (i, abstract_c))
	if (is_constructed_with
	    (element_type (i, 0), constructor_for_abstract_type))
	  {
	    type_t instantiated = instantiate_defined_abstract_type (i);

	    //fprintf(stderr,"Ho istanziato ");output_type(stderr,i);fprintf(stderr," con ");       
	    //output_type(stderr,instantiated);
	    //fprintf(stderr,"\n");

	    replace_term_with_term (i, instantiated);
	  }
    }				/* for */
}

/* returns:
   0 on success
   -1 for 'already exists'
   -2 for 'not compatible with declaration'
   -3 for 'warning: not declared'
*/
int
define_abstract_type (char *maybe_qualified_name,
		      type_t formal_parameters, type_t definition)
{
  /* Definition can only occour in the module where the abstract type is declared: */
  char *qualified_name = qualify_with_this_module (maybe_qualified_name);
  type_t declared_formal_parameters;
  type_t formal_parameters_arrow_definition;
  int was_type_declared = 1;

  if (!is_abstract_type_declared (maybe_qualified_name))
    {
      create_undefined_abstract_type (qualified_name, formal_parameters);
      was_type_declared = 0;
    }

  declared_formal_parameters =
    abstract_type_name_to_abstract_type_parameters (qualified_name);

  /*/
     fprintf(stderr,"STO DEFINENDO %s (",qualified_name);
     output_type(stderr,formal_parameters);
     fprintf(stderr,") COME ");
     output_type(stderr,definition);
     fprintf(stderr,"\n");
     dump_global_environment();
     /* */

  /* Freshen formal_parameters and definition in a consistent way: */
  formal_parameters_arrow_definition =
    freshen_type (new_function_type (formal_parameters, definition));
  if (is_error_type (formal_parameters_arrow_definition))
    return -2;
  formal_parameters = element_type (formal_parameters_arrow_definition, 0);
  definition = element_type (formal_parameters_arrow_definition, 1);

  //fprintf(stderr,"define_abstract_type(): init\n");

  if (is_abstract_type_defined (qualified_name))
    return -1;
  if (!is_abstract_type_declared (qualified_name))
    {
      fprintf (stderr,
	       "types.c: warning: define_abstract_type(): type not declared\n");
      create_undefined_abstract_type (qualified_name, formal_parameters);	/* a fake declaration */
    }
  if (is_error (unify_types (declared_formal_parameters, formal_parameters)))
    return -2;			/* not compatible with declaration */

  update_abstract_type_name_to_abstract_type_definition (qualified_name,
							 definition);

  /* This also removes the previous binding for parameters: */
  update_abstract_type_name_to_abstract_type_parameters (qualified_name,
							 formal_parameters);

  instantiate_all_references_to_abstract_type (qualified_name);

  //fprintf(stderr,"define_abstract_type(): done\n");
  /*
     fprintf(stderr,"HO DEFINITO %s (",maybe_qualified_name);
     output_type(stderr,formal_parameters);
     fprintf(stderr,") COME ");
     output_type(stderr,definition);
     fprintf(stderr,"\n");
   */
  if (was_type_declared)
    return 0;
  else
    {
      bind_abstract_type_name (qualified_name);
      return -3;		/* warning: type was not declared */
    }
}

/* 'exists' means 'is declared or is defined' */
int
does_abstract_type_exist (char *maybe_qualified_name)
{
  return is_abstract_type_declared (maybe_qualified_name) ||
    is_abstract_type_defined (maybe_qualified_name);
}


/* Just for uniformity in naming */
int
is_an_abstract_type_name (char *maybe_qualified_name)
{
  return does_abstract_type_exist (maybe_qualified_name);
}

int
is_an_exception_name (char *maybe_qualified_name)
{
  return is_exception_defined (qualify_exception (maybe_qualified_name));
}

/* Returns the abstract type with actual parameters if a definition is not available,
   else the instantiation or
   -1 for 'parameters don't match'
   -2 for 'undeclared abstract type' */
type_t
lookup_abstract_type (char *maybe_qualified_name, type_t actual_parameters)
{
  char *qualified_name = qualify_abstract_type_name (maybe_qualified_name);
  type_t declared_formal_parameters;

  if (!is_abstract_type_declared (qualified_name))
    return -2;			/* undeclared abstract type */

  declared_formal_parameters =
    abstract_type_name_to_abstract_type_parameters (qualified_name);

  if (is_error (unify_types (declared_formal_parameters, actual_parameters)))
    return -1;			/* parameters don't match */

  if (is_abstract_type_defined (qualified_name))
    {
      /* instantiate the existing definition: */
      return instantiate_defined_abstract_type (term (abstract_c,
						      term
						      (abstract_type_name_to_abstract_type_constructor
						       (qualified_name),
						       actual_parameters)));
    }
  else
    /* instantiate the term for the not-yet-defined abstract type: */
    return term (abstract_c,
		 term (abstract_type_name_to_abstract_type_constructor
		       (qualified_name), actual_parameters));
}

/* ************************************* CONCRETE TYPES ***************************************** */

/* Compute an instantiated concrete type from a given instantiated case:
*/
type_t
concrete_type_constructor_instance_to_concrete_type (char
						     *maybe_qualified_constructor_name,
						     type_t
						     case_actual_definition)
{
  type_t temp;
  type_t concrete_type_formal_parameters;
  type_t concrete_type_actual_parameters = new_generic_type ();
  type_t case_formal_definition;
  int concrete_type_index;
  int encoding;			/* not used */

  lookup_concrete_type_constructor_no_refresh
    (maybe_qualified_constructor_name, &concrete_type_index,
     &case_formal_definition, &encoding);

  concrete_type_formal_parameters =
    concrete_types[concrete_type_index].formal_parameters;

  temp =
    unify_types (freshen_type
		 (new_function_type
		  (concrete_type_formal_parameters, case_formal_definition)),
		 new_function_type (concrete_type_actual_parameters,
				    case_actual_definition));
  concrete_type_actual_parameters = element_type (temp, 0);

  return
    new_concrete_type (term (concrete_types[concrete_type_index].constructor,
			     concrete_type_actual_parameters));
}

/* Lookup a concrete type case

   returns:
   0 on success
   -1 for 'not found'
*/
int
lookup_concrete_type_constructor (char *maybe_qualified_constructor_name,
				  int *concrete_type_index,
				  type_t * definition, int *encoding)
{
  int r =
    lookup_concrete_type_constructor_no_refresh
    (maybe_qualified_constructor_name,
     concrete_type_index,
     definition,
     encoding);
  *definition = freshen_type (*definition);
  return r;
}

/* Lookup a concrete type case
   
   returns:
   0 on success
   -1 for 'not found'
*/
int
lookup_concrete_type_constructor_no_refresh (char
					     *maybe_qualified_constructor_name,
					     int *concrete_type_index,
					     type_t * definition,
					     int *encoding)
{
  char *qualified_constructor_name =
    qualify_concrete_type_constructor (maybe_qualified_constructor_name);
  int i;

  for (i = 0; i < next_concrete_type_constructor; i++)
    {
      //fprintf(stderr,"%2i. %s",i,concrete_types_constructors[i].name);
      if (!strcmp
	  (concrete_types_constructors[i].name, qualified_constructor_name))
	{
	  *concrete_type_index = concrete_types_constructors[i].concrete_type;
	  *definition = concrete_types_constructors[i].case_definition;
	  *encoding = concrete_types_constructors[i].encoding;
	  //fprintf(stderr," OK, TROVATO!\n");
	  return 0;
	}
      //fprintf(stderr,"\n");
    }

  return -1;
}

/* Lookup a concrete type

   returns:
   0 on success
   -1 for 'not found'
*/
int
lookup_concrete_type (char *maybe_qualified_name,
		      int *concrete_type_index, constructor_t * constructor)
{
  char *qualified_name = qualify_concrete_type_name (maybe_qualified_name);
  int i;

  for (i = 0; i < next_concrete_type; i++)
    if (!strcmp (concrete_types[i].name, qualified_name))
      {
	*concrete_type_index = i;
	*constructor = concrete_types[i].constructor;
	return 0;
      }

  return -1;
}


/* Returns nonzero iff the name represents a concrete type constructor */
int
is_a_concrete_type_constructor_name (char *maybe_qualified_constructor_name)
{
  type_t concrete_type;		/* not used */
  type_t definition;		/* not used */
  int encoding;			/* not used */

  return
    lookup_concrete_type_constructor_no_refresh
    (maybe_qualified_constructor_name, &concrete_type, &definition,
     &encoding) == 0;
}


/* Returns nonzero iff the name represents a concrete type */
int
is_a_concrete_type_name (char *maybe_qualified_name)
{
  char *qualified_name = qualify_concrete_type_name (maybe_qualified_name);
  int i;

  for (i = 0; i < next_concrete_type; i++)
    if (!strcmp (concrete_types[i].name, qualified_name))
      return 1;
  return 0;
}

/* Ugly and inefficient. We don't care very much, we will implement all of
   this better in the epsilon meta-compiler. */
int
is_concrete_type_defined (char *maybe_qualified_name)
{
  char *qualified_name = qualify_concrete_type_name (maybe_qualified_name);
  int i;

  for(i = 0; i < next_concrete_type; i++)
    if (!strcmp (concrete_types[i].name, qualified_name)){
      if(concrete_types[i].cases_no == -1)
	return 0; /* declared but not defined */
      else
	return 1; /* defined */
    }
  return 0;
}

/* Ugly and inefficient. We don't care very much, we will implement all of
   this better in the epsilon meta-compiler. */
int
is_concrete_type_declared_and_not_defined (char *maybe_qualified_name)
{
  char *qualified_name = qualify_concrete_type_name (maybe_qualified_name);
  int i;

  for(i = 0; i < next_concrete_type; i++)
    if (!strcmp (concrete_types[i].name, qualified_name)){
      if(concrete_types[i].cases_no == -1)
	return 1; /* declared but not defined */
      else
	return 0; /* defined */
    }
  return 0;
}

/* Ugly and inefficient. We don't care very much, we will implement all of
   this better in the epsilon meta-compiler. */
int
is_concrete_type_declared_or_defined (char *maybe_qualified_name)
{
  char *qualified_name = qualify_concrete_type_name (maybe_qualified_name);
  int i;

  for(i = 0; i < next_concrete_type; i++)
    if (!strcmp (concrete_types[i].name, qualified_name))
      return 1; /* declared or defined */
  return 0;
}

/* Returns the index of qualified_name in the concrete_types table, or -1 if
   the name is not found. */
int concrete_type_name_to_index(char *maybe_qualified_name){
  int i;
  char* qualified_name = qualify_concrete_type_name(maybe_qualified_name);

  for(i = 0; i < next_concrete_type; i++)
    if (!strcmp (concrete_types[i].name, qualified_name))
      return i;
  return -1;
}


/* Create a new undefined concrete type, to be qualified in _this_ module:

   returns:
   0 on success
   -1 for 'already exists'
*/
int
create_undefined_concrete_type (char *maybe_qualified_name,
				type_t formal_parameters
				/* or tuple_terminator_type */,
                                int only_declared )
{
  /*
  char *qualified_name = qualify_with_this_module (maybe_qualified_name);
  constructor_t c = constructor (qualified_name, 1);

  // Refresh type variables in formal parameters:
  formal_parameters = freshen_type (formal_parameters);

  if (is_abstract_type_defined (qualified_name))
    return -1;

  update_abstract_type_constructor_to_abstract_type_name (c, qualified_name);
  update_abstract_type_name_to_abstract_type_constructor (qualified_name, c);
  update_abstract_type_name_to_abstract_type_parameters (qualified_name,
							 formal_parameters);

  bind_abstract_type_name (qualified_name);

  return 0;
*/
  //???????????????
  char *qualified_name = qualify_with_this_module (maybe_qualified_name);
  int i;
  
  if(is_concrete_type_defined(qualified_name))
    return -1; /* declaring or defining an already defined type is an error */
  if(only_declared && is_concrete_type_declared_and_not_defined(qualified_name))
    return -1; /* declaring twice is an error */

  if(is_concrete_type_declared_and_not_defined(qualified_name)){ // updated an entry
    i = concrete_type_name_to_index(qualified_name);

    concrete_types[i].formal_parameters = formal_parameters;
    /* Don't touch the constructor: it was already created */
    concrete_types[i].first_case = next_concrete_type_constructor;
    if(only_declared){
      fprintf(stderr, "types.c: this is impossible\n\a");
      exit(EXIT_FAILURE);
    }
    concrete_types[i].cases_no = 0; /* only_declared must be false here */
    return 0;
  }
  else{ /* this is a new entry (declared or defined, but new) */
    if (next_concrete_type == concrete_types_size)
      {
	concrete_types_size *= 2;
	concrete_types =
	  (struct concrete_types_element *) realloc (concrete_types,
						     sizeof (struct
							     concrete_types_element)
						     * concrete_types_size);
      }
    
    strcpy (concrete_types[next_concrete_type].name, qualified_name);
    concrete_types[next_concrete_type].formal_parameters = formal_parameters;
    concrete_types[next_concrete_type].constructor = constructor (qualified_name, 1);
    concrete_types[next_concrete_type].first_case = next_concrete_type_constructor; // by now
    concrete_types[next_concrete_type].cases_no = (only_declared ? -1 : 0);
    next_concrete_type++;
    
    return 0;
  }
}


/* Add a case to the latest concrete type:

   returns:
   0 on success
   -1 for 'case already exists'
*/
int
define_concrete_type_case (char *concrete_type_name,
                           char *maybe_qualified_constructor_name,
			   type_t case_definition)
{
  int this_concrete_type = concrete_type_name_to_index(concrete_type_name);
  char *qualified_constructor_name = qualify_with_this_module (maybe_qualified_constructor_name);
  
  if (is_a_concrete_type_constructor_name (qualified_constructor_name))
    return -1;

  if (next_concrete_type_constructor == concrete_types_constructors_size)
    {
      concrete_types_constructors_size *= 2;
      concrete_types_constructors =
	(struct concrete_types_constructors_element *)
	realloc (concrete_types_constructors,
		 sizeof (struct concrete_types_constructors_element) *
		 concrete_types_constructors_size);
    }

  strcpy (concrete_types_constructors[next_concrete_type_constructor].name,
	  qualified_constructor_name);
  concrete_types_constructors[next_concrete_type_constructor].concrete_type =
    this_concrete_type;

  /* Find an integer coding for this case: */
  if (next_concrete_type_constructor == 0)
    concrete_types_constructors[next_concrete_type_constructor].encoding = 0;
  else if (this_concrete_type == 0)//(next_concrete_type == 1) // To do: I don't understand this anymore !!!!!!!!!!!!!!!!!!!
    concrete_types_constructors[next_concrete_type_constructor].encoding =
      concrete_types_constructors[next_concrete_type_constructor -
				  1].encoding + 1;
  else if (concrete_types_constructors[next_concrete_type_constructor - 1].
	   concrete_type != this_concrete_type)
    concrete_types_constructors[next_concrete_type_constructor].encoding = 0;
  else
    concrete_types_constructors[next_concrete_type_constructor].encoding =
      concrete_types_constructors[next_concrete_type_constructor -
				  1].encoding + 1;

  concrete_types_constructors[next_concrete_type_constructor].
    case_definition = case_definition;

  next_concrete_type_constructor++;

  concrete_types[this_concrete_type].cases_no++;

  bind_concrete_type_constructor (qualified_constructor_name);

  return 0;
}


/* Returns the concrete type instantiated with the given parameters,
   or type_error if actual parameters are not compatible.

   maybe_qualified_name must be defined: */
type_t
instantiate_concrete_type (char *maybe_qualified_type_name,
			   type_t actual_parameters)
{
  constructor_t constructor;
  int concrete_type_index;
  type_t formal_parameters;
  type_t r;

  lookup_concrete_type (maybe_qualified_type_name,
			&concrete_type_index, &constructor);
  formal_parameters = concrete_types[concrete_type_index].formal_parameters;
  r = unify (freshen_type (formal_parameters), actual_parameters);

  //fprintf(stderr,"[");
  //output_type(stderr,r);
  //fprintf(stderr,"]\n");

  if (is_error_type (r))
    return type_error;
  else
    return new_concrete_type (term (constructor, r));
}



/* Returns the concrete type instantiated with the given parameters for the given case,
   or type_error if actual parameters are not compatible.

   maybe_qualified_name must be defined: */
type_t
instantiate_concrete_type_given_a_constructor (char *maybe_qualified_type_name,	// returned
					       char
					       *maybe_qualified_constructor_name,
					       type_t actual_definition)
{
  type_t formal_definition;
  type_t concrete_type_formal_parameters;
  type_t concrete_type_actual_parameters = new_generic_type ();
  int concrete_type_index;
  int encoding;			/* not used */
  constructor_t term_constructor;

  type_t temp;

  lookup_concrete_type_constructor_no_refresh
    (maybe_qualified_constructor_name, &concrete_type_index,
     &formal_definition, &encoding);
  concrete_type_formal_parameters =
    concrete_types[concrete_type_index].formal_parameters;
  term_constructor = concrete_types[concrete_type_index].constructor;

  temp =
    unify_types (freshen_type
		 (new_function_type
		  (concrete_type_formal_parameters, formal_definition)),
		 new_function_type (concrete_type_actual_parameters,
				    actual_definition));
  concrete_type_actual_parameters = element_type (temp, 0);

  return new_concrete_type (term (term_constructor,
				  concrete_type_actual_parameters));
}

char *
concrete_type_term_constructor_to_name (constructor_t c)
{
  int i;
  for (i = 0; i < next_concrete_type; i++)
    if (concrete_types[i].constructor == c)
      return concrete_types[i].name;
  return "{THIS MUST NOT HAPPEN sdnvcaskln129}";
}

char *
concrete_type_to_name (type_t x)
{
  return
    concrete_type_term_constructor_to_name (term_to_constructor
					    (element_type (x, 0)));
}

int
concrete_type_to_cases_number (type_t x)
{				/* x _must_ be concrete */
  char *name =
    concrete_type_term_constructor_to_name (term_to_constructor
					    (element_type (x, 0)));
  constructor_t unused_constructor;
  int index;

  lookup_concrete_type (name, &index, &unused_constructor);
  return concrete_types[index].cases_no;
}

void
concrete_type_to_nth_case (type_t intantiated_type,	/* _must_ be concrete */
			   int n,	/* n must be in range */
			   char **constructor_name,
			   int *encoding, type_t * actual_parameters)
{
  int i;
  int concrete_type_index;
  int concrete_type_case_index;
  constructor_t term_constructor =
    term_to_constructor (element_type (intantiated_type, 0));
  type_t temp;

  /* This search always succeeds: */
  for (i = 0; i < next_concrete_type; i++)
    if (concrete_types[i].constructor == term_constructor)
      {
	concrete_type_index = i;
	break;
      }
  concrete_type_case_index =
    concrete_types[concrete_type_index].first_case + n - 1;
  *constructor_name =
    concrete_types_constructors[concrete_type_case_index].name;
  *encoding = concrete_types_constructors[concrete_type_case_index].encoding;

  temp =
    unify_types (new_function_type
		 (concrete_types[concrete_type_index].formal_parameters,
		  concrete_types_constructors[concrete_type_case_index].
		  case_definition),
		 new_function_type (element_type
				    (element_type (intantiated_type, 0), 0),
				    new_generic_type ()));

  *actual_parameters = element_type (temp, 1);
}

/* ******************** OUTPUT PROCEDURES MEMOIZATION ********************* */

void
create_output_procedures_stack ()
{
  output_procedures_current_size = INITIAL_OUTPUT_PROCEDURES_STACK_SIZE;
  output_procedures_stack =
    (struct output_procedures_stack_element *)
    malloc (sizeof (struct output_procedures_stack_element)
	    * output_procedures_current_size);
  output_procedures_next_position = 0;
}

void
destroy_output_procedures_stack ()
{
  free (output_procedures_stack);
}

/* The code is returned. Here types don't need to be normalized: */
int
memoize_output_procedure_for_concrete_type (type_t t)
{
  if (output_procedures_next_position == output_procedures_current_size)
    {
      int i;

      output_procedures_current_size *= 2;
      output_procedures_stack =
	(struct output_procedures_stack_element *)
	realloc (output_procedures_stack,
		 output_procedures_current_size
		 * sizeof (struct output_procedures_stack_element));
    }

  output_procedures_stack[output_procedures_next_position].index =
    output_procedures_index;
  output_procedures_stack[output_procedures_next_position++].type =
    normalize_type (t);
  return output_procedures_index++;
}

/* Here types don't need to be normalized.
   Returns 0 if the procedure does not exist, else its index (!= 0)
*/
int
concrete_type_output_procedure_to_code (type_t t)
{
  int i;

  t = normalize_type (t);
  for (i = 0; i < output_procedures_next_position; i++)
    if (are_types_equal (t, output_procedures_stack[i].type))
      return output_procedures_stack[i].index;
  return 0;
}


/* ************************************* OUTPUT  ***************************************** */

void
output_type (FILE * f, type_t t)
{
  //output_term(f,t);
   /**/ if (is_a_generic_type (t))
    {
      output_term (f, t);;
    }
  else if (is_a_function_type (t))
    {
      fprintf (f, "(");
      output_type (f, element_type (t, 0));
      fprintf (f, " -> ");
      output_type (f, element_type (t, 1));
      fprintf (f, ")");
    }
  else if (is_a_pair_type (t))
    {
      type_t x;

      fprintf (f, "(");
      for (x = t;
	   (!is_a_tuple_terminator_type (x)) && (!is_a_generic_type (x));
	   x = element_type (x, 1))
	{
	  output_type (f, element_type (x, 0));
	  if (!is_a_tuple_terminator_type (element_type (x, 1)))
	    fprintf (f, " * ");
	}
      if (is_a_generic_type (x))
	fprintf (f, "...");
      fprintf (f, ")");
    }
  else if (is_an_array_type (t))
    {
      fprintf (f, "array of ");
      output_type (f, element_type (t, 0));
    }
  else if (is_an_io_type (t))
    {
      fprintf (f, "i/o of ");
      output_type (f, element_type (t, 0));
    }
  else if (is_a_list_type (t))
    {
      fprintf (f, "list of ");
      output_type (f, element_type (t, 0));
    }
  else if (is_a_promise_type (t))
    {
      fprintf (f, "promise of ");
      output_type (f, element_type (t, 0));
    }
  else if (is_an_abstract_type (t))
    {
      //fprintf(f,"<< ");
      //output_term(f,t);
      //fprintf(f," >>\n");
      fprintf (f, "%s",
	       bare_identifier
	       (abstract_type_constructor_to_abstract_type_name
		(term_to_constructor (element_type (t, 0)))));
      if (!is_a_tuple_terminator_type (element_type (element_type (t, 0), 0)))
	{
	  fprintf (f, " of ");
	  output_type (f, element_type (element_type (t, 0), 0));
	}
    }
  else if (is_a_concrete_type (t))
    {
      char *concrete_type_name =
	concrete_type_term_constructor_to_name (term_to_constructor
						(element_type (t, 0)));
      fprintf (f, "%s", bare_identifier (concrete_type_name));

      if (!is_a_tuple_terminator_type (element_type (element_type (t, 0), 0)))
	{
	  fprintf (f, " of ");
	  output_type (f, element_type (element_type (t, 0), 0));
	}
    }
  else
    output_term (f, t);
 /**/}
