/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002, 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include "multibuffer.h"

#define INITIAL_BUFFERS_NO  64
#define INITIAL_BUFFER_SIZE 4096
#define INITIAL_STACK_SIZE  64

struct buffer_struct
{
  char *buffer;
  int buffer_size;
  int allocated_size;
}
 *buffers;

int buffers_no;
int allocated_buffers_no;

int stack_size;
int stack_allocated_size;
buffer_t *buffers_stack;

void
init_multibuffer ()
{
  int i;

  allocated_buffers_no = INITIAL_BUFFERS_NO;
  buffers_no = 0;
  buffers =
    (struct buffer_struct *) malloc (sizeof (struct buffer_struct) *
				     INITIAL_BUFFERS_NO);
  for (i = 0; i < allocated_buffers_no; i++)
    {
      buffers[i].buffer =
	(char *) malloc (sizeof (char) * INITIAL_BUFFER_SIZE);
      strcpy (buffers[i].buffer, "");
      buffers[i].buffer_size = 0;
      buffers[i].allocated_size = INITIAL_BUFFER_SIZE;
    }

  stack_allocated_size = INITIAL_STACK_SIZE;
  stack_size = 0;
  buffers_stack =
    (buffer_t *) malloc (sizeof (buffer_t) * stack_allocated_size);
}

void
free_multibuffer ()
{
  int i;

  for (i = 0; i < allocated_buffers_no; i++)
    free (buffers[i].buffer);
  free (buffers);
  free (buffers_stack);
}

buffer_t
new_buffer ()
{
  if (buffers_no == allocated_buffers_no)
    {
      int i;

      buffers =
	(struct buffer_struct *) realloc (buffers,
					  sizeof (struct buffer_struct) *
					  (allocated_buffers_no *= 2));
      for (i = allocated_buffers_no / 2; i < allocated_buffers_no; i++)
	{
	  buffers[i].buffer =
	    (char *) malloc (sizeof (char) * INITIAL_BUFFER_SIZE);
	  strcpy (buffers[i].buffer, "");
	  buffers[i].buffer_size = 0;
	  buffers[i].allocated_size = INITIAL_BUFFER_SIZE;
	}
    }

  return buffers_no++;
}

void
write_to_buffer (buffer_t b, const char *s)
{
  int s_size = strlen (s);

  if (s_size + buffers[b].buffer_size + 1 >= buffers[b].allocated_size)
    {
      buffers[b].allocated_size = (s_size + buffers[b].buffer_size + 1) * 2;
      buffers[b].buffer =
	(char *) realloc (buffers[b].buffer, buffers[b].allocated_size);
    }

  strcpy (buffers[b].buffer + buffers[b].buffer_size, s);
  buffers[b].buffer_size += s_size;
}

void
flush_multibuffer (FILE * f)
{
  int i;

  fprintf (f, "# This is an eAML -*- asm -*- file.\n\n");
  for (i = 0; i < buffers_no; i++)
    fprintf (f, "%s", buffers[i].buffer);
}

void
push_buffer (buffer_t x)
{
  if (stack_size == stack_allocated_size)
    {
      stack_allocated_size *= 2;
      buffers_stack =
	(buffer_t *) realloc (buffers_stack,
			      sizeof (buffer_t) * stack_allocated_size);
    }
  buffers_stack[stack_size++] = x;
}

buffer_t
top_buffer ()
{
  return buffers_stack[stack_size - 1];
}

buffer_t
pop_buffer ()
{
  return buffers_stack[--stack_size];
}
