/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002, 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include "epsilon.h"
#include "types.h"
#include "environment.h"
#include "multibuffer.h"
#include "../commontext.h"

#include "../common/string_map.h"
#include "../common/search_path.h"
#include "../common/command_line.h"

#include "importedmodules.h"

#define MODULES_TABLE_INITIAL_SIZE 1
struct modules_table_element
{
  char name[MODULES_PATH_LENGTH + 1];
  int children_number;
  int children[MAX_INCLUDED_MODULES];
  struct string_map *children_by_name;	// name -> unused
}
 *modules_table;
int modules_table_size;
int modules_table_pointer;

struct string_map *modules_by_name;
struct string_map *visited_modules_by_name;	// name -> unused

extern search_path_t interfaces_search_path; /* defined in compiler.c */

extern int verbose; /* defined in epsilonc.c */

void
create_modules_table ()
{
  modules_table_size = MODULES_TABLE_INITIAL_SIZE;
  modules_table_pointer = 0;
  modules_table = (struct modules_table_element *)
    malloc (modules_table_size * sizeof (struct modules_table_element));

  modules_by_name = create_string_map (sizeof (int));
  visited_modules_by_name = create_string_map (sizeof (int));
}

int
new_module (char *name)
{
  /* Is the module already present? */
  int *x;
  if ((x = access_string_map (modules_by_name, name)) != NULL)
    return *x;

  /* The module is not present yet. */
  if (modules_table_pointer == modules_table_size)
    {
      modules_table_size *= 2;
      modules_table = (struct modules_table_element *)
	realloc (modules_table,
		 modules_table_size * sizeof (struct modules_table_element));
    }
  strncpy (modules_table[modules_table_pointer].name, name,
	   MODULES_PATH_LENGTH);
  modules_table[modules_table_pointer].name[MODULES_PATH_LENGTH] = '\0';
  modules_table[modules_table_pointer].children_number = 0;
  modules_table[modules_table_pointer].children_by_name =
    create_string_map (sizeof (int));

  /* Update the associative table: */
  insert_into_string_map (modules_by_name, name, &modules_table_pointer);
  return modules_table_pointer++;
}

int non_null = 1;

void
make_child_of (int child, int parent)
{
  /* Is 'child' already child of 'parent'? */
  int *x = access_string_map (modules_table[parent].children_by_name,
			      modules_table[child].name);
  if (x != NULL)
    return;

  /* Is 'child' the same as 'parent'? */
  if (child == parent)
    return;

  /* Ok, make 'child' child of 'parent': */
  insert_into_string_map (modules_table[parent].children_by_name,
			  modules_table[child].name, &non_null);
  if (modules_table[parent].children_number > MAX_INCLUDED_MODULES)
    {
      fprintf (stderr, "FATAL: %s imports more than %i modules.\n",
	       MAX_INCLUDED_MODULES);
      exit (EXIT_FAILURE);
    }
  modules_table[parent].children[modules_table[parent].children_number++] =
    child;
}

void
destroy_modules_table ()
{
  int i;
  for (i = 0; i < modules_table_pointer; i++)
    destroy_string_map (modules_table[i].children_by_name);
  free (modules_table);
  destroy_string_map (modules_by_name);
  destroy_string_map (visited_modules_by_name);
}

struct modules_table_element *
get_module (int x)
{
  return &(modules_table[x]);
}

void
dump_modules_table ()
{
  int i;
  fprintf (stderr,
	   "------------------------------------------------------------------------------------\n");
  for (i = 0; i < modules_table_pointer; i++)
    {
      int j;
      fprintf (stderr, "%s, parent of [ ", get_module (i));
      for (j = 0; j < modules_table[i].children_number; j++)
	fprintf (stderr, "%s ",
		 modules_table[modules_table[i].children[j]].name);
      fprintf (stderr, "]\n");
    }
  fprintf (stderr,
	   "------------------------------------------------------------------------------------\n");
}

int *stack;

int
is_child_admittable (int x, int stack_pointer)
{
  int i;
  stack[stack_pointer++] = x;

  /*
     fprintf(stderr,"[ ");
     for(i=0;i<stack_pointer;i++)
     fprintf(stderr,"%s ",get_module(stack[i])->name);
     fprintf(stderr,"]\n");
   */

  /* If the path stored in the stack contains x twice, then there is a cyclic 
     inclusion: */
  for (i = 0; i < stack_pointer - 1; i++)
    if (stack[i] == x)
      {
	fprintf (stderr, "%s imports a module which imports %s.\n",
		 get_module (stack[i])->name, get_module (stack[i])->name);
	return 0;
      }

  /* If even a single path from a child is not admittable, then x is not admittable: */
  for (i = 0; i < modules_table[x].children_number; i++)
    if (!is_child_admittable (modules_table[x].children[i], stack_pointer))
      return 0;

  /* Ok, if we are here x is admittable. */
  return 1;
}

/* No node must compare twice in a root-leaf path: */
int
is_inclusion_admittable ()
{
  int r;

  stack = (int *) malloc (sizeof (int) * modules_table_pointer);	/* more than it's needed */
  r = is_child_admittable (0 /* root */ ,
			   0 /* empty stack */ );

  free (stack);
  return r;
}

int *sorted_modules;
int sorted_modules_pointer;

void
breadth_first_visit_child (int x)
{
  int i;
  int *p;
  for (i = 0; i < modules_table[x].children_number; i++)
    breadth_first_visit_child (modules_table[x].children[i]);

  //fprintf(stderr,"Q1\n");
  p = access_string_map (visited_modules_by_name, modules_table[x].name);
  //fprintf(stderr,"Q2\n");
  if (p == NULL)
    {
      sorted_modules[sorted_modules_pointer++] = x;
      /* fprintf(stderr,"%s ",modules_table[x].name); */
      //fprintf(stderr,"Q2.7\n");
      insert_into_string_map (visited_modules_by_name,
			      modules_table[x].name,
			      &i /* any value: not used */ );
    }
}

/* Each module must be loaded after the ones it includes: */
void
sort_modules_to_load (int x)
{
  breadth_first_visit_child (x);
}

extern FILE *yyin;
extern int yylineno;

extern char** epb_arguments; /* defined in epsilonc.c */

/* If we are compiling XXXX.epb, then the interface XXXX.epi is allowed
   not to exist. */
int is_interface_really_needed(char* interface){
  return strncmp(epb_arguments[0],
		 interface,
		 strlen(epb_arguments[0]) - 1);
}

/* Returns 0 on success: */
int
load_module (char *bare_file_name)
{
  char* file_name =
    search_for_file_in_search_path(bare_file_name,
				   interfaces_search_path);
  FILE *f;
  FILE *previous_yyin = yyin;
  int r;
  int l;
  int i;
  if(file_name == NULL){
    if(is_interface_really_needed(bare_file_name)){
      /* This interface was needed but was not found: */
      fprintf(stderr, "About %s:\n", bare_file_name);
      fatal("could not find interface");
    }
    else{ /* Ok, bare_file_name was not strictly needed */
      return 0; /* return with success */
    }
  }
  f = fopen(file_name, "r");
  l = strlen(file_name);

  if (f == NULL)
    {
      fprintf (stderr, "load_module(): Could not load %s\n", file_name);
      return -1;
    }
  yyin = f;
  is_this_an_interface = (file_name[l - 1] == 'i');	/* is this file an interface module? */
  yylineno = 1;
  strcpy (current_module_name, bare_file_name);

  /* Replace '/' with MODULES_SEPARATOR: */
  for (i = 0; current_module_name[i] != '\0'; i++)
    if (current_module_name[i] == '/')
      current_module_name[i] = MODULES_SEPARATOR;

  /* file_name _does_ contain a dot; I have tested this condition before. */
  for (i = strlen (current_module_name) - 1; current_module_name[i] != '.';
       i--)
    /* do nothing */ ;
  current_module_name[i] = '\0';	/* drop dot and extension */

  r = yyparse ();
  yyin = previous_yyin;

  fclose (f);

#ifdef DEBUG
  dump_global_environment ();
#endif

  return r;
}

int
exists (char *file_name)
{
  FILE *f = fopen (file_name, "r");
  if (f == NULL)
    {
      return 0;
    }
  else
    {
      fclose (f);
      return 1;
    }
}

extern FILE *modin;
extern int modlex ();
extern int current_module;	/* defined in modulesscanner.l */

char *append_extension (char *name, char *e);


/* Returns 0 on success.
   Returns -1 if first_module is an interface and it does not exits.
   Returns 1 on error.

   If first_module is an interface and does not exist then it's _not_
   an error. Any other non_existing module is an error.
   Scan every module (also indirectly) included by first_module: */
int
scan_included_modules_for_inclusions (char *first_module)
{
  // To do: rewrite this loop without unrolling of the first
  // iteration?
  FILE *f = fopen (first_module, "r");

  if (f == NULL)
    {
      if (first_module[strlen (first_module) - 1] == 'i')
	{
#ifdef VERBOSE
	  fprintf (stderr, "Could not load %s: no matter, it's optional.\n",
		   first_module);
#endif
	  return -1;		/* first_module is an interface and does */
	  /* not exist. */
	}

      fprintf (stderr, "Could not load %s.\n", first_module);
      return 1;
    }

  current_module = new_module (first_module);
  modin = f;
  modlex ();
  fclose (f);

  if(long_option_to_value("use-prelude"))
    make_child_of(new_module("prelude.epi"),current_module); ///???????????????????

  ////////////////////////////
  while (current_module < modules_table_pointer - 1)
    {
      char* path_to_file = 
	search_for_file_in_search_path(modules_table[++current_module].name,
				       interfaces_search_path);
      if(path_to_file == NULL){ /* interface was not found */
	if(! is_interface_really_needed(modules_table[current_module].name))
	  continue; /* we did not find <epb_arguments[0]>.epi; no matter */
	else{ /* we did not find a really needed module: abort */
	  fprintf(stderr, "About %s:\n",
		  modules_table[current_module].name);
	  fatal("could not find interface");
	}
      }

      f = fopen (path_to_file, "r");
      if (f == NULL)
	{
	  fprintf (stderr, "Could not load %s, imported by %s.\n",
		   path_to_file, first_module);
	  return 1;
	}
      new_module (first_module);
      modin = f;
      modlex ();
      fclose (f);
    }
  ////////////////////////////
  return 0;
}

/* Returns 0 on success: */
int
load_modules (char *first_module)
{
  int was_first_epi_found = 0, x;
  current_module = -1;

  /* Load every module (also indirectly) included by first_module: */
#ifdef VERBOSE
  fprintf (stderr, "Searching for inclusions... \n");
#endif

  if ((x =
       scan_included_modules_for_inclusions (replace_extension
					     (first_module, "epi"))) == -1)
    {				/* this can safely fail */
#ifdef VERBOSE
      fprintf (stderr, " No matter, it is optional.\n");
#endif
    }
  else if (x == 1)
    {
      /* The error message is shown by scan_included_modules_for_inclusions(): */
      return -1;
    }
  else
    {
      was_first_epi_found = 1;
#ifdef DEBUG
      fprintf (stderr, "{OK, %s scanned}\n",
	       replace_extension (first_module, "epi"));
#endif
    }
  if(verbose)
    fprintf(stderr, "Scanning for inclusion of modules...\n");

  if (scan_included_modules_for_inclusions (first_module) != 0)
    {				/* this cannot */
      fprintf (stderr, "Could not load %s or a module included by it.\n",
	       first_module);
      return -1;
    }

  if(verbose)
    fprintf(stderr, "  done.\n");

  if (was_first_epi_found)
    {
      int *main_body = access_string_map (modules_by_name, first_module),
	*main_interface =
	access_string_map (modules_by_name,
			   replace_extension (first_module, "epi"));

      make_child_of (*main_interface, *main_body);
      //fprintf(stderr,"{OK, %s is child of ",replace_extension(first_module,"epi"));fprintf(stderr,"%s}\n",first_module);
    }

  /* Test for cycles in inclusion: */
#ifdef VERBOSE
  fprintf (stderr, "Checking depencencies... ");
#endif
  if (is_inclusion_admittable ())
    {
      int i;

#ifdef VERBOSE
      fprintf (stderr, "ok\n");
#endif
      sorted_modules = (int *) malloc (sizeof (int) * modules_table_pointer);
      sorted_modules_pointer = 0;

#ifdef VERBOSE
      fprintf (stderr, "Sorting interfaces using the precedence graph... ");
#endif
      sort_modules_to_load (*(int *)
			    access_string_map (modules_by_name,
					       first_module));
#ifdef VERBOSE
      fprintf (stderr, "ok.\n");
#endif

#ifdef DEBUG
      dump_modules_table ();
#endif

      //fprintf(stderr,"-- %i --\n",sorted_modules_pointer);//////////////
      for (i = 0; i < sorted_modules_pointer; i++)
	{
	  //fprintf(stderr,"sort: %i |--> %i\n",i,sorted_modules[i]);//////////////
#ifdef VERBOSE
	  fprintf (stderr, "Loading %s... ",
		   modules_table[sorted_modules[i]].name);
#endif
	  if (load_module (modules_table[sorted_modules[i]].name) == 0)
	    {
#ifdef VERBOSE
	      fprintf (stderr, "ok.\n");
#endif
	    }
	  else
	    {
#ifdef VERBOSE
	      fprintf (stderr, "failed.\n");
#endif
	      free (sorted_modules);
	      return -1;
	    }
	}
      //fprintf(stderr,"- X2\n");//////////////
      free (sorted_modules);
      return 0;
    }
  else
    return -1;
}

int
is_a_body_name (const char *f)
{
  return has_extension (f, "epb");
}

int
is_an_interface_name (const char *f)
{
  return has_extension (f, "epi");
}

char *
input_file_name_to_output_name (char *f)
{
  if (strlen (f) >= MAX_FILENAME_LENGTH - 5)
    {
      fprintf (stderr, "FATAL: file name is too long.\n");
      exit (EXIT_FAILURE);
    }

  return replace_extension (f, "eaml");
}
