/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002, 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

%{
#include <stdio.h>
#include "epsilon.h"
#include "types.h"
#include "environment.h"
#include "multibuffer.h"
#include "epsilonparser.h"

/*IDENTIFIER {LETTER}({LETTER}|{DIGIT}|"_")* */
/*
IDENTIFIER [^\ \n\r()\\\:\.]+
*/

char current_module_name[MAX_FILENAME_LENGTH+1];

int are_we_in_the_repl=0; /* In this case the environment is not initialized */

int yyerror(char* message);

/* This function modifies its argument: */
void remove_thousands_separators(char* number, int length /* NOT including the '\0' */);
%}

%option yylineno


DIGIT   [0-9]
LOWER   [a-z]
UPPER   [A-Z]
LETTER  {LOWER}|{UPPER}
UNDERSCORE \_

COMMA     \,
SEMICOLON \;

SIGN       (\-|\+)
THOUSANDS_SEPARATOR \'

TRIPLET             {DIGIT}{DIGIT}{DIGIT}
TRIPLET_OR_LESS     {DIGIT}|{DIGIT}{DIGIT}|{TRIPLET}

INTEGER    {SIGN}?{DIGIT}+|{SIGN}?{TRIPLET_OR_LESS}({THOUSANDS_SEPARATOR}{TRIPLET})*

MODULE_IDENTIFIER {LETTER}({LETTER}|{DIGIT}|"_"|\-)*
MODULE_SEPARATOR  ":"

MODULE {MODULE_IDENTIFIER}({MODULE_SEPARATOR}{MODULE_IDENTIFIER})*

/* This definition must be kept synchronized with 'IDENTIFIER'
   in modulesscanner.l: */
/*BARE_IDENTIFIER [^0-9\ \r\n()\\\:\.]([^\ \r\n()\\\:\.]*)*/
BARE_IDENTIFIER ({LETTER}|{DIGIT}|{UNDERSCORE}|\<|\>|\`|\~|\!|\@|\#|\$|\%|\^|\&|\*|\_|\+|\-|\=|\\|\{|\}|\?|\/|\[|\])+

IDENTIFIER ({MODULE}{MODULE_SEPARATOR})?{BARE_IDENTIFIER}

GENERIC    T{DIGIT}+

CESCAPE    \\.|\\{DIGIT}|\\{DIGIT}{DIGIT}|\\{DIGIT}{DIGIT}{DIGIT}
CSTRING    \"([^\n\\"]|{CESCAPE})*\"
STRING     {CSTRING}
CHARACTER  \'([^\n\r\t\a]|\\a|\\n|\\t|\\')\'
POINTED    {INTEGER}"."{DIGIT}*
FLOAT      {SIGN}?{POINTED}|{POINTED}e{INTEGER}+

WHITESPACE       (" "|\t|\n|\r|\v)+
CCOMMENT         "/*"([^/]|"/"[^*])*"*/"
CPLUSPLUSCOMMENT "//"[^\n\r]*(\n|\r)
/* An hack to recognize a C++ comment at the EOF without newline: */
CPLUSPLUSCOMMENT_AT_EOF "//"[^\n\r]*
COMMENT          {CCOMMENT}|{CPLUSPLUSCOMMENT}
IMPORTATION          import{WHITESPACE}{MODULE}{WHITESPACE}?({COMMA}{WHITESPACE}?{MODULE}{WHITESPACE}?)*{SEMICOLON}
UNIMPORTATION        unimport{WHITESPACE}{MODULE}{WHITESPACE}?({COMMA}{WHITESPACE}?{MODULE}{WHITESPACE}?)*{SEMICOLON}

%%
{WHITESPACE}  {/* Do nothing */}
{COMMENT}     {/*fprintf(stderr,"[[[COMMENT<%s>]]]\n",yytext);*/}
{CPLUSPLUSCOMMENT_AT_EOF} {if(! feof(yyin))
                             yyerror("scan error (malformed C++-style comment): this should not happen");
                             /* else we are at EOF, so do nothing */
                          }
{IMPORTATION} { /* Do nothing if we are not in the REPL */
                if(are_we_in_the_repl)
		  return IMPORTATION;
              }
{UNIMPORTATION} { /* Do nothing if we are not in the REPL */
                  if(are_we_in_the_repl)
		    return UNIMPORTATION;
                }
"()"          { return VOID_CONSTANT; }
{INTEGER}     {
                remove_thousands_separators(yytext,yyleng);
                return INTEGER_CONSTANT;
              }
{FLOAT}       {
                remove_thousands_separators(yytext,yyleng);
                return FLOAT_CONSTANT;
              }
{STRING}      {return STRING_CONSTANT;}
{CHARACTER}   {return CHARACTER_CONSTANT;}

"+"           {return PLUS;}
"-"           {return MINUS;}
"*"           {return TIMES;}
"**"          {return POWER;}
"/"           {return DIVIDED;}
"mod"         {return MODULO;}

"+i"          {return PLUS;}
"-i"          {return MINUS;}
"*i"          {return TIMES;}
"**i"         {return POWER;}
"/i"          {return DIVIDED;}
"modi"        {return MODULO;}

"+f"          {return PLUS_FLOAT;}
"-f"          {return MINUS_FLOAT;}
"*f"          {return TIMES_FLOAT;}
"**f"         {return POWER_FLOAT;}
"/f"          {return DIVIDED_FLOAT;}

"("           {return OPEN_PAR;}
")"           {return CLOSE_PAR;}

"<"           {return LESS;}
">"           {return GREATER;}
"="           {return EQUAL;}
"<="          {return LESS_OR_EQUAL;}
">="          {return GREATER_OR_EQUAL;}
"=/="         {return DIFFERENT;}

"<i"          {return LESS;}
">i"          {return GREATER;}
"=i"          {return EQUAL;}
"<=i"         {return LESS_OR_EQUAL;}
">=i"         {return GREATER_OR_EQUAL;}
"=/=i"        {return DIFFERENT;}

"<f"          {return LESS_FLOAT;}
">f"          {return GREATER_FLOAT;}
"=f"          {return EQUAL_FLOAT;}
"<=f"         {return LESS_OR_EQUAL_FLOAT;}
">=f"         {return GREATER_OR_EQUAL_FLOAT;}
"=/=f"        {return DIFFERENT_FLOAT;}

"<s"          {return LESS_STRING;}
">s"          {return GREATER_STRING;}
"=s"          {return EQUAL_STRING;}
"<=s"         {return LESS_OR_EQUAL_STRING;}
">=s"         {return GREATER_OR_EQUAL_STRING;}
"=/=s"        {return DIFFERENT_STRING;}

"<c"          {return LESS_CHARACTER;}
">c"          {return GREATER_CHARACTER;}
"=c"          {return EQUAL_CHARACTER;}
"<=c"         {return LESS_OR_EQUAL_CHARACTER;}
">=c"         {return GREATER_OR_EQUAL_CHARACTER;}
"=/=c"        {return DIFFERENT_CHARACTER;}

"=b"          {return EQUAL_BOOLEAN;}
"=/=b"        {return DIFFERENT_BOOLEAN;}

";"           {return SEMICOLON;}

"assert"      {return ASSERT;}
"in"          {return IN;}
"debug"       {return DEBUG;}

abstract      {return ABSTRACT;}
synonym       {return ABSTRACT; /* dirty trick */ }
concrete      {return CONCRETE;}
type          {return TYPE;}

exception     {return EXCEPTION;}
try           {return TRY;}
catch         {return CATCH;}
into          {return INTO;}
throw         {return THROW;}

impossible    {return IMPOSSIBLE;}

true          {return TRUE;}
false         {return FALSE;}
and           {return AND;}
or            {return OR;}
and_strict    {return AND_STRICT;}
or_strict     {return OR_STRICT;}
xor           {return XOR;}
not           {return NOT;}

if            {return IF;}
then          {return THEN;}
else          {return ELSE;}

match         {return MATCH;}
with          {return WITH;}
"_"           {return UNDERSCORE;}


"assign"      {return ASSIGN;}
":="          {return TAKES;}
skip          {return SKIP;}
begin         {return BEGIN_IO;}
end           {return END_IO;}
make_io       {return MAKE_IO;}
throw_io      {return THROW_IO;}
try_io        {return TRY_IO;}
catch_io      {return CATCH_IO;}
impossible_io {return IMPOSSIBLE_IO;}

define        {return DEFINE;}
undefine      {
               if(are_we_in_the_repl)
                 return UNDEFINE;
	       else
		 yyerror("'undefine' can only used in interactive mode");
              }
declare       {return DECLARE;}

"@@"          {return CONCATENATE_ARRAY;}
"@@s"         {return CONCATENATE_STRING;}

"::"          {return CONS;}
head          {return HEAD;}
tail          {return TAIL;}

\\            {return LAMBDA;}
\.            {return DOT;}

":"           {return COLON;}

let           {return LET;}
letrec        {return LETREC;}
be            {return BE;}
in            {return IN;}

fix           {return FIX;}

from_array    {return FROM_ARRAY; /* Luca Saiu, 2004 */ }
to_array      {return TO_ARRAY;   /* Luca Saiu, 2004 */ }
from          {return FROM;}
to            {return TO;}
"@s"          {return AT_STRING;}
"@"           {return AT_ARRAY;}
length        {return LENGTH_STRING;}
length_array  {return LENGTH_ARRAY;}

character_to_integer {return CHARACTER_TO_INTEGER;}
integer_to_character {return INTEGER_TO_CHARACTER;}
character_to_string  {return CHARACTER_TO_STRING;}
array_to_list        {return ARRAY_TO_LIST;}
list_to_array        {return LIST_TO_ARRAY;}

succ          {return SUCC;}
pred          {return PRED;}

void          {return VOID;}
integer       {return INTEGER;}
float         {return FLOAT;}
string        {return STRING;}
character     {return CHARACTER;}
boolean       {return BOOLEAN;}
c_type        {return C_TYPE;}

of            {return OF;}
"i/o"         {return IO;}
list          {return LIST;}
array         {return ARRAY;}
promise       {return PROMISE;}

empty         {return EMPTY;}

force         {return FORCE;}
delay         {return DELAY;}

integer_to_float {return INTEGER_TO_FLOAT;}
float_to_integer {return FLOAT_TO_INTEGER;}

"->"          {return ARROW;}

","           {return COMMA;}

"["           {return OPEN_BRACKET;}
"]"           {return CLOSE_BRACKET;}
"[]"          {return EMPTY_LIST;}

"<|"          {return OPEN_ARRAY;}
"|>"          {return CLOSE_ARRAY;}

"|"           {return PIPE;}
"&"           {return AMPERSAND;}
"^"           {return SELECTOR;}

"postfix"     {return POSTFIX;}
"infix"       {return INFIX;}
"left"        {return LEFT;}
"right"       {return RIGHT;}

"c_object"    {return C_OBJECT;}
"c_function"  {return C_FUNCTION;}
"c_action"    {return C_ACTION;}
"c_function_to_action" {return C_FUNCTION_TO_ACTION;}

{GENERIC}     {return GENERIC;}

{IDENTIFIER}  {
                type_t type;
                int unused;
                int precedence;
                
                if(are_we_in_the_repl)
                  return IDENTIFIER;

		if(is_a_concrete_type_name(yytext)){
                  //fprintf(stderr,"%s E` UN TIPO CONCRETO\n",yytext);
		  return CONCRETE_TYPE_NAME;
		}

		if(is_an_abstract_type_name(yytext)){
		  //fprintf(stderr,"%s E` UN TIPO ASTRATTO\n",yytext);
		  return ABSTRACT_TYPE_NAME;
		}

                if(is_a_concrete_type_constructor_name(yytext)){
		  //fprintf(stderr,"%s E` UN COSTRUTTORE DI TIPO CONCRETO\n",yytext);
		  return CONSTRUCTOR;
		}

		if(is_an_exception_name(yytext)){
                  //fprintf(stderr,"%s E` UN'ECCEZIONE\n",yytext);
                  return EXCEPTION_NAME;
		}

                if(lookup_with_precedence(yytext, &unused, &unused, &type, &precedence) == 0){
		  if(is_an_infix_operator_left_type(type)){
		    switch(precedence){
		      case 1:  return INFIX_OPERATOR_LEFT_1;
		      case 2:  return INFIX_OPERATOR_LEFT_2;
		      case 3:  return INFIX_OPERATOR_LEFT_3;
		      case 4:  return INFIX_OPERATOR_LEFT_4;
		      case 5:  return INFIX_OPERATOR_LEFT_5;
		      case 6:  return INFIX_OPERATOR_LEFT_6;
		      case 7:  return INFIX_OPERATOR_LEFT_7;
		      case 8:  return INFIX_OPERATOR_LEFT_8;
		      case 9:  return INFIX_OPERATOR_LEFT_9;
		      case 10: return INFIX_OPERATOR_LEFT_10;
		      case 11: return INFIX_OPERATOR_LEFT_11;
		      case 12: return INFIX_OPERATOR_LEFT_12;
		      case 13: return INFIX_OPERATOR_LEFT_13;
		      case 14: return INFIX_OPERATOR_LEFT_14;
		      case 15: return INFIX_OPERATOR_LEFT_15;
		      case 16: return INFIX_OPERATOR_LEFT_16;
		      case 17: return INFIX_OPERATOR_LEFT_17;
		      case 18: return INFIX_OPERATOR_LEFT_18;
  		      case 19: return INFIX_OPERATOR_LEFT_19;
		      case 20: return INFIX_OPERATOR_LEFT_20;
		      default:{
  		        fprintf(stderr,"(precedence is %i)\n",precedence);
		        yyerror("{IDENTIFIER} 1 in epsilon.l: this cannot happen");
		      }
		    } /* switch */
		  } /* if */
                  else if(is_an_infix_operator_right_type(type)){
                    switch(precedence){
                      case 1:  return INFIX_OPERATOR_RIGHT_1;
                      case 2:  return INFIX_OPERATOR_RIGHT_2;
                      case 3:  return INFIX_OPERATOR_RIGHT_3;
                      case 4:  return INFIX_OPERATOR_RIGHT_4;
                      case 5:  return INFIX_OPERATOR_RIGHT_5;
                      case 6:  return INFIX_OPERATOR_RIGHT_6;
                      case 7:  return INFIX_OPERATOR_RIGHT_7;
                      case 8:  return INFIX_OPERATOR_RIGHT_8;
                      case 9:  return INFIX_OPERATOR_RIGHT_9;
                      case 10: return INFIX_OPERATOR_RIGHT_10;
                      case 11: return INFIX_OPERATOR_RIGHT_11;
                      case 12: return INFIX_OPERATOR_RIGHT_12;
                      case 13: return INFIX_OPERATOR_RIGHT_13;
                      case 14: return INFIX_OPERATOR_RIGHT_14;
                      case 15: return INFIX_OPERATOR_RIGHT_15;
                      case 16: return INFIX_OPERATOR_RIGHT_16;
                      case 17: return INFIX_OPERATOR_RIGHT_17;
                      case 18: return INFIX_OPERATOR_RIGHT_18;
                      case 19: return INFIX_OPERATOR_RIGHT_19;
                      case 20: return INFIX_OPERATOR_RIGHT_20;
                      default:{
                        fprintf(stderr,"(precedence is %i)\n",precedence);
                        yyerror("{IDENTIFIER} 1 in epsilon.l: this cannot happen");
                      }
                    } /* switch */
                  }
                  else if(is_a_postfix_operator_type(type)){
                    switch(precedence){
                      case 1:  return POSTFIX_OPERATOR_1;
                      case 2:  return POSTFIX_OPERATOR_2;
                      case 3:  return POSTFIX_OPERATOR_3;
                      case 4:  return POSTFIX_OPERATOR_4;
                      case 5:  return POSTFIX_OPERATOR_5;
                      case 6:  return POSTFIX_OPERATOR_6;
                      case 7:  return POSTFIX_OPERATOR_7;
                      case 8:  return POSTFIX_OPERATOR_8;
                      case 9:  return POSTFIX_OPERATOR_9;
                      case 10: return POSTFIX_OPERATOR_10;
                      case 11: return POSTFIX_OPERATOR_11;
                      case 12: return POSTFIX_OPERATOR_12;
                      case 13: return POSTFIX_OPERATOR_13;
                      case 14: return POSTFIX_OPERATOR_14;
                      case 15: return POSTFIX_OPERATOR_15;
                      case 16: return POSTFIX_OPERATOR_16;
                      case 17: return POSTFIX_OPERATOR_17;
                      case 18: return POSTFIX_OPERATOR_18;
                      case 19: return POSTFIX_OPERATOR_19;
                      case 20: return POSTFIX_OPERATOR_20;
		      default:{
                        fprintf(stderr,"(precedence is %i)\n",precedence);
                        yyerror("{IDENTIFIER} 3 in epsilon.l: this cannot happen");
  		      }
		    } /* switch */
                  } /* if */
                  else
                    return IDENTIFIER;
		}
                else /* this identifier is not bound */
                  return IDENTIFIER;
              }

%%

char* static_identifier=NULL;
char* static_definition=NULL;
int static_definition_length=0;

int is_a_define_line(char* text,char** identifier, char** definition){
  int token;
  YY_BUFFER_STATE bufferState=yy_scan_string(text);

  static_definition_length=0;

  are_we_in_the_repl=1; /* don't use environment in yylex() */


  /* Skip the DEFINE token */
  if(yylex()!=DEFINE){
    *identifier="";
    *definition="";
    return 0;
  }

  /* Get the IDENTIFIER token: */
  if(yylex()!=IDENTIFIER){
    *identifier="";
    *definition="";
    return 1;
  }
  static_identifier=(char*)realloc(static_identifier,sizeof(char)*(yyleng+1));
  strcpy(static_identifier,yytext);
  *identifier=static_identifier;

  /* Get the EQUAL token: */
  if(yylex()!=EQUAL){
    *identifier="";
    *definition="";
    return 1;
  }

  /* Add the following tokens to the definition, until the SEMICOLON: */
  static_definition_length=0;
  do{
    token=yylex();
    static_definition=(char*)realloc(static_definition,
                                     sizeof(char)*(static_definition_length+yyleng+2));
    strcpy(static_definition+static_definition_length,yytext);
    static_definition[static_definition_length+yyleng]=' '; /* separate tokens */
    static_definition[static_definition_length+yyleng+1]='\0';
    static_definition_length+=yyleng+1;
  }while(token != SEMICOLON);
  *definition=static_definition;

  yy_delete_buffer(bufferState);
  return 1;
}

int is_an_undefine_line(char* text,char** identifier){
  int token;
  YY_BUFFER_STATE bufferState=yy_scan_string(text);

  static_definition_length=0;
  are_we_in_the_repl=1; /* don't use environment in yylex() */

  /* Skip the UNDEFINE token */
  if(yylex()!=UNDEFINE){
    *identifier="";
    return 0;
  }

  /* Get the IDENTIFIER token: */
  if(yylex()!=IDENTIFIER){
    *identifier="";
    return 1;
  }
  static_identifier=(char*)realloc(static_identifier,sizeof(char)*(yyleng+1));
  strcpy(static_identifier,yytext);
  *identifier=static_identifier;
  
  /* Get the IDENTIFIER token: */
  if(yylex()!=SEMICOLON){
    *identifier="";
    return 1;
  }
  
  yy_delete_buffer(bufferState);
  return 1;
}

void check_entered_line(char* text,
			int*  is_last_token_a_semicolon,
			int*  is_there_a_semicolon_before_last_token){
  int previous_token,current_token=-1;
  int semicolons_no=0;
  int is_this_the_first_token=1;
  YY_BUFFER_STATE bufferState=yy_scan_string(text);

  are_we_in_the_repl=1; /* don't use environment in yylex() */

  do{
    previous_token=current_token;
    if((current_token=yylex())==SEMICOLON)
      semicolons_no++;
/*    
    if(is_this_the_first_token){
      // To do: check for REPL commands.
      if(current_token==DEFINE)
	*is_a_define_line=1;
      else if(current_token==UNDEFINE)
        *is_an_undefine_line=1;
      else if(current_token==IMPORTATION){
	*is_an_import_line=1;
	semicolons_no++;
      }
      else if(current_token==UNIMPORTATION){
        *is_an_unimport_line=1;
	semicolons_no++;
      }
    }
*/
    is_this_the_first_token=0;
  }while(current_token!=0);
  yy_delete_buffer(bufferState);
  
  (*is_last_token_a_semicolon)=((previous_token==SEMICOLON)
                                ||(previous_token==IMPORTATION)
                                ||(previous_token==UNIMPORTATION));
  (*is_there_a_semicolon_before_last_token)=(semicolons_no>1);

  are_we_in_the_repl=0; /* next time use environment again in yylex() */
}

#define THOUSANDS_SEPARATOR '\''

/* This is O(n): */
void remove_thousands_separators(char* string,int length /* NOT including the '\0' */){
  int i,j,shift_amount=1;

  for(i=0;string[i]!='\0';i++)
    if(string[i]==THOUSANDS_SEPARATOR){
      /* Overwrite the separator found at i left-shifting a substring: */
      for(j=i+1;(string[j]!=THOUSANDS_SEPARATOR);j++){
	string[j-shift_amount]=string[j];
	if(string[j]=='\0')
	  break;
      }
      shift_amount++;
      i=j-1;
    }
  //fprintf(stderr,"[%s]\n",string);
}

int yywrap(){
  return 1;
}
