/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002, 2003, 2004 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

%{
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include "epsilon.h"
#include "types.h"
#include "environment.h"
#include "multibuffer.h"
#include "importedmodules.h"

#include "../commontext.h"

extern int yylineno;
extern char* yytext;
extern int yyleng;
extern FILE *yyin, *yyout;

extern int unescaped_strings; /* defined in epsilonc.c */
extern int show_types; /* defined in epsilonc.c */
int is_this_a_synonym_type;
int is_this_module_main = 0;

int is_this_an_interface; /* declared in importedmodules.h */

int current_if_label=1;
int current_assert_label=1;
int current_non_strict_and_label=1;
int current_non_strict_or_label=1;
int current_function_label=1;
int current_promise_label=1;
int current_io_label=1;

int current_let_label=1;

char current_concrete_type_name[IDENTIFIER_LENGTH+1];

void output(const char* s);
void output_integer(int x);
void output_output(type_t t);

int precedence;

char current_module_name[MAX_FILENAME_LENGTH+1];

char last_bound_identifier[IDENTIFIER_LENGTH+1];
int are_we_in_a_recursive_definition = 0;

type_t force_type_to_be(type_t t1, type_t t2);
type_t force_types_to_be_equal(type_t t1, type_t t2);

int yytype_mismatch_error(type_t actual, type_t attended);

void yywarning(char* message);

void output_action_beginning();
void output_action_end();
void output_current_point();
void make_current_point_failure();
%}

%union{
  struct{ 
          int label; 
        } labeled_form;
  struct{
          char identifier[IDENTIFIER_LENGTH+1];
          int label;
        } binding_form;
  struct{
          char identifier[IDENTIFIER_LENGTH+1];
          int label;
          int order; /* IDENTIFIER, INFIX_OPERATOR_LEFT,
                        INFIX_OPERATOR_RIGHT, or POSTFIX_OPERATOR */
          int precedence;
        } binding_form_pos;
  struct{
          char identifier[IDENTIFIER_LENGTH+1];
          int label;
          type_t type;
        } binding_form_with_type;
  struct{
          char identifier[IDENTIFIER_LENGTH+1];
          int label;
          int order; /* IDENTIFIER, INFIX_OPERATOR_LEFT,
                        INFIX_OPERATOR_RIGHT, or POSTFIX_OPERATOR */
          int precedence;
          type_t type;
        } binding_form_pos_with_type;
  struct{ 
          buffer_t buffer; 
          environment_id environment;
          type_t type;
        } typed_expression;
  struct{
          type_t type;
          int size;
        } expression_with_type_and_size;
  struct{
          type_t pattern_type;
          type_t expression_type;
        } double_typed_expression;
  struct{
          int bindings_no;
        } multibinding_form;
  struct{
          char identifier[IDENTIFIER_LENGTH+1];
        } just_an_identifier;
  type_t type_expression;
}

%left LOWEST_PRECEDENCE

%nonassoc C_OBJECT C_FUNCTION C_ACTION C_FUNCTION_TO_ACTION

%nonassoc ABSTRACT CONCRETE TYPE EXCEPTION

%token ASSERT IN DEBUG

%nonassoc THROW INTO THROW_IO
%nonassoc IMPOSSIBLE IMPOSSIBLE_IO

%left TRY CATCH TRY_IO CATCH_IO

%left BEGIN_IO END_IO
%left MAKE_IO
%nonassoc ASSIGN TAKES
%nonassoc SKIP

%left ELSE
%left <labeled_form> IF  ASSERT
%token THEN

%token MATCH WITH PIPE

%nonassoc UNDERSCORE

%token AMPERSAND

%left <binding_form> POSTFIX_OPERATOR_1
%left <binding_form> INFIX_OPERATOR_LEFT_1
%right <binding_form> INFIX_OPERATOR_RIGHT_1

%nonassoc <binding_form> FIX
%right <binding_form> LET LETREC

%left <binding_form> POSTFIX_OPERATOR_2
%left <binding_form> INFIX_OPERATOR_LEFT_2
%right <binding_form> INFIX_OPERATOR_RIGHT_2

%right BE
%right IN
%right COLON

%left <binding_form> POSTFIX_OPERATOR_3
%left <binding_form> INFIX_OPERATOR_LEFT_3
%right <binding_form> INFIX_OPERATOR_RIGHT_3

%right <labeled_form> LAMBDA
%right DOT 
%left EQUAL DIFFERENT LESS GREATER LESS_OR_EQUAL GREATER_OR_EQUAL EQUAL_FLOAT DIFFERENT_FLOAT LESS_FLOAT GREATER_FLOAT 
      LESS_OR_EQUAL_FLOAT GREATER_OR_EQUAL_FLOAT EQUAL_STRING DIFFERENT_STRING LESS_STRING GREATER_STRING LESS_OR_EQUAL_STRING 
      GREATER_OR_EQUAL_STRING EQUAL_BOOLEAN DIFFERENT_BOOLEAN EQUAL_CHARACTER DIFFERENT_CHARACTER LESS_CHARACTER 
      GREATER_CHARACTER LESS_OR_EQUAL_CHARACTER GREATER_OR_EQUAL_CHARACTER

/* %left APPLICATION */

%left <binding_form> POSTFIX_OPERATOR_4
%left <binding_form> INFIX_OPERATOR_LEFT_4
%right <binding_form> INFIX_OPERATOR_RIGHT_4

%nonassoc VOID_CONSTANT  INTEGER_CONSTANT  STRING_CONSTANT  CHARACTER_CONSTANT  FLOAT_CONSTANT

%left <binding_form> POSTFIX_OPERATOR_5
%left <binding_form> INFIX_OPERATOR_LEFT_5
%right <binding_form> INFIX_OPERATOR_RIGHT_5
/*
%left ELSE
%left <labeled_form> IF
%token THEN
*/

%right ARROW

%left UNARY_MINUS UNARY_PLUS /* only used for precedence */

%left <binding_form> POSTFIX_OPERATOR_6
%left <binding_form> INFIX_OPERATOR_LEFT_6
%right <binding_form> INFIX_OPERATOR_RIGHT_6

/* %left APPLICATION */

%left COMMA /* 'x' and ' ' must associate the same way */
%left PLUS MINUS PLUS_FLOAT MINUS_FLOAT
%left TIMES DIVIDED MODULO TIMES_FLOAT DIVIDED_FLOAT MODULO_FLOAT /* 'x' and ' ' must associate the same way */
/* %left COMMA*/ /* 'x' and ' ' must associate the same way */

%left <binding_form> POSTFIX_OPERATOR_7
%left <binding_form> INFIX_OPERATOR_LEFT_7
%right <binding_form> INFIX_OPERATOR_RIGHT_7

%right CONS

%left <binding_form> POSTFIX_OPERATOR_8
%left <binding_form> INFIX_OPERATOR_LEFT_8
%right <binding_form> INFIX_OPERATOR_RIGHT_8

%left OPEN_PAR CLOSE_PAR
%left OPEN_BRACKET CLOSE_BRACKET OPEN_ARRAY CLOSE_ARRAY EMPTY_LIST

%left <binding_form> POSTFIX_OPERATOR_9
%left <binding_form> INFIX_OPERATOR_LEFT_9
%right <binding_form> INFIX_OPERATOR_RIGHT_9

%left <binding_form> IDENTIFIER  INFIX_OPERATOR_LEFT  INFIX_OPERATOR_RIGHT 
                     POSTFIX_OPERATOR  CONSTRUCTOR  ABSTRACT_TYPE_NAME 
                     CONCRETE_TYPE_NAME  EXCEPTION_NAME
%nonassoc GENERIC
%left TRUE FALSE

%left APPLICATION

%left <binding_form> POSTFIX_OPERATOR_10
%left <binding_form> INFIX_OPERATOR_LEFT_10
%right <binding_form> INFIX_OPERATOR_RIGHT_10

%left <labeled_form> OR
%left OR_STRICT  XOR
%left <labeled_form> AND
%left AND_STRICT

%token DEFINE UNDEFINE IMPORTATION UNIMPORTATION DECLARE

%left <binding_form> POSTFIX_OPERATOR_11
%left <binding_form> INFIX_OPERATOR_LEFT_11
%right <binding_form> INFIX_OPERATOR_RIGHT_11


%right SEMICOLON

%left <binding_form> POSTFIX_OPERATOR_12
%left <binding_form> INFIX_OPERATOR_LEFT_12
%right <binding_form> INFIX_OPERATOR_RIGHT_12

%left CONCATENATE_STRING CONCATENATE_ARRAY
%token VOID INTEGER FLOAT STRING CHARACTER C_TYPE BOOLEAN IO ARRAY LIST PROMISE

%left <binding_form> POSTFIX_OPERATOR_13
%left <binding_form> INFIX_OPERATOR_LEFT_13
%right <binding_form> INFIX_OPERATOR_RIGHT_13

/*%right ARROW*/
%right OF

%left <binding_form> POSTFIX_OPERATOR_14
%left <binding_form> INFIX_OPERATOR_LEFT_14
%right <binding_form> INFIX_OPERATOR_RIGHT_14

%left FROM TO FROM_ARRAY TO_ARRAY AT_STRING LENGTH_STRING
%left AT_ARRAY LENGTH_ARRAY

%left CHARACTER_TO_INTEGER INTEGER_TO_CHARACTER CHARACTER_TO_STRING
      ARRAY_TO_LIST LIST_TO_ARRAY

%left <binding_form> POSTFIX_OPERATOR_15
%left <binding_form> INFIX_OPERATOR_LEFT_15
%right <binding_form> INFIX_OPERATOR_RIGHT_15

%nonassoc EMPTY
%left HEAD TAIL

%left <binding_form> POSTFIX_OPERATOR_16
%left <binding_form> INFIX_OPERATOR_LEFT_16
%right <binding_form> INFIX_OPERATOR_RIGHT_16

%nonassoc INTEGER_TO_FLOAT FLOAT_TO_INTEGER
%left POWER POWER_FLOAT
%left FORCE DELAY

%left <binding_form> POSTFIX_OPERATOR_17
%left <binding_form> INFIX_OPERATOR_LEFT_17
%right <binding_form> INFIX_OPERATOR_RIGHT_17

%left NOT
/*%left UNARY_MINUS UNARY_PLUS /* only used for precedence */

%left <binding_form> POSTFIX_OPERATOR_18
%left <binding_form> INFIX_OPERATOR_LEFT_18
%right <binding_form> INFIX_OPERATOR_RIGHT_18

%left PRED SUCC/* APPLICATION            /* APPLICATION is only used for precedence */

%left <binding_form> POSTFIX_OPERATOR_19
%left <binding_form> INFIX_OPERATOR_LEFT_19
%right <binding_form> INFIX_OPERATOR_RIGHT_19

%right SELECTOR
%nonassoc POSTFIX
%nonassoc INFIX  LEFT  RIGHT

%left <binding_form> POSTFIX_OPERATOR_20
%left <binding_form> INFIX_OPERATOR_LEFT_20
%right <binding_form> INFIX_OPERATOR_RIGHT_20

%nonassoc SCAN_ERROR


%nonassoc HIGHEST_PRECEDENCE

%start module

%type <typed_expression> module
%type <typed_expression> expression
%type <typed_expression> optional_expression
%type <typed_expression> exception_case
%type <typed_expression> exception_cases
%type <typed_expression> optional_more_exception_cases
%type <typed_expression> sequence_of_actions_with_end
%type <typed_expression> sequence_of_actions_with_catch_io

%type <typed_expression> pattern

%type <typed_expression> optional_exception_parameter

%type <typed_expression> impossible

%type <expression_with_type_and_size> sequence_of_elements
%type <expression_with_type_and_size> nonempty_sequence_of_elements

%type <double_typed_expression> pattern_arrow_expression
%type <double_typed_expression> patterns_arrow_expressions
%type <double_typed_expression> more_patterns_arrow_expressions

%type <type_expression> type
%type <type_expression> generic
%type <type_expression> optional_type
%type <type_expression> optional_equal_type_for_exception_definition

%type <typed_expression> application
%type <typed_expression> list_without_left_bracket
%type <typed_expression> tuple

%type <multibinding_form> bindings
%type <binding_form> identifier_or_operator
%type <binding_form> identifier_or_operator_with_more_optional_parentheses
%type <binding_form> optional_identifier_or_operator_with_more_optional_parentheses
%type <binding_form> optional_identifier_or_operator_with_into
%type <binding_form_pos> identifier_declaration
%type <binding_form> postfix_operator
%type <binding_form> infix_operator
%type <binding_form> exception_name

%type <binding_form_with_type> identifier_or_operator_with_optional_type
%type <binding_form_pos_with_type> identifier_declaration_with_optional_type

%type <type_expression> optional_follow_of_formal_type_parameters_for_userdefined_type_without_of
%type <type_expression> optional_formal_type_parameters_for_userdefined_type_with_of

%type <type_expression> actual_type_parameters_for_userdefined_type_without_of
%type <type_expression> optional_follow_of_actual_type_parameters_for_userdefined_type_without_of
%type <type_expression> optional_actual_type_parameters_for_userdefined_type_with_of

%type <type_expression> optional_concrete_type_case_definition_with_of

%type <just_an_identifier> abstract_type_name concrete_type_name constructor;

%%

optional_expression:
  /* nothing */
  {
    $$.type=tuple_terminator_type;
  }
  |
  OPEN_PAR expression CLOSE_PAR
  {
    $$.type=$2.type;
  }
  ;

constructor:
  CONSTRUCTOR
  {
    strcpy($$.identifier,yytext);
  }
  ;

identifier_or_operator_with_more_optional_parentheses:
  OPEN_PAR identifier_or_operator CLOSE_PAR
  {
    strcpy($$.identifier,$2.identifier);
  }
  |
  OPEN_PAR identifier_or_operator_with_more_optional_parentheses CLOSE_PAR
  {
    strcpy($$.identifier,$2.identifier);
  }
  ;

optional_identifier_or_operator_with_more_optional_parentheses:
  /* nothing */
  {
    strcpy($$.identifier,""); /* a special dummy value */
  }
  |
  identifier_or_operator_with_more_optional_parentheses
  {
    strcpy($$.identifier,$1.identifier);
  }
  ;

pattern:
  constructor
  optional_identifier_or_operator_with_more_optional_parentheses
  {
    int unused_int;
    int encoding;
    type_t instantiated_parameter;

    lookup_concrete_type_constructor($1.identifier,
                                     &unused_int,
                                     &instantiated_parameter,
				     &encoding);
    if(is_a_tuple_terminator_type(instantiated_parameter)&&
       strcmp($2.identifier,""))
      yyerror("Constructor is nullary"); 
    if(!(is_a_tuple_terminator_type(instantiated_parameter))&&
       !strcmp($2.identifier,""))
      yyerror("Constructor is not nullary");

    $$.type=concrete_type_constructor_instance_to_concrete_type(
                $1.identifier,
                instantiated_parameter);
    output("PATTERN_");output_integer(current_pattern_matching());
    output("_");output_integer(current_pattern_matching_case_counter());output(":\n");

    output("\ts_jnm\t");output_integer(encoding);
    output("\tPATTERN_");output_integer(current_pattern_matching());
    output("_");output_integer(current_pattern_matching_case_counter()+1);
    output(":\n\t# ok, the test succeeded\n");

    if(!strcmp($2.identifier,"")){
      output("\ts_pshl\t0\t# make a block\n");
      push_environment(bind_not_refreshable(top_environment(),
					    "",
					    0,
					    type_error));
    }
    else{
      output("\ts_lkp_i\t1\t# take the argument ");output($2.identifier);
      output("\n");
      push_environment(bind_not_refreshable(top_environment(),
                                            $2.identifier,
					    0,
                                            instantiated_parameter));
      output("\ts_pshl\t1\t# make a block with it\n");
    }
    increment_pattern_matching_case_counter();
    if(is_pattern_matching_case_present(encoding))
      yyerror("pattern-matching: duplicate case");
    if(memoize_pattern_matching_case(encoding,
				     concrete_type_to_cases_number($$.type))
       !=0)
      yyerror("pattern-matching: type mismatch in the left part");
  }
  |
  UNDERSCORE
  {
    if(is_current_pattern_matching_exhaustive())
      yyerror("pattern-matching is already exhaustive before \'_\'");
    
    make_current_pattern_matching_exhaustive();

    output("PATTERN_");output_integer(current_pattern_matching());
    output("_");output_integer(current_pattern_matching_case_counter());output(":\n");
    increment_pattern_matching_case_counter();
    output("# _\n");
    output("\tpop\t\t# we don't need the argument any more\n");
    output("\ts_pshl\t0\t# make a block\n");
    push_environment(bind_not_refreshable(top_environment(),
					  "",
					  0,
					  type_error));
    $$.type=new_generic_type();
  }
  ;

pattern_arrow_expression: %prec LOWEST_PRECEDENCE
  pattern
  ARROW
  {
    output("# ->\n");
  }
  expression 
  {
    pop_environment();
    output("\ts_popl\t\n");
    output("\tj\tEND_OF_MATCH_");output_integer(current_pattern_matching());
    output(":\n");

    $$.pattern_type=$1.type;
    $$.expression_type=$4.type;
  }
  ;

more_patterns_arrow_expressions:
  /* nothing */
  {
    $$.pattern_type=new_generic_type();
    $$.expression_type=new_generic_type();
  }
  |
  PIPE
  {
    output("# |\n");
  }
  pattern_arrow_expression
  more_patterns_arrow_expressions
  {

    $$.pattern_type=force_types_to_be_equal($3.pattern_type,
					    $4.pattern_type);
    $$.expression_type=force_types_to_be_equal($3.expression_type,
					       $4.expression_type);

    if(is_error_type($$.pattern_type))
      yyerror("type mismatch in pattern-matching left sides");
    if(is_error_type($$.expression_type))
      yyerror("type mismatch in pattern-matching right sides");
  }
  ;

patterns_arrow_expressions:
  pattern_arrow_expression
  more_patterns_arrow_expressions
  {

    $$.pattern_type=force_types_to_be_equal($1.pattern_type,
					    $2.pattern_type);
    $$.expression_type=force_types_to_be_equal($1.expression_type,
					       $2.expression_type);

    if(is_error_type($$.pattern_type))
      yyerror("type mismatch in pattern-matching left sides");
    if(is_error_type($$.expression_type))
      yyerror("type mismatch in pattern-matching right sides");
  }
  ;

exception_cases:
  exception_case
  optional_more_exception_cases
  {
    $$.type=force_types_to_be_equal($1.type, $2.type);
  }
  ;

optional_more_exception_cases:
  /* nothing */
  {
    $$.type=new_generic_type();
  }
  |
  PIPE
  exception_case
  optional_more_exception_cases
  {
    $$.type=force_types_to_be_equal($2.type, $3.type);
  }
  ;

exception_name:
  EXCEPTION_NAME
  {
    strcpy($$.identifier,yytext);
  }
  ;

optional_identifier_or_operator_with_into:
  /* nothing */
  {
    strcpy($$.identifier, "");
  }
  |
  INTO identifier_or_operator
  {
    strcpy($$.identifier, $2.identifier);
  }
  ;

exception_case:
  exception_name optional_identifier_or_operator_with_into
  {
    increment_exception_handling_cases_number();
    output("EXCEPTION_DISPATCHER_");output_integer(current_exception_handling());output("_CASE_");
    output_integer(current_exception_handling_cases_number());output(":\n");
    output("\tjde\t\"");output(qualify_exception($1.identifier));output("\"\t");
    output("EXCEPTION_DISPATCHER_");output_integer(current_exception_handling());output("_CASE_");
    output_integer(current_exception_handling_cases_number()+1);output(":\n");
    if(strcmp($2.identifier,"")){
      /* There is an 'INTO variable' */
      type_t t;

      lookup_exception($1.identifier, &t);
      if(is_a_void_type(t))
	yywarning("extracting value from an exception defined as void");
      output("\ts_pshl\t1\t# block for ");output($2.identifier);output("\n");
      push_environment(bind_not_refreshable(top_environment(),
                                            $2.identifier,
                                            0,
                                            t));
    }
    else{
      output("\tpop\t\t# pop unused exception value\n");
    }
  }
  ARROW expression
  {
    if(strcmp($2.identifier,"")){
      /* There is an 'INTO variable' */
      pop_environment();
      output("\ts_popl\t\n");
    }    
    output("\tj\tEXCEPTION_DISPATCHER_");output_integer(current_exception_handling());output("_OUT:\n");
    $$.type=$5.type;
  }
  ;

optional_exception_parameter:
  /* nothing */
  {
    $$.type=void_type;
    output("\tpshnll\t\t# no parameter for throw\n");
  }
  |
  expression
  {
    $$.type=$1.type;
  }
  ;


optional_semicolon:
  SEMICOLON
  |
  /* nothing */
  ;

nonempty_sequence_of_elements:
  expression
  {
    $$.type = $1.type;
    $$.size = 1;
  }
  |
  nonempty_sequence_of_elements
  SEMICOLON
  expression
  {
    $$.type = force_types_to_be_equal($1.type, $3.type);
    $$.size = $1.size + 1;
  }
sequence_of_elements:
  /* nothing */
  {
    $$.type = new_generic_type();
    $$.size = 0;
  }
  |
  nonempty_sequence_of_elements
  optional_semicolon
  {
    $$.type = $1.type;
    $$.size = $1.size;
  }
  ;
/*
SEQ ::=
  ACT
| ACT FOLLOW
  ;

FOLLOW ::=
  //
| ACT FOLLOW
  ;
*/
/* ????? */
sequence_of_actions_with_end:
  ASSIGN identifier_or_operator TAKES
  expression
  {
    if(is_qualified( $2.identifier ))
      yyerror("identifiers bound by := cannot be qualified");
    force_types_to_be_equal($4.type,
			    new_io_type(new_generic_type()));
    output("\ts_cll\t0\t# evaluate right side of :=\n");
    output("\tpsho\t\t# push right side of :=\n");
    output("\ts_pshl\t1\t# assign ");output($2.identifier);output(" := top\n");
    push_environment(bind_not_refreshable(top_environment(),
					  $2.identifier,
					  0,
					  element_type($4.type, 0))); /* strip 'i/o'*/
  }
  SEMICOLON
  { output("# ;\n"); }
  sequence_of_actions_with_end
  {
    /* sequence_of_actions_with_end must have i/o type: */
    force_types_to_be_equal($8.type,
			    new_io_type(new_generic_type()));

    output("\ts_popl\t\t\t# end of ");output($2.identifier);output(" scope\n");
    pop_environment();
    $$.type = $8.type; /* the type is the type of the last action */
  }
  |
  expression
  SEMICOLON
  END_IO
  {
    output("\ts_cll\t0\t# evaluate action\n");
    output("# ;\n");
    output("# end\n");
    /* $1 must have i/o type: */
    force_types_to_be_equal($1.type,
			    new_io_type(new_generic_type()));
    $$.type = $1.type;
  }
  |
  expression
  SEMICOLON
  {
    /* expression must have i/o type: */
    force_types_to_be_equal($1.type,
			    new_io_type(new_generic_type()));
    output("\ts_cll\t0\t# evaluate action\n");
    output("# ;\n");
  }
  sequence_of_actions_with_end
  {
    /* the sequence of actions must have i/o type: */
    force_types_to_be_equal($4.type,
			    new_io_type(new_generic_type()));
    /* The type of the whole sequence of actions is the same type as the
       last action: */
    $$.type = $4.type;
  }
  ;

/* Horrible code duplication: begin */
sequence_of_actions_with_catch_io:
  ASSIGN identifier_or_operator TAKES
  expression
  {
    if(is_qualified( $2.identifier ))
      yyerror("identifiers bound by := cannot be qualified");
    force_types_to_be_equal($4.type,
			    new_io_type(new_generic_type()));
    output("\ts_cll\t0\t# evaluate right side of :=\n");
    output("\tpsho\t\t# push right side of :=\n");
    output("\ts_pshl\t1\t# assign ");output($2.identifier);output(" := top\n");
    push_environment(bind_not_refreshable(top_environment(),
					  $2.identifier,
					  0,
					  element_type($4.type, 0))); /* strip 'i/o'*/
  }
  SEMICOLON
  { output("# ;\n"); }
  sequence_of_actions_with_catch_io
  {
    /* sequence_of_actions_with_end must have i/o type: */
    force_types_to_be_equal($8.type,
			    new_io_type(new_generic_type()));

    output("\ts_popl\t\t\t# end of ");output($2.identifier);output(" scope\n");
    pop_environment();
    $$.type = $8.type; /* the type is the type of the last action */
  }
  |
  expression
  SEMICOLON
  CATCH_IO
  {
    output("\ts_cll\t0\t# evaluate action\n");
    output("# ;\n");
    output("# catch_io\n");
    /* $1 must have i/o type: */
    force_types_to_be_equal($1.type,
			    new_io_type(new_generic_type()));
    $$.type = $1.type;
  }
  |
  expression
  SEMICOLON
  {
    /* expression must have i/o type: */
    force_types_to_be_equal($1.type,
			    new_io_type(new_generic_type()));
    output("\ts_cll\t0\t# evaluate action\n");
    output("# ;\n");
  }
  sequence_of_actions_with_catch_io
  {
    /* the sequence of actions must have i/o type: */
    force_types_to_be_equal($4.type,
			    new_io_type(new_generic_type()));
    /* The type of the whole sequence of actions is the same type as the
       last action: */
    $$.type = $4.type;
  }
  ;
/* Horrible code duplication: end */



impossible:
  IMPOSSIBLE
  {
    $$.type = new_generic_type();
    output("\tpshcs\t\"From ");output(current_module_name);output(".epb:");
    output_integer(yylineno);output(": code \\\"impossible\\\" to reach was reached\\n\"\n\ts_outs\n");
    output("\tpshnll\n\ts_thrw\t\"exceptions:assertion_violated_exception\"\t # this should not be possible\n ");
  }
  ;

expression:
  C_FUNCTION STRING_CONSTANT
  {
    push_buffer(new_buffer());
    output("\n# Code for c_function (lambda-abstraction #");
    output_integer(current_function_label);output("):\n");
    output("FUN_");output_integer(current_function_label++);output(":\n");
    output("\ts_lcl\t1\t# Push the argument\n");
    output("\ts_cfun\t");output(yytext);output(" ");
  }
  FROM STRING_CONSTANT
  {
    output(yytext); output(" # call C function\n");
    output("\ts_ret\t# return result from C function\n");
    pop_buffer();
    output("\ts_cls\tFUN_");output_integer(current_function_label - 1);output(":\t# make a closure with the current environment\n");
    $$.type = new_function_type(new_generic_type(), new_generic_type());
  }
  |
  C_ACTION STRING_CONSTANT
  {
    push_buffer(new_buffer());
    output("\n# Code for c_action #");
    output_integer(current_function_label);output(":\n");
    output("ACT_");output_integer(current_function_label++);output(":\n");
    //output("\ts_lcl\t1\t# Push the argument\n");
    output("\ts_cact\t");output(yytext);output(" ");
  }
  FROM STRING_CONSTANT
  {
    output(yytext); output(" # execute C action\n");
    output("\ts_retn\t# actions don't return anything on the stack\n");
    pop_buffer();
    output("\ts_cls\tACT_");output_integer(current_function_label - 1);output(":\t# make a closure with the current environment\n");
    $$.type = new_io_type(new_generic_type());
  }
  |
  C_FUNCTION_TO_ACTION STRING_CONSTANT
  {
    push_buffer(new_buffer());
    output("\n# Code for c_action #");
    output_integer(current_function_label);output(":\n");
    output("FUN_TO_ACT_");output_integer(current_function_label);output(":\n");
    output("\ts_cls\tACT_");output_integer(current_function_label);output(":\n");
    output("\ts_ret\t\t# Return the action\n");
    output("ACT_");output_integer(current_function_label);output(":\n");
    output("\ts_nlcl\t1 1\t# Push the argument\n");
    output("\ts_cactp\t");output(yytext);output(" ");
  }
  FROM STRING_CONSTANT
  {
    output(yytext); output(" # execute C action\n");
    output("\ts_retn\t# actions don't return anything on the stack\n");
    pop_buffer();
    output("\ts_cls\tFUN_TO_ACT_");output_integer(current_function_label);output(":\t# make a closure with the current environment\n");
    current_function_label++;
    $$.type = new_function_type(new_generic_type(),
				new_io_type(new_generic_type()));
  }
  |
  C_OBJECT STRING_CONSTANT
  {
    output("\ts_cobj\t");output(yytext);output(" ");
  }
  FROM STRING_CONSTANT
  {
    output(yytext); output(" # push C object\n");
    $$.type = new_generic_type();
  }
  |
  impossible
  {
    $$.type = $1.type;
  }
  |
  /* beginning of I/O cases */
  SKIP
  {
    output_action_beginning();
    output_action_end();
    $$.type = new_io_type(void_type);
  }
  |
  MAKE_IO 
  {
    output_action_beginning();
  }
  expression
  {
    output("\ts_seto\t\t# move top into output_register\n");
    output_action_end();
    output("# make_io\n");
    $$.type = new_io_type($3.type);
  }
  |


  BEGIN_IO
  {
    output_action_beginning();
    output("# begin\n");
  }
  sequence_of_actions_with_end
  {
    output_action_end();
    $$.type = $3.type;
  }
/* ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ */
  |
  TRY_IO
  {
    output_action_beginning();
    output("\tpshtry\tEXCEPTION_DISPATCHER_");output_integer(push_exception_handling());output("_CASE_1:\t# try_io\n");
  }
  sequence_of_actions_with_catch_io
  {
    output("\tpoptry\n");
    output("\tj\tEXCEPTION_DISPATCHER_");output_integer(current_exception_handling());output("_AFTER_HANDLERS:\n");
    $$.type = $3.type;
  }
  exception_cases
  {
    output("EXCEPTION_DISPATCHER_");output_integer(current_exception_handling());output("_CASE_");
    output_integer(current_exception_handling_cases_number()+1);output(":\n");
    output("\trthrw\n");
    output("EXCEPTION_DISPATCHER_");output_integer(current_exception_handling());output("_OUT:\n");

    output("\ts_cll\t0\t# execute the exception handler action\n");

    output("EXCEPTION_DISPATCHER_");output_integer(current_exception_handling());output("_AFTER_HANDLERS:\n");

    pop_exception_handling();
    output_action_end();
    $$.type=force_types_to_be_equal($3.type, $5.type);
  }
  |
  THROW_IO 
  {
    output_action_beginning();
  }
  exception_name optional_exception_parameter
  {
    type_t t;
    make_current_point_failure();
    output("\ts_thrw\t\"");output(qualify_exception($3.identifier));output("\"\n");
    lookup_exception($3.identifier, &t);
    force_types_to_be_equal($4.type,t);
    output_action_end();
    $$.type=new_io_type(new_generic_type());
  }
  /* end of I/O cases */
  |
  LIST_TO_ARRAY expression
  {
    force_types_to_be_equal($2.type, new_list_type(new_generic_type()));
    $$.type = new_array_type(element_type($2.type, 0));
    output("\ts_ltbar\t\t# list_to_array\n");
  }
  |
  ARRAY_TO_LIST expression
  {
    force_types_to_be_equal($2.type, new_array_type(new_generic_type()));
    $$.type = new_list_type(element_type($2.type, 0));
    output("\ts_barlt\t\t# array_to_list\n");
  }
  |
  CHARACTER_TO_INTEGER expression
  {
    force_type_to_be($2.type, character_type);
    $$.type=integer_type;
    /* No need to generate any code */
    output("# character_to_integer\n");
  }
  |
  INTEGER_TO_CHARACTER expression
  {
    force_type_to_be($2.type, integer_type);
    $$.type=character_type;
    /* No need to generate any code */
    output("# integer_to_character\n");
  }
  |
  CHARACTER_TO_STRING expression
  {
    force_type_to_be($2.type, character_type);
    $$.type=string_type;
    output("\ts_chst\t\t# character_to_string\n");
  }
  |
  DEBUG expression
  {
    output("\tpshcs\t\"Debug from ");output(current_module_name);output(".epb:");
    output_integer(yylineno);output(": \"\n\ts_outs\n");
    output_output($2.type);
    output("\tpshcs\t\"\\n\"\n\ts_outs\n");
  }
  IN expression
  {
    $$.type=$5.type;
  }
  |
  ASSERT 
  {
    output("# assert\n");
  }
  expression
  {
    force_type_to_be($3.type, boolean_type);
    $1.label=current_assert_label++;
    output("\ts_jnz\tAFTER_ASSERT_");output_integer($1.label);output(":\n");
    output("\tpshcs\t\"Assertion in ");output(current_module_name);output(".epb:");output_integer(yylineno);
    output(" violated\\a\\n\"\n");
    output("\ts_outs\n");
    output("\tpshnll\n\ts_thrw\t\"exceptions:violated_assertion_exception\"\n");
    output("AFTER_ASSERT_");output_integer($1.label);output(":\n");
  }
  IN expression
  {
    output("# in\n");
    $$.type=$6.type;
  }
  |
  OPEN_ARRAY sequence_of_elements CLOSE_ARRAY
  {
    $$.type = new_array_type($2.type);
    output("\ts_mkba_i\t");output_integer($2.size);output("\t# make an array\n");
  }
  |
  TRY
  {
    output("\tpshtry\tEXCEPTION_DISPATCHER_");output_integer(push_exception_handling());output("_CASE_1:\n");
  }
  expression
  CATCH
  {
    output("\tpoptry\n");
    output("\tj\tEXCEPTION_DISPATCHER_");output_integer(current_exception_handling());output("_OUT:\n");
  }
  exception_cases
  {
    output("EXCEPTION_DISPATCHER_");output_integer(current_exception_handling());output("_CASE_");
    output_integer(current_exception_handling_cases_number()+1);output(":\n");
    output("\trthrw\n");
    output("EXCEPTION_DISPATCHER_");output_integer(current_exception_handling());output("_OUT:\n");
    pop_exception_handling();
    $$.type=force_types_to_be_equal($3.type,$6.type);
  }
  |
  THROW exception_name optional_exception_parameter
  {
    type_t t;
    make_current_point_failure();
    output("\ts_thrw\t\"");output(qualify_exception($2.identifier));output("\"\n");
    lookup_exception($2.identifier, &t);
    force_types_to_be_equal($3.type,t);
    $$.type=new_generic_type();
  }
  |
  MATCH
  {
    output("# match\n");
  }
  expression
  {
    push_pattern_matching();
    output("# with\n");
  }
  WITH
  patterns_arrow_expressions
  {
    output("PATTERN_");output_integer(current_pattern_matching());
    output("_");output_integer(current_pattern_matching_case_counter());
    output(":\t\t# fallback case\n");
    increment_pattern_matching_case_counter();

    //output("\tpop\n");
    output("\tpshcs\t\"pattern matching: this is not possible (");output(current_module_name);
    output(":");output_integer(yylineno);output(")\\n\"\n");
    output("\ts_outs\n\thlt\t-1\n");
    output("END_OF_MATCH_");output_integer(current_pattern_matching());
    output(":\n");
    force_type_to_be($3.type,$6.pattern_type);
    $$.type=$6.expression_type;
    if(!is_current_pattern_matching_exhaustive())
      yyerror("pattern-matching is not exhaustive");
    pop_pattern_matching();

    //fprintf(stderr,"PATTERNS:    ");output_type(stderr,$6.pattern_type);fprintf(stderr,"\n");
    //fprintf(stderr,"EXPRESSIONS: ");output_type(stderr,$6.expression_type);fprintf(stderr,"\n");
  }
  |
  constructor
  {
    type_t unused_type;
    int unused_int;
    int encoding; /* only this one is actually used */

    lookup_concrete_type_constructor($1.identifier,
                                     &unused_int,
                                     &unused_type,
                                     &encoding);

    output("\tpshci\t");output_integer(encoding);output("\t# encoding of ");output($1.identifier);output("\n");
  }
  optional_expression
  {
    /* Look for a type to use as a parameter to constructor, reading from the constructors table; then
       make the type of optional_expression match with that type.
       
       The whole $$ expression has the suitable concrete type, instantiated with the above type. */
    type_t expected_type_of_optional_expression;
    int unused;
    int encoding;
    char *concrete_type_name;

    lookup_concrete_type_constructor($1.identifier,
				     &unused,
				     &expected_type_of_optional_expression,
				     &encoding);
    force_types_to_be_equal($3.type,
			    expected_type_of_optional_expression); // To do: reverse? Test...
    $$.type=instantiate_concrete_type_given_a_constructor(concrete_type_name,
							  $1.identifier,
							  $3.type);

    if(is_a_tuple_terminator_type($3.type)){
      output("\ts_mka_i\t1\t# ");output($1.identifier);output(" has no parameters\n");
    }
    else{
      output("\ts_mka_i\t2\t# build ");output($1.identifier);output("\n");
    }
  }
  |
  INTEGER_TO_FLOAT expression
  {
    force_type_to_be($2.type,integer_type);
    output("\ts_infl\t\t# integer_to_float\n");
    $$.type = float_type;
  }
  |
  FLOAT_TO_INTEGER expression
  {
    force_type_to_be($2.type,float_type);
    output("\ts_flin\t\t# float_to_integer\n");
    $$.type = integer_type;
  }
  |
  FORCE expression
  {
    force_type_to_be($2.type,
		     new_promise_type(new_generic_type()));
    output("\ts_cll\t0\t# force\n");

    $$.type = element_type($2.type,0);
  }
  |
  DELAY
  {
    output("\ts_cls\tPROM_");output_integer(current_promise_label);output(":\t# closure for the promise\n");
    push_buffer(new_buffer());
    output("\n# Code for promise #");output_integer(current_promise_label);output(":\n");
    output("PROM_");output_integer(current_promise_label++);output(":\n");

    /* Create a new fake environment binding: the promise body sees the locals of this environment as non-locals of depth 1: */
    push_environment(bind_not_refreshable(top_environment(),""/* a dummy identifier */,0,type_error /* not used */));
  }
  expression
  {
    pop_environment();
    output("\ts_ret\t\t# return the value of the promise\n");
    pop_buffer();
    $$.type = new_promise_type($3.type);
  }
  |
  EMPTY expression
  {
    force_type_to_be($2.type,
                     new_list_type(new_generic_type()));
    output("\ts_nll\t\t# empty\n");
    $$.type = boolean_type;
  }
  |
  PRED expression
  {
    force_type_to_be($2.type,
                     integer_type);
    output("\ts_subi_i\t1\t# pred\n");
    $$.type = integer_type;
  }
  |
  SUCC expression
  {
    force_types_to_be_equal($2.type, integer_type);
    output("\ts_addi_i\t1\t# succ\n");
    $$.type = integer_type;
  }
  |
  LENGTH_STRING expression
  {
    force_types_to_be_equal($2.type, string_type);
    output("\ts_lkp_i\t0\t# length of string\n");
    $$.type=integer_type;
  }
  |
  LENGTH_ARRAY expression
  {
    force_types_to_be_equal($2.type, new_array_type(new_generic_type()));
    output("\ts_lkp_i\t0\t# length of array\n");
    $$.type=integer_type;
  }
  |
  expression CONCATENATE_STRING expression
  {
    force_types_to_be_equal($1.type, string_type);
    force_types_to_be_equal($3.type, string_type);
    $$.type=string_type;
    
    output("\ts_cctba\t\t# @@s\n");
  }
  |
  expression CONCATENATE_ARRAY expression
  {
    force_types_to_be_equal($1.type, new_array_type(new_generic_type()));
    force_types_to_be_equal($1.type, $3.type);
    $$.type=$1.type;
    
    output("\ts_cctba\t\t# @@\n");
  }
  |
  expression AT_STRING expression
  {
    force_types_to_be_equal($1.type, string_type);
    force_types_to_be_equal($3.type, integer_type);

    $$.type = character_type;
    make_current_point_failure();
    output("\ts_lkpb\t\t# @s\n");
  }
  |
  expression AT_ARRAY expression
  {
    force_types_to_be_equal($1.type, new_array_type(new_generic_type()));
    force_types_to_be_equal($3.type, integer_type);

    $$.type = element_type($1.type, 0);
    make_current_point_failure();
    output("\ts_lkpb\t\t# @\n");
  }
  |
  expression FROM
  { 
    force_types_to_be_equal($1.type, string_type);
  }
  expression TO
  {
    force_types_to_be_equal($4.type, integer_type);
  }
  expression
  {
    force_types_to_be_equal($7.type, integer_type);
    make_current_point_failure();
    output("\ts_frtoba\t\t# from..to operator\n");
    $$.type = string_type;
  }
  |

  expression FROM_ARRAY
  { 
    force_types_to_be_equal($1.type, new_array_type(new_generic_type()));
  }
  expression TO_ARRAY
  {
    force_types_to_be_equal($4.type, integer_type);
  }
  expression
  {
    force_types_to_be_equal($7.type, integer_type);
    make_current_point_failure();
    output("\ts_frtoba\t\t# from_array..to_array operator\n");
    $$.type = $1.type;
  }

  |
  expression AND
  {
    $2.label=current_non_strict_and_label++;
    output("\ts_jnz\tEVALUATE_SECOND_TERM_IN_AND_");output_integer($2.label);output(":\n");
    output("\tpshci\t0\t# false, don\'t evaluate the second part\n\tj\tEND_OF_AND_");
    output_integer($2.label);output(":\n");
    output("EVALUATE_SECOND_TERM_IN_AND_");output_integer($2.label);output(":\n");
  }
  expression
  {
    output("END_OF_AND_");output_integer($2.label);output(":\n");
    
    force_types_to_be_equal($1.type, boolean_type);
    force_types_to_be_equal($4.type, boolean_type);
    $$.type=boolean_type;
  }
  |
  expression OR
  {
    $2.label=current_non_strict_or_label++;
    output("\ts_jz\tEVALUATE_SECOND_TERM_IN_OR_");output_integer($2.label);output(":\n");
    output("\tpshci\t1\t# true, don\'t evaluate the second part\n\tj\tEND_OF_OR_");
    output_integer($2.label);output(":\n");
    output("EVALUATE_SECOND_TERM_IN_OR_");output_integer($2.label);output(":\n");
  }
  expression
  {
    output("END_OF_OR_");output_integer($2.label);output(":\n");

    force_types_to_be_equal($1.type, boolean_type);
    force_types_to_be_equal($4.type, boolean_type);
    $$.type=boolean_type;
  }
  |
  expression AND_STRICT expression
  {
    force_types_to_be_equal($1.type, boolean_type);
    force_types_to_be_equal($3.type, boolean_type);
    $$.type=boolean_type;

    output("\ts_andi\t\t# and_strict\n");
  }
  |
  expression OR_STRICT expression
  {
    force_types_to_be_equal($1.type, boolean_type);
    force_types_to_be_equal($3.type, boolean_type);
    $$.type=boolean_type;

    output("\ts_ori\t\t# or_strict\n");
  }
  |
  expression XOR expression
  {
    force_types_to_be_equal($1.type, boolean_type);
    force_types_to_be_equal($3.type, boolean_type);
    $$.type=boolean_type;
    
    output("\ts_xori\t\t# xor\n");
  }
  |
  NOT expression
  {
    force_types_to_be_equal($2.type, boolean_type);
    $$.type=boolean_type;

    output("\ts_noti\t\t# not\n");
  }
  |
  expression EQUAL expression
  {
    force_types_to_be_equal($1.type,integer_type);
    force_types_to_be_equal($3.type,integer_type);

    output("\ts_eqi\t\t# =\n");
    $$.type=boolean_type;
  }
  |
  expression DIFFERENT expression
  {
    force_types_to_be_equal($1.type,integer_type);
    force_types_to_be_equal($3.type,integer_type);

    output("\ts_neqi\t\t# =/=\n");
    $$.type=boolean_type;
  }
  |
  expression LESS expression
  {
    force_types_to_be_equal($1.type,integer_type);
    force_types_to_be_equal($3.type,integer_type);

    output("\ts_lti\t\t# <\n");
    $$.type=boolean_type;
  }
  |
  expression GREATER expression
  {
    force_types_to_be_equal($1.type,integer_type);
    force_types_to_be_equal($3.type,integer_type);

    output("\ts_gti\t\t# >\n");
    $$.type=boolean_type;
  }
  |
  expression LESS_OR_EQUAL expression
  {
    force_types_to_be_equal($1.type,integer_type);
    force_types_to_be_equal($3.type,integer_type);

    output("\ts_ltei\t\t# <=\n");
    $$.type=boolean_type;
  }
  |
  expression GREATER_OR_EQUAL expression
  {
    force_types_to_be_equal($1.type,integer_type);
    force_types_to_be_equal($3.type,integer_type);

    output("\ts_gtei\t\t# >=\n");
    $$.type=boolean_type;
  }
  |
  expression EQUAL_FLOAT expression
  {
    force_types_to_be_equal($1.type,float_type);
    force_types_to_be_equal($3.type,float_type);

    output("\ts_eqf\t\t# =f\n");
    $$.type=boolean_type;
  }
  |
  expression DIFFERENT_FLOAT expression
  {
    force_types_to_be_equal($1.type,float_type);
    force_types_to_be_equal($3.type,float_type);

    output("\ts_neqf\t\t# =/=f\n");
    $$.type=boolean_type;
  }
  |
  expression LESS_FLOAT expression
  {
    force_types_to_be_equal($1.type,float_type);
    force_types_to_be_equal($3.type,float_type);

    output("\ts_ltf\t\t# <f\n");
    $$.type=boolean_type;
  }
  |
  expression GREATER_FLOAT expression
  {
    force_types_to_be_equal($1.type,float_type);
    force_types_to_be_equal($3.type,float_type);

    output("\ts_gtf\t\t# >f\n");
    $$.type=boolean_type;
  }
  |
  expression LESS_OR_EQUAL_FLOAT expression
  {
    force_types_to_be_equal($1.type,float_type);
    force_types_to_be_equal($3.type,float_type);

    output("\ts_ltef\t\t# <=f\n");
    $$.type=boolean_type;
  }
  |
  expression GREATER_OR_EQUAL_FLOAT expression
  {
    force_types_to_be_equal($1.type,float_type);
    force_types_to_be_equal($3.type,float_type);

    output("\ts_gtef\t\t# >=f\n");
    $$.type=boolean_type;
  }
  |
  expression EQUAL_STRING expression
  {
    force_types_to_be_equal($1.type,string_type);
    force_types_to_be_equal($3.type,string_type);

    output("\ts_eqs\t\t# =f\n");
    $$.type=boolean_type;
  }
  |
  expression DIFFERENT_STRING expression
  {
    force_types_to_be_equal($1.type,string_type);
    force_types_to_be_equal($3.type,string_type);

    output("\ts_neqs\t\t# =/=f\n");
    $$.type=boolean_type;
  }
  |
  expression LESS_STRING expression
  {
    force_types_to_be_equal($1.type,string_type);
    force_types_to_be_equal($3.type,string_type);

    output("\ts_lts\t\t# <f\n");
    $$.type=boolean_type;
  }
  |
  expression GREATER_STRING expression
  {
    force_types_to_be_equal($1.type,string_type);
    force_types_to_be_equal($3.type,string_type);

    output("\ts_gts\t\t# >f\n");
    $$.type=boolean_type;
  }
  |
  expression LESS_OR_EQUAL_STRING expression
  {
    force_types_to_be_equal($1.type,string_type);
    force_types_to_be_equal($3.type,string_type);

    output("\ts_ltes\t\t# <=f\n");
    $$.type=boolean_type;
  }
  |
  expression GREATER_OR_EQUAL_STRING expression
  {
    force_types_to_be_equal($1.type,string_type);
    force_types_to_be_equal($3.type,string_type);

    output("\ts_gtes\t\t# >=f\n");
    $$.type=boolean_type;
  }
  |
  expression EQUAL_CHARACTER expression
  {
    force_types_to_be_equal($1.type,character_type);
    force_types_to_be_equal($3.type,character_type);

    output("\ts_eqi\t\t# =c\n");
    $$.type=boolean_type;
  }
  |
  expression DIFFERENT_CHARACTER expression
  {
    force_types_to_be_equal($1.type,character_type);
    force_types_to_be_equal($3.type,character_type);

    output("\ts_neqi\t\t# =/=c\n");
    $$.type=boolean_type;
  }
  |
  expression LESS_CHARACTER expression
  {
    force_types_to_be_equal($1.type,character_type);
    force_types_to_be_equal($3.type,character_type);

    output("\ts_lti\t\t# <c\n");
    $$.type=boolean_type;
  }
  |
  expression GREATER_CHARACTER expression
  {
    force_types_to_be_equal($1.type,character_type);
    force_types_to_be_equal($3.type,character_type);

    output("\ts_gti\t\t# >c\n");
    $$.type=boolean_type;
  }
  |
  expression LESS_OR_EQUAL_CHARACTER expression
  {
    force_types_to_be_equal($1.type,character_type);
    force_types_to_be_equal($3.type,character_type);

    output("\ts_ltei\t\t# <=c\n");
    $$.type=boolean_type;
  }
  |
  expression GREATER_OR_EQUAL_CHARACTER expression
  {
    force_types_to_be_equal($1.type,character_type);
    force_types_to_be_equal($3.type,character_type);

    output("\ts_gtei\t\t# >=c\n");
    $$.type=boolean_type;
  }
  |
  expression EQUAL_BOOLEAN expression
  {
    force_types_to_be_equal($1.type,boolean_type);
    force_types_to_be_equal($3.type,boolean_type);

    output("\ts_nxori\t\t# =b\n");
    $$.type=boolean_type;
  }
  |
  expression DIFFERENT_BOOLEAN expression
  {
    force_types_to_be_equal($1.type,boolean_type);
    force_types_to_be_equal($3.type,boolean_type);

    output("\ts_xori\t\t# =/=b\n");
    $$.type=boolean_type;
  }
  |
  VOID_CONSTANT
  {
    output("\tpshnll\t\t# ()\n");
    $$.type=void_type;
  }
  |
  TRUE
  {
    output("\tpshci\t1\t# true\n");
    $$.type=boolean_type;
  }
  |
  FALSE
  {
    output("\tpshci\t0\t# false\n");
    $$.type=boolean_type;
  }
  |
  FLOAT_CONSTANT
  {
    output("\tpshcf\t");output(yytext);output("\n");
    $$.type=float_type;
  }
  |
  STRING_CONSTANT
  {
    output("\tpshcs\t");output(yytext);output("\n");
    $$.type=string_type;
  }
  |
  CHARACTER_CONSTANT
  {
    char c;
    /* Remove "'" at the beginning and at the end: */
    yytext[yyleng-1]='\0'; yytext++;
    
    if(yytext[0]=='\\')
      switch(yytext[1]){
      case 'a':  { c='\a'; break; }
      case 't':  { c='\t'; break; }
      case 'n':  { c='\n'; break; }
      case '\'': { c='\''; break; }
      default:   { yyerror("undefined escape sequence in character"); }
      }/* switch */
    else
      c=yytext[0];
    output("\tpshci\t");output_integer(c);output("\t# '");output(yytext);output("'\n");
    $$.type=character_type;
  }
  |
  INTEGER_CONSTANT
  {
    output("\tpshci\t");output(yytext);output("\n");
    $$.type=integer_type;
  }
  |
  expression PLUS expression
  {
    force_types_to_be_equal($1.type,integer_type);
    force_types_to_be_equal($3.type,integer_type);

    output("\ts_addi\t\t# +\n");
    $$.type=integer_type;
  }
  |
  expression MINUS expression
  {
    force_types_to_be_equal($1.type,integer_type);
    force_types_to_be_equal($3.type,integer_type);

    output("\ts_subi\t\t# -\n");
    $$.type=integer_type;
  }
  |
  expression TIMES expression
  {
    force_types_to_be_equal($1.type,integer_type);
    force_types_to_be_equal($3.type,integer_type);

    output("\ts_muli\t\t# *\n");
    $$.type=integer_type;
  }
  |
  expression DIVIDED expression
  {
    force_types_to_be_equal($1.type,integer_type);
    force_types_to_be_equal($3.type,integer_type);

    make_current_point_failure();
    output("\ts_divi\t\t# /\n");
    $$.type=integer_type;
  }
  |
  expression POWER expression
  {
    force_types_to_be_equal($1.type,integer_type);
    force_types_to_be_equal($3.type,integer_type);

    output("\ts_powi\t\t# **\n");
    $$.type=integer_type;
  }
  |
  expression MODULO expression
  {
    force_types_to_be_equal($1.type,integer_type);
    force_types_to_be_equal($3.type,integer_type);
    $$.type=integer_type;

    make_current_point_failure();
    output("\ts_modi\t\t# mod\n");
  }
  |
  expression PLUS_FLOAT expression
  {
    force_types_to_be_equal($1.type,float_type);
    force_types_to_be_equal($3.type,float_type);

    output("\ts_addf\t\t# +f\n");
    $$.type=float_type;
  }
  |
  expression MINUS_FLOAT expression
  {
    force_types_to_be_equal($1.type,float_type);
    force_types_to_be_equal($3.type,float_type);

    output("\ts_subf\t\t# -f\n");
    $$.type=float_type;
  }
  |
  expression TIMES_FLOAT expression
  {
    force_types_to_be_equal($1.type,float_type);
    force_types_to_be_equal($3.type,float_type);

    output("\ts_mulf\t\t# *f\n");
    $$.type=float_type;
  }
  |
  expression DIVIDED_FLOAT expression
  {
    force_types_to_be_equal($1.type,float_type);
    force_types_to_be_equal($3.type,float_type);

    make_current_point_failure();
    output("\ts_divf\t\t# /f\n");
    $$.type=float_type;
  }
  |
  expression POWER_FLOAT expression
  {
    force_types_to_be_equal($1.type,float_type);
    force_types_to_be_equal($3.type,float_type);

    output("\ts_powf\t\t# **f\n");
    $$.type=float_type;
  }
  |
  IF 
  {
    $1.label=current_if_label++;
    output("# Start of if...then...else #");output_integer($1.label);output("\n");
  }
  expression THEN
  {
    force_types_to_be_equal($3.type, boolean_type);

    output("\ts_jz\tELSE_BRANCH_");output_integer($1.label);output(":\n");
    output("# then branch #");output_integer($1.label);output("\n");
  }
  expression ELSE
  {
    output("\tj\tEND_OF_IF_THEN_ELSE_");output_integer($1.label);output(":\nELSE_BRANCH_");output_integer($1.label);output(":\n");
  }
  expression
  { 
    /* This jump wouldn't actually be needed, but it allows tail-call
       optimizations; if peephole optimizations are not needed then the
       peephole optimizator removes it. */
    output("\tj\tEND_OF_IF_THEN_ELSE_");output_integer($1.label);output(":\n");
    output("END_OF_IF_THEN_ELSE_");output_integer($1.label);output(":\n");
    output("# End of if...then...else #");output_integer($1.label);output("\n");
    $$.type=force_types_to_be_equal($6.type,$9.type);
    if(is_error_type($$.type))
      yyerror("then and else branches have incompatible types");
  }
  |
  expression CONS expression
  {
    $$.type=force_types_to_be_equal($3.type,new_list_type($1.type));

    output("\ts_mka_i\t2\t# ::\n");
  }
  |
  HEAD expression
  {
    $$.type=element_type(force_types_to_be_equal(new_list_type(new_generic_type()),$2.type), 0);
    make_current_point_failure();
    output("\ts_car\t\t# head\n");
  }
  |
  TAIL expression
  {
    $$.type=force_types_to_be_equal(new_list_type(new_generic_type()),$2.type);
    make_current_point_failure();
    output("\ts_cdr\t\t# tail\n");
  }
  |
  tuple
  {
    int n=arity_of_tuple_type($1.type);
    
    $$.type=$1.type;
    output("\ts_mka_i\t");output_integer(n);output("\t# make a tuple of ");output_integer(n);output(" elements \n");
  }
  |
  expression SELECTOR INTEGER_CONSTANT
  {
    type_t t=new_generic_type();
    int n;
    
    sscanf(yytext,"%i",&n);
    if(n<1)
      yyerror("the selector integer argument must be greater than 0");
    
    output("\ts_lkp_i\t");output_integer(n-1);output("\t# ^ ");output_integer(n);output("\n");
    
    force_types_to_be_equal(new_tuple_type_from_selector(n,t), $1.type);
    $$.type = element_of_tuple_type($1.type,n);
  }
  |
  OPEN_BRACKET 
  {
    output("# [\n");
  }
  list_without_left_bracket
  {
    $$.type = $3.type;
    output("# ]\n");
  }
  |
  EMPTY_LIST
  {
    $$.type = new_list_type(new_generic_type());
    output("\tpshnll\t# []\n");
  }
  |
  OPEN_PAR
  {
    output("# (\n");
  } 
  expression CLOSE_PAR
  { 
    $$.type = $3.type;
    output("# )\n");
  }
  |
  LAMBDA
  {
    push_buffer(new_buffer());
    $1.label=current_function_label;
    output("\n# Code for lambda-abstraction #");output_integer(current_function_label);output(":\n");
    output("FUN_");output_integer(current_function_label++);output(":\n");
    if(are_we_in_a_recursive_definition){
      output("\tgcin\t\t# GC if needed\n");
      are_we_in_a_recursive_definition = 0;
    }
  }
  identifier_or_operator_with_optional_type
  {
    if(is_qualified($3.identifier))
      yyerror("formal parameters cannot be qualified");
    push_environment(bind_not_refreshable(top_environment(),$3.identifier,0,$3.type));
  }
  DOT expression
  {
    pop_environment();

    output("\ts_ret\t\t# return the value on top\n");
    pop_buffer();
    output("\ts_cls\tFUN_");output_integer($1.label);output(":\t# make a closure with the current environment\n");
    $$.type=new_function_type($3.type,$6.type);
  }
  |
  LET
  {
    $1.label=current_let_label++;
    output("# let\n");
    push_bindings();
  }
  bindings 
  {
    pop_bindings();
    output("# in\n"); 
  } 
  IN 
  {
    output("\ts_pshl\t");output_integer($3.bindings_no);output("\t# create environment to hold let bindings\n");
  }
  expression
  {
    pop_environment();
    $$.type=$7.type;

    output("# end of let\n\ts_popl\t\n");
  }
  |
  FIX LAMBDA identifier_declaration_with_optional_type
  {
    if(is_qualified($3.identifier))
      yyerror("identifiers bound by 'fix' cannot be qualified");
    output("# fix \\ ");output($3.identifier);output("\n");
    strncpy($1.identifier,$3.identifier,IDENTIFIER_LENGTH);$1.identifier[IDENTIFIER_LENGTH-1]='\0';
    
    if($3.order == IDENTIFIER)
      push_environment(bind_not_refreshable(top_environment(),$1.identifier,0,$3.type));
    else if($3.order == POSTFIX_OPERATOR)
      push_environment(bind_not_refreshable_with_precedence(top_environment(),
							    $1.identifier,0,new_postfix_operator_type(element_type($3.type,0),
												      element_type($3.type,1)),
							    $3.precedence));
    
    output("\ts_clsr\tFUN_");output_integer(current_function_label);
    output(":\t# recursively close with current environment\n");
    output("\ts_pshl\t1\t# create a frame for the recursive definition\n");
    
    output("# .\n");
    are_we_in_a_recursive_definition = 1;
  }
  DOT expression
  {
    type_t t;
    int unused;
    
    lookup($3.identifier,&unused,&unused,&t);
    
    force_types_to_be_equal(t,$6.type);
    
    output("\ts_lcl\t1\t# ");output($1.identifier);output("\n");
    output("\ts_popl\t\t\t# end of fix expression\n");
    pop_environment();
    
    if(!is_a_function_type(t))
      yyerror("in (fix \\ f . E) E must have function type");
    
    $$.type=t;
  }
  |
  IDENTIFIER
  {
    int jumps,n;
    type_t t;
    int return_code=lookup(yytext,&jumps,&n,&t);
    
    if(return_code==-1)
      yyerror("unbound identifier");
    else if(return_code==-2){
      yyerror("non-qualified reference to global is ambiguous");
    }
    else if(jumps==0){
      output("\ts_lcl\t");output_integer(n+1);output("\t# ");output(yytext);output("\n");
    }
    else if(jumps==-1){ /* it's a global */
      output("\tpshgl\t\"");output(qualify(yytext));output("\"\t# ");output(yytext);output("\n");
    }
    else{
      output("\ts_nlcl\t");output_integer(jumps);output("\t");output_integer(n+1);output("\t# ");
      output(yytext);output("\n");
    }

    $$.type=t;
  }
  |
  PLUS expression %prec UNARY_PLUS
  {
    if((!is_an_integer_type($2.type))&&(!is_a_float_type($2.type)))
      yyerror("type mismatch in unary plus operand");
    $$.type=integer_type;
    /* unary plus: there is no need to translate it into assembly: */
    output("# unary +\n");
  } 
  |
  MINUS expression  %prec UNARY_MINUS
  {
    if(is_an_integer_type($2.type))
      output("\ts_umini\t\t# unary minus\n");
    else if(is_a_float_type($2.type))
      output("\ts_uminf\t\t# unary minus\n");
    else
      yyerror("type mismatch in unary minus operand");

    $$.type=$2.type;
  }
  |
  application %prec APPLICATION
  {
    $$.type = $1.type;
  }
  ;

application:
  expression expression %prec APPLICATION
  {
    force_types_to_be_equal($1.type,new_function_type($2.type,new_generic_type())); // restored on 23rd Jun 2002
    $$.type=element_type($1.type,1);
    
    output("\ts_cll\t1\t# apply the closure on the stack\n");
  }
  |
  expression postfix_operator
  {
    type_t operator_type; int jumps_no,pos;
    int return_code=lookup(yytext, &jumps_no, &pos, &operator_type);

    if(return_code==-1){
      yyerror("operator is not bound");
    }
    else if(return_code==-2)
      yyerror("reference to postfix operator is ambiguous");

    force_types_to_be_equal(operator_type,new_postfix_operator_type($1.type,new_generic_type()));
    $$.type=element_type(operator_type,1);
        
    if(jumps_no==0){ /* local */
      output("\ts_lcl\t");output_integer(pos+1);output("\n");
    }
    else if(jumps_no == -1){ /* global*/
      output("\tpshgl\t\"");output(qualify(yytext));output("\"\t# ");output(yytext);output("\n");
    }
    else{ /* non-local */
      output("\ts_nlcl\t");output_integer(jumps_no);output("\t");output_integer(pos+1);output("\t# ");
      output(yytext);output("\n");
    }

    output("\ts_swp\t\t# put the postfix operator under the operand\n\ts_cll\t1\t# apply the closure on the stack\n");
  }
  |
  expression infix_operator expression
  {
    type_t operator_type; int jumps_no,pos;
    int return_code=lookup($2.identifier, &jumps_no, &pos, &operator_type);

    if(return_code==-1){
      yyerror("operator is not bound");
    }
    else if(return_code==-2)
      yyerror("referece to infix operator is ambiguous");
    
    if(is_an_infix_operator_left_type(operator_type))
      force_types_to_be_equal(operator_type,
			      new_infix_operator_left_type($1.type,$3.type,new_generic_type()));
    else
      force_types_to_be_equal(operator_type,
                              new_infix_operator_right_type($1.type,$3.type,new_generic_type()));
    $$.type=element_type(operator_type,2);
    
    output("\ts_mka_i\t2\n");
    
    if(jumps_no==0){ /* local */
      output("\ts_lcl\t");output_integer(pos+1);output("\n");
    }
    else if(jumps_no == -1){ /* global*/
      output("\tpshgl\t\"");output(qualify($2.identifier));output("\"\t# ");output($2.identifier);output("\n");
    }
    else{ /* non-local */
      output("\ts_nlcl\t");output_integer(jumps_no);output("\t");output_integer(pos+1);output("\t# ");
      output(yytext);output("\n");
    }
    
    output("\ts_swp\n\ts_cll\t1\t# apply the infix operator\n");
  }
  ;

bindings:
  identifier_declaration
  {
    if(is_qualified($1.identifier))
      yyerror("identifiers bound by 'let' cannot be qualified");

    strncpy($1.identifier,yytext,IDENTIFIER_LENGTH);$1.identifier[IDENTIFIER_LENGTH-1]='\0';
    output("# ");output(yytext);output(" be... \n");

    if((*current_bindings()) != 0)
      push_environment(previous_environment()); // hide bindings from this same let
    else
      push_environment(top_environment()); // just for being able to always use pop_environment(), later
  }
  BE expression
  {
    pop_environment(); // ok, return to the environment we are building

    if($1.order==IDENTIFIER){
      if((*current_bindings()) == 0)
        push_environment(bind_not_refreshable(top_environment(),$1.identifier,(*current_bindings())++,$4.type));
      else
        push_environment(bind_not_refreshable(pop_environment(),$1.identifier,(*current_bindings())++,$4.type));
    }
    else if($1.order==POSTFIX_OPERATOR){
      if(!is_a_function_type($4.type))
        yyerror("postfix operators must have function type");

      if((*current_bindings()) == 0)
        push_environment(bind_not_refreshable_with_precedence(top_environment(),$1.identifier,(*current_bindings())++,
					      new_postfix_operator_type(element_type($4.type,0),element_type($4.type,1)),
                                              $1.precedence));
      else
        push_environment(bind_not_refreshable_with_precedence(pop_environment(),$1.identifier,(*current_bindings())++,
                                              new_postfix_operator_type(element_type($4.type,0),element_type($4.type,1)),
                                              $1.precedence));
    }
    else if($1.order==INFIX_OPERATOR_LEFT){
      type_t arg1,arg2;

      if(!is_a_function_type($4.type))
        yyerror("infix operators must have function type");
      else if(!is_a_pair_type(element_type($4.type,0)))
	yyerror("infix operators must be binary");

      if((*current_bindings()) == 0)
        push_environment(bind_not_refreshable_with_precedence(top_environment(),$1.identifier,(*current_bindings())++,
                                              new_infix_operator_left_type(element_type(element_type($4.type,0),0),
									   element_type(element_type($4.type,0),1),
                                                                           element_type($4.type,1)),
                                              $1.precedence));
      else
        push_environment(bind_not_refreshable_with_precedence(pop_environment(),$1.identifier,(*current_bindings())++,
                                              new_infix_operator_left_type(element_type(element_type($4.type,0),0),
                                                                           element_type(element_type($4.type,0),1),
                                                                           element_type($4.type,1)),
                                              $1.precedence));
    }
    else if($1.order==INFIX_OPERATOR_RIGHT){
      type_t arg1,arg2;

      if(!is_a_function_type($4.type))
        yyerror("infix operators must have function type");
      else if(!is_a_pair_type(element_type($4.type,0)))
        yyerror("infix operators must be binary");

      if((*current_bindings()) == 0)
        push_environment(bind_not_refreshable_with_precedence(top_environment(),$1.identifier,(*current_bindings())++,
                                              new_infix_operator_right_type(element_type(element_type($4.type,0),0),
                                                                           element_type(element_type($4.type,0),1),
                                                                           element_type($4.type,1)),
                                              $1.precedence));
      else
        push_environment(bind_not_refreshable_with_precedence(pop_environment(),$1.identifier,(*current_bindings())++,
                                              new_infix_operator_right_type(element_type(element_type($4.type,0),0),
                                                                           element_type(element_type($4.type,0),1),
                                                                           element_type($4.type,1)),
                                              $1.precedence));
    }
    $$.bindings_no=1;
  }
  |
  bindings AMPERSAND bindings
  {
    $$.bindings_no = $1.bindings_no + $3.bindings_no;
  }
  ;

postfix_operator:
    POSTFIX_OPERATOR_1 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_2 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_3 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_4 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_5 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_6 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_7 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_8 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_9 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_10 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_11 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_12 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_13 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_14 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_15 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_16 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_17 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_18 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_19 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | POSTFIX_OPERATOR_20 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  ;

infix_operator:
    INFIX_OPERATOR_LEFT_1 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_1 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_2 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_2 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_3 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_3 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_4 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_4 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_5 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_5 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_6 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_6 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_7 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_7 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_8 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_8 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_9 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_9 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_10 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_10 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_11 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_11 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_12 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_12 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_13 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_13 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_14 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_14 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_15 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_15 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_16 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_16 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_17 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_17 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_18 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_18 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_19 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_19 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_LEFT_20 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  | INFIX_OPERATOR_RIGHT_20 {strcpy($$.identifier, yytext); $$.label = $1.label;}
  ;

identifier_or_operator:
  IDENTIFIER
  {
    strcpy($$.identifier, yytext);
    $$.label = $1.label;
    //fprintf(stderr, "identifier_or_operator: yytext is %s, $$.identifier is %s\n", yytext, $$.identifier);
  }
  |
  postfix_operator
  {
    strcpy($$.identifier, yytext);
    $$.label = $1.label;
  }
  |
  infix_operator
  {
    strcpy($$.identifier, $1.identifier);
    $$.label = $1.label;
  }
  |
  CONCRETE_TYPE_NAME
  {
    strcpy($$.identifier, yytext);
    $$.label = $1.label;
  }
  |
  ABSTRACT_TYPE_NAME
  {
    strcpy($$.identifier, yytext);
    $$.label = $1.label;
  }
  |
  CONSTRUCTOR
  {
    strcpy($$.identifier, yytext);
    $$.label = $1.label;
  }
  |
  EXCEPTION_NAME
  {
    strcpy($$.identifier, yytext);
    $$.label = $1.label;
  }
  ;

optional_type:
  /* nothing */
  { $$ = new_generic_type(); }
  |
  COLON type
  { $$ = $2; }
  ;

identifier_or_operator_with_optional_type:
  identifier_or_operator
  {
    $$.label=$1.label;
    strcpy($$.identifier,$1.identifier);
  }
  optional_type
  {
    $$.type=$3;
  }
  ;

identifier_declaration_with_optional_type:
  identifier_declaration
  {
    $$.label=$1.label;
    strcpy($$.identifier,$1.identifier);
  }
  optional_type
  {
    $$.type=$3;
    $$.order=$1.order;
    $$.precedence=$1.precedence;
  }
  ;

infix_with_optional_left:
  INFIX
  |
  INFIX LEFT
  ;

infix_with_optional_precedence:
  infix_with_optional_left
  |
  infix_with_optional_left INTEGER_CONSTANT
  ;

infix_right_with_optional_precedence:
  INFIX RIGHT
  |
  INFIX RIGHT INTEGER_CONSTANT
  ;

postfix_with_optional_precedence:
  POSTFIX
  | 
  POSTFIX INTEGER_CONSTANT
  ;

identifier_declaration:
  identifier_or_operator
  {
    strcpy($$.identifier, $1.identifier);
    $$.label = $1.label;
    $$.order=IDENTIFIER;
  }
  |
  postfix_with_optional_precedence
  {
    if(sscanf(yytext,"%i",&precedence) > 0){
      if((precedence<1)||(precedence>20))
        yyerror("operator precedence must be greater than 0 and less than 21");
    }
    else
      precedence=10;
  }
  identifier_or_operator
  {
    strcpy($$.identifier, $3.identifier);
    $$.label = $3.label;
    $$.order=POSTFIX_OPERATOR;
    $$.precedence=precedence;
  }
  |
  infix_with_optional_precedence
  {
    if(sscanf(yytext,"%i",&precedence) > 0){
      if((precedence<1)||(precedence>20))
        yyerror("operator precedence must be greater than 0 and less than 21");
    }
    else
      precedence=10;
  }
  identifier_or_operator
  {
    strcpy($$.identifier, $3.identifier);
    $$.label = $3.label;
    $$.order=INFIX_OPERATOR_LEFT;
    $$.precedence=precedence;
  }
  |
  infix_right_with_optional_precedence
  {
    if(sscanf(yytext,"%i",&precedence) > 0){
      if((precedence<1)||(precedence>20))
        yyerror("operator precedence must be greater than 0 and less than 21");
    }
    else
      precedence=10;
  }
  identifier_or_operator
  {
    strcpy($$.identifier, $3.identifier);
    $$.label = $3.label;
    $$.order=INFIX_OPERATOR_RIGHT;
    $$.precedence=precedence;
  }
  ;

tuple:
  expression COMMA expression
  {
    $$.type = new_pair_type($1.type,new_pair_type($3.type,tuple_terminator_type));
  }
  |
  tuple COMMA expression
  {
    $$.type = append_to_tuple_type($1.type,$3.type);
  }
  ;

list_without_left_bracket:
  CLOSE_BRACKET
  {
    output("\tpshnll\t\t# empty list\n");
    $$.type=new_list_type(new_generic_type());
  }
  |
  expression CLOSE_BRACKET
  {
    output("\tpshnll\t\t# trailing empty list\n\ts_mka_i\t2\t# ;\n");
    $$.type=new_list_type($1.type);
  }
  |
  expression SEMICOLON list_without_left_bracket
  {
    if(!is_a_list_type($3.type))
      yyerror("The [...;...] notation should build a homogeneous list");
    else if(!do_types_match(element_type($3.type,0),$1.type))
      yyerror("The [...;...] notation should build a homogeneous list");
    output("\ts_mka_i\t2\t# ;\n");
    $$.type=new_list_type(unify_types($1.type,element_type($3.type,0)));
  }
  ;

type:
  TYPE OF
  {
    push_buffer(new_buffer());
    output("# This code will never be executed: begin\n");
  }
  expression
  {
    output("# This code will never be executed: end\n");
    pop_buffer();
    $$ = $4.type;
  }
  |
  VOID { $$ = void_type; }
  |
  INTEGER { $$ = integer_type; }
  |
  FLOAT   { $$ = float_type; }
  |
  BOOLEAN { $$ = boolean_type; }
  |
  STRING  { $$ = string_type; }
  |
  CHARACTER  { $$ = character_type; }
  |
  C_TYPE  { $$ = c_type; }
  |
  IO OF type { $$ = new_io_type($3); }
  |
  LIST OF type { $$ = new_list_type($3); }
  |
  ARRAY OF type { $$ = new_array_type($3); }
  |
  PROMISE OF type { $$ = new_promise_type($3); }
  |
  type ARROW type { $$ = new_function_type($1,$3); }
  |
  type TIMES type
  {
    if(is_a_pair_type($1))
      $$ = append_to_tuple_type($1,$3);
    else
      $$ = new_pair_type($1,new_pair_type($3,tuple_terminator_type)); 
  }
  |
  generic { $$ = $1; }
  |
  OPEN_PAR type CLOSE_PAR { $$ = $2; }
  |
  abstract_type_name
  optional_actual_type_parameters_for_userdefined_type_with_of
  {
    type_t t=lookup_abstract_type($1.identifier,$2);
    
    if(t==-1)
      yyerror("mismatch in abstract type parameters");
    else if(t==-2)
      yyerror("type identifier not declared");

    $$=t;
  }
  |
  concrete_type_name
  optional_actual_type_parameters_for_userdefined_type_with_of
  {
    $$=instantiate_concrete_type($1.identifier, $2);
    if(is_error_type($$))
      yyerror("mismatch in concrete type parameters");
  }
  ;


concrete_type_name:
  CONCRETE_TYPE_NAME
  {
    strcpy($$.identifier, yytext);
  }
  ;

abstract_type_name:
  ABSTRACT_TYPE_NAME
  {
    strcpy($$.identifier, yytext);
  }
  ;

generic:
  GENERIC { $$ = generic_type(yytext); }
  ;

optional_formal_type_parameters_for_userdefined_type_with_of:
  /* nothing */
  {
    $$=tuple_terminator_type;
  }
  |
  OF generic
  optional_follow_of_formal_type_parameters_for_userdefined_type_without_of
  {
    $$ = new_tuple_type($2,$3);
  }
  ;

optional_follow_of_formal_type_parameters_for_userdefined_type_without_of:
  /* nothing */
  { $$ = tuple_terminator_type; }
  |
  COMMA generic
  optional_follow_of_formal_type_parameters_for_userdefined_type_without_of
  {
    $$ = new_tuple_type($2,$3);
  }
  ;

optional_actual_type_parameters_for_userdefined_type_with_of:
  /* nothing */
  {
    $$=tuple_terminator_type;
  }
  |
  OF type
  optional_follow_of_actual_type_parameters_for_userdefined_type_without_of
  {
    $$ = new_tuple_type($2,$3);
  }
  ;

actual_type_parameters_for_userdefined_type_without_of:
  OPEN_PAR actual_type_parameters_for_userdefined_type_without_of CLOSE_PAR
  {
    $$ = $2;
  }
  |
  type
  {
    $$ = $1;
  }
  optional_follow_of_actual_type_parameters_for_userdefined_type_without_of
  {
    $$ = new_tuple_type($$,$3);
  }
  ;

optional_follow_of_actual_type_parameters_for_userdefined_type_without_of:
  /* nothing */
  { $$ = tuple_terminator_type; }
  |
  COMMA type
  optional_follow_of_actual_type_parameters_for_userdefined_type_without_of
  {
    $$ = new_tuple_type($2,$3);
  }
  ;

///////////////////////////////////////////////////////////////////////////////////////////////

naming_or_declaration:
  DEFINE identifier_declaration
  {
    int unused; type_t unused_type;

    if(is_this_an_interface)
      yyerror("namings are not admitted in interfaces");
    
    if(is_qualified($2.identifier))
      yyerror("identifiers in definitions cannot be qualified");

    if(has_been_defined($2.identifier))
      yyerror("cannot override a previous global definition");

    strncpy(last_bound_identifier,$2.identifier,IDENTIFIER_LENGTH);
    last_bound_identifier[IDENTIFIER_LENGTH-1]='\0';
    output("# Naming of the global ");output($2.identifier);output(":\n");
    
    /* The 'main' global identifier is reserved to identify the main program: */
    if(! strcmp($2.identifier, "main"))
      is_this_module_main = 1;
  }
  EQUAL
  expression
  {
    int unused;
    type_t declared_type;
    lookup($2.identifier,&unused,&unused,&declared_type);
    
    //fprintf(stderr,"%i: \'%s\' IS DEFINED TO HAVE TYPE ",yylineno,$2.identifier);
    //output_type(stderr, $5.type);fprintf(stderr,";\n");
    
    if($2.order == POSTFIX_OPERATOR){
      if(!is_a_function_type($5.type))
	yyerror("postfix objects must be functions");
      if(bind_global_with_precedence(last_bound_identifier,
 		                     new_postfix_operator_type(element_type($5.type,0),
							       element_type($5.type,1)),
				     $2.precedence)!=0){
	fprintf(stderr, "There is a mismatch with the declaration: ");
	yytype_mismatch_error($5.type,declared_type);
      }
    }
    else if($2.order == INFIX_OPERATOR_LEFT){
      force_types_to_be_equal(
	$5.type,
	new_function_type(
          new_tuple_type(new_generic_type(),
			 new_tuple_type(new_generic_type(),
					tuple_terminator_type)),
	  new_generic_type()));
      force_types_to_be_equal(
        declared_type,
	new_infix_operator_left_type(
          element_of_tuple_type(element_type($5.type,0),1),
	  element_of_tuple_type(element_type($5.type,0),2),
	  element_type($5.type,1))); 

      if(bind_global_with_precedence(
           last_bound_identifier,
	   declared_type,
	   $2.precedence)!=0){
	fprintf(stderr, "There is a mismatch with the declaration: ");
        yytype_mismatch_error($5.type,declared_type);
      }
    }
    else if($2.order == INFIX_OPERATOR_RIGHT){
      force_types_to_be_equal(
	$5.type,
	new_function_type(
          new_tuple_type(new_generic_type(),
			 new_tuple_type(new_generic_type(),
					tuple_terminator_type)),
	  new_generic_type()));
      force_types_to_be_equal(
        declared_type,
	new_infix_operator_right_type(
          element_of_tuple_type(element_type($5.type,0),1),
	  element_of_tuple_type(element_type($5.type,0),2),
	  element_type($5.type,1))); 

      if(bind_global_with_precedence(
           last_bound_identifier,
	   declared_type,
	   $2.precedence)!=0){
	fprintf(stderr, "There is a mismatch with the declaration: ");
        yytype_mismatch_error($5.type,declared_type);
      }
    }
    else{ /* $2.order == IDENTIFER */
      if(is_type_more_general_than_or_equal_to($5.type, declared_type)){
        //force_types_to_be_equal($5.type, declared_type);
	force_types_to_be_equal(declared_type, $5.type);
	//yywarning("ok1");
	//fprintf(stderr,"{{");output_type(stderr,normalize_type($5.type));fprintf(stderr,"}}\n");
      }

      if(bind_global(last_bound_identifier,$5.type)!=0){
	fprintf(stderr, "There is a mismatch with the declaration: ");
        yytype_mismatch_error($5.type,declared_type);
      }
    }

    output("\ts_gl\t\"");output(qualify_with_this_module($2.identifier));output("\"\t# update global environment\n");

    /* The 'main' global identifier must have an i/o type: */
    if(! strcmp($2.identifier, "main"))
      force_types_to_be_equal($5.type, new_io_type(new_generic_type()));

    /* Output type if requested: */
    if(show_types){
      fprintf(stderr,"%s : ",$2.identifier);
      output_type(stderr,normalize_type($5.type));
      fprintf(stderr,"\n");
    }
  }
  SEMICOLON
  {
    output("\n");
  }
  | /* Constant declaration. */
  DECLARE
  {
    //fprintf(stderr,"\n<%s>",yytext);
  }
  identifier_declaration
  {
    //fprintf(stderr,"{%s}\n",$3.identifier);
    if(is_qualified($3.identifier))
      yyerror("declared identifiers cannot be qualified");
    //fprintf(stderr,"<%s>",yytext);
  }
  COLON 
  {
    //fprintf(stderr,"<%s>",yytext);
  }
  type
  {
    /* Ok, remember this identifier: */
    if($3.order==IDENTIFIER){
      if(declare_global($3.identifier,$7)!=0){
	if(has_been_defined($3.identifier)) 
	  yyerror("declaring an already defined identifier");
        else
          yyerror("re-declaring identifier");
      }
    }
    else{
      type_t t;
      
      if(!is_a_function_type($7))
	yyerror("operators must have function type");
      
      if((($3.order==INFIX_OPERATOR_LEFT)||($3.order==INFIX_OPERATOR_RIGHT))
	 &&(!is_a_pair_type(element_type($7,0))))
	yyerror("infix operators must be binary, of the form (X * Y) -> Z");
      
      switch($3.order){
        case POSTFIX_OPERATOR:{
	  t=new_postfix_operator_type(element_type($7,0),
				      element_type($7,1));
	  break;
        }
        case INFIX_OPERATOR_LEFT:{
	  t=new_infix_operator_left_type(element_of_tuple_type(element_type($7,0),1),
					 element_of_tuple_type(element_type($7,0),2),
					 element_type($7,1));
	  break;
        }
        case INFIX_OPERATOR_RIGHT:{
          t=new_infix_operator_right_type(element_of_tuple_type(element_type($7,0),1),
					  element_of_tuple_type(element_type($7,0),2),
					  element_type($7,1));
          break;
        }
        default:
	  yyerror("this cannot happen! ABCDE");
      } /* switch */

      if(declare_global_with_precedence($3.identifier,t,$3.precedence)!=0)
        yyerror("re-declaring identifier");
    }
    //fprintf(stderr,"Declared %s : ",$3.identifier);output_type(stderr,$7);fprintf(stderr,"\n");
  }
  SEMICOLON
  {
    //fprintf(stderr,"\n"); 
  }
  |
  DECLARE ABSTRACT TYPE identifier_or_operator
  optional_formal_type_parameters_for_userdefined_type_with_of
  {
    if(!is_this_an_interface)
      yyerror("type declarations can only occour in interfaces");
    if(is_qualified($4.identifier))
      yyerror("type name should not be qualified in declaration");
    
    if(create_undefined_abstract_type(qualify_with_this_module($4.identifier),$5)!=0)
      yyerror("this abstract type is already defined");

    //fprintf(stderr,"epsilon.y: --- fjkheskfnasdk ");
    //output_type(stderr,lookup_abstract_type($4.identifier,$5));
    //fprintf(stderr,"\n");
  }
  SEMICOLON
  |
  DECLARE CONCRETE TYPE identifier_or_operator
  optional_formal_type_parameters_for_userdefined_type_with_of
  SEMICOLON
  {
    if(is_qualified($4.identifier))
      yyerror("type name should not be qualified in declaration");

    if(create_undefined_concrete_type(qualify_with_this_module($4.identifier),$5, 1 /* only declared */)!=0)
      yyerror("this concrete type is already declared or defined");
    //yyerror("concrete types can be defined but not declared");
  }
  |
  DEFINE ABSTRACT
  {
    is_this_a_synonym_type = strcmp(yytext, "abstract");      
  }
  TYPE identifier_or_operator
  optional_formal_type_parameters_for_userdefined_type_with_of
  {
    if(is_this_an_interface && ! is_this_a_synonym_type)
      yyerror("abstract type definition can not occour in interfaces");
    if(is_qualified($5.identifier))
      yyerror("type name should not be qualified in definition");
  }
  EQUAL type
  {
    //fprintf(stderr,"PARAMETERS IN DEFINE: ");
    //output_type(stderr,$5);
    //fprintf(stderr,"\n");
    type_t parameters_arrow_definition;
    int return_code=define_abstract_type($5.identifier,
					 $6,
					 $9);
    if(return_code==-1)
      yyerror("can not define twice the abstract or synonym type");
    else if(return_code==-2)
      yyerror("definition is not compatible with declaration");
    else if((return_code==-3) && ! is_this_a_synonym_type){
      yywarning("abstract type is defined but not declared");
    }
    
    parameters_arrow_definition=normalize_type(new_function_type($6,$9));
    if(show_types){
      fprintf(stderr,"%s type %s",
	      is_this_a_synonym_type ? "synonym" : "abstract",
	      $5.identifier);
      if(!is_a_tuple_terminator_type(element_type(parameters_arrow_definition,0))){
	fprintf(stderr," of ");
	output_type(stderr,element_type(parameters_arrow_definition,0));
      }
      fprintf(stderr," = ");
      output_type(stderr,element_type(parameters_arrow_definition,1));
      fprintf(stderr,"\n");
    }
  }
  SEMICOLON  
  |
  DEFINE CONCRETE TYPE identifier_or_operator
  optional_formal_type_parameters_for_userdefined_type_with_of
  EQUAL
  {
    if(is_qualified($4.identifier))
      yyerror("type name should not be qualified in definition");
    if(create_undefined_concrete_type($4.identifier, $5, 0 /* declared and DEFINED */) == -1)
      yyerror("type already exists");

    strcpy(current_concrete_type_name,$4.identifier);
    if(show_types && !is_this_an_interface){
      fprintf(stderr,"concrete type %s",$4.identifier);

      if(!is_a_tuple_terminator_type($5)){
	fprintf(stderr," of ");
	output_type(stderr,$5);
      }
      fprintf(stderr," = ");
    }
  }
  concrete_type_case
  concrete_type_cases
  SEMICOLON
  {
    if(bind_concrete_type_name($4.identifier)!=0)
      yyerror("concrete type is already defined");
    else if(show_types && !is_this_an_interface)
      fprintf(stderr,"\n");
  }
  |
  DEFINE EXCEPTION identifier_or_operator optional_equal_type_for_exception_definition SEMICOLON
  {
    if(is_qualified($3.identifier))
      yyerror("identifiers must not be qualified in their definitions");
    if(is_an_exception_name(qualify_with_this_module($3.identifier)))
      yyerror("exception is already defined");
    
    create_exception($3.identifier,$4);
  }
  ;

optional_equal_type_for_exception_definition:
  /* nothing */
  {
    $$ = void_type;
  }
  |
  EQUAL type
  {
    $$ = $2;
  }
  ;
concrete_type_case:
  identifier_or_operator
  optional_concrete_type_case_definition_with_of
  {
    if(show_types && !is_this_an_interface){
      fprintf(stderr,"%s",$1.identifier);
      if(!is_a_tuple_terminator_type($2)){
	fprintf(stderr," of ");
	output_type(stderr,$2);
      }
    }
    
    if(is_qualified($1.identifier))
      yyerror("constructor name should not be qualified in definition");
    if(define_concrete_type_case(current_concrete_type_name,
				 $1.identifier,$2) == -1)
      yyerror("duplicate case in abstract type definition");
  }
  ;

concrete_type_cases:
  /* nothing */
  |
  PIPE
  {
    if(show_types && !is_this_an_interface)
      fprintf(stderr," | ");
  }
  concrete_type_case
  concrete_type_cases
  ;

optional_concrete_type_case_definition_with_of:
  OF type
  {
    $$=$2;
  }
  |
  /* nothing */
  {
    $$=tuple_terminator_type;
  }
  ;

namings_or_declarations:
  naming_or_declaration
  namings_or_declarations
  |
  /* nothing */
  ;

optional_main_expression:
  expression
  {
    if(is_this_an_interface)
      yyerror("expressions are not admitted in interfaces");

    if(is_error_type($1.type))
      yyerror("type error in expression");

    if(show_types){
      fprintf(stderr,"- : ");
      output_type(stderr, normalize_type($1.type));
      fprintf(stderr,"\n");
    }
    output("\n# Output the result:\n");
    output_output($1.type);
    /* Don't show a new-line if --unescaped-strings is set: */
    if(! unescaped_strings)
      output("\tpshcs\t\"\\n\"\n\ts_outs\n");
  }
  SEMICOLON
  |
  /* nothing */
  ;

module:
  namings_or_declarations
  {
    if(!is_this_an_interface)
      output("# Main expression:\nmain:\n");
  }
  optional_main_expression
  {
    if(!is_this_an_interface)
      if(are_there_undefined_but_declared_identifiers(current_module_name)){
	fprintf(stderr,"About the identifier '%s':\n",first_undefined_identifier(current_module_name));
	yyerror("an identifier was declared but not defined");
      }
    //dump_global_environment();
    if(is_this_module_main){
      output("\n# Call the main action:\n");
      output("\tpshgl\t\"");
      output(qualify_with_this_module("main"));
      output("\"\n");
      output("\ts_cll\t0\n\n");
    }
    if(!is_this_an_interface)
      output("\n# Jump to the end of the module:\n\tj\tEND_OF_MODULE:\n");
  }
  ;

%%

int already_output_boolean=0;
int last_output_array_label=0;
int last_output_list_label=0;

void output_output_for_concrete(type_t type, int index){
  int cases_no;
  char* case_name;
  type_t case_definition;
  int case_encoding;
  int i;

  push_buffer(new_buffer());
  output("OUTPUT_CONCRETE_");output_integer(index);output(":\n");
  
  cases_no=concrete_type_to_cases_number(type);
  for(i=0;i<cases_no;i++){
    concrete_type_to_nth_case(type,
			      i+1,
			      &case_name,
			      &case_encoding,
			      &case_definition);
    output("CONCRETE_TYPE_");output_integer(index);
    output("_CASE_");output_integer(i+1);output(":\n");
    output("\ts_lcl\t1\n\ts_lkp_i\t0\n");
    output("\tpshci\t");output_integer(case_encoding);output("\n\ts_eqi\n");
    output("\ts_jz\tCONCRETE_TYPE_");output_integer(index);output("_CASE_");
    output_integer(i+2);output(":\n");
    output("\tpshcs\t\"");output(bare_identifier(case_name));
    output("\"\n\ts_outs\n");

    if(!is_a_tuple_terminator_type(case_definition)){
      output("\tpshcs\t\"(\"\n\ts_outs\n");
      output("\ts_lcl\t1\n\ts_lkp_i\t1\n");
      output_output(case_definition);
      output("\tpshcs\t\")\"\n\ts_outs\n");
    }
    output("\ts_retn\n");
  }
  
  output("CONCRETE_TYPE_");output_integer(index);output("_CASE_");
  output_integer(i+1);output(":\n");
  output("\tpshcs\t\"This cannot happen\\n\"\n\ts_outs\n");
  output("\thlt\t-1\n");
  pop_buffer();
}

void output_output(type_t t){
  if(is_an_io_type(t)){
    output("\tpop\t\n\tpshcs\t\"<I/O action>\"\n\ts_outs\n");
  }
  else if(is_a_generic_type(t)){
    output("\t# This should not happen:\n");
    output("\tpop\t\n\tpshcs\t\"<generic>\"\n\ts_outs\n");
  }
  else if(is_a_function_type(t)){
    output("# Output a function object:\n\tpop\n");
    output("\tpshcs\t\"<function>\"\n");
    output("\ts_outs\n");
  }
  else if(is_a_promise_type(t)){
    output("# Output a function object:\n\tpop\n");
    output("\tpshcs\t\"<promise>\"\n");
    output("\ts_outs\n");
  }
  else if(is_a_pair_type(t)){
    type_t i,j=0;
    int length=arity_of_tuple_type(t);
    
    output("# Output a tuple object:\n");
    output("\tpshcs\t\"(\"\n");
    output("\ts_outs\n");
    
    for(i = t; ! is_a_tuple_terminator_type(i); i = element_type(i,1)){
      output("\tcpy\t\t# keep another reference to the tuple\n");
      output("# Output element #");output_integer(++j);output("\n");
      output("\ts_lkp_i\t");output_integer(j-1);output("\n");
      output_output(element_type(i,0));
      if(j+1<=length)
        output("\tpshcs\t\", \"\n\ts_outs\n");
    }
    output("\tpop\t\t# we don't need the tuple anymore.\n");
    output("\tpshcs\t\")\"\n");
    output("\ts_outs\n");
  }
  else if(is_an_array_type(t)){
    int label = ++last_output_array_label;

    output("# Output an array object:\n");
    output("\ts_barlt\t\t# convert into list\n");

    output("\tpshcs\t\"<| \"\n\ts_outs\n");

    output("OUTPUT_ARRAY_");output_integer(label);output(":\n");
    output("\tcpy\n");
    output("\ts_nll\n");
    output("\ts_jnz\tOUTPUT_ARRAY_OUT_");output_integer(label);output(":\n");
    output("\tcpy\n\ts_car\n");
    output_output(element_type(t,0));
    output("\tpshcs\t\"; \"\n\ts_outs\n");
    output("\ts_cdr\n");
    output("\tj\tOUTPUT_ARRAY_");output_integer(label);output(":\n");
    output("OUTPUT_ARRAY_OUT_");output_integer(label);output(":\n");
    output("\tpshcs\t\"|>\"\n\ts_outs\n");
    output("\tpop\n");
  }
  else if(is_a_list_type(t)){
    int label=++last_output_list_label;

    output("# Output a list object:\n");
    output("\tpshcs\t\"[ \"\n\ts_outs\n");
    output("OUTPUT_LIST_");output_integer(label);output(":\n");
    output("\tcpy\n");
    output("\ts_nll\n");
    output("\ts_jnz\tOUTPUT_LIST_OUT_");output_integer(label);output(":\n");
    output("\tcpy\n\ts_car\n");
    output_output(element_type(t,0));
    output("\tpshcs\t\"; \"\n\ts_outs\n");
    output("\ts_cdr\n");
    output("\tj\tOUTPUT_LIST_");output_integer(label);output(":\n");
    output("OUTPUT_LIST_OUT_");output_integer(label);output(":\n");
    output("\tpshcs\t\"]\"\n\ts_outs\n");
    output("\tpop\n");
  }
  else if(is_a_boolean_type(t)){
    output("# Output a boolean object:\n");
    output("\ts_cls\tOUTPUT_BOOLEAN:\n");
    output("\ts_swp\t\t# swap argument and closure\n");
    output("\ts_cll\t1\n");
    if(!already_output_boolean){
      push_buffer(new_buffer());
      output("\nOUTPUT_BOOLEAN:\n");
      output("\ts_lcl\t1\n");
      output("\ts_jnz\tOUTPUT_TRUE:\n");
      output("\tpshcs\t\"false\"\n");
      output("\tj\tEND_OF_OUTPUT_BOOLEAN:\n");
      output("OUTPUT_TRUE:\n");
      output("\tpshcs\t\"true\"\n");
      output("END_OF_OUTPUT_BOOLEAN:\n");
      output("\ts_outs\n");
      output("\ts_retn\n\n");
      pop_buffer();    
      already_output_boolean=1;
    }
  }
  else if(is_a_string_type(t)){
    if(unescaped_strings){
      output("# Output a string object (unescaped):\n");
      output("\ts_outs\n");
    }
    else{
      output("# Output a string object:\n");
      output("\ts_outes\n");
    }
  }
  else if(is_a_character_type(t)){
    if(unescaped_strings){
      output("# Output a character object (unescaped):\n");
      output("\ts_outc\n");
    }
    else{
      output("# Output a character object:\n");
      output("\ts_outec\n");
    }
  }
  else if(is_a_c_type(t)){
    output("# Output a C object:\n");
    output("\tpshcs\t\"<C>\"\n");
    output("\ts_outs\n");
  }
  else if(is_an_integer_type(t)){
    output("# Output an integer object:\n");
    output("\ts_outi\n");
  }
  else if(is_a_float_type(t)){
    output("# Output a float object:\n");
    output("\ts_outf\n");
  }
  else if(is_a_void_type(t)){
    output("# Output a void object:\n");
    output("\tpop\n");
    output("\tpshcs\t\"()\"\n");
    output("\ts_outs\n");
  }
  else if(is_an_abstract_type(t)){
    output("\t# Output an abstract type:\n");
    output("\tpop\n\tpshcs\t\"<abstract>\"\n\ts_outs\n");
  }
  else if(is_a_concrete_type(t)){
    int index_of_t=concrete_type_output_procedure_to_code(t);

    if(index_of_t==0){
      index_of_t=memoize_output_procedure_for_concrete_type(t);
      output_output_for_concrete(t,index_of_t);
    }

    output("\ts_cls\tOUTPUT_CONCRETE_");output_integer(index_of_t);output(":\n");
    output("\ts_swp\n\ts_cll\t1\n");
  }
  else{
    fprintf(stderr,"epsilon.y: This cannot happen! for type ");
    output_type(stderr,t);fprintf(stderr,". 234234\n");
    exit(EXIT_FAILURE);
  }  
  /*
  else
    output("\toutf\t\t# output a primitive object\n");
  */
}

char* current_module_name_with_extension(){
  static char r[IDENTIFIER_LENGTH+10];
  if(is_this_an_interface)
    sprintf(r,"%s.epi",current_module_name);
  else
    sprintf(r,"%s.epb",current_module_name);

  return r;
}

void output_current_point(){
  output("\"");output(current_module_name);output(".epb:");
  output_integer(yylineno);output("\"");
}

void make_current_point_failure(){
  output("\tsetfp\t");
  output_current_point();
  output("\n");
}

void output_action_beginning(){
  output("\ts_cls\tIO_");output_integer(current_io_label);output(":\t# closure for the action\n");
  push_buffer(new_buffer());
  output("\n# Code for action #");output_integer(current_io_label);output(":\n");
  output("IO_");output_integer(current_io_label++);output(":\n");
  push_environment(bind_not_refreshable(top_environment(),""/* a dummy identifier */,0,type_error /* not used */));
}

void output_action_end(){
  output("\ts_retn\t\t# end of action\n");
  pop_environment();
  pop_buffer();
}

void yywarning(char* message){
  fprintf(stderr,"%s:%i: warning: %s near \'%s\'\n",current_module_name_with_extension(),yylineno,message,yytext);
}

int yyerror(char* message){
  fprintf(stderr,"%s:%i: %s near \'%s\'\n",current_module_name_with_extension(),yylineno,message,yytext);
  exit(EXIT_FAILURE);
}

int yytype_mismatch_error(type_t actual, type_t attended){
  fprintf(stderr,"%s:%i: type is\n",current_module_name_with_extension(),yylineno);output_type(stderr,normalize_type(actual));
  fprintf(stderr,"\ninstead of\n");output_type(stderr,normalize_type(attended));
  fprintf(stderr,"\nnear \'%s\'\n",yytext);
  exit(EXIT_FAILURE);
}

void output_integer(int x){
  char s[10];
  sprintf(s,"%i",x);
  output(s);
}

void output(const char* s){
  write_to_buffer(top_buffer(),s);
}

type_t force_type_to_be(type_t t1, type_t t2){
  //fprintf(stderr,"Forcing ");output_type(stderr,t1);
  //fprintf(stderr," to be ");output_type(stderr,t2);
  //fprintf(stderr," ===> ");
  
  make_type_unify_with_type(t1,t2);
  if(is_error_type(t1))
    //yyerror("could not unify t1 with t2");
    yytype_mismatch_error(t2,t1);
  
  //output_type(stderr,t1); fprintf(stderr,"\n");
  
  return t1;
}

type_t force_types_to_be_equal(type_t t1, type_t t2){
  type_t t=unify_types(t1,t2);
  
  if(is_error_type(t))
    //yyerror("could not make equal t1 and t2");
    yytype_mismatch_error(t1,t2);
  
  force_type_to_be(t1,t);
  force_type_to_be(t2,t);
  
  return t;
}
