/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002, 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include "epsilon.h"
#include "types.h"
#include "environment.h"
#include "multibuffer.h"
#include "../commontext.h"
#include "../common/command_line.h"
#include "../common/search_path.h"
#include "prefixes.h"
#include "../common/malloc.h"

#include "importedmodules.h"

extern FILE *yyin;
extern FILE *yyout;

int unescaped_strings;
int verbose;
int show_types;

search_path_t interfaces_search_path;
search_path_t modules_search_path;
search_path_t libraries_search_path;


int epb_arguments_no = 0;
char **epb_arguments;
int eaml_arguments_no = 0;
char **eaml_arguments;
int eamo_arguments_no = 0;
char **eamo_arguments;
int eamx_arguments_no = 0;
char **eamx_arguments;
int eama_arguments_no = 0;
char **eama_arguments;

/* For each argument copy (a pointer to) it into a suitable array
   based on its extension, and increment the variable holding the
   dimension of the array.  Fail if extension is unknown. */
void classify_arguments_by_extension(int argc, char** argv){
  int i;
  
  /* Allocate arrays: */
  epb_arguments = malloc(sizeof(char*) * (argc - 1));
  eaml_arguments = malloc(sizeof(char*) * (argc - 1));
  eamo_arguments = malloc(sizeof(char*) * (argc - 1));
  eamx_arguments = malloc(sizeof(char*) * (argc - 1));
  eama_arguments = malloc(sizeof(char*) * (argc)); /* one more is needed
						      for default library */
  /* Add the default library if needed: */
  if(long_option_to_value("use-library")
     && (! long_option_to_value("generate-eaml"))
     && (! long_option_to_value("generate-eamo"))
     && (! long_option_to_value("generate-eama")))
    eama_arguments[eama_arguments_no ++] =
      search_for_file_or_fail("epsilon.eama", libraries_search_path);
  
  /* For each element of argv, copy it (a pointer) into the
     suitable array and increment the array size, or fail
     if the extension is not known: */
  for(i = 1; i < argc; i++)
    if (has_extension(argv[i], "epb"))
      epb_arguments[epb_arguments_no ++] = argv[i];
    else if (has_extension(argv[i], "eaml"))
      eaml_arguments[eaml_arguments_no ++] = argv[i];
    else if (has_extension(argv[i], "eamo"))
      eamo_arguments[eamo_arguments_no ++] = 
	search_for_file_or_fail(argv[i], modules_search_path);
    else if (has_extension(argv[i], "eamx"))
      eamx_arguments[eamx_arguments_no ++] = argv[i];
    else if (has_extension(argv[i], "eama"))
      eama_arguments[eama_arguments_no ++] =
	search_for_file_or_fail(argv[i], libraries_search_path);
    else if (has_extension(argv[i], "epi")){
      fprintf(stderr, "About the argument %s:\n", argv[i]);
      command_line_fatal(".epi files should not be included");
    }
    else{
      fprintf(stderr, "About the argument %s:\n", argv[i]);
      command_line_fatal("Unrecognized extension");
    }

  /* Do some additional consistency checks which depend upon classification: */

  /* If an output file was given then check its extension: */
  if(strcmp(long_option_to_value("output"), "")){ /* --output was specified*/
    char* right_extension = NULL;

    if(long_option_to_value("generate-eaml"))
      right_extension = "eaml";
    else if(long_option_to_value("generate-eamo"))
      right_extension = "eamo";
    else if(long_option_to_value("generate-eamx"))
      right_extension = "eamx";
    else if(long_option_to_value("generate-eama"))
      right_extension = "eama";
    else if(long_option_to_value("generate-c"))
      right_extension = "c";
    else if(long_option_to_value("generate-scheme"))
      right_extension = "scm";
    else /* Generate native code */
      if(has_some_extension(long_option_to_value("output")))
	command_line_fatal("Native executables should not have extension");

    /* If we arrived here, then we are not generating native code OR the 
       given extension is correct */
    if(right_extension != NULL) /* don't check if we're generating native */
      if(! has_extension(long_option_to_value("output"),
			 right_extension)){
	fprintf(stderr, "The output file name %s should have extension %s.\n",
		long_option_to_value("output"),
		right_extension);
	command_line_fatal("Extension is not correct");
      }
  }
  
  if(long_option_to_value("generate-eama") &&
     strcmp(long_option_to_value("main"), ""))
    command_line_fatal("--main, -m is incompatible with --generate-eama, -a");

  /* The parameter of --main, -m must not have extension: */
  if(has_some_extension(long_option_to_value("main")))
    command_line_fatal("The parameter of main must be a module, not a file name");

  /* Set global variables holding the value of some commonly used options, to save
     lookups later: */
  unescaped_strings = (int) long_option_to_value("unescaped-strings");
  verbose = (int) long_option_to_value("verbose");
  show_types = (int) long_option_to_value("show-types");
}

/* Check consistency of command-line arguments: */
void check_command_line(int argc, char** argv){
  int number_of_mutually_exclusive_options = 0;

  number_of_mutually_exclusive_options += (int) long_option_to_value("generate-eaml");
  number_of_mutually_exclusive_options += (int) long_option_to_value("generate-eamo");
  number_of_mutually_exclusive_options += (int) long_option_to_value("generate-eamx");
  number_of_mutually_exclusive_options += (int) long_option_to_value("generate-c");
  number_of_mutually_exclusive_options += (int) long_option_to_value("generate-scheme");
  number_of_mutually_exclusive_options += (int) long_option_to_value("generate-eama");
  if(number_of_mutually_exclusive_options > 1)
    command_line_fatal("You can't supply more than one --generate-* option");
  if(argc < 2)
    command_line_fatal("At least an argument is required");

  if(! long_option_to_value("use-library")){
    if(long_option_to_value("generate-eaml")
       || long_option_to_value("generate-eamo")
       )
    command_line_fatal("no eamx is created: can't use --no-use-library");
  }
}

void compile_epsilon_body_in_this_process(char* filename){
  FILE *f;
  char *output_file_name;
  
  if(verbose)
    fprintf(stderr, "Compiling %s...\n", filename);
  if ((yyin = fopen (filename, "r")) == NULL)
    {
      fprintf (stderr, "Could not load %s\n", filename);
      exit (EXIT_FAILURE);
    }
  output_file_name = input_file_name_to_output_name (filename);
  if ((f = fopen (output_file_name, "w")) == NULL)
    {
      fprintf (stderr, "Could not open %s for output.\n", output_file_name);
      exit (EXIT_FAILURE);
    }

  initialize_types_table ();
  init_multibuffer ();
  create_environment_structures ();
  push_buffer (new_buffer ());
  create_modules_table ();

#ifdef DEBUG
  dump_modules_table ();
#endif

  if (load_modules (filename) != 0)
    {
      fprintf(stderr,"Failed loading modules for %s.\n", filename);
      exit (EXIT_FAILURE);
    }

  destroy_modules_table ();
  free_environment_structures ();
  push_buffer (new_buffer ());
  output ("\nEND_OF_MODULE:\n\tnop\n\n");

  flush_multibuffer (f);
  free_multibuffer ();
  destroy_types_table ();

  fclose (yyin);
  fclose (f);
}

void process_files_in_a_subprocess(char* command,
				   char* options, /* to be passed to command */
				   char* filenames /* separated by spaces */){
  char *command_line; /* The command to execute, with arguments */
  int length_of_command = strlen(command);
  int length_of_options = strlen(options);

  /* Allocate and fill the buffer for the command line: */
  command_line = (char*)
    malloc(sizeof(char) * 
	   (length_of_command + 3 + length_of_options + strlen(filenames)));
  strcpy(command_line, command);
  command_line[length_of_command] = ' ';
  strcpy(command_line + length_of_command + 1, options);
  command_line[length_of_command + 1 + length_of_options] = ' ';
  strcpy(command_line + length_of_command + 2 + length_of_options,
	 filenames);

  if(verbose)
    fprintf(stderr, "  Executing '%s'...\n", command_line);

  /* Execute the command line: */
  if(system(command_line) != EXIT_SUCCESS)
    {fprintf(stderr, "Command-line %s:\n", command_line);fatal("Failed");}
    //exit(EXIT_FAILURE); /* The error message was written by the subprocess */

  /* Free the buffer: */
  free(command_line);
}

void compile_epsilon_body_in_a_subprocess(char* filename){
  char * options;

  /* Allocate and build the options string: */
  options = (char*) 
    malloc(sizeof(char) * 
	   (200 + 
	    strlen(long_option_to_value("modules-path")) + 4 + 
	    strlen(long_option_to_value("libraries-path")) + 4 + 
	    strlen(long_option_to_value("interfaces-path")) + 4));
  sprintf(options, "--generate-eaml %s --interfaces-path=%s --modules-path=%s --libraries-path=%s %s",
	  show_types ? "--show-types" : "--no-show-types",
	  long_option_to_value("interfaces-path"),
	  long_option_to_value("modules-path"),
	  long_option_to_value("libraries-path"),
          long_option_to_value("use-prelude") ? "--use-prelude" : "--no-use-prelude");
  /* Call the subprocess: */
  process_files_in_a_subprocess(
      "epsilonc", options, filename);
  /* Free the heap-allocated options string:*/
  free(options);
}

void assemble_eaml_in_a_subprocess(char* filename){
  if(verbose)
    fprintf(stderr, "Assembling %s...\n", filename);
  process_files_in_a_subprocess("eamlas", "", filename);
}

void peephole_optimize_in_a_subprocess(char* filename){
  if(verbose)
    fprintf(stderr, "Making peephole optimizations in %s...\n", filename);
  process_files_in_a_subprocess("eampo", verbose ? "--verbose" : "", filename);
}

/* Returns the first non-library input file which is assumed to be the main
   module (returns a pointer to a static buffer), or the parameter of
   --main, -m; returns NULL if no module is found in the command-line or in
   --main, -m: */
char* find_the_main_module(int argc, char** argv){
  static char* buffer;
  char* r = NULL;
  char* parameter_of_main = long_option_to_value("main");
  int i;

  if(strcmp(parameter_of_main, ""))
    r = parameter_of_main;
  else /* --main, -m was not specified */
    for(i = 1; i < argc; i++)
      if(! has_extension(argv[i], "eama")){
	/* remove extension: */
	r = replace_extension_with_part_of_filename(argv[i], "");
	break;
      }
  
  if(r == NULL){ /* no non-library parameter was found */
    return NULL;
  }
  
  buffer = xmalloc(sizeof(char) * (strlen(r) + 1));
  strcpy(buffer, r);
    
  return buffer;
}

/* Returns the output file name for the linker, as specified by command-line options,
   or the default name (returns a pointer to a static buffer): */
char* compute_linked_file(){
  char* command_line_name = long_option_to_value("output");
  char* extension;
  char* r;
  static char* buffer;

  if(long_option_to_value("generate-eama"))
    extension = "eama";
  else
    extension = "eamx";
  if(! strcmp(command_line_name, "")) /* No name passed on the command line */
    r = replace_extension("a.XXXX", extension);
  else
    r = replace_extension(command_line_name, extension);
  buffer = xmalloc(sizeof(char) * (strlen(r) + 1));
  strcpy(buffer, r);
  return buffer;
}

void compile_epsilon_bodies(){
  int i;
  
  /* If we have more than one .epb to compile, then compile each of
     them in a separate subprocess, else compile the only .epb here
     in this process: */
  if (epb_arguments_no > 1)
    for(i = 0; i < epb_arguments_no; i++)
      compile_epsilon_body_in_a_subprocess(epb_arguments[i]);
  else if (epb_arguments_no == 1)
    compile_epsilon_body_in_this_process(epb_arguments[0]);
  /* else epb_arguments_no == 0, do nothing */
}

void assemble_eamls(){
  int i;
  
  /* Assemble the .eaml files generated from the .epb files: */
  for(i = 0; i < epb_arguments_no; i++)
    assemble_eaml_in_a_subprocess(replace_extension(epb_arguments[i], "eaml"));

  /* Assemble the .eaml files on the command line: */
  for(i = 0; i < eaml_arguments_no; i++)
    assemble_eaml_in_a_subprocess(eaml_arguments[i]);
}

void make_peephole_optimizations(){
  int i;
  
  /* Make peephole optimizations in .epb modules: */
  for(i = 0; i < epb_arguments_no; i++)
    peephole_optimize_in_a_subprocess(replace_extension(epb_arguments[i], "eamo"));
  
  /* Make peephole optimizations in .eaml modules: */
  for(i = 0; i < eaml_arguments_no; i++)
    peephole_optimize_in_a_subprocess(replace_extension(eaml_arguments[i], "eamo"));

  /* Make peephole optimizations in .eamo modules: */
  for(i = 0; i < eamo_arguments_no; i++)
    peephole_optimize_in_a_subprocess(eamo_arguments[i]);
}

/* Files are linked in a subprocess; files are processed all at once, so the
   common ACTION_EXTENSION_in_a_subprocess() scheme can not be used here. */
void link_eamos_and_eamas(const char* linker_option, int argc, char** argv){
  char* command_line;
  int i, offset;
  int size_of_command_line; /* in char's */
  char* main_module = find_the_main_module(argc, argv);
  int length_of_main_module = main_module ? strlen(main_module) : 0;
  char* output_file = compute_linked_file();
  int length_of_output_file = strlen(output_file);
  
  if((main_module == NULL) && (!strcmp(linker_option, "")))
    command_line_fatal("A main module is needed when not making a library");
  
  #define LINKER_COMMAND_WITH_SPACE "eamld "
  
  if(verbose)
    fprintf(stderr, "Linking eAM modules and libraries...\n");

  /* Compute the size of command_line; we must link all supplied files,
     *.epb, *.eaml, *.eamo and *.eama; we must pass linked_option, 
     "-m" main_module, and a whitespace between each pair of adjacent
     parameters/options: */
  size_of_command_line = strlen(LINKER_COMMAND_WITH_SPACE) 
    + 1 /* for the trailing '\0' */;
  size_of_command_line += strlen(linker_option) + 1;
  for(i = 0; i < epb_arguments_no; i++)
    size_of_command_line += 2 /* ".eamo" is longer than ".epb" */
      + strlen(epb_arguments[i]);
  for(i = 0; i < eaml_arguments_no; i++)
    size_of_command_line += 1 + strlen(eaml_arguments[i]);
  for(i = 0; i < eamo_arguments_no; i++)
    size_of_command_line += 1 + strlen(eamo_arguments[i]);
  for(i = 0; i < eama_arguments_no; i++)
    size_of_command_line += 1 + strlen(eama_arguments[i]);
  size_of_command_line += sizeof("-m") + length_of_main_module + 1;
  size_of_command_line += sizeof("-o") + length_of_output_file + 1;
  
  /* Allocate command_line: */
  command_line = (char*)
    xmalloc(sizeof(char) * size_of_command_line);
  
  /* Fill the command_line: */
  offset = 0;
  strcpy(command_line, LINKER_COMMAND_WITH_SPACE);
  offset += strlen(LINKER_COMMAND_WITH_SPACE); /* "eamld" */
  strcpy(command_line + offset, linker_option);
  offset += strlen(linker_option); /* "--library" or nothing */
  strcpy(command_line + offset, " ");
  offset++; /* " " */
  for(i = 0; i < epb_arguments_no; i++){
    char* argument = replace_extension(epb_arguments[i], "eamo");
    strcpy(command_line + offset, argument);
    offset += strlen(argument);
    strcpy(command_line + offset, " "); offset++;
  }
  for(i = 0; i < eaml_arguments_no; i++){
    char* argument = replace_extension(eaml_arguments[i], "eamo");
    strcpy(command_line + offset, argument);
    offset += strlen(argument);
    strcpy(command_line + offset, " "); offset++;
  }
  for(i = 0; i < eamo_arguments_no; i++){
    char* argument = eamo_arguments[i];
    strcpy(command_line + offset, argument);
    offset += strlen(argument);
    strcpy(command_line + offset, " "); offset++;
  }
  for(i = 0; i < eama_arguments_no; i++){
    char* argument = eama_arguments[i];
    strcpy(command_line + offset, argument);
    offset += strlen(argument);
    strcpy(command_line + offset, " "); offset++;
  }
  if(! strcmp(linker_option, "")){ /* No --library implies --main */
    strcpy(command_line + offset, "-m"); offset+=2;
    strcpy(command_line + offset, main_module);
    offset += length_of_main_module;
    strcpy(command_line + offset, " "); offset++;
  }
  strcpy(command_line + offset, "-o"); offset+=2;
  strcpy(command_line + offset, output_file);
  offset += length_of_output_file;

  if(verbose)
    fprintf(stderr, "  Executing '%s'...\n", command_line);
  
  /* Execute the command line: */
  if(system(command_line) != EXIT_SUCCESS){
    fprintf(stderr, "About the command-line %s:\n", command_line); fatal("Failed");
  }
  
  /* Free the allocated string: */
  free(command_line);
}

void generate_c(){
  char* eamx_file_name = compute_linked_file();
  
  if(verbose)
    fprintf(stderr, "Generating C code from %s...\n", eamx_file_name);
  process_files_in_a_subprocess("eamx2c", "", eamx_file_name);
}

void generate_native(){
  char* c_file_name = replace_extension(compute_linked_file(), "c");
  char* arguments;
  char* output_file_name = long_option_to_value("output");
  if(verbose)
    fprintf(stderr, "Compiling the C source %s...\n", c_file_name);
  
  /* Arguments actually contains also some options; this trick is needed since
     the order of arguments/options is important when calling a C compiler: */
  arguments = (char*) /* we don't know the size of the buffer in advance */
    xmalloc(sizeof(char) * (strlen(C_LIBS) + strlen(c_file_name) +
			    strlen(output_file_name) + 30));
  sprintf(arguments,
	  "-o %s %s -lepsilon %s",
	  strlen(output_file_name) ? output_file_name : "a.out" ,
	  c_file_name,
	  C_LIBS);
  process_files_in_a_subprocess(C_CC,
                                C_CFLAGS" "
                                " -L"EPSILON_C_LIBRARIES_PATH
                                " -I"EPSILON_C_HEADERS_PATH,
                                arguments);
  free(arguments); /* release storage for the buffer */
}

void generate_scheme(){
  char* eamx_file_name = compute_linked_file();
  
  if(verbose)
    fprintf(stderr, "Generating Scheme code from %s...\n", eamx_file_name);
  process_files_in_a_subprocess("eamx2scheme", "", eamx_file_name);
}

void create_search_paths(){
  /* Create and initialize structures: */
  interfaces_search_path = create_search_path();
  modules_search_path = create_search_path();
  libraries_search_path = create_search_path();
  
  /* Prepend user-supplied search paths: */
  append_path_to_search_path(long_option_to_value("interfaces-path"),
			     interfaces_search_path);
  append_path_to_search_path(long_option_to_value("modules-path"),
			     modules_search_path);
  append_path_to_search_path(long_option_to_value("libraries-path"),
			     libraries_search_path);

  /* Append default search paths, if not forbidden by 
     --no-use-default-paths: */
  if(long_option_to_value("use-default-paths")){
    append_path_to_search_path(EPSILON_INTERFACES_PATH ":.",
			       interfaces_search_path);
    append_path_to_search_path(".",
			       modules_search_path);
    append_path_to_search_path(EPSILON_LIBRARIES_PATH ":.",
			       libraries_search_path);
  }
  else{
    /* The user supplied --no-use-default-paths: use at least the 
       current directory: */
    append_path_to_search_path(".",
			       interfaces_search_path);
    append_path_to_search_path(".",
			       modules_search_path);
    append_path_to_search_path(".",
			       libraries_search_path);
  }
}

int
main (int argc, char **argv)
{
  /* Define command-line parameters: */
  set_program_name ("epsilonc (" PACKAGE_NAME ")");
  set_general_help_message
    ("Compiles the given files.");
  set_synopsis_string ("epsilonc [OPTIONS] FILE1 ... FILEn");
  set_version_string (VERSION_STRING);
  set_copyright_string (COPYRIGHT_STRING);
  set_license_message (LICENSE_STRING);
  set_bug_reporting_message (BUG_REPORTING_MESSAGE);
  
  /* Define command line options: */
  add_toggle_option ("verbose", 'v', 0,
                     "Be verbose while processing");
  add_toggle_option ("show-types", 't', 1,
                     "Show inferred and user-defined types");
  add_toggle_option ("unescaped-strings", 's', 0,
		     "Show strings without quotes and escapes");
  add_toggle_option ("generate-eaml", 'S', 0,
		     "Stop before assembling");
  add_toggle_option ("generate-eamo", 'c', 0,
                     "Stop before linking");
  add_toggle_option ("generate-eamx", 'x', 0,
                     "Stop before generating C code");
  add_toggle_option ("generate-c", 'C', 0,
                     "Stop before compiling C code");
  add_toggle_option ("generate-scheme", '\0', 0,
                     "Generate Scheme instead of C code");
  add_toggle_option ("generate-eama", 'a', 0,
                     "Generate an eAM library");
  add_toggle_option("peephole", 'p', 1,
		    "Make peephole optimizations");
  add_toggle_option("use-library", '\0', 1,
		    "Link with the default epsilon library");
  add_toggle_option("use-prelude", '\0', 1,
		    "Import the default prelude interface");
  add_option_with_nonoptional_parameter ("cc-options", '\0', "",
					 "Set options to pass to the C compiler");
  add_option_with_nonoptional_parameter ("main", 'm', "",
					 "Specify which module is the main one");
  add_option_with_nonoptional_parameter ("output", 'o', "",
					 "Set the output file name");
  add_option_with_nonoptional_parameter("interfaces-path", 'I', "",
					"Prepend XXXX to the directories to search");
  add_option_with_nonoptional_parameter("libraries-path", 'L', "",
					"Prepend XXXX to the directories to search");
  add_option_with_nonoptional_parameter("modules-path", 'M', "",
					"Prepend XXXX to the directories to search");
  add_toggle_option("use-default-paths", '\0', 1,
		    "Search default directories for interfaces, etc.");

  /* Parse command line and check consistency: */
  parse_command_line (&argc, argv);
  check_command_line (argc, argv);

  /* Everything is ok if we got here, proceed: */
  
  /* Create search paths; they are used in the following phases. */
  create_search_paths();

  /* Classify command line arguments: */
  classify_arguments_by_extension(argc, argv);
  
  /* Compile: */
  compile_epsilon_bodies();
  /* Stop here on --generate-eaml: */
  if(long_option_to_value("generate-eaml"))
    return EXIT_SUCCESS;
  
  /* Assemble: */
  assemble_eamls();
  
  /* Make peephole optimizations, if requested: */
  if(long_option_to_value("peephole"))
    make_peephole_optimizations();
  /* Stop here on --generate-eamo: */
  if(long_option_to_value("generate-eamo"))
    return EXIT_SUCCESS;

  /* Link: */
  if(long_option_to_value("generate-eama")){
    link_eamos_and_eamas("--library" /* make an eama */, argc, argv);
    /* If we are generating an eama then we can stop here. */
    return EXIT_SUCCESS;
  }
  else
    link_eamos_and_eamas("" /* make an eamx */, argc, argv);
  /* Stop here on --generate-eamx: */
  if(long_option_to_value("generate-eamx"))
    return EXIT_SUCCESS;

  /* Generate C or Scheme: */
  if(long_option_to_value("generate-scheme")){
    generate_scheme();
    return EXIT_SUCCESS;
  }
  generate_c();
  /* Stop here on --generate-c: */
  if(long_option_to_value("generate-c"))
    return EXIT_SUCCESS;

  /* Compile C: */
  generate_native();

  /* Done */
  return EXIT_SUCCESS;
}
