/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002, 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

#include "environment.h"
#include <string.h>
#include <stdlib.h>

#include "types.h"


#define INITIAL_ENVIRONMENT_STACK_SIZE   64
#define INITIAL_ENVIRONMENTS_STACK_SIZE  64
#define INITIAL_BINDINGS_STACK_SIZE      64

//--------------------------------------------------------------

#define INITIAL_EXCEPTION_HANDLING_STACK_SIZE      64

struct exception_handling
{
  int no;
  int cases_no;
}
 *exception_handling_stack;

int exception_handling_next_position;
int exception_handling_current_size;
int exception_handling_counter = 1;

void
create_exception_handling_stack ()
{
  exception_handling_current_size = INITIAL_EXCEPTION_HANDLING_STACK_SIZE;
  exception_handling_stack = (struct exception_handling *)
    malloc (sizeof (struct exception_handling) *
	    exception_handling_current_size);
  exception_handling_next_position = 0;
}

void
destroy_exception_handling_stack ()
{
  free (exception_handling_stack);
}

int
push_exception_handling ()
{
  if (exception_handling_next_position == exception_handling_current_size)
    {
      int i;

      exception_handling_current_size *= 2;
      exception_handling_stack =
	(struct exception_handling *) realloc (exception_handling_stack,
					       exception_handling_current_size
					       *
					       sizeof (struct
						       exception_handling));
    }

  exception_handling_stack[exception_handling_next_position].no =
    exception_handling_counter++;
  exception_handling_stack[exception_handling_next_position].cases_no = 0;
  exception_handling_next_position++;

  return current_exception_handling ();
}

void
increment_exception_handling_cases_number ()
{
  exception_handling_stack[exception_handling_next_position - 1].cases_no++;
}

int
current_exception_handling_cases_number ()
{
  return exception_handling_stack[exception_handling_next_position -
				  1].cases_no;
}

void
pop_exception_handling ()
{
  exception_handling_next_position--;
}

int
current_exception_handling ()
{
  return exception_handling_stack[exception_handling_next_position - 1].no;
}

//--------------------------------------------------------------

#define INITIAL_PATTERN_MATCHING_STACK_SIZE      64

struct pattern_matching
{
  int no;
  int case_no;			/* last case which was dealt with using pattern matching */
  int cases_no;
  int *cases;
}
 *pattern_matching_stack;

int pattern_matching_next_position;
int pattern_matching_current_size;
int pattern_matching_counter = 1;

void
create_pattern_matching_stack ()
{
  pattern_matching_current_size = INITIAL_PATTERN_MATCHING_STACK_SIZE;
  pattern_matching_stack = (struct pattern_matching *)
    malloc (sizeof (struct pattern_matching) * pattern_matching_current_size);
  pattern_matching_next_position = 0;
}

void
destroy_pattern_matching_stack ()
{
  int i;
  for (i = 0; i < pattern_matching_next_position; i++)
    free (pattern_matching_stack[i].cases);
  free (pattern_matching_stack);
}

void
push_pattern_matching ()
{
  if (pattern_matching_next_position == pattern_matching_current_size)
    {
      int i;

      pattern_matching_current_size *= 2;
      pattern_matching_stack =
	(struct pattern_matching *) realloc (pattern_matching_stack,
					     pattern_matching_current_size *
					     sizeof (struct
						     pattern_matching));
    }

  pattern_matching_stack[pattern_matching_next_position].no =
    pattern_matching_counter++;
  pattern_matching_stack[pattern_matching_next_position].case_no = 1;
  pattern_matching_stack[pattern_matching_next_position].cases_no = 0;
  pattern_matching_stack[pattern_matching_next_position++].cases = NULL;
}

void
increment_pattern_matching_case_counter ()
{
  pattern_matching_stack[pattern_matching_next_position - 1].case_no++;
}

int
current_pattern_matching_case_counter ()
{
  return pattern_matching_stack[pattern_matching_next_position - 1].case_no;
}

void
pop_pattern_matching ()
{
  free (pattern_matching_stack[--pattern_matching_next_position].cases);
}

int
current_pattern_matching ()
{
  return pattern_matching_stack[pattern_matching_next_position - 1].no;
}

void
make_current_pattern_matching_exhaustive ()
{
  int i;
  for (i = 0;
       i <
       pattern_matching_stack[pattern_matching_next_position - 1].cases_no;
       i++)
    pattern_matching_stack[pattern_matching_next_position - 1].cases[i] = 1;
}

int
is_current_pattern_matching_exhaustive ()
{
  int i;
  for (i = 0;
       i <
       pattern_matching_stack[pattern_matching_next_position - 1].cases_no;
       i++)
    if (pattern_matching_stack[pattern_matching_next_position - 1].cases[i] ==
	0)
      return 0;
  return 1;
}

/* Returns:

   0  on success
   -1 on type mismatch */
int
memoize_pattern_matching_case (int encoding, int cases_no)
{
  if ((pattern_matching_stack[pattern_matching_next_position - 1].cases_no !=
       0)
      && (pattern_matching_stack[pattern_matching_next_position - 1].
	  cases_no != cases_no))
    return -1;
  if (pattern_matching_stack[pattern_matching_next_position - 1].cases_no ==
      0)
    {
      int i;
      pattern_matching_stack[pattern_matching_next_position - 1].cases_no =
	cases_no;
      pattern_matching_stack[pattern_matching_next_position - 1].cases =
	(int *) malloc (sizeof (int) * cases_no);
      for (i = 0; i < cases_no; i++)
	pattern_matching_stack[pattern_matching_next_position - 1].cases[i] =
	  0;
    }

  pattern_matching_stack[pattern_matching_next_position - 1].cases[encoding] =
    1;
  return 0;
}

int
is_pattern_matching_case_present (int encoding)
{
  if ((pattern_matching_stack[pattern_matching_next_position - 1].cases_no ==
       0)
      || (pattern_matching_stack[pattern_matching_next_position - 1].
	  cases_no <= encoding))
    return 0;
  else
    return pattern_matching_stack[pattern_matching_next_position - 1].
      cases[encoding];
}

//--------------------------------------------------------------

struct element
{
  char name[IDENTIFIER_LENGTH + 1];
  environment_id slink;
  int pos;
  type_t type;
  int precedence;		/* not used for all bindings */
  int refreshable;		/* "boolean" */
};

struct element *environment_stack;
environment_id next_position;
int current_size;

environment_id *environments_stack;
int environments_next_position;
int environments_current_size;

enum identifier_sort
{
  abstract_type_name_sort,
  concrete_type_name_sort,
  concrete_type_constructor_sort,
  exception_sort,
  variable_sort
};

/* *********************************** */
/* To do: rewrite using an hash table? */
struct global_element
{
  char name[IDENTIFIER_LENGTH];
  type_t type;
  int precedence;		/* not used for all bindings */
  int has_been_defined;		/* set to 0 if declared but not defined */
  enum identifier_sort sort;
};

struct global_element *global_environment_stack;
global_environment_id global_next_position;
int global_current_size;

/* returns 0 iff succeeds */
int lookup_global (char *identifier, int *address, type_t * t,
		   int *precedence);

/* Returns zero on success: */
int
declare_global (char *identifier, type_t type)
{
  int unused_int, already_exists;
  type_t unused_type;

  identifier = qualify_with_this_module (identifier);
  already_exists =
    !lookup_global ((char *) identifier, &unused_int, &unused_type,
		    &unused_int);
  if (already_exists)		/* Object was already declared (or even defined!) */
    return -1;

  bind_global (identifier, type);
  global_environment_stack[global_next_position - 1].has_been_defined = 0;
  return 0;
}

/* Returns zero on success: */
int
declare_global_with_precedence (char *identifier, type_t type, int precedence)
{
  int r = declare_global (identifier, type);

  if (r == 0)
    global_environment_stack[global_next_position - 1].precedence =
      precedence;
  return r;
}

/* Return zero on success */
int
bind_abstract_type_name (char *identifier)
{
  int r = bind_global (identifier, type_error);
  if (r == 0)
    global_environment_stack[global_next_position - 1].sort =
      abstract_type_name_sort;
  return r;
}

/* Return zero on success */
int
bind_concrete_type_name (char *identifier)
{
  int r = bind_global (identifier, type_error);
  if (r == 0)
    global_environment_stack[global_next_position - 1].sort =
      concrete_type_name_sort;
  return r;
}

/* Return zero on success */
int
bind_concrete_type_constructor (char *identifier)
{
  int r = bind_global (identifier, type_error);
  if (r == 0)
    global_environment_stack[global_next_position - 1].sort =
      concrete_type_constructor_sort;
  return r;
}

/* Returns zero on success */
int
bind_exception_name (char *unqualified_name)
{
  char *qualified_name = qualify_with_this_module (unqualified_name);
  int r = bind_global (unqualified_name, type_error);
  if (r == 0)
    global_environment_stack[global_next_position - 1].sort = exception_sort;

  return r;
}

/* Return zero on success */
int
bind_global (char *identifier, type_t type)
{
  return bind_global_with_precedence (identifier, type, 0);	/* An invalid value for precedence, to catch errors. */
}

/* Return zero on success */
int
bind_global_with_precedence (char *identifier, type_t type, int precedence)
{
  identifier = qualify_with_this_module (identifier);

  /* First test whether identifier has been declared but not yet defined: */
  if (has_been_declared (identifier))
    {
      int i;
      for (i = 0; i < global_next_position; i++)
	if (!strcmp (global_environment_stack[i].name, identifier))
	  break;		// I will surely hit the break in the loop

      strcpy (global_environment_stack[i].name, identifier);
      if (are_types_equal_excluding_variables
	  (global_environment_stack[i].type, type)
	  && (global_environment_stack[i].precedence == precedence))
	{
	  global_environment_stack[i].has_been_defined = 1;
	  return 0;
	}
      else
	{
	  fprintf(stderr,"ALEPH 1: declared ");
	  output_type(stderr,global_environment_stack[i].type);
	  fprintf(stderr," (prec %i)\n", 
		         global_environment_stack[i].precedence);
	  fprintf(stderr,"ALEPH 2: actual   ");
	  output_type(stderr,type);
	  fprintf(stderr," (prec %i)\n",
		         global_environment_stack[i].precedence);

	  return -1;		/* Type or precedence does not match */
	}
    }
  /* Ok, identifier has not been declared; add it: */

  if (global_next_position == global_current_size)
    {
      global_current_size *= 2;
      global_environment_stack =
	(struct global_element *) realloc (global_environment_stack,
					   global_current_size *
					   sizeof (struct global_element));
    }

  strcpy (global_environment_stack[global_next_position].name, identifier);
  global_environment_stack[global_next_position].type = freshen_type (type);	// To do: is refreshing needed? I think so
  global_environment_stack[global_next_position].sort = variable_sort;

  global_environment_stack[global_next_position].precedence = precedence;
  global_environment_stack[global_next_position].has_been_defined = 1;
  global_next_position++;

  return 0;
}

int
has_been_declared (char *identifier)
{
  int i;
  identifier = qualify_with_this_module (identifier);

  for (i = 0; i < global_next_position; i++)
    if (!strcmp (global_environment_stack[i].name, identifier))
      {
	if (global_environment_stack[i].has_been_defined == 0)
	  return 1;
	else
	  return 0;
      }

  return 0;
}

int
has_been_defined (char *identifier)
{
  int i;
  identifier = qualify_with_this_module (identifier);

  for (i = 0; i < global_next_position; i++)
    if (!strcmp (global_environment_stack[i].name, identifier))
      {
	if (global_environment_stack[i].has_been_defined == 1)
	  return 1;
	else
	  return 0;
      }

  return 0;
}

int
has_been_declared_or_defined (char *identifier)
{
  int i;

  identifier = qualify_with_this_module (identifier);
  for (i = 0; i < global_next_position; i++)
    if (!strcmp (global_environment_stack[i].name, identifier))
      return 1;

  return 0;
}

char *
sort_to_char_star (enum identifier_sort sort)
{
  if (sort == variable_sort)
    return "variable";
  else if (sort == abstract_type_name_sort)
    return "abstract";
  else if (sort == concrete_type_name_sort)
    return "con. name";
  else if (sort == concrete_type_constructor_sort)
    return "con.const.";
  else
    {
      fprintf (stderr, "environment.c: This cannot happen! saadnklj\n");
      exit (EXIT_FAILURE);
    }
}

void
dump_global_environment ()
{
  int i;

  fprintf (stderr,
	   "  i  | %10s |     name     |  type\n----------------------------------------------------------------\n",
	   "sort");
  for (i = global_next_position - 1; i >= 0; i--)
    {
      fprintf (stderr, "%3i  | %10s |%12s|"
	       " |  ", i,
	       sort_to_char_star (global_environment_stack[i].sort),
	       global_environment_stack[i].name);
      output_type (stderr, global_environment_stack[i].type);
      if (!global_environment_stack[i].has_been_defined)
	fprintf (stderr, " [ONLY DECLARED]");
      fprintf (stderr, "\n");
    }
  fprintf (stderr, "\n");
}

char *
qualify_with_this_module (char *identifier)
{
  if (is_qualified (identifier))
    return identifier;
  else
    {
      static char qualified_identifier[MAX_QUALIFIED_LENGTH];
      int identifier_length = strlen (identifier);
      int current_module_length = strlen (current_module_name);

      strcpy (qualified_identifier, current_module_name);
      qualified_identifier[current_module_length] = MODULES_SEPARATOR;
      strcpy (qualified_identifier + current_module_length + 1, identifier);

      return qualified_identifier;
    }
}

char *
qualify_with_sort (char *identifier, enum identifier_sort sort)
{
  if (is_qualified (identifier))
    return identifier;
  else
    {
      char *first_found_identifier;
      int address;

      //fprintf(stderr,"qualifying %s... ",identifier);
      for (address = global_next_position - 1; address >= 0; address--)
	{
	  if (!strcmp (identifier,
		       bare_identifier (global_environment_stack[address].
					name))
	      && (global_environment_stack[address].sort == sort))
	    {
	      //fprintf(stderr,"found %s.\n",global_environment_stack[address].name);
	      first_found_identifier = global_environment_stack[address].name;
	      return global_environment_stack[address].name;
	    }			/* if */
	}			/* for */

      //fprintf(stderr,"\n");
      return qualify_with_this_module (identifier);
    }
}

char *
qualify (char *identifier)
{
  return qualify_with_sort (identifier, variable_sort);
}

char *
qualify_abstract_type_name (char *identifier)
{
  char *r;
  //fprintf(stderr,"QUALIFICO IL TIPO ASTRATTO %s:\n",identifier);
  //dump_global_environment();
  r = qualify_with_sort (identifier, abstract_type_name_sort);

  //fprintf(stderr,"%s |--> %s\n-----------------------------------------------------------------\n\n",identifier,r);

  return r;
}

char *
qualify_concrete_type_name (char *identifier)
{
  char *r;
  //fprintf(stderr,"QUALIFICO IL TIPO CONCRETO %s:\n",identifier);
  //dump_global_environment();
  r = qualify_with_sort (identifier, concrete_type_name_sort);

  //fprintf(stderr,"%s |--> %s\n-----------------------------------------------------------------\n\n",identifier,r);

  return r;
}

char *
qualify_concrete_type_constructor (char *identifier)
{
  char *r;
  //fprintf(stderr,"QUALIFICO IL COSTRUTTORE DI TIPO CONCRETO %s:\n",identifier);
  //dump_global_environment();
  r = qualify_with_sort (identifier, concrete_type_constructor_sort);

  //fprintf(stderr,"%s |--> %s\n-----------------------------------------------------------------\n\n",identifier,r);

  return r;
}

char *
qualify_exception (char *identifier)
{
  char *r;
  r = qualify_with_sort (identifier, exception_sort);
  return r;
}

/* returns 0 iff succeeds */
int
lookup_global (char *identifier, int *address, type_t * t, int *precedence)
{
  //fprintf(stderr,"LOOKUP GLOBAL %s:\n",identifier);
  //dump_global_environment();

  if (is_qualified (identifier))
    {
      /* The identifier is qualified, so I can assume there are not more than one binding for it: */
      for ((*address) = global_next_position - 1; (*address) >= 0;
	   (*address)--)
	if (!strcmp (identifier, global_environment_stack[*address].name))
	  {
	    // ?????????????? Commented out on 11 Jun 2002
	    //if(!global_environment_stack[*address].has_been_defined)
	    //  return -1;
	    // ?????????????? Commented out on 11 Jun 2002
	    (*t) = freshen_type (global_environment_stack[*address].type);
	    (*precedence) = global_environment_stack[*address].precedence;
	    return 0;
	  }

      /* identifier does not exist in the global environment. */
      return -1;
    }
  else
    {
      /* identifier is not qualified */
      int found = 0;
      char *first_found_identifier;

      //fprintf(stderr,"lookup_global %s... ",identifier);
      for ((*address) = global_next_position - 1; (*address) >= 0;
	   (*address)--)
	{
	  if (!strcmp (identifier,
		       bare_identifier (global_environment_stack[*address].
					name)))
	    {
	      //fprintf(stderr,"found %s.\n",global_environment_stack[*address].name);
	      if (found)
		return -2;

	      found = 1;
	      first_found_identifier =
		global_environment_stack[*address].name;
	      /* I will refresh the type later, in case of no errors: */
	      (*t) = global_environment_stack[*address].type;
	      (*precedence) = global_environment_stack[*address].precedence;
	    }			/* if */
	}			/* for */

      //fprintf(stderr,"\n");

      if (found)
	{
	  *t = freshen_type (*t);
	  return 0;
	}
      else			/* identifier does not exist in the global environment. */
	return -1;
    }				/* else */
}

/* *********************************** */

void create_bindings_stack ();
void destroy_bindings_stack ();

void
create_environment_structures ()
{
  int i;

  current_size = INITIAL_ENVIRONMENT_STACK_SIZE;
  environment_stack =
    (struct element *) malloc (sizeof (struct element) * current_size);
  next_position = 0;

  environments_current_size = INITIAL_ENVIRONMENTS_STACK_SIZE;
  environments_stack =
    (environment_id *) malloc (sizeof (environments_stack) *
			       environments_current_size);
  environments_next_position = 0;

  push_environment (-1);

  create_bindings_stack ();
  create_pattern_matching_stack ();
  create_exception_handling_stack ();
  /* Create global environment: */
  global_current_size = INITIAL_ENVIRONMENT_STACK_SIZE;
  global_environment_stack =
    (struct global_element *) malloc (sizeof (struct global_element) *
				      global_current_size);
  global_next_position = 0;
}

environment_id
top_environment ()
{
  if (environments_stack[environments_next_position - 1] == -1)
    return -1;
  else
    return environments_stack[environments_next_position - 1];
}

environment_id
previous_environment ()
{
  if (environments_next_position < 2)
    return -1;
  else
    return environments_stack[environments_next_position - 2];
}

environment_id
pop_environment ()
{
  return environments_stack[environments_next_position-- - 1];
}

void
push_environment (environment_id e)
{
  if (environments_next_position == environments_current_size)
    {
      int i;

      environments_current_size *= 2;
      environments_stack =
	(environment_id *) realloc (environments_stack,
				    environments_current_size *
				    sizeof (environment_id));
    }

  environments_stack[environments_next_position++] = e;
}

environment_id
bind (environment_id e, char *identifier, int n, type_t type, int refreshable)
{
  if (next_position == current_size)
    {
      current_size *= 2;
      environment_stack =
	(struct element *) realloc (environment_stack,
				    current_size * sizeof (struct element));
    }

  strcpy (environment_stack[next_position].name, identifier);
  environment_stack[next_position].slink = e;
  environment_stack[next_position].pos = n;
  environment_stack[next_position].type = type;
  environment_stack[next_position].refreshable = refreshable;

  //fprintf(stderr,"%i: [%s @ %i slink = %i]\n",next_position,identifier,n,e);

  return next_position++;
}

environment_id
bind_with_precedence (environment_id e,
		      char *identifier,
		      int n, type_t type, int precedence, int refreshable)
{
  int r = bind (e, identifier, n, type, refreshable);
  environment_stack[next_position - 1].precedence = precedence;

  return r;
}

environment_id
bind_not_refreshable (environment_id e, char *identifier, int pos,
		      type_t type)
{
  return bind (e, identifier, pos, type, /* not refreshable */ 0);
}

environment_id
bind_not_refreshable_with_precedence (environment_id e, char *identifier,
				      int pos, type_t type, int precedence)
{
  return bind_with_precedence (e, identifier, pos, type, precedence,
			       /* not refreshable */ 0);
}

environment_id
bind_refreshable (environment_id e, char *identifier, int pos, type_t type)
{
  return bind (e, identifier, pos, type, /* refreshable */ 1);
}

environment_id
bind_refreshable_with_precedence (environment_id e, char *identifier, int pos,
				  type_t type, int precedence)
{
  return bind_with_precedence (e, identifier, pos, type, precedence,
			       /* refreshable */ 1);
}

void
free_environment_structures ()
{
  free (environment_stack);

  destroy_bindings_stack ();
  destroy_pattern_matching_stack ();
  destroy_exception_handling_stack ();

  /* Destroy globals stack: */
  free (global_environment_stack);
}

/* returns 0 iff succeeds */
int
lookup (char *identifier, int *jumps_no, int *n, type_t * t)
{
  int precedence;		/* ignored */
  int r = lookup_with_precedence (identifier, jumps_no, n, t, &precedence);
  if (r != 0)
    *t = new_generic_type ();	/* no declaration is the same as ": T1" */

  return r;
}

/* returns 0 iff succeeds */
int
lookup_with_precedence (char *identifier, int *jumps_no, int *n, type_t * t,
			int *precedence)
{
  //fprintf(stderr,"lookup_with_precedence: init %s\n",identifier);

  environment_id e = environments_stack[environments_next_position - 1];
  int r;

  //fprintf(stderr,"e = %i, environments_next_position-1 = %i\n",e,environments_next_position-1);

  *jumps_no = 0;
  //fprintf(stderr,"lookup_with_precedence: while init\n");
  while (e != ((environment_id) - 1))
    {
      //fprintf(stderr,". e = %i\n",e);
      if (!strcmp (identifier, environment_stack[e].name))
	{
	  *n = environment_stack[e].pos;
	  *t = environment_stack[e].type;
	  if (environment_stack[e].refreshable)
	    *t = freshen_type (*t);
	  *precedence = environment_stack[e].precedence;
	  //fprintf(stderr,"lookup_with_precedence: while done A\n");
	  //fprintf(stderr,"lookup_with_precedence: done A\n");
	  return 0;
	}			/* if */
      if (environment_stack[e].pos == 0)
	(*jumps_no)++;
      e = environment_stack[e].slink;
    }				/* while */
  //fprintf(stderr,"lookup_with_precedence: while done B\n");

  /* We have not found identifier as local nor as non-local: */
  //fprintf(stderr,"{{{{LOOKING UP %s AS A GLOBAL}}}}\n",identifier);
  *jumps_no = -1;
  r = lookup_global (identifier, n, t, precedence);
  //fprintf(stderr,"lookup_with_precedence: done B %s\n",(!r)?"FOUND":"NOT found");
  return r;
}

int *bindings_stack;
int bindings_next_position;
int bindings_current_size;

void
create_bindings_stack ()
{
  bindings_current_size = INITIAL_BINDINGS_STACK_SIZE;
  bindings_stack = (int *) malloc (sizeof (int) * bindings_current_size);
  bindings_next_position = 0;
}

void
destroy_bindings_stack ()
{
  free (bindings_stack);
}

void
push_bindings ()
{
  if (bindings_next_position == bindings_current_size)
    {
      int i;

      bindings_current_size *= 2;
      bindings_stack =
	(int *) realloc (bindings_stack,
			 bindings_current_size * sizeof (int));
    }

  bindings_stack[bindings_next_position++] = 0;	/* push an empty bindings set */
}

void
pop_bindings ()
{
  --bindings_next_position;
}

int *
current_bindings ()
{
  //  fprintf(stderr,"[Now pos is %i]\n",bindings_next_position);

  return &(bindings_stack[bindings_next_position - 1]);
}

void
dump_environment ()
{
  int i;

  fprintf (stderr,
	   "  i  |    name    pos   slink \n----------------------------------\n");
  for (i = top_environment (); i >= 0; i--)
    fprintf (stderr, "%3i  |%7s%7i%7i\n", i, environment_stack[i].name,
	     environment_stack[i].pos, environment_stack[i].slink);
  fprintf (stderr, "\n");
}

int
next_free_position (environment_id e)
{
  return environment_stack[e].pos + 1;
}

int
are_there_undefined_but_declared_identifiers (char *module)
{
  int i;

  for (i = 0; i < global_next_position; i++)
    if (!global_environment_stack[i].has_been_defined)
      if (!strcmp (module, bare_module (global_environment_stack[i].name)))
	return 1;

  return 0;
}

char *
first_undefined_identifier (char *module)
{
  int i;

  for (i = 0; i < global_next_position; i++)
    if (!global_environment_stack[i].has_been_defined)
      if (!strcmp (module, bare_module (global_environment_stack[i].name)))
	return global_environment_stack[i].name;

  return "";
}

/* I assume the identifier to be valid */
int
is_qualified (char *identifier)
{
  int i, l = strlen (identifier);

  for (i = 0; i < l; i++)
    if (identifier[i] == MODULES_SEPARATOR)
      return 1;

  return 0;
}

/* The identifier may not be qualified */
char *
bare_identifier (char *maybe_qualified_identifier)
{
  int i, l = strlen (maybe_qualified_identifier =
		     qualify_with_this_module (maybe_qualified_identifier));

  for (i = l - 1; i >= 0; i--)
    if (maybe_qualified_identifier[i] == MODULES_SEPARATOR)
      return maybe_qualified_identifier + i + 1;

  return maybe_qualified_identifier;
}

char *
bare_module (char *maybe_qualified_identifier)
{
  static char qualified_identifier[MAX_QUALIFIED_LENGTH];
  int i, l;

  strcpy (qualified_identifier,
	  qualify_with_this_module (maybe_qualified_identifier));
  l = strlen (qualified_identifier);

  for (i = l - 1; i >= 0; i--)
    if (qualified_identifier[i] == MODULES_SEPARATOR)
      {
	qualified_identifier[i] = '\0';
	return qualified_identifier;
      }

  fprintf (stderr, "environment.c: bare_module(): This cannot happen!\n");
  exit (EXIT_FAILURE);
}
