/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002, 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

#include <stdio.h>
#include <malloc.h>
#include <string.h>

#include "string_map.h"

#define STRING_MAP_SIZE 512

struct sm_element
{
  char *key;
  void *value;

  struct sm_element *next;
};

struct string_map
{
  struct sm_element *buckets[STRING_MAP_SIZE];
  size_t size_of_value;
};

long
string_hash_function (const char *x)
{
  long i, r = 0;
  for (i = 0; x[i] != '\0'; i++)
    r += x[i] * (i + 1);

  //fprintf(stderr,"string_hash_function(): %i\n",r%STRING_MAP_SIZE);
  return r % STRING_MAP_SIZE;
}

void
destroy_sm_bucket (struct sm_element *b)
{
  struct sm_element *next;
  struct sm_element *current = b;

  while (current != NULL)
    {
      //fprintf(stderr,"%s ",current->key);
      next = current->next;
      free (current->key);
      free (current->value);
      free (current);
      current = next;
    }				/* while */
}

struct string_map *
create_string_map (size_t size_of_value_)
{
  int i;
  struct string_map *r =
    (struct string_map *) malloc (sizeof (struct string_map));

  r->size_of_value = size_of_value_;

  for (i = 0; i < STRING_MAP_SIZE; i++)
    r->buckets[i] = NULL;

  //  printf("<C>");
  return r;
}

void
destroy_string_map (struct string_map *m)
{
  int i;
  int fb = 0;
  for (i = 0; i < STRING_MAP_SIZE; i++)
    {
      //fprintf(stderr,"DESTROYING BUCKET %i: ",i);
      if (m->buckets[i] != NULL)
	fb++;
      destroy_sm_bucket (m->buckets[i]);
      //fprintf(stderr,"\n");
    }

  //  printf("<D %f>",(float)fb/(float)STRING_MAP_SIZE);
  //fprintf(stderr,"destroy_string_map()\n");

  free (m);
}

struct sm_element *
search_in_sm_bucket (struct sm_element *bucket, const char *k)
{
  struct sm_element *x = bucket;

  //fprintf(stderr,"\tStart searching for >%s<...\n",k);
  while (x != NULL)
    {
      //fprintf(stderr,"\tSearching for >%s< in bucket list... \n",k);
      //fprintf(stderr,"reached >%s<.\n",x->key);
      if (!strcmp (x->key, k))
	{
	  //fprintf(stderr,"found >%s<.\n",x->key);
	  return x;		/* We have found k */
	}
      else
	x = x->next;
    }
  //fprintf(stderr,"\t>%s< not found\n",k);
  /* x is NULL and we have not found k: */
  return NULL;
}

void
show_bucket (struct string_map *m, int bucket)
{
  struct sm_element *x = m->buckets[bucket];

  while (x != NULL)
    {
      fprintf (stderr, "%s->%p ", x->key, x->value);
      x = x->next;
    }
}

struct sm_element *
remove_from_bucket (struct sm_element *bucket, const char *k)
{
  struct sm_element *x = bucket;

  if (bucket == NULL)
    return 0;
  else if (!strcmp (bucket->key, k))
    {
      struct sm_element *next = bucket->next;

      free (bucket->key);
      free (bucket->value);
      free (bucket);
      return next;
    }

  while (x->next != NULL)
    {
      if (!strcmp (x->next->key, k))
	{
	  struct sm_element *next_of_next = x->next->next;

	  free (x->next->key);
	  free (x->next->value);
	  free (x->next);
	  x->next = next_of_next;
	  return bucket;
	}
      x = x->next;
    }

  return bucket;
}

void
insert_into_string_map (struct string_map *m, const char *k, void *v)
{
  int bucketNo = string_hash_function (k);

  if (search_in_sm_bucket (m->buckets[bucketNo], k) == NULL)
    multinsert_into_string_map (m, k, v);
}

void
multinsert_into_string_map (struct string_map *m, const char *k, void *v)
{
  int bucketNo = string_hash_function (k);

  struct sm_element *e =
    (struct sm_element *) malloc (sizeof (struct sm_element));

  e->key = (char*) malloc (strlen(k) + 1);
  strcpy (e->key, k);

  e->value = malloc (m->size_of_value);
  memcpy (e->value, v, m->size_of_value);

  e->next = m->buckets[bucketNo];
  m->buckets[bucketNo] = e;
}

void
remove_from_string_map (struct string_map *m, const char *k)
{
  int bucket_no = string_hash_function (k);

  m->buckets[bucket_no] = remove_from_bucket (m->buckets[bucket_no], k);
  //fprintf(stderr,"remove_from_string_map(): [");
  //show_bucket(m,bucket_no);
  //fprintf(stderr,"]\n");
}

/*
void remove_from_string_map(struct string_map* m,char* k){
  int bucket_no=string_hash_function(k);
  struct sm_element* e;

  e=search_in_sm_bucket(m->buckets[bucket_no],k);
  //printf("remove_from_string_map: %p\n",e);
  if(e==NULL)
    return;
  else{
    struct sm_element* another_pointer_to_e=e;
    e=e->next;
    free(another_pointer_to_e->key);
    free(another_pointer_to_e->value);
    free(another_pointer_to_e);
  }
  //printf("OK\n");
}
*/

void *
access_string_map (struct string_map *m, const char *k)
{
  int bucket_no = string_hash_function (k);
  struct sm_element *e;

  if ((e = search_in_sm_bucket (m->buckets[bucket_no], k)) == NULL)
    return NULL;
  else
    return e->value;
}

int is_string_map_defined_on (struct string_map *map, const char *key)
{
  return access_string_map(map, key) != NULL;
}
