/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

#include "library.h"

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>
#include <signal.h>
#include <SDL/SDL.h>

void initialize_SDL();

/* The following functions are dynamically loaded from libSDL: */
int (*lSDL_LockSurface)(SDL_Surface *surface);
int (*lSDL_UnlockSurface)(SDL_Surface *surface);
int (*lSDL_Init)(Uint32 flags);
char* (*lSDL_GetError)(void);
SDL_Surface* (*lSDL_SetVideoMode)(int width, int  height,
                                  int bpp, Uint32 flags);
void (*lSDL_Quit)(void);
Uint32 (*lSDL_MapRGB)(SDL_PixelFormat *fmt, Uint8 r, Uint8 g, Uint8 b);
void (*lSDL_GetRGB)(Uint32 pixel, SDL_PixelFormat *fmt,
                    Uint8 *r, Uint8 *g, Uint8 *b);
int (*lSDL_FillRect)(SDL_Surface  *dst,  SDL_Rect   *dstrect,
                     Uint32 color);
void (*lSDL_UpdateRect)(SDL_Surface *screen, Sint32 x,
                        Sint32 y, Sint32 w, Sint32 h);
int (*lSDL_InitSubSystem)(Uint32 flags);
int (*lSDL_QuitSubSystem)(Uint32 flags);

void initialize_c_library(){
  dynamic_library_handle_t handle;
  
#ifdef DEBUG
  fprintf(stderr, "Initializing the C-library sdl.c... ");
#endif

  /* Open the SDL library: */
  handle = open_dynamic_library("libSDL.so");
  
  /* Load functions from libSDL: */
  lSDL_LockSurface = load_symbol_from_handle(handle, "SDL_LockSurface");
  lSDL_UnlockSurface = load_symbol_from_handle(handle, "SDL_UnlockSurface");
  lSDL_Init = load_symbol_from_handle(handle, "SDL_Init");
  lSDL_GetError = load_symbol_from_handle(handle, "SDL_GetError");
  lSDL_SetVideoMode = load_symbol_from_handle(handle, "SDL_SetVideoMode");
  lSDL_Quit = load_symbol_from_handle(handle, "SDL_Quit");
  lSDL_MapRGB = load_symbol_from_handle(handle, "SDL_MapRGB");
  lSDL_GetRGB = load_symbol_from_handle(handle, "SDL_GetRGB");
  lSDL_FillRect = load_symbol_from_handle(handle, "SDL_FillRect");
  lSDL_UpdateRect = load_symbol_from_handle(handle, "SDL_UpdateRect");
  lSDL_InitSubSystem = load_symbol_from_handle(handle, "SDL_InitSubSystem");
  lSDL_QuitSubSystem = load_symbol_from_handle(handle, "SDL_QuitSubSystem");

  /* Initialize the SDL library using some of the functions we loaded: */
  initialize_SDL();

  /* For some reason which deserves investigation, once the SDL library
     is initialized the eAM becomes immune to some signals. Here we
     reset them to their default behaviour: */
  signal(SIGINT, SIG_DFL);
  signal(SIGTERM, SIG_DFL);

#ifdef DEBUG
  fprintf(stderr, "done\n");
#endif
}


/* A color is represented by SDL as an Uint32; an Uint32 is not wider than
   an unsigned_integer_t, so we are going to wrap colors into eAM words. */

/* The displayed image: */
static SDL_Surface *surface;

/* Screen parameters, set up at initialization time: */
static integer_t screen_bpp;
static integer_t do_we_need_screen_locking;
static SDL_Rect screen_size;

/* Called when no surface locking/unlocking is needed: */
static void do_nothing(){
}

static void actually_lock_screen(){
  lSDL_LockSurface(surface);
}
static void actually_unlock_screen(){
  lSDL_UnlockSurface(surface);
}

void (*lock_screen)();
void (*unlock_screen)();

/* Only one of these can be used for a given surface: */
static Uint32 get_pixel_1bpp(integer_t x, integer_t y){
  Uint8 *p = (Uint8 *)surface->pixels + y * surface->pitch + x * 1;
  return *p;
}
static Uint32 get_pixel_2bpp(integer_t x, integer_t y){
  Uint8 *p = (Uint8 *)surface->pixels + y * surface->pitch + x * 2;
  return *(Uint16 *)p;
}
static Uint32 get_pixel_3bpp(integer_t x, integer_t y){
  Uint8 *p = (Uint8 *)surface->pixels + y * surface->pitch + x * 3;
  if(SDL_BYTEORDER == SDL_BIG_ENDIAN) /* decided at compile time */
    return p[0] << 16 | p[1] << 8 | p[2];
  else
    return p[0] | p[1] << 8 | p[2] << 16;
}
static Uint32 get_pixel_4bpp(integer_t x, integer_t y){
  Uint8 *p = (Uint8 *)surface->pixels + y * surface->pitch + x * 4;
  return *(Uint32 *)p;
}

/* Locking versions of the functions above: */
static Uint32 get_pixel_1bpp_locking(integer_t x, integer_t y){
  static Uint32 r;
  actually_lock_screen(); r = get_pixel_1bpp(x, y); actually_unlock_screen();
  return r;
}
static Uint32 get_pixel_2bpp_locking(integer_t x, integer_t y){
  static Uint32 r;
  actually_lock_screen(); r = get_pixel_2bpp(x, y); actually_unlock_screen();
  return r;
}
static Uint32 get_pixel_3bpp_locking(integer_t x, integer_t y){
  static Uint32 r;
  actually_lock_screen(); r = get_pixel_3bpp(x, y); actually_unlock_screen();
  return r;
}
static Uint32 get_pixel_4bpp_locking(integer_t x, integer_t y){
  static Uint32 r;
  actually_lock_screen(); r = get_pixel_4bpp(x, y); actually_unlock_screen();
  return r;
}

/* A pointer to the function which can actually be used among the ones
   ones above: */
Uint32 (*get_pixel)(integer_t x, integer_t y);

/* Only one of these can be used for a given surface: */
static void set_pixel_1bpp(integer_t x, integer_t y, Uint32 pixel){
  Uint8 *p = (Uint8 *)surface->pixels + y * surface->pitch + x * 1;
  *p = pixel;
}
static void set_pixel_2bpp(integer_t x, integer_t y, Uint32 pixel){
  Uint8 *p = (Uint8 *)surface->pixels + y * surface->pitch + x * 2;
  *(Uint16 *)p = pixel;
}
static void set_pixel_3bpp(integer_t x, integer_t y, Uint32 pixel){
  Uint8 *p = (Uint8 *)surface->pixels + y * surface->pitch + x * 3;
  if(SDL_BYTEORDER == SDL_BIG_ENDIAN) { /* decided at compile time */
    p[0] = (pixel >> 16) & 0xff;
    p[1] = (pixel >> 8) & 0xff;
    p[2] = pixel & 0xff;
  } else {
    p[0] = pixel & 0xff;
    p[1] = (pixel >> 8) & 0xff;
    p[2] = (pixel >> 16) & 0xff;
  }
}
static void set_pixel_4bpp(integer_t x, integer_t y, Uint32 pixel){
  Uint8 *p = (Uint8 *)surface->pixels + y * surface->pitch + x * 4;
  *(Uint32 *)p = pixel;
}

/* Locking versions of the functions above: */
static void set_pixel_1bpp_locking(integer_t x, integer_t y, Uint32 pixel){
  actually_lock_screen();
  set_pixel_1bpp_locking(x, y, pixel);
  actually_unlock_screen();
}
static void set_pixel_2bpp_locking(integer_t x, integer_t y, Uint32 pixel){
  actually_lock_screen();
  set_pixel_2bpp_locking(x, y, pixel);
  actually_unlock_screen();
}
static void set_pixel_3bpp_locking(integer_t x, integer_t y, Uint32 pixel){
  actually_lock_screen();
  set_pixel_3bpp_locking(x, y, pixel);
  actually_unlock_screen();
}
static void set_pixel_4bpp_locking(integer_t x, integer_t y, Uint32 pixel){
  actually_lock_screen();
  set_pixel_4bpp_locking(x, y, pixel);
  actually_unlock_screen();
}

/* A pointer to the function which can actually be used among the ones
   ones above: */
void (*set_pixel)(integer_t x, integer_t y, Uint32 color);

void initialize_SDL(){
  if((lSDL_Init(SDL_INIT_VIDEO | SDL_INIT_NOPARACHUTE) == -1)) { 
    fprintf(stderr, "Could not initialize SDL: %s\n", lSDL_GetError());
    fatal("sdl:c: fatal SDL error");
  }
}

/* Decide once and for all whether we need surface locking, so we won't
   test for each call of lock_screen()/unlock_screen(): */
static void setup_SDL_surface_locking(){
  if(do_we_need_screen_locking = SDL_MUSTLOCK(surface)){ /* we need locking */
#ifdef DEBUG
    fprintf(stderr, "We DO need screen locking\n");
#endif
    lock_screen = actually_lock_screen;
    unlock_screen = actually_unlock_screen;
  } else{ /* we don't need locking */
#ifdef DEBUG
    fprintf(stderr, "We DON'T need screen locking\n");
#endif
    lock_screen = do_nothing;
    unlock_screen = do_nothing;
  }
}

/* This requires that do_we_need_screen_locking is already set: */
static void setup_SDL_bpp(){
#ifdef DEBUG
  fprintf(stderr, "Using a %i bytes per pixel framebuffer\n",
	  surface->format->BytesPerPixel);
#endif
  switch(screen_bpp = surface->format->BytesPerPixel){
  case 1:
    if(do_we_need_screen_locking){
      set_pixel = set_pixel_1bpp_locking;
      get_pixel = get_pixel_1bpp_locking;
    } else {
      set_pixel = set_pixel_1bpp;
      get_pixel = get_pixel_1bpp;
    }
    break;  
    
  case 2:
    if(do_we_need_screen_locking){
      set_pixel = set_pixel_2bpp_locking;
      get_pixel = get_pixel_2bpp_locking;
    } else {
      set_pixel = set_pixel_2bpp;
      get_pixel = get_pixel_2bpp;
    }
    break;  
  
  case 3:
    if(do_we_need_screen_locking){
      set_pixel = set_pixel_3bpp_locking;
      get_pixel = get_pixel_3bpp_locking;
    } else {
      set_pixel = set_pixel_3bpp;
      get_pixel = get_pixel_3bpp;
    }
    break;  
    
  case 4:
    if(do_we_need_screen_locking){
      set_pixel = set_pixel_4bpp_locking;
      get_pixel = get_pixel_4bpp_locking;
    } else {
      set_pixel = set_pixel_4bpp;
      get_pixel = get_pixel_4bpp;
    }
    break;  
    
  default:
    fprintf(stderr, "sdl.c: this should not happen\n");
    fatal("sdl:c: impossible");
    
  }; /* switch */
}

static void setup_SDL_screen_size(){
  screen_size.x = 0;
  screen_size.y = 0;
  screen_size.w = surface->w;
  screen_size.h = surface->h;
}

void initialize_SDL_graphics(integer_t width, integer_t height){
  /* This may fail if the video support was already initialized; don't care: */
  lSDL_InitSubSystem(SDL_INIT_VIDEO | SDL_INIT_NOPARACHUTE);

  /* Create an SDL window or switch to video mode: */
  surface = lSDL_SetVideoMode((int)width,
			      (int)height, 
			      0, /* Use the current color depth */
			      SDL_ANYFORMAT);
  if ( surface == NULL ) {
    fprintf(stderr, "Couldn't set video mode: %s\n", lSDL_GetError());
    exit(EXIT_FAILURE);
  }

  /* Make all initialization-time decisions, in order not to introduce
     tests later: */
  setup_SDL_surface_locking();
  setup_SDL_bpp();
  setup_SDL_screen_size();
}

void finalize_SDL(){
  /* Shutdown all subsystems */
  lSDL_Quit();
}

Uint32 RGB_to_color(Uint8 r, Uint8 g, Uint8 b){
  return lSDL_MapRGB(surface->format, r, g, b);
}

void color_to_RGB(Uint32 color, integer_t *r, integer_t *g, integer_t *b){
  static Uint8 r_, g_, b_;
  lSDL_GetRGB(color, surface->format, &r_, &g_, &b_);
  *r = (integer_t) r_;
  *g = (integer_t) g_;
  *b = (integer_t) b_;
}

void clear_screen(Uint32 color){
  lSDL_FillRect(surface, &screen_size, color);
}

void show_pixel(integer_t x, integer_t y, Uint32 color){
  set_pixel(x, y, color);
  lSDL_UpdateRect(surface, x, y, 1, 1);
}

void update_screen(){
  lSDL_UpdateRect(surface, 0, 0, 0, 0);
}


/* The following objects are exported to epsilon: */

word_t e_initialize_graphics(word_t w_h){
  initialize_SDL_graphics(((integer_t*)w_h)[0],
			  ((integer_t*)w_h)[1]);
  return NULL; 
}
word_t e_finalize_graphics(){
  lSDL_QuitSubSystem(SDL_INIT_VIDEO);
  return NULL;
}
word_t rgb_to_color(word_t r_g_b){
  return (word_t)(unsigned_integer_t)
    RGB_to_color((Uint8)((integer_t*)r_g_b)[0],
		 (Uint8)((integer_t*)r_g_b)[1],
		 (Uint8)((integer_t*)r_g_b)[2]);
}
word_t e_color_to_rgb(word_t color){
  integer_t* r_g_b = allocate_exact(3);
  color_to_RGB((Uint32)(unsigned_integer_t)color,
	       &(r_g_b[0]), &(r_g_b[1]), &(r_g_b[2]));
  return r_g_b;
}

word_t e_set_pixel(word_t x_y_c){
  set_pixel(((integer_t*)x_y_c)[0],
	    ((integer_t*)x_y_c)[1],
	    (Uint32)((unsigned_integer_t*)x_y_c)[2]);
  return NULL; 
}
void show_line_by_x(integer_t min_x, integer_t max_x,
		    integer_t x1, integer_t y1,
		    integer_t x2, integer_t y2,
		    Uint32 c){
  float_t m =
    (float_t)(y2 - y1) / (float_t)(x2 - x1);
  float_t q = 
    (float_t)(x2 * y1 - x1 * y2) / (float_t)(x2 - x1);
  integer_t x;
  for(x = min_x; x <= max_x; x++)
    set_pixel(x,
	      (integer_t)(m * x + q),
	      c);
}
void show_line_by_y(integer_t min_y, integer_t max_y,
		    integer_t x1, integer_t y1,
		    integer_t x2, integer_t y2,
		    Uint32 c){
  float_t m =
    (float_t)(x2 - x1) / (float_t)(y2 - y1);
  float_t q = 
    (float_t)(y2 * x1 - y1 * x2) / (float_t)(y2 - y1);
  integer_t y;
  for(y = min_y; y <= max_y; y++)
    set_pixel((integer_t)(m * y + q),
	      y,
	      c);
}
word_t e_show_line(word_t x1_y1_x2_y2_c){
  integer_t x1 = ((integer_t*)x1_y1_x2_y2_c)[0];
  integer_t y1 = ((integer_t*)x1_y1_x2_y2_c)[1];
  integer_t x2 = ((integer_t*)x1_y1_x2_y2_c)[2];
  integer_t y2 = ((integer_t*)x1_y1_x2_y2_c)[3];
  Uint32 c = (Uint32)rgb_to_color(((integer_t**)x1_y1_x2_y2_c)[4]);
  integer_t min_x, max_x, min_y, max_y;
  integer_t x;
  float_t m, q;

  if(x1 < x2){
    min_x = x1; max_x = x2;
  } else {
    min_x = x2; max_x = x1;
  }

  if(y1 < y2){
    min_y = y1; max_y = y2;
  } else {
    min_y = y2; max_y = y1;
  }
  
  if((max_x - min_x) >= (max_y - min_y))
    show_line_by_x(min_x, max_x, x1, y1, x2, y2, c);
  else
    show_line_by_y(min_y, max_y, x1, y1, x2, y2, c);

  lSDL_UpdateRect(surface, min_x, min_y, max_x - min_x + 1, max_y - min_y + 1);

  return NULL; 
}
word_t e_show_pixel(word_t x_y_c){ /* c is in RGB */
  Uint32 color = (Uint32)rgb_to_color(((integer_t**)x_y_c)[2]);
  show_pixel(((integer_t*)x_y_c)[0],
	     ((integer_t*)x_y_c)[1],
	     color);
  return NULL;
}

void e_update_screen(){
  update_screen();
}

