/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

#include "library.h"
#include <stdio.h>
/* Note that there is no need to include math.h; every symbol is 
   explicitly loaded, so any declaration would be overridden. */


/* The following pointers to functions are needed because we must explicitly
   load some symbols from libm.so which are different depending on the word
   width: in initialize_c_library() we test the word width, then we load the
   appropriate version. */
static float_t (*loaded_sin)(float_t);
static float_t (*loaded_cos)(float_t);
static float_t (*loaded_tan)(float_t);
static float_t (*loaded_tan)(float_t);
static float_t (*loaded_asin)(float_t);
static float_t (*loaded_acos)(float_t);
static float_t (*loaded_atan)(float_t);
static float_t (*loaded_sinh)(float_t);
static float_t (*loaded_cosh)(float_t);
static float_t (*loaded_tanh)(float_t);
static float_t (*loaded_tanh)(float_t);
static float_t (*loaded_asinh)(float_t);
static float_t (*loaded_acosh)(float_t);
static float_t (*loaded_atanh)(float_t);
static float_t (*loaded_exp)(float_t);
static float_t (*loaded_log)(float_t);
static float_t (*loaded_sqrt)(float_t);
static float_t (*loaded_cbrt)(float_t);
static float_t (*loaded_pow)(float_t, float_t);

/* The following constants are visible from epsilon: */
word_t c_pi;
word_t c_e;

void initialize_c_library(){
  dynamic_library_handle_t handle;

#ifdef DEBUG
  fprintf(stderr, "Initializing C library float_math... ");
#endif

  /* Open the math library: */
  handle = open_dynamic_library("libm.so");
  
  /* Load functions: this is slightly tricky because the width of
     a float_t depends on the platform: we want functions taking and
     returning float_t objects, where float may be float or double: */
  if(sizeof(float_t) == sizeof(float)){
    loaded_sin = load_symbol_from_handle(handle, "sinf");
    loaded_cos = load_symbol_from_handle(handle, "cosf");
    loaded_tan = load_symbol_from_handle(handle, "tanf");
    loaded_asin = load_symbol_from_handle(handle, "asinf");
    loaded_acos = load_symbol_from_handle(handle, "acosf");
    loaded_atan = load_symbol_from_handle(handle, "atanf");

    loaded_sinh = load_symbol_from_handle(handle, "sinhf");
    loaded_cosh = load_symbol_from_handle(handle, "coshf");
    loaded_tanh = load_symbol_from_handle(handle, "tanhf");
    loaded_asinh = load_symbol_from_handle(handle, "asinhf");
    loaded_acosh = load_symbol_from_handle(handle, "acoshf");
    loaded_atanh = load_symbol_from_handle(handle, "atanhf");

    loaded_exp = load_symbol_from_handle(handle, "expf");
    loaded_log = load_symbol_from_handle(handle, "logf");
    loaded_pow = load_symbol_from_handle(handle, "powf");
    loaded_sqrt = load_symbol_from_handle(handle, "sqrtf");
    loaded_cbrt = load_symbol_from_handle(handle, "cbrtf");
  }else if(sizeof(float_t) == sizeof(double)){
    loaded_sin = load_symbol_from_handle(handle, "sin");
    loaded_cos = load_symbol_from_handle(handle, "cos");
    loaded_tan = load_symbol_from_handle(handle, "tan");
    loaded_asin = load_symbol_from_handle(handle, "asin");
    loaded_acos = load_symbol_from_handle(handle, "acos");
    loaded_atan = load_symbol_from_handle(handle, "atan");

    loaded_sinh = load_symbol_from_handle(handle, "sinh");
    loaded_cosh = load_symbol_from_handle(handle, "cosh");
    loaded_tanh = load_symbol_from_handle(handle, "tanh");
    loaded_asinh = load_symbol_from_handle(handle, "asinh");
    loaded_acosh = load_symbol_from_handle(handle, "acosh");
    loaded_atanh = load_symbol_from_handle(handle, "atanh");

    loaded_exp = load_symbol_from_handle(handle, "exp");
    loaded_log = load_symbol_from_handle(handle, "log");
    loaded_pow = load_symbol_from_handle(handle, "pow");
    loaded_sqrt = load_symbol_from_handle(handle, "sqrt");
    loaded_cbrt = load_symbol_from_handle(handle, "cbrt");
  }else
    fatal("float_math.c: sizeof(float_t) == sizeof(long double)");

  /* Exported constants: */
  c_pi = float_to_word((float_t)3.14159265358979323846264338327);
  c_e = float_to_word((float_t)2.7182818284590452353602874713526625);

#ifdef DEBUG
  fprintf(stderr, "done.\n");
#endif
}

/* Here begin definitions of symbols to be exported. All of these are
   visible from epsilon: */

word_t c_sin(word_t x){
  return float_to_word(loaded_sin(word_to_float(x)));
}
word_t c_cos(word_t x){
  return float_to_word(loaded_cos(word_to_float(x)));
}
word_t c_tan(word_t x){
  return float_to_word(loaded_tan(word_to_float(x)));
}
word_t c_asin(word_t x){
  return float_to_word(loaded_asin(word_to_float(x)));
}
word_t c_acos(word_t x){
  return float_to_word(loaded_acos(word_to_float(x)));
}
word_t c_atan(word_t x){
  return float_to_word(loaded_atan(word_to_float(x)));
}
word_t c_sinh(word_t x){
  return float_to_word(loaded_sinh(word_to_float(x)));
}
word_t c_cosh(word_t x){
  return float_to_word(loaded_cosh(word_to_float(x)));
}
word_t c_tanh(word_t x){
  return float_to_word(loaded_tanh(word_to_float(x)));
}
word_t c_asinh(word_t x){
  return float_to_word(loaded_asinh(word_to_float(x)));
}
word_t c_acosh(word_t x){
  return float_to_word(loaded_acosh(word_to_float(x)));
}
word_t c_atanh(word_t x){
  return float_to_word(loaded_atanh(word_to_float(x)));
}
word_t c_exp(word_t x){
  return float_to_word(loaded_exp(word_to_float(x)));
}
word_t c_log(word_t x){
  return float_to_word(loaded_log(word_to_float(x)));
}
word_t c_sqrt(word_t x){
  return float_to_word(loaded_sqrt(word_to_float(x)));
}
word_t c_cbrt(word_t x){
  return float_to_word(loaded_cbrt(word_to_float(x)));
}
word_t c_pow(word_t b_e){
  return float_to_word(loaded_pow(((float_t*)b_e)[0],
				  ((float_t*)b_e)[1]));
}


////////// Just a test


word_t output_something(){
  printf("Something.\n");
  return (word_t)(integer_t)19;
}

word_t output_integer_with_parameter(word_t x){
  printf("<"INTEGER_T_FORMAT">\n", (integer_t)x);
  return x;
}

/*
// just a driver
int main(){
  initialize_dynamic_libraries();

  initialize_c_module();
  printf("sin(PI/2) = %f\n", word_to_float(sine(float_to_word(M_PI/2))));
  printf("cos(PI/2) = %f\n", word_to_float(cosine(float_to_word(M_PI/2))));

  close_dynamic_libraries();
}
*/
