/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


#include "library.h"

#include <stdio.h>
#include <errno.h>

/* This returns a pair <FILE*, success>: */
word_t e_open_file(word_t epsilon_filename, char* access){
  char* filename = epsilon_string_to_c_string(epsilon_filename);
  word_t* r = allocate_exact(2); /* the pair to be returned */
  r[0] = r[1] = fopen(filename, access);
  free(filename); /* filename was malloc()'d */
  return r;
}

/* The following objects are visible from epsilon: */

word_t e_standard_input;
word_t e_standard_output;
word_t e_standard_error;

/* This returns an eAM 'boolean' integer meaning 'success': */
word_t e_close_file(word_t filestar){
  return (word_t)(integer_t) ! fclose((FILE*)filestar);
}

/* All of these return a pair <FILE*, success>: */
word_t e_open_file_r(word_t filename){
  return e_open_file(filename, "r");
}
word_t e_open_file_w(word_t filename){
  return e_open_file(filename, "w");
}
word_t e_open_file_rw(word_t filename){
  return e_open_file(filename, "rw");
}
word_t e_open_file_a(word_t filename){
  return e_open_file(filename, "a");
}

/* This one takes a FILE* and returns a pair <integer, success> where
   the eAM integer stands for a character: */
word_t e_read_8bit_character(word_t filestar){
  integer_t *r = allocate_exact(2);
  char c;
  
  r[1] = (integer_t)
    fread(&c, 1 /* we assume a char is 8 bit */, 1, (FILE*)filestar);
  r[0] = (integer_t)c;
  
  return r;
}

/* This one takes a pair <c, FILE*> and returns an eAM integer standing for
   'success': */
word_t e_write_8bit_character(word_t c_filestar){
  char c = (char)(((integer_t*)c_filestar)[0]);
  
  return (word_t)(integer_t)
    fwrite(&c, 1 /* we assume a char is 8 bit */, 1, ((FILE**)c_filestar)[1]);
}

void initialize_c_library(){
#ifdef DEBUG
  fprintf(stderr, "Initializing the C-library file.c... ");
#endif

  /* We don't need dynamic libraries here, since the random numbers
     functionality is built into the C library. */

  /* Initialize globals which are visisble from epsilon: */
  e_standard_input = (word_t)stdin;
  e_standard_output = (word_t)stdout;
  e_standard_error = (word_t)stderr;

#ifdef DEBUG
  fprintf(stderr, "done.\n");
#endif
}
