/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: display.hh,v 1.31 2004/05/27 20:34:50 dheck Exp $
 */
#ifndef DISPLAY_HH
#define DISPLAY_HH

#include "enigma.hh"
#include "px/px.hh"

//----------------------------------------
// Definition of models
//----------------------------------------
namespace display
{
    class DisplayLayer;
    class ModelLayer;

    /*! Animations can invoke a callback of this type on completion.
      Note that you may not delete or replace models other than the
      one that induced the callback from inside the callback--use a
      timer or a flag to do this. */
    class ModelCallback {
    public:
        virtual ~ModelCallback() {}
        virtual void animcb() = 0;
    };

    class Animation {
    public:
        virtual ~Animation() {}
        virtual void set_callback (ModelCallback *) {}
        virtual void reverse() {}

	virtual bool is_garbage() const { return false; }
        virtual void tick(double /*dtime*/) {}
        virtual bool has_changed(px::Rect &/*changed_region*/) { return false; }
    };

    class Model : public Animation {
    public:

        virtual void draw(px::GC &/*gc*/, int /*x*/, int /*y*/) {}
        virtual void draw_shadow(px::GC &/*gc*/, int /*x*/, int /*y*/) {}

        virtual Model *get_shadow() const { return 0; }

        virtual void expose (ModelLayer */*ml*/, const px::V2 &/*pos*/) {}
        virtual void remove (ModelLayer */*ml*/) {}

        virtual Model *clone() = 0;
    };

/* -------------------- Functions -------------------- */

    void InitModels();
    void ShutdownModels();

    Model * MakeModel (const std::string &name);


    int DefineImage (const char *name, const char *fname,
                     int xoff, int yoff);
    int DefineSubImage (const char *name, const char *fname,
                        int xoff, int yoff, px::Rect r);
    void DefineRandModel (const char *name, int n, const char **names);
    void DefineShadedModel (const char *name, const char *model, const char *shade);
    void DefineOverlayImage (const char *name, int n, const char **images);
    void DefineComposite (const char *name, const char *bgname, const char *fgname);
    void DefineAnim (const char *name, bool loop_p);
    void AddFrame (const char *name, const char *model, double time);
    void DefineAlias (const char *name, const char *othername);
}

//----------------------------------------
// Models on the grid
//----------------------------------------
namespace display
{
    using enigma::GridPos;
    using enigma::GridLayer;
    using enigma::GridLoc;

    Model*  SetModel (const GridLoc & l, const std::string &modelname);
    Model*  SetModel (const GridLoc & l, Model *m);
    void    KillModel (const GridLoc & l);
    Model*  GetModel (const GridLoc & l);
    Model*  YieldModel (const GridLoc & l);
}

//----------------------------------------
// Sprites (movable models)
//----------------------------------------
namespace display
{
    enum SpriteLayer
    {
        SPRITE_ACTOR, SPRITE_EFFECT, SPRITE_DEBRIS
    };

    enum FollowMode
    {
        FOLLOW_NONE,            // Don't follow any sprite
        FOLLOW_SCROLLING,       // Scroll the screen
        FOLLOW_SCREEN,          // Flip the screen region
        FOLLOW_SCREENSCROLLING,  // Scroll to the next screen
        FOLLOW_SMOOTH,          // Follow pixel by pixel
    };

    typedef unsigned int SpriteId;

    class DL_Sprites;

    class SpriteHandle {
        DL_Sprites *layer;
        unsigned id;
    public:
        SpriteHandle (DL_Sprites *l, unsigned spriteid)
        : layer(l), id(spriteid)
        {}
        SpriteHandle();

        void kill();
        void move (const px::V2 &newpos) const;
        void replace_model (Model *m) const;
        Model *get_model () const;
        void set_callback (ModelCallback *cb) const;
        void hide() const;
        void show() const;
    };

    /*! Add a new effect sprite. */
    SpriteHandle AddEffect (const px::V2 &pos, const char *modelname);

    /*! Create a new sprite.  If modelname==0, the sprite is considered
      invisible.  */
    SpriteHandle AddSprite (const px::V2 &pos, const char *modelname=0);


    void SetReferencePoint (const px::V2 &point);
    void SetFollowMode (FollowMode m);
    void FocusReferencePoint();
    void GetReferencePointCoordinates(int *x, int *y);
}

/* -------------------- Rubber bands -------------------- */
namespace display
{
    class DL_Lines;

    class RubberHandle {
    public:
        RubberHandle (DL_Lines *layer=0, unsigned id=0);
        operator unsigned() { return id; }

        void update_first (const px::V2 &p1);
        void update_second (const px::V2 &p2);
        void kill();

        DL_Lines *line_layer;
        unsigned id;
    };

    RubberHandle AddRubber (const px::V2 &p1, const px::V2 &p2);
}


/* -------------------- Status bar -------------------- */
namespace display
{
    using enigma_player::Inventory;

    class StatusBar {
    public:
        virtual ~StatusBar() {}
        virtual void set_inventory (const std::vector<std::string> &modelnames) = 0;

        virtual void show_text (const std::string &str, 
                                bool scrolling, 
                                double duration = -1) = 0;
        virtual void hide_text() = 0;

        virtual void show_move_counter (bool active) = 0;
        virtual void show_odometer (bool active)     = 0;

        virtual void set_time (double time) = 0;
        virtual void set_speed (double speed) = 0;
        virtual void set_travelled_distance (double distance) = 0;
        virtual void set_counter (int nummoves) = 0;
    };

    StatusBar *GetStatusBar();

#define STATUSBAR display::GetStatusBar()


/* -------------------- Interface to display engine -------------------- */

    enum DisplayFlags
    {
        SHOW_FLOOR     = 0x01,
        SHOW_STONES    = 0x02,
        SHOW_ITEMS     = 0x04,
        SHOW_SHADES    = 0x08,
        SHOW_SPRITES   = 0x10,
	SHOW_TIME      = 0x20,
	SHOW_INVENTORY = 0x40,
        SHOW_ALL       = 0x7f
    };

    void ToggleFlag(DisplayFlags flag);


    void Init();
    void Shutdown();

    void            NewWorld (int w, int h);
    void            ResizeGameArea (int w, int h);
    const px::Rect& GetGameArea ();

    void DrawAll(px::GC &gc);
    void RedrawAll(px::Screen *sfc);
    void Redraw(px::Screen *sfc);
    void Tick(double dtime);
}

#endif
