/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: gui.cc,v 1.22 2004/01/09 21:37:56 dheck Exp $
 */

#include "gui.hh"
#include "enigma.hh"
#include "sound.hh"
#include "video.hh"
#include "px/px.hh"
#include <memory>
#include <cassert>

using namespace gui;
using namespace px;
using namespace std;

#define SCREEN px::Screen::get_instance()

/* -------------------- Widget -------------------- */

Widget::Widget(Widget *parent)
: area(), menu(0), m_parent(parent), m_listener(0)
{}

void Widget::redraw() {
    if (menu) 
        menu->invalidate_area(get_area());
}

void Widget::redraw(const px::Rect &r) {
    if (menu) 
        menu->invalidate_area(r);
}

void Widget::invoke_listener() {
    if (m_listener)
        m_listener->on_action(this);
}

/* -------------------- Image -------------------- */

void Image::draw (px::GC &gc, const px::Rect &/*r*/) {
    if (px::Surface *s = enigma::GetImage(imgname.c_str()))
        blit(gc, get_x(), get_y(), s);
}


/* -------------------- Container -------------------- */

Container::~Container() {
    clear();
}

void Container::clear() {
    delete_sequence(m_widgets.begin(), m_widgets.end());
    m_widgets.clear();
}

void Container::add_child (Widget *w) {
    if (w) {
        m_widgets.push_back(w);
        w->set_parent(this);
        w->move (get_x()+w->get_x(),
                 get_y()+w->get_y());
    }
}


void Container::draw (px::GC& gc, const px::Rect &r) {
    for (iterator i=begin(); i!=end(); ++i) {
        Widget *w = *i;
        Rect rr = intersect(r, w->get_area());
        clip(gc, rr);
        w->draw(gc,rr);
    }
}

Widget * Container::find_widget(int x, int y) {
    for (iterator i=begin(); i!=end(); ++i) {
        Widget *w = *i;
        if (w->get_area().contains(x,y))
            return w;
    }
    return 0;
}

Widget * Container::find_adjacent_widget(Widget *from, int x, int y) {
    // valid values for x/y : 1/0, -1/0, 0/1, 0/-1
    assert(from && x>=-1 && x<=1 && y>=-1 && y<=1 && abs(x+y) == 1);

    if (!from) return 0;

    int       best_distance = INT_MAX;
    Widget   *best_widget   = 0;
    px::Rect  farea         = from->get_area();

    for (iterator i=begin(); i!=end(); ++i) {
        Widget   *w        = *i;
        px::Rect  warea    = w->get_area();
        bool      adjacent = true;
        int       distance = 0;

        if (x) { // check for y-overlap
            if (farea.y>(warea.y+warea.h-1) || warea.y>(farea.y+farea.h-1)) {
                adjacent = false;
            }
            else {
                distance = (warea.x-farea.x)*x;
            }
        }
        else { // check for x-overlap
            if (farea.x>(warea.x+warea.h-1) || warea.x>(farea.x+farea.h-1)) {
                adjacent = false;
            }
            else {
                distance = (warea.y-farea.y)*y;
            }
        }

        if (adjacent && distance>0 && distance<best_distance) {
            best_distance = distance;
            best_widget   = w;
        }
    }

    return best_widget;
}

void Container::redraw_child (Widget */*w*/, const px::Rect &/*r*/) {
}

void Container::move (int x, int y) {
    int dx = x-get_x();
    int dy = y-get_y();

    for (iterator i=begin(); i!=end(); ++i) {
        Widget *w = *i;
        w->move(dx + w->get_x(), dy+w->get_y());
    }
}

px::Rect Container::boundingbox() {
    if (!m_widgets.empty()) {
        iterator i=begin();
        Rect bbox=(*i)->get_area();
        for (++i; i!=end(); ++i)
            bbox = px::boundingbox(bbox, (*i)->get_area());
        return bbox;
    } else
        return get_area();
}


/* -------------------- Label -------------------- */

Label::Label (const std::string &text,
              HAlignment halign, 
              VAlignment valign)
: m_text (text),
  m_font(enigma::GetFont("menufont")),
  m_halign(halign),
  m_valign(valign)
{}


void Label::set_text (const std::string &text) {
    if (text != m_text) {
        m_text = text;
        redraw();
    }
}

void Label::set_font (px::Font *font) {
    if (m_font != font) {
        m_font = font;
        redraw();
    }
}

void Label::draw (px::GC &gc, const px::Rect &) 
{
    Font *f = m_font;
    int h = f->get_height();
    int w = f->get_width(m_text.c_str());

    int x = get_x(), y=get_y();
    switch (m_halign) {
    case HALIGN_LEFT: break;
    case HALIGN_RIGHT: x += get_w() - w; break;
    case HALIGN_CENTER: x += (get_w()-w)/2; break;
    }
    switch (m_valign) {
    case VALIGN_TOP: break;
    case VALIGN_BOTTOM: y += get_h() - h; break;
    case VALIGN_CENTER: y += (get_h()-h)/2; break;
    }

    f->render (gc, x, y, m_text.c_str());
}

void Label::set_alignment (HAlignment halign, VAlignment valign) {
    if (halign != m_halign || valign != m_valign) {
        m_halign = halign;
        m_valign = valign;
        redraw();
    }
}


/* -------------------- Button -------------------- */

Button::Button() : m_activep (false) {
}

void Button::activate() 
{
    sound::PlaySoundGlobal ("menuswitch");
    m_activep = true;
    redraw();
}

void Button::deactivate() {
    m_activep = false;
    redraw();
}

void Button::draw(px::GC &gc, const px::Rect &r) {
    const int borderw = 4;

    px::Surface *s = enigma::GetImage (m_activep ? "buttonhl" : "button");

    if (s) {                    // Ugly, but hey, it works
        set_color (gc, 0,0,0);
        Rect srcrect (0,0,borderw, borderw);
        Rect area = get_area();

        // background
        box (gc, smaller(area, borderw));

        // corners
        blit (gc, area.x, area.y, s, srcrect);
        srcrect.x += s->width()-borderw;
        blit (gc, area.x+area.w-borderw, area.y, s, srcrect);
        srcrect.x = 0;
        srcrect.y += s->height()-borderw;
        blit (gc, area.x, area.y+area.h-borderw, s, srcrect);
        srcrect.x += s->width()-borderw;
        blit (gc, area.x+area.w-borderw, area.y+area.h-borderw, s, srcrect);

        // horizontal borders
        {
            int tilew = s->width() - 2*borderw;
            int ntiles = (area.w - 2*borderw) / tilew;
            int x = area.x + borderw;
            for (int i=0; i<ntiles; ++i) {
                blit (gc, x, area.y, s, Rect (borderw, 0, tilew, borderw));
                blit (gc, x, area.y+area.h-borderw, s,
                      Rect (borderw, s->height()-borderw, tilew, borderw));
                x += tilew;
            }
            int restw = (area.w - 2*borderw) - tilew*ntiles;
            blit (gc, x, area.y, s, Rect (borderw, 0, restw, borderw));
            blit (gc, x, area.y+area.h-borderw, s,
                  Rect (borderw, s->height()-borderw, restw, borderw));
        }
        // vertical borders
        {
            int tileh = s->height() - 2*borderw;
            int ntiles = (area.h - 2*borderw) / tileh;
            int y = area.y + borderw;
            for (int i=0; i<ntiles; ++i) {
                blit (gc, area.x, y, s, Rect (0, borderw, borderw, tileh));
                blit (gc, area.x+area.w-borderw, y, s,
                      Rect (s->width()-borderw, borderw, borderw, tileh));
                y += tileh;
            }
            int resth = (area.h - 2*borderw) - tileh*ntiles;
            blit (gc, area.x, y, s, Rect (0, borderw, borderw, resth));
            blit (gc, area.x+area.w-borderw, y, s,
                  Rect (s->width()-borderw, borderw, borderw, resth));
        }
    }
    else {
        set_color (gc, 0,0,0);
        box (gc, r);
        set_color (gc, 160,160,160);
        frame (gc, r);
        frame (gc, smaller(r, 1));
    }
}


/* -------------------- PushButton -------------------- */

PushButton::PushButton() : m_pressedp (false) {
}

bool PushButton::on_event(const SDL_Event &e) {
    bool was_pressed = m_pressedp;

    switch (e.type) {
    case SDL_KEYDOWN:
        if (e.key.keysym.sym != SDLK_RETURN &&
            e.key.keysym.sym != SDLK_SPACE) break;
        // fall-through
    case SDL_MOUSEBUTTONDOWN:
        m_pressedp = true;
        break;

    case SDL_KEYUP:
        if (e.key.keysym.sym != SDLK_RETURN &&
            e.key.keysym.sym != SDLK_SPACE) break;
        // fall-through
    case SDL_MOUSEBUTTONUP:
        m_pressedp = false;
        break;
    }

    bool changed = (was_pressed != m_pressedp);
    if (changed) {
        redraw();
        if (!m_pressedp) {
            sound::PlaySoundGlobal ("menuok");
            invoke_listener();
        }
    }

    return changed;
}

void PushButton::deactivate() {
    m_pressedp = false;
    redraw();
    Button::deactivate();
}


/* -------------------- TextButton -------------------- */

px::Font *TextButton::menufont = 0;
px::Font *TextButton::menufont_pressed = 0;

TextButton::TextButton(const string &t, ActionListener *al)
: text(t)
{
    if (menufont == 0) {
        menufont = enigma::GetFont("menufont");
        menufont_pressed = enigma::GetFont("menufontsel");
    }
    set_listener(al);
}

void TextButton::set_text(const std::string &t) {
    if (t != text) {
        text = t;
        redraw();
    }
}

void TextButton::draw(px::GC &gc, const px::Rect &r) {
    Button::draw(gc,r);
    Font *f = is_pressed() ? menufont_pressed : menufont;
    int h = f->get_height();
    int w = f->get_width(text.c_str());
    int x = get_x() + (get_w()-w)/2;
    int y = get_y() + (get_h()-h)/2;

    f->render (gc, x, y, text.c_str());
}


/* -------------------- ValueButton -------------------- */

ValueButton::ValueButton(const std::string &t, int min_value_, int max_value_)
: TextButton(t, this),
  min_value(min_value_),
  max_value(max_value_)
{}

bool ValueButton::inc_value(int offset) {
    int old_value = get_value();
    return update_value(old_value, old_value+offset);
}

void ValueButton::update() {
    update_value(-1, get_value());
}

bool ValueButton::update_value(int old_value, int new_value) {
    new_value = Clamp(new_value, min_value, max_value);
    if (new_value != old_value) {
        set_value(new_value);
        set_text(build_text(new_value));
        return true;
    }
    return false;
}

bool ValueButton::on_event (const SDL_Event &e) {
    // handles button movement and
    bool handled = PushButton::on_event(e);

    if (e.type == SDL_KEYDOWN) {
        bool keyhandled   = true;
        bool changed = false;

        switch (e.key.keysym.sym) {
        case SDLK_PAGEUP: changed = inc_value(1); break;
        case SDLK_PAGEDOWN:  changed = inc_value(-1); break;
        default : keyhandled = false; break;
        }

        if (keyhandled) {
            handled = true;
            sound::PlaySoundGlobal (changed ? "menuswitch" : "menustop");
        }
    }
    return handled;
}

void ValueButton::on_action(Widget *) {
    if (!inc_value(1))
        update_value(get_value(), min_value);
}


/* -------------------- ImageButton -------------------- */

ImageButton::ImageButton(const string &unselected,
                         const string &selected,
                         ActionListener *al)
: fname_sel(selected), fname_unsel(unselected)
{
    set_listener(al);
}

void ImageButton::draw(px::GC &gc, const px::Rect &r) {
    Button::draw(gc, r);
    string &fname = is_pressed() ? fname_sel : fname_unsel;

    if (Surface *s = enigma::GetImage(fname.c_str())) {
        int w=s->width();
        int h=s->height();
        int x = get_x() + (get_w()-w)/2;
        int y = get_y() + (get_h()-h)/2;
        blit(gc, x, y, s);
    }
}


/* -------------------- Menu -------------------- */

Menu::Menu()
: active_widget(0), quitp(false), abortp(false) {
}

void Menu::add(Widget *w) {
    Container::add_child(w); 
    w->set_menu(this);
}

void Menu::add(Widget *w, px::Rect r) {
    w->set_area(r);
    add(w);
}


void Menu::quit() {
    quitp=true;
}

void Menu::abort() {
    abortp=true;
}

void Menu::invalidate_area(const px::Rect &r) {
    dirtyrects.add(r);
}

void Menu::invalidate_all() {
    dirtyrects.clear();
    dirtyrects.push_back(SCREEN->size());
}


bool Menu::manage() {
    draw_all();
    quitp=abortp=false;
    while (!(quitp || abortp)) {
        SCREEN->flush_updates();
        SDL_Event e;
        while (SDL_PollEvent(&e)) {
            handle_event(e);
        }
        if (!dirtyrects.empty()) {
            RectList::iterator i    = dirtyrects.begin();
            RectList::iterator end_ = dirtyrects.end();
            for (; i!=end_; ++i)
                redraw_area(*i);
            dirtyrects.clear();
        }
        SDL_Delay(10);
        tick (0.01);
    }
    sound::PlaySoundGlobal ("menuexit");
    return !abortp;
}

void Menu::goto_adjacent_widget(int xdir, int ydir) {
    Widget *next_widget = 0;
    if (active_widget) {
        next_widget = find_adjacent_widget(active_widget, xdir, ydir);
    }
    else { // no active_widget yet
        if ((xdir+ydir)>0) { // take first
            next_widget = *begin();
        }
        else { // take last
            iterator e = end();
            for (iterator i = begin(); i != e; ++i) {
                next_widget = *i;
            }
        }
    }

    if (next_widget) {
        switch_active_widget(next_widget);
    }
    else { // no more widgets into that direction found
        sound::PlaySoundGlobal ("menustop");
    }
}

void Menu::handle_event(const SDL_Event &e) {
    if (on_event(e))
        return;

    switch (e.type) {
    case SDL_QUIT:
        abort();
        break;
    case SDL_MOUSEMOTION:
        track_active_widget( e.motion.x, e.motion.y );
        break;
    case SDL_KEYDOWN:
        if (!active_widget || !active_widget->on_event(e)) {
            // if not handled by active_widget
            switch (e.key.keysym.sym) {
            case SDLK_ESCAPE:
                abort();
                break;
            case SDLK_DOWN:  goto_adjacent_widget( 0,  1); break;
            case SDLK_UP:    goto_adjacent_widget( 0, -1); break;
            case SDLK_RIGHT: goto_adjacent_widget( 1,  0); break;
            case SDLK_LEFT:  goto_adjacent_widget(-1,  0); break;
            default:
                break;
            }
        }

        break;
    case SDL_MOUSEBUTTONDOWN:
    case SDL_MOUSEBUTTONUP:
        track_active_widget( e.button.x, e.button.y );
        if (active_widget) active_widget->on_event(e);
        break;
    default:
        if (active_widget) active_widget->on_event(e);
    }
}

void Menu::switch_active_widget(Widget *to_activate) {
    if (to_activate != active_widget) {
        if (active_widget)
            active_widget->deactivate();
        if (to_activate)
            to_activate->activate();
        active_widget = to_activate;
    }
}

void Menu::track_active_widget( int x, int y ) {
    switch_active_widget(find_widget(x, y));
}


void Menu::center() {
    if (m_widgets.size() > 0) {
        using std::min;
        using std::max;

        Rect a = m_widgets[0]->get_area();
        for (unsigned i=1; i<m_widgets.size(); ++i)
        {
            Rect r = m_widgets[i]->get_area();
            a.x = min(r.x, a.x);
            a.y = min(r.y, a.y);
            a.w += max(0, r.x+r.w-a.x-a.w);
            a.h += max(0, r.y+r.h-a.y-a.h);
        }
        Rect c=px::center(SCREEN->size(), a);
        int dx = c.x-a.x;
        int dy = c.y-a.y;

        for (unsigned i=0; i<m_widgets.size(); ++i) {
            Rect r = m_widgets[i]->get_area();
            r.x += dx;
            r.y += dy;
            m_widgets[i]->set_area(r);
        }
    }
}


void Menu::redraw_area(const px::Rect &r) {
    video::HideMouse();
    GC gc(SCREEN->get_surface());
    draw(gc, r);
    SCREEN->update_rect(r);
    video::ShowMouse();
}

void Menu::draw_all() {
    redraw_area(SCREEN->size());
    SCREEN->update_all();
}

void Menu::draw (px::GC &gc, const px::Rect &r)
{
    clip(gc, r);
    draw_background(gc);
    Container::draw(gc,r);
}
