(define-module (emacsy util)
  #:use-module (ice-9 optargs)
  #:use-module (oop goops)
  #:use-module (ice-9 pretty-print)
  #:use-module (ice-9 receive)
  #:use-module (srfi srfi-1)
  #:use-module (debugging assert)
  #:use-module (system repl error-handling)
  ;#:export-syntax (define-syntax-public)
)
;; I want to get rid of this state if I can.
(define-public continue-command-loop? (make-unbound-fluid))
(define-public debug-on-error? #f)
(define-syntax define-syntax-public
  (syntax-rules ()
    ((define-syntax-public name . body)
     (begin
       (define-syntax name . body)
       (export-syntax name)))))
(export-syntax define-syntax-public)
(define-syntax-public string-case
  (syntax-rules (else)
    ((_ str (else e1 ...))
     (begin e1 ...))
    ((_ str (e1 e2 ...))
     (when (string=? str e1) e2 ...))
    ((_ str (e1 e2 ...) c1 ...)
     (if (string=? str e1)
         (begin e2 ...)
         (string-case str c1 ...)))))
(define-syntax-public define-class-public
  (syntax-rules ()
    ((define-class-public name . body)
     (begin
       (define-class name . body)
       (export name)
       ))))
(define-syntax-public define-method-public
  (syntax-rules ()
    ((define-method-public (name . args) . body)
     (begin
       (define-method (name . args) . body)
       (export name)
       ))))
(define-syntax define-generic-public
  (syntax-rules ()
    ((define-generic-public name)
     (begin
       (define-generic name)
       (export name)))))
(export define-generic-public)
(define-syntax-public repeat
  (syntax-rules ()
    ((repeat c e ...)
     (repeat-func c (lambda () e ...)))))
(define-syntax-public in-out
  (syntax-rules ()
    ((in-out in thunk out)
     (in-out-guard (lambda () in)
                   (lambda () thunk)
                   (lambda () out)))
    ((in-out in thunk out pass-keys)
     (in-out-guard (lambda () in)
                   (lambda () thunk)
                   (lambda () out)
                   pass-keys))))
(define-syntax-public incr!
  (syntax-rules ()
    ((incr! variable inc)
     (begin
       (set! variable (+ variable inc))
       variable))
    ((incr! variable)
     (incr! variable 1))))
(define-syntax-public decr!
  (syntax-rules ()
    ((decr! variable inc)
     (incr! variable (- inc)))
    ((decr! variable)
     (decr! variable 1))))
(define-syntax-public cons!
  (syntax-rules ()
    ((cons! elm list)
     (begin
       (set! list (cons elm list))
       list))))
(define-public pp pretty-print)
(define-public (repeat-func count func)
  (if (<= count 0)
      #f
      (begin
        (func)
        (repeat-func (1- count) func))))
(define-public (emacsy-time)
  (exact->inexact (/ (tms:clock (times)) internal-time-units-per-second)))
(define-public (find-first f lst)
  "Return the first result f which is not false and #f if no such result is found."
  (if (null? lst)
      #f
      (or (f (car lst))
          (find-first f (cdr lst)))))
(define-public (alist-values alist)
  (map cdr alist))

(define-public (alist-keys alist)
  (map car alist))
(define-public (intersect-order list ordered-list )
  "Returns the intersection of the two lists ordered according to the
second argument."
  (filter (lambda (x) (memq x list)) 
          ordered-list))
(define-public (rcar lst)
  (car (reverse lst)))

(define-public (rcdr lst)
  (reverse (cdr (reverse lst))))
(define-public (emacsy-log-info format-msg . args)
  (apply format (current-error-port) format-msg args)
  (newline (current-error-port)))
(define-public (member-ref x list)
  (let ((sublist (member x list)))
    (if sublist
        (- (length list) (length sublist)) 
        #f)))
(define*-public (in-out-guard in thunk out #:optional (pass-keys '(quit quit-command)))
  (run-thunk in)
  ;if debug-on-error?
  ;; Don't run this as robustly so that we can debug the errors
  ;; more easily.
  (when #f
    (receive (result . my-values) (run-thunk thunk)
    (run-thunk out)
    (apply values result my-values)))
  
  (receive (result . my-values) 
      (catch #t
        (if debug-on-error? 
            (lambda ()
              (call-with-error-handling thunk #:pass-keys pass-keys))
            thunk)
        (lambda (key . args)
          (run-thunk out)
          (apply throw key args)))
    (run-thunk out)
    (apply values result my-values)))

;; Make code a little more obvious.
(define-public (run-thunk t)
  (t))
(define-public (vector= a b)
  (assert (vector? a) (vector? b) report: "vector= ")
  (let ((len (vector-length a)))
   (and (= (vector-length a) (vector-length b))
        (let loop ((i 0))
          (if (>= i len)
              #t
              (if (= (vector-ref a i) (vector-ref b i))
               (loop (1+ i))
               #f))))))
(define*-public (with-backtrace* thunk #:optional (no-backtrace-for-keys '()))
  (with-throw-handler 
   #t
   thunk
   (lambda (key . args)
     (when (not (memq key no-backtrace-for-keys))
       (emacsy-log-error 
        "ERROR: Throw to key `~a' with args `~a'." key args)
       (backtrace)))))
(define-public (emacsy-log-error format-msg . args)
  (apply format (current-error-port) format-msg args)
  (newline (current-error-port)))

(define-public (emacsy-log-trace format-msg . args)
  (apply format (current-error-port) format-msg args)
  (newline (current-error-port)))
(define-public (pp-string obj)
  (call-with-output-string (lambda (port) (pp obj port))))
(define-public (emacsy-log-debug format-msg . args)
  (apply format (current-error-port) format-msg args)
  (newline (current-error-port)))
(define-public (list-insert! lst k val)
  "Insert val into list such that (list-ref list k) => val."
  (receive (pre post) (split-at! lst k)
    (append! pre (list val) post)))
(define-public (read-from-string string)
  (call-with-input-string string (lambda (port) (read port))))
;; object-tracker :: (a -> b) -> ((a -> b), (b -> a))
(define-public (object-tracker a->b)
  (define (swap-cons c)
    (cons (cdr c) (car c)))
  (let ((cache (make-hash-table)))
    (values
     (lambda (x)
       (let ((y (a->b x)))
         (if (hash-ref cache y)
             (emacsy-log-warning "object-tracker has a duplicate for pairs ~a ~a" (cons x y) (swap-cons (hash-get-handle cache y))))
         (hash-set! cache y x)
         y))
     (lambda (y)
       (or (hash-ref cache y) y)))))
(define-public (emacsy-log-warning format-msg . args)
  (apply format (current-error-port) format-msg args)
  (newline (current-error-port)))
