(define-module (emacsy buffer)
  #:use-module (ice-9 optargs)
  #:use-module (ice-9 q)
  #:use-module (ice-9 gap-buffer)
  #:use-module (ice-9 receive)
  #:use-module (ice-9 regex)
  #:use-module (srfi srfi-26)
  #:use-module (string completion)
  #:use-module (oop goops)
  #:use-module (emacsy util)
  #:use-module (emacsy mru-stack)
  #:use-module (emacsy self-doc)
  #:use-module (emacsy event)
  #:use-module (emacsy keymap)
  #:use-module (emacsy command)
  #:use-module (emacsy klecl)
  #:use-module (emacsy mode)
  #:use-module (rnrs base))
(define-syntax-public with-buffer
  (syntax-rules ()
    ((with-buffer buffer e ...)
     (let ((old-buffer (current-buffer)))
       (in-out-guard
         (lambda () (set-buffer! buffer))
         (lambda () e ...)
         (lambda () (set-buffer! old-buffer)))))))
(define-class-public <buffer> ()
  (name #:init-keyword #:name)
  (keymap #:accessor local-keymap #:init-keyword #:keymap #:init-form (make-keymap))
  (locals #:accessor local-variables #:init-form '())
  (buffer-modified? #:accessor buffer-modified? #:init-value #f)
  (buffer-modified-tick #:accessor buffer-modified-tick #:init-value 0)
  (buffer-enter-hook #:accessor buffer-enter-hook #:init-form (make-hook 0))
  (buffer-exit-hook #:accessor buffer-exit-hook #:init-form (make-hook 0))
  (buffer-modes #:accessor buffer-modes #:init-form '()))
(export local-keymap local-variables buffer-enter-hook buffer-exit-hook before-buffer-change-hook after-buffer-change-hook after-change-hook before-change-hook buffer-modified-tick buffer-modes)
(define-class-public <text-buffer> (<buffer>)
  (gap-buffer #:accessor gap-buffer #:init-form (make-gap-buffer "")))
(export gap-buffer)
(define-variable before-buffer-change-hook (make-hook 1) "This hook is called prior to the buffer being changed with one argument, the buffer.")
(define-variable after-buffer-change-hook (make-hook 1) "This hook is called after to the buffer has changed with one argument, the buffer.")
(define-public buffer-stack (make <mru-stack>))
(define-public last-buffer #f)
(define-public aux-buffer #f)
(define-method-public (buffer-name)
  (buffer-name (current-buffer)))

(define-method-public (buffer-name (buffer <buffer>))
  (slot-ref buffer 'name))

(define-method-public (set-buffer-name! name)
  (set-buffer-name! name (current-buffer)))

(define-method-public (set-buffer-name! name (buffer <buffer>))
  (slot-set! buffer 'name name))

(define-method-public (buffer-modified?)
  (buffer-modified? (current-buffer)))

(define-method-public (buffer-modified-tick)
  (buffer-modified-tick (current-buffer)))

(define-method (write (obj <buffer>) port)
  (write (string-concatenate (list "#<buffer '" (buffer-name obj) "'>")) port))
(define-public (current-local-map)
  (local-keymap (current-buffer)))

(define-public (use-local-map keymap)
  (set! (local-keymap (current-buffer)) keymap))
(define-public (buffer-list)
  (mru-list buffer-stack))
(define-public (current-buffer)
  ;; Perhaps instead of returning #f for no buffer there should be an
  ;; immutable void-buffer class.
  (or aux-buffer
      (mru-ref buffer-stack)))
(define-public (add-buffer! buffer)
  (mru-add! buffer-stack buffer))
(define-public (remove-buffer! buffer)
  (mru-remove! buffer-stack buffer))

(define-interactive (next-buffer #:optional (incr 1))
  (mru-next! buffer-stack incr)
  (switch-to-buffer (mru-ref buffer-stack)))

(define-interactive (prev-buffer #:optional (incr 1))
  (next-buffer (- incr)))

(define-public (set-buffer! buffer)
  ;;(emacsy-log-debug "set-buffer! to ~a" buffer)
  (if (mru-set! buffer-stack buffer)
      (set! aux-buffer #f)
      (set! aux-buffer buffer)))

(define-interactive (kill-buffer #:optional (buffer (current-buffer)))
  (remove-buffer! buffer))

(define-interactive (other-buffer #:optional (count 1))
  (next-buffer count))
(define (primitive-switch-to-buffer buffer)
  (emacsy-log-debug "Running exit hook for ~a" (current-buffer))
  (run-hook (buffer-exit-hook (current-buffer)))
  (set! last-buffer (current-buffer))
  (if (mru-contains? buffer-stack buffer)
      (begin
        (emacsy-log-debug "Recall buffer ~a" buffer)
        (mru-recall! buffer-stack buffer)
        (set! aux-buffer #f))
      (begin 
        (emacsy-log-debug "Set buffer to ~a" buffer)
        (set-buffer! buffer)))
  (emacsy-log-debug "Running enter hook for ~a" (current-buffer))
  (run-hook (buffer-enter-hook (current-buffer)))
  (current-buffer))

(define-public switch-to-buffer primitive-switch-to-buffer) 
(define (local-var-ref symbol)
  (let ((result (assq symbol (local-variables (current-buffer)))))
    (if (pair? result)
     (cdr result)
     ;(variable-ref (make-undefined-variable))
     #f #;(throw 'no-such-local-variable symbol)))) ;; how can I throw an undefined value?

;; If buffers were in their own modules I could dynamically add variables
;; to their namespace.  Interesting idea.

(define (local-var-set! symbol value)
  (slot-set! (current-buffer) 
             'locals 
             (assq-set! (local-variables (current-buffer)) symbol value)))

(define-public local-var 
               (make-procedure-with-setter local-var-ref local-var-set!))
(define-method-public (buffer-string (buffer <text-buffer>))
  (gb->string (gap-buffer buffer)))
(define-method-public (point)
  (point (current-buffer)))

(define-method-public (point (buffer <text-buffer>))
  (gb-point (gap-buffer buffer)))

(define-method-public (point-min)
  (point-min (current-buffer)))

(define-method-public (point-min (buffer <text-buffer>))
  (gb-point-min (gap-buffer buffer)))

(define-method-public (point-max)
  (point-max (current-buffer)))

(define-method-public (point-max (buffer <text-buffer>))
  (gb-point-max (gap-buffer buffer)))
(define-method-public (goto-char point)
  (goto-char point (current-buffer)))

(define-method-public (goto-char point (buffer <text-buffer>))
  (gb-goto-char (gap-buffer buffer) point))
;(define goto-point! goto-char)
;; XXX define-method needs to allow for the definition of optional arguments.
;; This is getting ridiculous.
(define-method-public (char-before)
  (char-before (point) (current-buffer)))

(define-method-public (char-before point)
  (char-before point (current-buffer)))

(define (gb-char-before gb point)
  ;; Avert thy eyes.
  (if (<= point (gb-point-min gb))
      #f
      (string-ref (gb->string gb) (- point 2))))

(define (gb-char-after gb point)
  ;; Avert thy eyes.
  (if (>= point (gb-point-max gb))
      #f
      (string-ref (gb->string gb) (- point 1))))

(define-method-public (char-before point (buffer <text-buffer>))
  (gb-char-before (gap-buffer buffer) point))

(define-method-public (char-after)
  (char-after (point) (current-buffer)))

(define-method-public (char-after point)
  (char-after point (current-buffer)))

(define-method-public (char-after point (buffer <text-buffer>))
  (gb-char-after (gap-buffer buffer) point))

(define*-public (gb-re-search-forward gb regex 
                                      #:optional (bound #f) (no-error? #f) (repeat 1))
  ;; This could be done better in the gap-buffer.scm itself.
  (if (= repeat 0)
      (gb-point gb)
      (let* ((string (gb->string gb))
             (pt (gb-point gb))
             (match (regexp-exec regex string (- pt 1))))
        #;(format #t "match ~a ~%" match)
        (if match
            (begin 
              (gb-goto-char gb (+ 1 (match:end match 0)))
              (gb-re-search-forward gb regex bound no-error? (1- repeat)))
            (if no-error?
                #f
                (scm-error 'no-match 'gb-re-search-forward
                           
                           "No match found for regex '~a' in ~s after point ~a" (list regex string pt) #f))))))

(define*-public (gb-re-search-backward gb regex 
                                       #:optional (bound #f) (no-error? #f) (repeat 1))
  ;; This could be done better in the gap-buffer.scm itself.
  (if (= repeat 0)
      (gb-point gb)
      (let loop ((start-search 0)
                 (last-match-start #f))
       (let* ((string (gb->string gb))
              (pt (gb-point gb))
              (match (regexp-exec regex string start-search)))
         (define (my-error)
           (if no-error?
               #f
               (scm-error 
                'no-match 'gb-re-search-forward
                "No match found for regex '~a' in ~s before point ~a" 
                (list regex string pt) #f)))
         (define (finish)
           (if last-match-start
               (begin 
                 (gb-goto-char gb (1+ last-match-start))
                 (gb-re-search-backward gb regex bound no-error? (1- repeat)))
               (my-error)))
         #;(format #t "match ~a ~%" match)
         (if match
             (if (< (match:start match 0) (1- pt))
                 ;; continue searching
                 (loop (match:end match 0) (match:start match 0))
                 (finish))
             (finish))))))
(define-interactive (kill-line #:optional (n 1))
  (gb-delete-char! (gap-buffer (current-buffer)) (- (point-max) (point))))

(define-interactive (delete-backward-char #:optional (n 1))
  (gb-delete-char! (gap-buffer (current-buffer)) (- n)))

(define-interactive (forward-delete-char #:optional (n 1))
  (gb-delete-char! (gap-buffer (current-buffer)) n))

(define-interactive (forward-char #:optional (n 1))
  (goto-char (+ (point) n)))

(define forward-word-regex (make-regexp "\\s*\\w+\\b"))
(define backward-word-regex (make-regexp "\\b\\w+\\s*"))

;; XXX where is gb-re-search-forward defined?  It has documentation.
;; but no implementation?
;; http://gnuvola.org/software/guile/doc/Gap-Buffer.html
(define-interactive (forward-word #:optional (n 1))
  (gb-re-search-forward (gap-buffer (current-buffer)) forward-word-regex #f #t n))

(define-interactive (backward-word #:optional (n 1))
  (gb-re-search-backward (gap-buffer (current-buffer)) backward-word-regex #f #t n))

(define-interactive (move-beginning-of-line #:optional (n 1))
  ;(gb-beginning-of-line (gap-buffer (current-buffer)) n)
  (goto-char (point-min)))

(define-interactive (move-end-of-line #:optional (n 1))
  ;(gb-beginning-of-line (gap-buffer (current-buffer)) n)
  (goto-char (point-max)))

(define-interactive (backward-char #:optional (n 1))
  (forward-char (- n)))
(define*-public (insert #:rest args)
  (and (current-buffer)
   (if (null? args)
       0 
       (let ((arg (car args)))
         (run-hook before-buffer-change-hook (current-buffer))
         (cond
          ((string? arg)
           (gb-insert-string! (gap-buffer (current-buffer)) arg))
          ((char? arg)
           (gb-insert-char! (gap-buffer (current-buffer)) arg))
          (else #f))
         (set! (buffer-modified? (current-buffer)) #t)
         (incr! (buffer-modified-tick (current-buffer)))
         (run-hook after-buffer-change-hook (current-buffer))))))
(define-interactive (self-insert-command #:optional (n 1))
  (if (< n 1)
      ;; We're done.
      #f
      (let* ((event this-command-event))
        ;; Do I have to do anything for shifted characters?
        (insert (command-char event)))))

