
/* 
 *  The Enhanced Linux Progress Patch
 *     by Prasad <prasad_s@students.iiit.net>, IIIT - Hyderabad, INDIA
 *
 *  elpp_theme.cpp   ELPP Theme Maker Implementation file
 *
 *  This file is part of the Enhanced Linux Progress Patch.
 *
 *  ELPP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Foobar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with ELPP; if not, write to the Free Software 
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 * 
 */ 


#include "elpp_theme.h"

#include <qmessagebox.h>
#include <qvariant.h>
#include <qbuttongroup.h>
#include <qcombobox.h>
#include <qframe.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qspinbox.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qwhatsthis.h>
#include <qaction.h>
#include <qmenubar.h>
#include <qpopupmenu.h>
#include <qtoolbar.h>
#include <qimage.h>
#include <qfiledialog.h>
#include <qstatusbar.h>
#include <qpainter.h>
#include <qcolordialog.h>
#include <qapplication.h>

ELPPTheme::ELPPTheme( QWidget* parent, const char* name, WFlags fl )
    : QMainWindow( parent, name, fl )
{
    if ( !name )
	setName( "ELPPTheme" );
    
    // This should be done before we create the Label - Display
    qApp->setMainWidget(this);

    setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)1, 
    	(QSizePolicy::SizeType)1, 0, 0, sizePolicy().hasHeightForWidth() ) );
    setCentralWidget( new QWidget( this, "qt_central_widget" ) );
    ELPPThemeLayout = new QGridLayout(centralWidget(), 2, 4, 11, 6, "Layout"); 

    colors = new QColor[14];
    colors[0] = red;	colors[7] = blue;	/* Bar */
    colors[1] = white;	colors[8] = black;
    colors[2] = red;	colors[9] = black;
    colors[3] = white;	colors[10] = black;
    colors[4] = yellow;	colors[11] = black;
    colors[5] = white;	colors[12] = black;
    colors[6] = green;	colors[13] = black;

    Display = new Label( centralWidget(), "Display" );
    Display->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)5, 
    	(QSizePolicy::SizeType)5, 4, 3, 
	Display->sizePolicy().hasHeightForWidth() ) );
    Display->setMinimumSize( QSize( 480, 360 ) );
    Display->setMaximumSize( QSize( 480, 360 ) );
    Display->setScaledContents( TRUE );
    ELPPThemeLayout->addMultiCellWidget( Display, 0, 0, 0, 2 );

    barLocation = new QPushButton( centralWidget() );
    barLocation->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    barLocation->setToggleButton(true);
    ELPPThemeLayout->addWidget( barLocation, 1, 1 );
    msgLocation = new QPushButton( centralWidget() );
    msgLocation->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    msgLocation->setToggleButton(true);
    ELPPThemeLayout->addWidget( msgLocation, 1, 2 );
    pointer = new QLabel( centralWidget() );
    ELPPThemeLayout->addWidget( pointer, 1, 0 );

    layout6 = new QVBoxLayout( 0, 0, 6, "layout6"); 

    barGroup = new QButtonGroup( centralWidget(), "barGroup" );
    barGroup->setColumnLayout(0, Qt::Vertical );
    barGroup->layout()->setSpacing( 6 );
    barGroup->layout()->setMargin( 11 );
    barGroupLayout = new QGridLayout( barGroup->layout() );
    barGroupLayout->setAlignment( Qt::AlignTop );

    barHvalue = new QSpinBox( barGroup, "barHvalue" );
    barGroupLayout->addWidget( barHvalue, 3, 1 );
    barXvalue = new QSpinBox( barGroup, "barXvalue" );
    barGroupLayout->addWidget( barXvalue, 0, 1 );
    barY = new QLabel( barGroup, "barY" );
    barGroupLayout->addWidget( barY, 1, 0 );
    barWvalue = new QSpinBox( barGroup, "barWvalue" );
    barGroupLayout->addWidget( barWvalue, 2, 1 );
    barW = new QLabel( barGroup, "barW" );
    barGroupLayout->addWidget( barW, 2, 0 );
    barYvalue = new QSpinBox( barGroup, "barYvalue" );
    barGroupLayout->addWidget( barYvalue, 1, 1 );
    barX = new QLabel( barGroup, "barX" );
    barGroupLayout->addWidget( barX, 0, 0 );
    barH = new QLabel( barGroup, "barH" );
    barGroupLayout->addWidget( barH, 3, 0 );
    
    layout6->addWidget( barGroup );

    msgGroup = new QButtonGroup( centralWidget(), "msgGroup" );
    msgGroup->setColumnLayout(0, Qt::Vertical );
    msgGroup->layout()->setSpacing( 6 );
    msgGroup->layout()->setMargin( 11 );
    msgGroupLayout = new QGridLayout( msgGroup->layout() );
    msgGroupLayout->setAlignment( Qt::AlignTop );

    msgLvalue = new QSpinBox( msgGroup, "msgLvalue" );
    msgGroupLayout->addWidget( msgLvalue, 2, 1 );
    msgNvalue = new QSpinBox( msgGroup, "msgNvalue" );
    msgGroupLayout->addWidget( msgNvalue, 3, 1 );
    msgYvalue = new QSpinBox( msgGroup, "msgYvalue" );
    msgGroupLayout->addWidget( msgYvalue, 1, 1 );
    msgY = new QLabel( msgGroup, "msgY" );
    msgGroupLayout->addWidget( msgY, 1, 0 );
    msgX = new QLabel( msgGroup, "msgX" );
    msgGroupLayout->addWidget( msgX, 0, 0 );
    msgXvalue = new QSpinBox( msgGroup, "msgXvalue" );
    msgGroupLayout->addWidget( msgXvalue, 0, 1 );
    msgN = new QLabel( msgGroup, "msgN" );
    msgGroupLayout->addWidget( msgN, 3, 0 );
    msgL = new QLabel( msgGroup, "msgL" );
    msgGroupLayout->addWidget( msgL, 2, 0 );
    layout6->addWidget( msgGroup );
    QSpacerItem* spacer = new QSpacerItem( 20, 20, 
    	QSizePolicy::Minimum, QSizePolicy::Expanding );
    layout6->addItem( spacer );

    colorGroup = new QGroupBox(centralWidget(), "colorGroup");
    colorGroup->setColumnLayout(0, Qt::Vertical );
    colorGroup->layout()->setSpacing( 6 );
    colorGroup->layout()->setMargin( 11 );
    colorGroupLayout = new QGridLayout( colorGroup->layout() );
    colorGroupLayout->setAlignment( Qt::AlignTop );
    
    colorCombo = new QComboBox( FALSE, colorGroup, "colorCombo" );
    colorGroupLayout->addMultiCellWidget( colorCombo, 0, 0, 0, 1 );
    bg = new QButton( colorGroup, "bg" );
    colorGroupLayout->addWidget( bg, 2, 1 );
    fgColor = new QLabel( colorGroup, "fgColor" );
    colorGroupLayout->addWidget( fgColor, 1, 0 );
    fg = new QButton( colorGroup, "fg" );
    colorGroupLayout->addWidget( fg, 1, 1 );
    bgColor = new QLabel( colorGroup, "bgColor" );
    colorGroupLayout->addWidget( bgColor, 2, 0 );
    
    layout6->addWidget( colorGroup );
    
    ELPPThemeLayout->addMultiCellLayout( layout6, 0, 1, 3, 3 );

    // actions
    fileOpenAction = new QAction( this, "fileOpenAction" );
    fileSaveAction = new QAction( this, "fileSaveAction" );
    fileExitAction = new QAction( this, "fileExitAction" );
    helpAboutAction = new QAction( this, "helpAboutAction" );
    helpThemeAction = new QAction( this, "helpThemeAction" );


    // menubar
    menubar = new QMenuBar( this, "menubar" );

    fileMenu = new QPopupMenu( this );

    fileOpenAction->addTo( fileMenu );
    fileSaveAction->addTo( fileMenu );
    fileMenu->insertSeparator();
    fileExitAction->addTo( fileMenu );
    menubar->insertItem( "", fileMenu, 0 );
    helpMenu = new QPopupMenu( this );

    helpThemeAction->addTo( helpMenu );
    helpAboutAction->addTo( helpMenu );
    menubar->insertItem( "", helpMenu, 1 );

    connect(fileOpenAction, SIGNAL(activated()), SLOT(fileOpen()));
    connect(fileExitAction, SIGNAL(activated()), qApp, SLOT(quit()));
    connect(helpAboutAction, SIGNAL(activated()), SLOT(helpAbout()));
    connect(helpThemeAction, SIGNAL(activated()), SLOT(helpTheme()));
    connect(barXvalue, SIGNAL(valueChanged(int)), SLOT(paintLbl()));
    connect(barYvalue, SIGNAL(valueChanged(int)), SLOT(paintLbl()));
    connect(barWvalue, SIGNAL(valueChanged(int)), SLOT(paintLbl()));
    connect(barHvalue, SIGNAL(valueChanged(int)), SLOT(paintLbl()));
    connect(msgXvalue, SIGNAL(valueChanged(int)), SLOT(paintLbl()));
    connect(msgYvalue, SIGNAL(valueChanged(int)), SLOT(paintLbl()));
    connect(msgNvalue, SIGNAL(valueChanged(int)), SLOT(paintLbl()));
    connect(msgLvalue, SIGNAL(valueChanged(int)), SLOT(paintLbl()));
    connect(colorCombo, SIGNAL(activated(int)), SLOT(setColors(int)));
    connect(fg, SIGNAL(clicked()), SLOT(fgClicked()));
    connect(bg, SIGNAL(clicked()), SLOT(bgClicked()));
    connect(fileSaveAction, SIGNAL(activated()), SLOT(fileSave()));
    connect(barLocation, SIGNAL(toggled(bool)), SLOT(barToggle(bool)));
    connect(msgLocation, SIGNAL(toggled(bool)), SLOT(textToggle(bool)));
    
    languageChange();
    setColors(0);
    resize( QSize(650, 450).expandedTo(minimumSizeHint()) );

    Display->setEnabled(false);
    barGroup->setEnabled(false);
    msgGroup->setEnabled(false);
    colorGroup->setEnabled(false);
    barLocation->setEnabled(false);
    msgLocation->setEnabled(false);

    setTabOrder(barXvalue, barYvalue);
    setTabOrder(barYvalue, barWvalue);
    setTabOrder(barWvalue, barHvalue);
    setTabOrder(barHvalue, msgXvalue);
    setTabOrder(msgXvalue, msgYvalue);
    setTabOrder(msgYvalue, msgLvalue);
    setTabOrder(msgLvalue, msgNvalue);
    setTabOrder(msgNvalue, colorCombo);
    setTabOrder(colorCombo, fg);
    setTabOrder(fg, bg);
    setTabOrder(bg, barLocation);
    setTabOrder(barLocation, msgLocation);
}

/*
 *  Destroys the object and frees any allocated resources
 */
ELPPTheme::~ELPPTheme()
{
    // no need to delete child widgets, Qt does it all for us
}

/*
 *  Sets the strings of the subwidgets using the current
 *  language.
 */
void ELPPTheme::languageChange()
{
    setCaption( tr( "ELPP Theme Maker" ) );
    barGroup->setTitle( tr( "Progress Bar" ) );
    barY->setText( tr( "Y" ) );
    barW->setText( tr( "Width" ) );
    barX->setText( tr( "X" ) );
    barH->setText( tr( "Height" ) );
    msgGroup->setTitle( tr( "Text Messages" ) );
    colorGroup->setTitle( tr( "Colors" ) );
    msgY->setText( tr( "Y" ) );
    msgX->setText( tr( "X" ) );
    msgN->setText( tr( "Number" ) );
    msgL->setText( tr( "Length" ) );
    colorCombo->clear();
    colorCombo->insertItem( tr( "Progress Bar" ) );
    colorCombo->insertItem( tr( "Error Text" ) );
    colorCombo->insertItem( tr( "Error Icon" ) );
    colorCombo->insertItem( tr( "Warning Text" ) );
    colorCombo->insertItem( tr( "Warning Icon" ) );
    colorCombo->insertItem( tr( "Success Text" ) );
    colorCombo->insertItem( tr( "Success Icon" ) );
    fgColor->setText( tr( "FG Color" ) );
    bgColor->setText( tr( "BG Color" ) );
    barLocation->setText( tr( "Bar Location" ));
    msgLocation->setText( tr( "Text Location" ));
    fileOpenAction->setText( tr( "Open" ) );
    fileOpenAction->setMenuText( tr( "&Open..." ) );
    fileOpenAction->setAccel( tr( "Ctrl+O" ) );
    fileSaveAction->setText( tr( "Save" ) );
    fileSaveAction->setMenuText( tr( "&Save" ) );
    fileSaveAction->setAccel( tr( "Ctrl+S" ) );
    fileExitAction->setText( tr( "Exit" ) );
    fileExitAction->setMenuText( tr( "E&xit" ) );
    fileExitAction->setAccel( QString::null );
    helpAboutAction->setText( tr( "About" ) );
    helpAboutAction->setMenuText( tr( "&About" ) );
    helpAboutAction->setAccel( QString::null );
    helpThemeAction->setText( tr( "Making Themes" ) );
    helpThemeAction->setMenuText( tr( "&Making Themes" ) );
    menubar->findItem( 0 )->setText( tr( "&File" ) );
    menubar->findItem( 1 )->setText( tr( "&Help" ) );
}

void ELPPTheme::fgClicked()
{
    QColor col = QColorDialog::getColor(colors[colorCombo->currentItem()]);
    if( ! col.isValid() )
	return;
    colors[colorCombo->currentItem()] = col;
    fg->setPaletteBackgroundColor(col);
}

void ELPPTheme::bgClicked()
{
    QColor col = QColorDialog::getColor(colors[colorCombo->currentItem() + 7]);
    if( ! col.isValid() )
	return;
    colors[colorCombo->currentItem() + 7] = col;
    bg->setPaletteBackgroundColor(col);
}

void ELPPTheme::setColors(int index)
{
    fg->setPaletteBackgroundColor(colors[index]);
    bg->setPaletteBackgroundColor(colors[index+7]);
}

void ELPPTheme::paintLbl()
{
    Display->repaint(false);
}

void ELPPTheme::fileOpen()
{
    QString file = QFileDialog::getOpenFileName(QDir::currentDirPath(),
    		"Images (*.png *.jpg);;"
		"Pixmaps (*.xpm *.bmp *.pnm *.pbm *pgm);;"
		"All Files (*.*)",
		this, "Open File Dialog", "Select an image for theme");
    img.load(file);
    if(img.isNull()) {
	qWarning("Not a valid Image/Pixmap");
	return;
    }
	
    barXvalue->setMaxValue(img.width());
    barYvalue->setMaxValue(img.height());
    barWvalue->setMaxValue(img.width());
    barHvalue->setMaxValue(img.height());
    msgXvalue->setMaxValue(img.width());
    msgYvalue->setMaxValue(img.height());
	
    Display->setPixmap(img);
    Display->repaint();
    
    Display->setEnabled(true);
    barGroup->setEnabled(true);
    msgGroup->setEnabled(true);
    colorGroup->setEnabled(true);
    barLocation->setEnabled(true);
    msgLocation->setEnabled(true);

}

void ELPPTheme::fileSave()
{
    if(img.isNull()) {
	qWarning("No Image/Pixmap loaded");
	return;
    }

    QString file = QFileDialog::getSaveFileName(QDir::currentDirPath(),
    		"ELPP Theme Image (*.eti)",
		this, "Save File Dialog", "Select a name for Theme Image");
    if(file.isNull())
	return;
    if(!file.endsWith(".eti"))
	file = file + ".eti";

    FILE *fp = fopen(file.latin1(), "w");
    fprintf(fp, "ELPP-THEME-START");
    writePreferences(fp);
    writeImage(fp);
    fprintf(fp, "ELPP-THEME-END");
    fclose(fp);
}

void ELPPTheme::writePreferences(FILE *fp)
{
    elpp_userprefs *prefs = new elpp_userprefs;

    prefs->bar_x = barXvalue->value();
    prefs->bar_y = barYvalue->value();
    prefs->bar_length = barWvalue->value();
    prefs->bar_width = barHvalue->value();
    prefs->msg_length = msgLvalue->value();
    prefs->num_messages = msgNvalue->value();
    prefs->msg_x = msgXvalue->value();
    prefs->msg_y = msgYvalue->value();
    
    prefs->bar_bg[0] = colors[7].red();
    prefs->bar_bg[1] = colors[7].green();
    prefs->bar_bg[2] = colors[7].blue();
    
    prefs->bar_fg[0] = colors[0].red();
    prefs->bar_fg[1] = colors[0].green();
    prefs->bar_fg[2] = colors[0].blue();
    
    prefs->warning_bg[0] = colors[10].red();
    prefs->warning_bg[1] = colors[10].green();
    prefs->warning_bg[2] = colors[10].blue();
    
    prefs->warning_fg[0] = colors[3].red();
    prefs->warning_fg[1] = colors[3].green();
    prefs->warning_fg[2] = colors[3].blue();
    
    prefs->warning_symbol_bg[0] = colors[11].red();
    prefs->warning_symbol_bg[1] = colors[11].green();
    prefs->warning_symbol_bg[2] = colors[11].blue();
    
    prefs->warning_symbol_fg[0] = colors[4].red();
    prefs->warning_symbol_fg[1] = colors[4].green();
    prefs->warning_symbol_fg[2] = colors[4].blue();
    
    prefs->error_bg[0] = colors[8].red();
    prefs->error_bg[1] = colors[8].green();
    prefs->error_bg[2] = colors[8].blue();
    
    prefs->error_fg[0] = colors[1].red();
    prefs->error_fg[1] = colors[1].green();
    prefs->error_fg[2] = colors[1].blue();
    
    prefs->error_symbol_bg[0] = colors[9].red();
    prefs->error_symbol_bg[1] = colors[9].green();
    prefs->error_symbol_bg[2] = colors[9].blue();
    
    prefs->error_symbol_fg[0] = colors[2].red();
    prefs->error_symbol_fg[1] = colors[2].green();
    prefs->error_symbol_fg[2] = colors[2].blue();
    
    prefs->success_bg[0] = colors[12].red();
    prefs->success_bg[1] = colors[12].green();
    prefs->success_bg[2] = colors[12].blue();
    
    prefs->success_fg[0] = colors[5].red();
    prefs->success_fg[1] = colors[5].green();
    prefs->success_fg[2] = colors[5].blue();
    
    prefs->success_symbol_bg[0] = colors[13].red();
    prefs->success_symbol_bg[1] = colors[13].green();
    prefs->success_symbol_bg[2] = colors[13].blue();
    
    prefs->success_symbol_fg[0] = colors[6].red();
    prefs->success_symbol_fg[1] = colors[6].green();
    prefs->success_symbol_fg[2] = colors[6].blue();
    
    fwrite(prefs, sizeof(struct elpp_userprefs), 1, fp);
    delete prefs;
}

void ELPPTheme::writeImage(FILE *fp)
{
    struct linux_logo logo;
    
    fprintf(fp, "ELPP-START-IMAGE");
    logo.type = LINUX_LOGO_CLUT224;	/* CLUT224 */
    logo.width = img.width();
    logo.height = img.height();
    logo.clutsize = img.numColors();
    if(logo.clutsize <= 0 || logo.clutsize > 200) {
	img = img.convertDepth(8);
	img.setNumColors(200);
    }
    logo.clutsize = img.numColors();
    logo.clut = NULL;
    logo.data = NULL;
    fwrite(&logo, sizeof(struct linux_logo), 1, fp);
    
    fprintf(fp, "ELPP-START-IMAGE-DATA");
    unsigned char *data = new unsigned char[img.width() * img.height()];
    int k = 0;
    for(int i=0; i<img.height(); i++)
	for(int j=0; j<img.width(); j++)
	    data[k++] = img.pixelIndex(j,i) + 32;
    fwrite(data, sizeof(char) * img.height() * img.width(), 1, fp);
    delete data;
    
    fprintf(fp, "ELPP-START-IMAGE-CLUT");
    data = new unsigned char[img.numColors() * 3];
    k = 0;
    for(int i=0; i<img.numColors(); i++) {
	data[k++] = qRed(img.color(i));
	data[k++] = qGreen(img.color(i));
	data[k++] = qBlue(img.color(i));
    }
    fwrite(data, sizeof(char) * img.numColors() * 3, 1, fp);
    delete data;
}

void ELPPTheme::fileExit()
{
    qApp->exit();
}

void ELPPTheme::helpAbout()
{
    QMessageBox::about(this,"ELPP Theme Maker", 
	    "This tool can be used along with\n"
	    "The Enhanced Linux Progress Patch\n"
	    "to give an impressive look to the\n"
	    "GNU/Linux boot system.\n\n"
	    "ELPP along with this tool can be downloaded\n"
	    "from http://students.iiit.net/~prasad_s/elpp/\n"
	    "\nAuthor: Prasad <prasad_s@students.iiit.net>\n"
	    "\nSuggestions and comments are welcome.\n"
	    "This package is distributed under GPL\n");
}

void ELPPTheme::helpTheme()
{

    QMessageBox::about(this, "Making a theme", 
	    "The \"Bar Location\" Button shown at the bottom of the tool\n"
	    "is a toggle button... and when pressed the mouse cursor can\n"
	    "be used to select (press & drag) the region where the\n"
	    "progress bar should be located. Similarly the \"Text Location\".\n"
	    "\nFor fine tuning purposes... you can use the \n"
	    "spin boxes on the right panel.\n"
	    "\nThe color box let you configure the colors for\n"
	    "various parts of the theme  (Click the color rectangles to\n"
	    "select a new color\n"
	    "\nUse the file menu to load and image and to save the theme.\n");
    
}

void ELPPTheme::barToggle(bool on)
{
    if(on && msgLocation->isOn())
	msgLocation->setOn(false);
}

void ELPPTheme::textToggle(bool on)
{
    if(on && barLocation->isOn())
	barLocation->setOn(false);
}

Label::Label(QWidget *parent, char *name)
    :QLabel(parent, name)
{
    setMouseTracking(true);
    mousePressed = false;
    mousePressPoint = QPoint(0,0);
    win = (ELPPTheme *)qApp->mainWidget();
    setCursor(Qt::CrossCursor);
}

Label::~Label()
{
    return;
}

void Label::paintEvent(QPaintEvent *ev)
{
    QLabel::paintEvent(ev);

    if(win->img.isNull())
        return;

#define SCALE(x) ((x) * width())/win->img.width()
    int barX = SCALE(win->barXvalue->value());
    int barY = SCALE(win->barYvalue->value());
    int barW = SCALE(win->barWvalue->value());
    int barH = SCALE(win->barHvalue->value());
    int msgX = SCALE(win->msgXvalue->value());
    int msgY = SCALE(win->msgYvalue->value());
    int msgN = SCALE(win->msgNvalue->value() * 16);
    int msgL = SCALE(win->msgLvalue->value() * 8);
#undef SCALE

    QPainter p;
    p.begin(this);
    p.fillRect(barX, barY, barW, barH, QBrush(green, Dense4Pattern));
    p.fillRect(msgX, msgY, msgL, msgN, QBrush(magenta, Dense5Pattern));
    p.end();
}

void Label::mousePressEvent(QMouseEvent *ev)
{
    if(ev->button() != LeftButton)
	return;

    mousePressed = true;
    mousePressPoint = ev->pos();
}

void Label::mouseReleaseEvent(QMouseEvent *ev)
{
    if(ev->button() != LeftButton)
	return;

    win->barLocation->setOn(false);
    win->msgLocation->setOn(false);
    mousePressed = false;
}

void Label::mouseMoveEvent(QMouseEvent *ev)
{
#define SCALE(x) ((x * win->img.width())/width())
    int x = SCALE(ev->x());
    int y = SCALE(ev->y());
    int px = SCALE(mousePressPoint.x());
    int py = SCALE(mousePressPoint.y());
#undef SCALE
    
    win->pointer->setText(QString::number(x)+":"+QString::number(y));
    if(mousePressed) {
	if(win->barLocation->isOn()) {
	    win->barXvalue->setValue(px);
	    win->barYvalue->setValue(py);
	    win->barWvalue->setValue(x - px);
	    win->barHvalue->setValue(y - py);
	} else if(win->msgLocation->isOn()) {
	    win->msgXvalue->setValue(px);
	    win->msgYvalue->setValue(py);
	    win->msgLvalue->setValue((x - px)/8);
	    win->msgNvalue->setValue((y - py)/16);
	}
    }
}

