/*
ecp is designed to copy files, and is named extended cp.
Copyright (C) 2006  Andre Reffhaug <areffhaug@areffhaug.org>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

ecp homepage   : https://savannah.nongnu.org/projects/ecp/
Contact author : areffhaug@gmail.com
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <limits.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "copyfun.h"
#include "dir.h"

#define _GNU_SOURCE
#include <getopt.h>


static const int VERSION_MAJOR = 0;
static const int VERSION_MINOR = 1;
static const int VERSION_RELEASE = 1;
static const char * const VERSION_TEXT = "beta";



inline static void author(void) {
    printf("Written by %s (2006).\n", AUTHOR);
}

inline static void version(void) {
    printf("%s %i.%i.%i %s\n", NAME, VERSION_MAJOR,
           VERSION_MINOR, VERSION_RELEASE, VERSION_TEXT);
}

inline static void usage(void) {
    printf("%s [OPTIONS] SOURCE(s) DESTINATION\n", NAME);
    printf("\n");
    printf("-R, --recursive          copies files/folder recursively\n");
    printf("-p, --preserve           preserves access and modification times\n");
    printf("-n, --no-hidden-files    omits hidden files when copying\n");
    printf("-c, --create             creates target directory if it does not \
exist\n");
    printf("-s, --d-syms             dereferences symlinks\n");
    printf("-k, --kill               removes SOURCE after copying (use with caution)\n");
    printf("-q, --quiet              no information to stdout\n");
    printf("-u, --update             only copies if SOURCE is newer or \
modified (semi functional)\n");
    printf("-h, --human-readable     output is formatted for humans\n");
    printf("\n");
    printf("--information            outputs some small information on ecp\n");
    printf("--no-zero-bytes          skips files of zero byte length\n");
    printf("--version                outputs version information and exits\n");
}

inline static void information(void) {
    version();
    author();
    printf("Compiled: %s %s\n", __DATE__, __TIME__);
    printf("Compiled with GCC version %d.%d.%d\n", __GNUC__, __GNUC_MINOR__, __GNUC_PATCHLEVEL__);
    printf("Glibc version %d.%d, ", __GLIBC__, __GLIBC_MINOR__);
    printf("GNU library %d\n", __GNU_LIBRARY__);
    printf("Latest ecp can be found at: http://ag0ny.net/~xpozed/ecp.tar.gz\n");
}

int main (int argc, char **argv)
{
  int opt;
  struct opt options;
  struct stat destination;
  struct stat target;
  char dest[path_max];

  /* hacked in to avoid infinite recursion */
  struct stat origtarget;

  struct option longopts[] = {
    {"version", 0, NULL, 0},
    {"help", 0, NULL, 1},
    {"information", 0, NULL, 'i'}, /* nothing */
    {"recursive", 0, NULL, 'R'}, /* for cp compatibility */
    {"preserve", 0, NULL, 'p'}, /* preserves timestamps */
    {"no-zero-bytes", 0, NULL, 2}, /* doesnt copy zero byte files */
    {"no-hidden-files", 0, NULL, 'n'}, /* omits hidden files */
    {"create", 0, NULL, 'c'}, /* creates destination directory if it does not exist */
    {"d-syms", 0, NULL, 's'}, /* dereference symlinks. change name */
    {"kill", 0, NULL, 'k'}, /* removes source directories/files */
    {"quiet", 0, NULL, 'q'},
    {"update", 0, NULL, 'u'},
    {"human-readable", 0, NULL, 'h'}
  };

  opterr = 0;
  memset(&options, 0, sizeof(options));

  while ((opt = getopt_long(argc, argv, "iRpcksquhn", longopts, NULL)) != -1) {
    switch (opt) {
    case 0:
      version();
      author();
      exit(EXIT_SUCCESS);
      break;
    case 1:
      usage();
      exit(EXIT_SUCCESS);
      break;
    case 2:
      options.no_zero_bytes = true;
      break;
    case 'n':
      options.no_hidden_files = true;
      break;
    case 'i':
      information();
      exit(EXIT_SUCCESS);
      break;
    case 'R':
      options.recursive = true;
      break;
    case 'p':
      options.preserve = true;
      break;
    case 'c':
      options.create = true;
      break;
    case 'k':
      options.kill = true;
      break;
    case 's':
      options.symlinks = true;
      break;
    case 'q':
      options.quiet = true;
      break;
    case 'u':
      options.update = true;
      break;
    case 'h':
      options.hreadable = true;
      break;
    case '?':
      fprintf(stderr, "%s: unknown command line option '%c'\n", 
              NAME, optopt);
      exit(EXIT_FAILURE);
      break;
    default:
      break;
    }
  }

  if (argc <= 2) {
    version();
    printf("%s: missing arguments\n", NAME);
    exit(EXIT_SUCCESS);
  }
  remove_trailing_slashes(argv[argc-1]);
  realpath(argv[argc-1], dest);

  lstat(dest, &origtarget);
  if (((argc-optind) > 3) && (!S_ISDIR(origtarget.st_mode)) 
      && options.create != true) {
    printf("%s: target directory '%s' doesn't seem to exist ==> \
aborting\n", NAME, dest);
    exit(EXIT_FAILURE);
  }

  if (options.create == true) {
    if (mkdir(dest, 0777) != 0) {
      printf("%s: can't create directory: %s\n", NAME, strerror(errno));
      exit(EXIT_SUCCESS);
    }
  }





  for(; optind < (argc-1); ++optind) {
    stat(argv[optind], &destination);
    if (options.recursive == true) {
      remove_trailing_slashes(argv[optind]);
      if (S_ISDIR(destination.st_mode)) {
        apptodir(dest, base(argv[optind]));
        if (stat(dest, &target) == 0) {
          if (S_ISREG(target.st_mode)) {
            printf("%s: target '%s' is a regular file ==> aborting\n", NAME, dest);
            exit(EXIT_FAILURE);
          }
        }
        mkdir(dest, 0777);
        copy_recursive(argv[optind], dest, 0, origtarget.st_ino, 
                       &options);
        chmod(dest, destination.st_mode);
        strcpy(dest, argv[argc-1]);
        if (options.kill == 1) remove(argv[optind]);
      } else
        do_copy(argv[optind], dest, &options, true);
    }
    else if (S_ISDIR(origtarget.st_mode)) {
      if (S_ISDIR(destination.st_mode)) {
        printf("%s: skipping directory '%s'\n", NAME, argv[optind]);
        continue;
      }
      do_copy(argv[optind], dest, &options, true);
    }
    else if (!S_ISDIR(origtarget.st_mode))
      do_copy(argv[optind], dest, &options, false);
  }
  return 0;
}
