/*
  ecp is designed to copy files, and is named extended cp.
  Copyright (C) 2006  Andre Reffhaug <areffhaug@gmail.com>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  ecp homepage   : https://savannah.nongnu.org/projects/ecp/
  Contact author : areffhaug@gmail.com
*/




/*
  FIXME:

  Own realpath function would also be good, but thats in the 
  works.

  include functionality for mounting, ie copying right into
  devices? holy crap, batman, that'll be strenous

  have a look at size of CHUNK

  if recursively copying into a directory directly beneath the one
  we are copying, how to prevent it from forever recursing into 
  itself? one thought is to check if it does, and recursively copy
  it first, then add its inode to the checking mechanism. seems a 
  bit dense, though.

  what about human readable outputs? halfdecent hack thrown together

  hidden files and shell globbing: * doesnt include hidden files, which
  makes that whole shebang a little more crazy

  what about some sort of mode to output more information on each
  file in a sort of readable format? that'd be nice.

*/

#include <syscall.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <string.h>
#include <dirent.h>
#include <utime.h>
#include <errno.h>
#include <sys/mman.h>
#include "copyfun.h"
#include "dir.h"
#include "ecp_info.h"
#include "ftp_conn.h"

extern int path_max;


void
init_progress (struct progress *prog, off_t *bsize, char *dstname) 
{
    prog->fname = malloc (path_max * sizeof (*prog->fname));

    if (prog->fname == NULL)
    {
        fprintf (stderr, "%s: Could not allocate memory ==> aborting...\n", NAME);
        exit (EXIT_FAILURE);
    }
  
    prog->percent = 0;
    prog->in_total.bytes = *bsize;
    prog->in_total.kbytes = prog->in_total.bytes / 1000;
    prog->in_total.mbytes = prog->in_total.kbytes / 1000;
    prog->total.bytes = 0;

    strcpy (prog->fname, dstname);
  
}

void
init_progress_total (struct progress *prog, unsigned long long *bsize, unsigned long long *total) 
{
    prog->in_total.bytes = *bsize;
    prog->in_total.kbytes = prog->in_total.bytes / 1000;
    prog->in_total.mbytes = prog->in_total.kbytes / 1000;
    prog->total.bytes = *total;
    prog->percent = ((float) prog->total.bytes * 100) 
        / (float) prog->in_total.bytes;
}

void update_progress (struct progress *prog, struct opt *options)
{
    prog->percent = ((float) prog->total.bytes * 100) 
        / (float) prog->in_total.bytes;

    if ((options->ftp_transfer == true) && prog->in_total.bytes > 10000000)
        prog->chunk += (prog->in_total.bytes / (CHUNK*8));
    else
        prog->chunk += (prog->in_total.bytes / CHUNK);

}

void update_progress_total (struct progress *prog, struct opt *options)
{
    prog->percent = ((float) prog->total.bytes * 100) 
        / (float) prog->in_total.bytes;
}

void do_progress (struct progress *prog, struct opt *options)
{

    if (options->hreadable == true) 
    {
        if ((prog->in_total.bytes < 100000))
            printf ("\r%6.2f%%  %3llu / %3llu kB ==> %s", prog->percent, 
                    prog->total.bytes / 1000, prog->in_total.kbytes, prog->fname);
        if (prog->in_total.bytes > 100000)
            printf ("\r%6.2f%%  %3llu / %3llu mB ==> %s", prog->percent, 
                    prog->total.bytes / 1000000, prog->in_total.mbytes, prog->fname);
    }
    else
        if ((prog->in_total.bytes < 100000) && options->hreadable != true)
            printf("\r%6.2f%%  %6llu / %5llu  B ==> %s", prog->percent, 
                   prog->total.bytes, prog->in_total.bytes, prog->fname);
        else printf("\r%6.2f%%  %6llu / %5llu kB ==> %s", prog->percent, 
                    prog->total.bytes / 1000, prog->in_total.kbytes, prog->fname);

    fflush (stdout);
}

void do_progress_total (struct progress *prog, struct opt *options,
                        int *file_count, int *total_files)
{

    if (options->hreadable == true) 
    {
        if ((prog->in_total.bytes < 100000))
            printf ("\r%6.2f%%  %3llu / %3llu kB ==> Copied %d file(s) out of %d total.", prog->percent, 
                    prog->total.bytes / 1000, prog->in_total.kbytes, *file_count, *total_files);
        if (prog->in_total.bytes > 100000)
            printf ("\r%6.2f%%  %3llu / %3llu mB ==> Copied %d file(s) out of %d total.", prog->percent, 
                    prog->total.bytes / 1000000, prog->in_total.mbytes, *file_count, *total_files);
    }
    else
        if ((prog->in_total.bytes < 100000) && options->hreadable != true)
            printf("\r%6.2f%%  %6llu / %5llu  B ==> Copied %d file(s) out of %d total.", prog->percent, 
                   prog->total.bytes, prog->in_total.bytes, *file_count, *total_files);
        else printf("\r%6.2f%%  %6llu / %5llu kB ==> Copied %d file(s) out of %d total.", prog->percent,
                    prog->total.bytes / 1000, prog->in_total.kbytes, *file_count, *total_files);

    fflush (stdout);
}

void destroy_progress (struct progress *prog)
{
    free (prog->fname);
    free (prog);
}

bool do_copy (char *srcname, 
              char *outfile,
              struct opt *options, 
              bool todir)
{
    ssize_t readbytes, bwritten, buffersize;
    char * __restrict__ buf = NULL;
    char temp[path_max]; /* for readlink */
  
    int srcdesc = 0;
    struct stat srcstat;

    char dstname[path_max];
    int dstdesc = 0;
    struct stat dststat;

    struct progress *prog = NULL;
    struct utimbuf utimdef;

    prog = malloc (sizeof (*prog));

    if (prog == NULL)
    {
        fprintf (stderr, "%s: Could not allocate memory for struct prog ==> aborting...\n",
                 NAME);
        exit (EXIT_FAILURE);
    }
  
    /* set up our local copies of the filenames */
    strcpy (dstname, outfile);

    if (options->no_hidden_files == true && srcname[0] == '.') 
        return 1;

    if ((srcdesc = open (srcname, O_RDONLY, 0)) == -1) 
    {
        fprintf (stderr, "%s: failed opening '%s' (%s) ==> aborting\n", 
                 NAME, srcname, strerror (errno));
        return 0;
    }
    /* get hold of file information */
    lstat (srcname, &srcstat);


    if (S_ISLNK (srcstat.st_mode) && options->symlinks == true) 
    {
        readlink (srcname, srcname, path_max);
        lstat (srcname, &srcstat);
    } 

    if (options->no_zero_bytes == true)
        if (srcstat.st_size == 0) 
        {
            close (srcdesc);
            return 1;
        }



    if (todir == true) 
    {
        apptodir (dstname, base(srcname));
        if (S_ISLNK (srcstat.st_mode) && options->symlinks != true) 
        {
            memset (temp, '\0', sizeof (temp));
            readlink (srcname, temp, path_max);
            symlink (temp, dstname);
            close (srcdesc);
            if (options->kill == true) {
                if (remove (srcname) == -1)
                    fprintf (stderr, "%s: couldn't remove source file '%s' (%s)\n", 
                             NAME, srcname, strerror (errno));
            }
            return 1;
        } 
    }

    /* atleast it responds, but it really does need to be improved upon */
    if (options->update == true) {
        stat (dstname, &dststat);
        if (srcstat.st_mtime < dststat.st_mtime)
        {
            printf ("%s: target is newer or the same as '%s'\n", NAME, dstname);
            return 1;
        }
    }



    if ((dstdesc = open (dstname, O_CREAT | O_TRUNC | O_WRONLY, 
                         srcstat.st_mode)) == -1) 
    { 
        fprintf (stderr, "%s: failed opening '%s' (%s) ==> aborting\n", 
                 NAME, dstname, strerror (errno));
        return 0;
    }


    /* initializing the progress struct for our current file */
    init_progress (prog, &srcstat.st_size, dstname);

    if ((srcstat.st_size == 0) && options->quiet != true) 
    {
        prog->percent = 100;
        do_progress (prog, options);
        close (srcdesc);
        close (dstdesc);
        chmod (dstname, srcstat.st_mode&
               (S_IRWXU | S_IRWXG | S_IRWXO | S_ISUID | S_ISGID));
        if (options->kill == true) 
        {
            if ((chmod (srcname, 0777)) == -1)
                fprintf (stderr, "%s: Chmod for file '%s' failed\n", NAME, srcname);
          
            if (remove (srcname) == -1)
                fprintf (stderr, "%s: Couldn't remove source file '%s' (%s)\n", 
                         NAME, srcname, strerror (errno));
        }
        printf("\n");

        return 1;
    } 
    else if ((srcstat.st_size == 0) && options->quiet == true) 
    {
        close (srcdesc);
        close (dstdesc);
        if (options->kill == true) 
        {
            if ((chmod (srcname, 0777)) == -1)
            {
                fprintf (stderr, "%s: Chmod failed for file '%s'\n", NAME, srcname);
            }
            if (remove (srcname) == -1)
                fprintf (stderr, "%s: couldn't remove source file '%s' (%s)\n", 
                         NAME, srcname, strerror (errno));
        }

        return 0;
    }

    


    /* set up buffersize, according to disks blocksize */
    buffersize = srcstat.st_blksize;  
    /* if the file is actually smaller than the chunk we would like to mmap, we
       map the whole thing to memory, and write it back in one big swoop */

    buf = malloc (buffersize * sizeof (*buf));

    if (buf == NULL)
    {
        fprintf (stderr, "%s: Could not allocate memory for buffer in do_copy\n", NAME);
        exit (EXIT_FAILURE);
    }
    do
    {
        readbytes = read (srcdesc, buf, buffersize);

        if (readbytes == -1)
        {
            fprintf (stderr, "%s: An error occured while reading file '%s'\n", NAME, srcname);
            printf ("%s\n", strerror (errno));
            return 1;
        }
        bwritten = write (dstdesc, buf, readbytes);

        if (bwritten != readbytes)
        {
            fprintf (stderr, "%s: Error while writing to file '%s'\n", NAME, dstname);
            return 1;
        }
        prog->total.bytes += bwritten;
      
        if (options->quiet != true) 
        {
            if ((prog->total.bytes >= prog->chunk*2) || (bwritten < buffersize)) 
            {
                update_progress (prog, options);
                do_progress (prog, options);
            }
        }

    } while (readbytes != 0);


    

    if (options->quiet != true) printf ("\n");

    if (options->preserve == true) 
    {
        utimdef.actime = srcstat.st_atime;
        utimdef.modtime = srcstat.st_mtime;
        if (!utime (dstname, &utimdef) == 0)
            fprintf (stderr, "%s: error setting utime of file '%s' (%s)\n", 
                     NAME, dstname, strerror (errno));
    }

    /* FIXME: still not working right, even though it now retains
       file permissions. Feels like some sort of cheap hack. what about
       S_ISUID (set-used-ID) and S_ISVTX (sticky bit) and S_ISGID? 
    */

    chmod (dstname, srcstat.st_mode&
           (S_IRWXU | S_IRWXG | S_IRWXO | S_ISUID | S_ISGID));

    /* clean up */
    if (options->kill == true) 
    {
        if ((chmod (srcname, 0777)) == -1)
            fprintf (stderr, "%s: Chmod for file '%s' failed.\n", NAME, srcname);
      
        if (remove (srcname) == -1)
            fprintf (stderr, "%s: couldn't remove source file '%s' (%s)\n", 
                     NAME, srcname, strerror (errno));
    }
    destroy_progress (prog);
    free (buf);
    close (srcdesc);
    close (dstdesc);

    return 1;
}

unsigned long long
do_copy_total (char *srcname, 
               char *outfile,
               struct opt *options, 
               bool todir, unsigned long long *total_progress,
               unsigned long long *total_size, int *file_count,
               int *total_files)
{
    ssize_t readbytes, bwritten, buffersize;
    char * __restrict__ buf = NULL;
    char temp[path_max]; /* for readlink */
  
    int srcdesc = 0;
    struct stat srcstat;

    char dstname[path_max];
    int dstdesc = 0;
    struct stat dststat;

    struct progress *prog = NULL;
    struct utimbuf utimdef;

    prog = malloc (sizeof (*prog));

    if (prog == NULL)
    {
        fprintf (stderr, "%s: Could not allocate memory for struct prog ==> aborting...\n",
                 NAME);
        exit (EXIT_FAILURE);
    }
  
    /* set up our local copies of the filenames */
    strcpy (dstname, outfile);

    if (options->no_hidden_files == true && srcname[0] == '.') 
        return *total_progress;

    if ((srcdesc = open (srcname, O_RDONLY, 0)) == -1) 
    {
        fprintf (stderr, "%s: failed opening '%s' (%s) ==> aborting\n", 
                 NAME, srcname, strerror (errno));
        return *total_progress;
    }
    /* get hold of file information */
    lstat (srcname, &srcstat);


    if (S_ISLNK (srcstat.st_mode) && options->symlinks == true) 
    {
        readlink (srcname, srcname, path_max);
        lstat (srcname, &srcstat);
    } 

    if (options->no_zero_bytes == true)
        if (srcstat.st_size == 0) 
        {
            close (srcdesc);
            return *total_progress;
        }



    if (todir == true) 
    {
        apptodir (dstname, base(srcname));
        if (S_ISLNK (srcstat.st_mode) && options->symlinks != true) 
        {
            memset (temp, '\0', sizeof (temp));
            readlink (srcname, temp, path_max);
            symlink (temp, dstname);
            close (srcdesc);
            if (options->kill == true) {
                if (remove (srcname) == -1)
                    fprintf (stderr, "%s: couldn't remove source file '%s' (%s)\n", 
                             NAME, srcname, strerror (errno));
            }
            free (prog);

            return *total_progress + srcstat.st_size;
        } 
    }

    /* atleast it responds, but it really does need to be improved upon */
    if (options->update == true) {
        stat (dstname, &dststat);
        if (srcstat.st_mtime < dststat.st_mtime)
        {
            printf ("%s: target is newer or the same as '%s'\n", NAME, dstname);
          
            free (prog);
         
            
            return *total_progress;
        }
    }



    if ((dstdesc = open (dstname, O_CREAT | O_TRUNC | O_WRONLY, 
                         srcstat.st_mode)) == -1) 
    { 
        fprintf (stderr, "%s: failed opening '%s' (%s) ==> aborting\n", 
                 NAME, dstname, strerror (errno));
        free (prog);
        return *total_progress;
    }


    /* initializing the progress struct for our current file */
    init_progress_total (prog, total_size, total_progress);

    
    if ((srcstat.st_size == 0) && options->quiet != true) 
    {
        prog->percent = 100;
        do_progress_total (prog, options, file_count, total_files);
        close (srcdesc);
        close (dstdesc);
        chmod (dstname, srcstat.st_mode&
               (S_IRWXU | S_IRWXG | S_IRWXO | S_ISUID | S_ISGID));
        if (options->kill == true) 
        {
            if ((chmod (srcname, 0777)) == -1)
                fprintf (stderr, "%s: Chmod for file '%s' failed\n", NAME, srcname);
          
            if (remove (srcname) == -1)
                fprintf (stderr, "%s: Couldn't remove source file '%s' (%s)\n", 
                         NAME, srcname, strerror (errno));
        }
//        printf("\n");
        free (prog);

        return *total_progress;
    } 
    else if ((srcstat.st_size == 0) && options->quiet == true) 
    {
        close (srcdesc);
        close (dstdesc);
        if (options->kill == true) 
        {
            if ((chmod (srcname, 0777)) == -1)
            {
                fprintf (stderr, "%s: Chmod failed for file '%s'\n", NAME, srcname);
            }
            if (remove (srcname) == -1)
                fprintf (stderr, "%s: couldn't remove source file '%s' (%s)\n", 
                         NAME, srcname, strerror (errno));
        }
        free (prog);

        return *total_progress;
    }

    

    int counter = 0;
    
    /* set up buffersize, according to disks blocksize */
    buffersize = srcstat.st_blksize;  
  
    buf = malloc (buffersize * sizeof (*buf));

    if (buf == NULL)
    {
        fprintf (stderr, "%s: Could not allocate memory for buffer in do_copy\n", NAME);
        exit (EXIT_FAILURE);
    }
    do
    {
        readbytes = read (srcdesc, buf, buffersize);

        if (readbytes == -1)
        {
            fprintf (stderr, "%s: An error occured while reading file '%s'\n", NAME, srcname);
            printf ("%s\n", strerror (errno));
            return *total_progress;
        }
        bwritten = write (dstdesc, buf, readbytes);

        if (bwritten != readbytes)
        {
            fprintf (stderr, "%s: Error while writing to file '%s'\n", NAME, dstname);
            return *total_progress;
        }
        prog->total.bytes += bwritten;
        counter++;
        
        if (options->quiet != true) 
        {

            if ((counter >= 2048) || (bwritten < buffersize)) 
            {
                update_progress_total (prog, options);
                do_progress_total (prog, options, file_count, total_files);
                counter = 0;
            }
        }

    } while (readbytes != 0);

    unsigned long long temp_value = prog->total.bytes;

    

//    if (options->quiet != true) printf ("\n");

    if (options->preserve == true) 
    {
        utimdef.actime = srcstat.st_atime;
        utimdef.modtime = srcstat.st_mtime;
        if (!utime (dstname, &utimdef) == 0)
            fprintf (stderr, "%s: error setting utime of file '%s' (%s)\n", 
                     NAME, dstname, strerror (errno));
    }

    /* FIXME: still not working right, even though it now retains
       file permissions. Feels like some sort of cheap hack. what about
       S_ISUID (set-used-ID) and S_ISVTX (sticky bit) and S_ISGID? 
    */

    chmod (dstname, srcstat.st_mode&
           (S_IRWXU | S_IRWXG | S_IRWXO | S_ISUID | S_ISGID));

    /* clean up */
    if (options->kill == true) 
    {
        if ((chmod (srcname, 0777)) == -1)
            fprintf (stderr, "%s: Chmod for file '%s' failed.\n", NAME, srcname);
      
        if (remove (srcname) == -1)
            fprintf (stderr, "%s: couldn't remove source file '%s' (%s)\n", 
                     NAME, srcname, strerror (errno));
    }
//    destroy_progress (prog);
    free (prog);
    free (buf);
    close (srcdesc);
    close (dstdesc);

    return temp_value;
}

void copy_recursive (char *real_src_dir, 
                     char *real_target_dir,
                     int depth, ino_t orig_inode, 
                     struct opt *options) 
{
    char *copy_realsrc, *copy_targetdir, *new_dir;
    char *mkd;

    DIR *directory;
    struct dirent *current;
    struct stat statfile;

    copy_realsrc = malloc (path_max * sizeof (*copy_realsrc));
    copy_targetdir = malloc (path_max * sizeof (*copy_targetdir));

    realpath (real_src_dir, copy_realsrc);
    realpath (real_target_dir, copy_targetdir);


    lstat (real_src_dir, &statfile);
    if (statfile.st_ino == orig_inode) 
    {
        printf ("%s: skipping target directory '%s'\n", NAME, real_src_dir);
        return;
    }
    remove_trailing_slashes (copy_targetdir);

    if ((directory = opendir (copy_realsrc)) == NULL) 
    {
        fprintf (stderr, "%s: error opening directory '%s' (%s)\n", 
                 NAME, copy_realsrc, strerror (errno));
        return;
    }

    chdir (copy_realsrc);

    while ((current = readdir (directory)) != NULL) 
    {
        lstat (current->d_name, &statfile);
        if (S_ISDIR (statfile.st_mode)) 
        {
            new_dir = malloc (path_max * sizeof *new_dir);
            mkd = malloc (path_max * sizeof *mkd);

            /* if we are seeing ., .., continue */
            if ((strcmp (".", current->d_name) == 0) || 
                (strcmp ("..", current->d_name) == 0)) continue;

            /* check for hidden files */ 
            if (options->no_hidden_files == true)
                if (current->d_name[0] == '.')
                    continue;
    
            realpath (current->d_name, new_dir);

            strcpy (mkd, copy_targetdir);
            apptodir (mkd, current->d_name);
            mkdir (mkd, 0777);
            chmod (mkd, statfile.st_mode);
            copy_recursive (new_dir, mkd, depth+4, orig_inode,
                            options);
            if (options->kill == true) 
            { 
                chmod (new_dir, 0777); /* precaution against readonly dirs */
                if (remove (new_dir) != 0)
                    fprintf (stderr, "%s: removing '%s' failed (%s)\n", 
                             NAME, new_dir, strerror (errno));
            }

        }
        else 
        {
            do_copy (current->d_name, copy_targetdir, options, true);
        }

    }
    chdir ("..");
    closedir (directory);
}

void copy_recursive_total (char *real_src_dir, 
                           char *real_target_dir,
                           int depth, ino_t orig_inode, 
                           struct opt *options, unsigned long long *total_progress,
                           unsigned long long *total_size, int *file_count,
                           int *total_files) 
{
    char *copy_realsrc, *copy_targetdir, *new_dir;
    char *mkd;

    DIR *directory;
    struct dirent *current;
    struct stat statfile;
    unsigned long long bwritten;

    copy_realsrc = malloc (path_max * sizeof (*copy_realsrc));
    copy_targetdir = malloc (path_max * sizeof (*copy_targetdir));

    realpath (real_src_dir, copy_realsrc);
    realpath (real_target_dir, copy_targetdir);


    lstat (real_src_dir, &statfile);
    if (statfile.st_ino == orig_inode) 
    {
        printf ("%s: skipping target directory '%s'\n", NAME, real_src_dir);
        return;
    }
    remove_trailing_slashes (copy_targetdir);

    if ((directory = opendir (copy_realsrc)) == NULL) 
    {
        fprintf (stderr, "%s: error opening directory '%s' (%s)\n", 
                 NAME, copy_realsrc, strerror (errno));
        return;
    }

    chdir (copy_realsrc);

    while ((current = readdir (directory)) != NULL) 
    {
        lstat (current->d_name, &statfile);
        if (S_ISDIR (statfile.st_mode)) 
        {

            /* if we are seeing ., .., continue */
            if ((strcmp (".", current->d_name) == 0) || 
                (strcmp ("..", current->d_name) == 0)) continue;

            /* check for hidden files */ 
            if (options->no_hidden_files == true)
                if (current->d_name[0] == '.')
                    continue;
            new_dir = malloc (path_max * sizeof *new_dir);
            mkd = malloc (path_max * sizeof *mkd);
    
            realpath (current->d_name, new_dir);

            strcpy (mkd, copy_targetdir);
            apptodir (mkd, current->d_name);
            mkdir (mkd, 0777);
            chmod (mkd, statfile.st_mode);
            copy_recursive_total (new_dir, mkd, depth+4, orig_inode,
                                  options, total_progress, total_size, file_count, total_files);
            if (options->kill == true) 
            { 
                chmod (new_dir, 0777); /* precaution against readonly dirs */
                if (remove (new_dir) != 0)
                    fprintf (stderr, "%s: removing '%s' failed (%s)\n", 
                             NAME, new_dir, strerror (errno));
            }
            free (mkd);
            free (new_dir);

        }
        else 
        {
            bwritten = do_copy_total (current->d_name, copy_targetdir, options, true, total_progress,
                                      total_size, file_count, total_files);
            *total_progress = bwritten;
            *file_count += 1;
        }

    }
    chdir ("..");
    closedir (directory);
    free (current);
    free (copy_realsrc);
    free (copy_targetdir);

}

void count_total (char *source_dir, 
                  unsigned long long *total_size,
                  int *filecount, struct opt *options) 
{
    DIR *directory;
    struct dirent *current;
    struct stat statfile;

 
    if ((directory = opendir (source_dir)) == NULL) 
    {
        fprintf (stderr, "%s: error opening directory '%s' (%s)\n", 
                 NAME, source_dir, strerror (errno));
        return;
    }

    chdir (source_dir);
    
    while ((current = readdir (directory)) != NULL) 
    {
        lstat (current->d_name, &statfile);
        if (S_ISDIR (statfile.st_mode)) 
        {
            if ((strcmp (".", current->d_name) == 0) || 
                (strcmp ("..", current->d_name) == 0)) continue;

            /* check for hidden files */ 
/*             if (options->no_hidden_files == true) */
/*                 if (current->d_name[0] == '.') */
/*                     continue; */
    
            count_total (current->d_name, total_size,
                         filecount, options);
        }
        else 
        {
            *total_size += statfile.st_size;
            *filecount = *filecount + 1;
        }

    }
    chdir ("..");
    closedir (directory);
    free (current);
}



