/*
 * dvdspanky - a video to DVD MPEG conversion front-end for transcode
 * Copyright (C) 2007  Jeffrey Grembecki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <string.h>
#include <pcre.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <stdarg.h>
#include <dirent.h>
#include <fnmatch.h>
#include "dvdspanky.h"
#include "vstr.h"
#include "config.h"

static char gp_title[] = "dvdspanky 1.0.3 - (C) 2007 Jeffrey Grembecki\n\n";
static char gp_licence[] =
"This program is free software; you can redistribute it and/or modify\n"
"it under the terms of the GNU General Public License as published by\n"
"the Free Software Foundation; either version 2 of the License, or\n"
"(at your option) any later version.\n"
"\n"
"This program is distributed in the hope that it will be useful,\n"
"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
"GNU General Public License for more details.\n"
"\n"
"You should have received a copy of the GNU General Public License\n"
"along with this program; if not, write to the Free Software\n"
"Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,\n"
"MA 02110-1301, USA.\n";

static char gp_help[] =
"%s -i file [OPTIONS]\n"
"\n"
"following; [ ] = default, { } = requirements, * = experimental\n"
"\n"
"-0, --avpass           auto volume pass only [off]\n"
"-1, --firstpass        1st pass only [off]\n"
"-2, --secondpass       2nd pass only including multiplex [off]\n"
"-a, --aspect=MODE      aspect ratio 2=4:3, 3=16:9 [auto] {2,3}\n"
"-A, --autovolume       automatically adjust volume [off]\n"
"-b, --border           automatic black border [off]\n"
"-B, --clip             clips edges to fit aspect [off]\n"
"-c, --channels=CHANS   number of audio channels [auto] {1 - 5}\n"
"-f, --filters=FILTERS  specify transcode filters [none]\n"
"-g, --gain=GAIN        multiply volume by GAIN [off/1.0]\n"
"-G, --gamma=GAMMA      set gamma, < 1.0 is brighter, > is darker [1.0]\n"
"-h, --help             this help text\n"
"-H, --hcrop=PIXELS     as above [off]\n"
"-i, --input=FILE       input file\n"
"-I, --nonice           do not run at nice priority 19 [off]\n"
"-L, --licence          show the program licence\n"
"-m, --multi            2 pass mode [off]\n"
"-M, --mplex            multiplex only [off]\n"
"-n, --ntsc             set ntsc [auto]\n"
"-N, --ntscfix          fix variable framerate ntsc video [off] *\n"
"-o, --output=FILE      output filename prefix (w/o extension)\n"
"-O, --normalise        dynamically normalise audio volume [off]\n"
"-r, --vrate=RATE       video bitrate [auto] {600 - 7500}\n"
"-p, --pal              set pal [auto]\n"
"-P, --preview          previews 5 frames using feh then exits [off]\n"
"-R, --arate=RATE       audio bitrate [auto] {64 - 448, 192 is good}\n"
"-s, --size=KB          final mpg size in kilobytes\n"
"-T, --postprocess      enable post processing filter [off]\n"
"-v, --verbose          be verbose [off]\n"
"-V, --vcrop=PIXELS     pos for crop, neg for black border [off]\n"
"\nsee man 1 dvdspanky for more information\n";

static struct option gp_longopts[] =
{
	{"avpass", 0, 0, '0'},
	{"firstpass", 0, 0, '1'},
	{"secondpass", 0, 0, '2'},
	{"aspect", 1, 0, 'a'},
	{"autovolume", 0, 0, 'A'},
	{"border", 0, 0, 'b'},
	{"clip", 0, 0, 'B'},
	{"channels", 1, 0, 'c'},
	{"filters", 1, 0, 'f'},
	{"frameadjust", 0, 0, 'F'},
	{"gain", 1, 0, 'g'},
	{"gamma", 1, 0, 'G'},
	{"hcrop", 1, 0, 'H'},
	{"help", 0, 0, 'h'},
	{"input", 1, 0, 'i'},
	{"nonice", 0, 0, 'I'},
	{"licence", 0, 0, 'L'},
	{"mplex", 0, 0, 'M'},
	{"multi", 0, 0, 'm'},
	{"multipass", 0, 0, 'm'},
	{"ntsc", 0, 0, 'n' },
	{"ntscfix", 0, 0, 'N'},
	{"output", 1, 0, 'o'},
	{"normalise", 0, 0, 'O'},
	{"normalize", 0, 0, 'O'},
	{"pal", 0, 0, 'p' },
	{"preview", 0, 0, 'P'},
	{"vrate", 1, 0, 'r'},
	{"arate", 1, 0, 'R'},
	{"size", 1, 0, 's'},
	{"postprocess", 0, 0, 'T'},
	{"vcrop", 1, 0, 'V'},
	{"verbose", 0, 0, 'v'},
};

static char gp_shortopts[] = "012a:AbBc:f:Fg:G:hH:i:ILmMnNo:OpPr:R:s:TvV:";

static pcre *gp_idregex = NULL;
static pcre *gp_transregex = NULL;
static pcre *gp_tcvideoregex = NULL;
static pcre *gp_tcaudioregex = NULL;

#define ML_ERROR 4
#define ML_WARNING 3
#define ML_INFO 2
#define ML_VERBOSE 1
#define ML_DEBUG 0

static int g_msglevel = ML_INFO;

static FILE *gp_logfile = NULL;

static void print(int msglevel, const char *p_format, ...)
{
	static vstr *p_str = NULL;
	va_list argp;

	if(!p_str)
	{
		if(!(p_str = vs_new()))
			return;
	}

	if(msglevel >= g_msglevel)
	{
		va_start(argp, p_format);
		vs_vprint(p_str, p_format, argp);
		va_end(argp);
		printf("%s", p_str->s);
	}

	if(gp_logfile)
	{
		vs_print(p_str, " dvdspanky | ");
		va_start(argp, p_format);
		vs_vcat(p_str, p_format, argp);
		va_end(argp);
		fwrite(p_str->s, 1, p_str->len, gp_logfile);
		fflush(gp_logfile);
	}
}

static void pprint(int msglevel, const char *p_prefix, const char *p_format, ...)
{
	static vstr *p_str = NULL;
	va_list argp;

	if(!p_str)
	{
		if(!(p_str = vs_new()))
			return;
	}

	vs_print(p_str, "%10s | ", p_prefix);
	va_start(argp, p_format);
	vs_vcat(p_str, p_format, argp);
	va_end(argp);

	if(msglevel >= g_msglevel)
		printf("%s", p_str->s);

	if(gp_logfile)
	{
		fwrite(p_str->s, 1, p_str->len, gp_logfile);
		fflush(gp_logfile);
	}
}

static void printerror(const char *p_format, ...)
{
	static vstr *p_str = NULL;
	va_list argp;

	if(!p_str)
	{
		if(!(p_str = vs_new()))
			return;
	}

	vs_print(p_str, "error: ");
	va_start(argp, p_format);
	vs_vcat(p_str, p_format, argp);
	va_end(argp);
	vs_cat(p_str, "\n");

	print(ML_ERROR, p_str->s);
}

/* termination cleanup */
static void cleanup()
{
	if(gp_idregex)
		pcre_free(gp_idregex);
	if(gp_transregex)
		pcre_free(gp_transregex);
	if(gp_tcvideoregex)
		pcre_free(gp_tcvideoregex);
	gp_idregex = gp_transregex = gp_tcvideoregex = NULL;

	if(gp_logfile)
		fclose(gp_logfile);

	gp_logfile = NULL;
}

/* encode / preview */
int encode(video_t *p_in, video_t *p_out, int options, const char *p_filters)
{
	vstr *p_generic, *p_video, *p_videoextra, *p_audio, *p_final;

	p_generic = vs_new();
	p_video = vs_new();
	p_videoextra = vs_new();
	p_audio = vs_new();
	p_final = vs_new();

	if(!p_generic || !p_video || !p_videoextra || !p_audio || !p_final)
	{
		if(p_generic) vs_free(p_generic);
		if(p_video) vs_free(p_video);
		if(p_videoextra) vs_free(p_videoextra);
		if(p_audio) vs_free(p_audio);
		if(p_final) vs_free(p_final);
		printerror("unable to allocate strings");
		return 1;
	}

#define freestrings() { vs_free(p_generic); vs_free(p_video); vs_free(p_videoextra); \
	vs_free(p_audio); vs_free(p_final); }

	/* generic */
	vs_cat(p_generic, "transcode -i %s", p_in->p_filename);
	if(!(options & SPANK_NONICE))
		vs_cat(p_generic, " --nice 19");
	if(options & SPANK_MPLAYER)
	{
		unlink("stream.yuv");
		vs_cat(p_generic, " -H 0 -x mplayer,mplayer -g %dx%d -n 0x1 -e %d,%d,%d", p_in->width,
			p_in->height, p_in->hz,	(p_in->abits) ? p_in->abits : 16, p_in->achans);
	}

	/* audio */
	vs_cat(p_audio, " -E 48000,%d,%d -N 0x2000 -b %d",
		(p_out->abits) ? p_out->abits : 16, p_out->achans, p_out->arate);

	/* video */
	{
		int asr = (p_out->aspect < 1.5) ? 2 : 3;
		vs_cat(p_video, " -w %d -f %.3f --export_fps %.3f --import_asr %d --export_asr %d"
			" --encode_fields %c", 	p_out->vrate, p_in->fps, p_out->fps, asr, asr,
			(p_out->p_vcodec[0] == 'P') ? 't' : 'b');
	}
	if(options & SPANK_DOUBLEFPS)
		vs_cat(p_video, " -J doublefps");
	if(options & SPANK_NEWFPS)
		vs_cat(p_video, " -J modfps=clonetype=1");
	if(options & SPANK_FIXNTSC)
		vs_cat(p_video, " -M 2 -J ivtc,32detect=force_mode=5,decimate --hard_fps");

	/* video extra */
	if(p_out->vcrop || p_out->hcrop)
		vs_cat(p_videoextra, " -j %d,%d,%d,%d",
			p_out->vcrop, p_out->hcrop, p_out->vcrop, p_out->hcrop);
	if(p_out->gamma != 1.0)
		vs_cat(p_videoextra, " -G %f", p_out->gamma);
	if(options & SPANK_PPROCESS)
		vs_cat(p_videoextra, " -J pp=de");
	if(options & SPANK_NORMALISE)
		vs_cat(p_videoextra, " -J normalize=algo=2 -s 0.5");
	if(p_filters[0])
		vs_cat(p_videoextra, " -J %s", p_filters);

	/* create ffmpeg.cfg */
	system("echo \"[mpeg2video]\nvrc_minrate=0\nvrc_maxrate=7500\nvrc_buf_size=1792\n\""
		" > ffmpeg.cfg");

	/* preview settings */
	if(options & SPANK_PREVIEW)
	{
		struct dirent **pp_dirent;
		int count;

		print(ML_INFO, "generating previews..\n"); /* generate previews */
		if(forkitf(NULL, NULL, TRANSCODE_BIN, "%s %s %s -y jpg,null -F 100 -Z %dx%d"
			" -c 300-301,600-601,900-901,1200-1201,1500-1501 -o pre",
			p_generic->s, p_video->s, p_videoextra->s,
			(int)((double)p_out->height * p_out->aspect), p_out->height, p_out->p_filename))
		{
			printerror("failed generating previews");
			freestrings();
			return 1;
		}

		count = scandir(".", &pp_dirent, 0, 0);
		if(count >= 0) /* display then remove previews */
		{
			int found = count;

			print(ML_INFO, "displaying previews..\n");

			vs_print(p_final, "feh");
			while(count--)
			{
				if(!fnmatch("pre*.jpg", pp_dirent[count]->d_name, 0))
					vs_cat(p_final, " %s", pp_dirent[count]->d_name);
			}
			forkit(NULL, NULL, FEH_BIN, p_final->s);

			count = found;
			while(count--)
			{
				if(!fnmatch("pre*.jpg", pp_dirent[count]->d_name, 0))
					unlink(pp_dirent[count]->d_name);
			}

			free(pp_dirent);
		}
		freestrings();
		return 0;
	}

	/* auto volume only pass */
	if(options & SPANK_AVPASS)
	{
		print(ML_INFO, "scanning volume..\n");
		vs_print(p_final, "%s-tmp.ac3", p_out->p_filename);
		if(forkitf(cb_transcode, (void*)p_out, TRANSCODE_BIN,
			"%s %s -y null,ac3 -m %s-tmp -J astat=%s.norm",
			p_generic->s, p_audio->s, p_out->p_filename, p_out->p_filename))
		{
			print(ML_INFO, "\n");
			printerror("error scanning volume");
			unlink(p_final->s);
			freestrings();
			return 1;
		}
		print(ML_INFO, "\n");
		unlink(p_final->s);
		return 0;
	}

	/* execute 1st pass */
	if(options & SPANK_FIRST && !(options & SPANK_MULTIPLEX))
	{
		print(ML_INFO, "starting 1st VBR pass..\n");

		vs_print(p_final, "%s %s %s", p_generic->s, p_video->s, p_audio->s);
		if(options & SPANK_AVOLUME)
			vs_cat(p_final,
				" -y ffmpeg -F mpeg2video -R 1,%s.pass1 -m %s.tmp -J astat=%s.norm",
				p_out->p_filename, p_out->p_filename, p_out->p_filename);
		else
			vs_cat(p_final, " -y ffmpeg,null -F mpeg2video -R 1,%s.pass1", p_out->p_filename);

		if(forkit(cb_transcode, (void*)p_out, TRANSCODE_BIN, p_final->s))
		{
			vs_print(p_final, "%s.tmp", p_out->p_filename);
			unlink(p_final->s);
			print(ML_INFO, "\n");
			printerror("transcode failed");
			freestrings();
			return 1;
		}
		vs_print(p_final, "%s.tmp", p_out->p_filename);
		unlink(p_final->s);

		print(ML_INFO, "\n");
	}

	/* execute 2nd pass */
	if(options & SPANK_SECOND && !(options & SPANK_MULTIPLEX))
	{
		/* volume auto adjust */
		if(options & SPANK_AVOLUME)
		{
			FILE *p_file;
			char p_buffer[320];

			vs_print(p_final, "%s.norm", p_out->p_filename);
			p_file = fopen(p_final->s, "r");
			if(!p_file)
			{
				printerror("unable to open %s.norm for autoaudio adjustment", p_out->p_filename);
				freestrings();
				return 1;
			}
			if(!fgets(p_buffer, sizeof(p_buffer) - 1, p_file))
			{
				printerror("%s.norm contains no relevant data", p_out->p_filename);
				fclose(p_file);
				freestrings();
				return 1;
			}
			p_out->volume = atof(p_buffer);
			if(p_out->volume >= 1.1)
				p_out->volume -= 0.1;
			else
				p_out->volume = 1.0;
			fclose(p_file);
		}

		/* check pass1 file exists */
		{
			struct stat sdata;
			vs_print(p_final, "%s.pass1", p_out->p_filename);
			if(stat(p_final->s, &sdata))
			{
				printerror("unable to open %s.pass1", p_out->p_filename);
				freestrings();
				return 1;
			}
		}

		/* transcode */
		print(ML_INFO, "starting 2nd VBR pass..\n");
		vs_print(p_final,
			"%s %s %s %s -Z %dx%d -y ffmpeg -F mpeg2video -R 2,%s.pass1 -o %s -m %s.ac3",
			p_generic->s, p_audio->s, p_video->s, p_videoextra->s,
			p_out->width, p_out->height, p_out->p_filename, p_out->p_filename, p_out->p_filename);
		if(p_out->volume != 1.0)
			vs_cat(p_final, " -s %f", p_out->volume);
		if(forkitf(cb_transcode, (void*)p_out, TRANSCODE_BIN, p_final->s))
		{
			print(ML_INFO, "\n");
			printerror("transcode failed");
			freestrings();
			return 1;
		}
		print(ML_INFO, "\n");
	}

	/* CBR encoding */
	if(!(options & (SPANK_FIRST | SPANK_SECOND)) && !(options & SPANK_MULTIPLEX))
	{
		/* volume auto adjust */
		if(options & SPANK_AVOLUME)
		{
			char p_buffer[320];
			FILE *p_file;

			print(ML_INFO, "scanning volume..\n");
			if(forkitf(cb_transcode, (void*)p_out, TRANSCODE_BIN,
				"%s %s -y null,ac3 -m %s-tmp -J astat=%s.norm",
				p_generic->s, p_audio->s, p_out->p_filename, p_out->p_filename))
			{
				print(ML_INFO, "\n");
				printerror("error scanning volume");
			}
			print(ML_INFO, "\n");

			vs_print(p_final, "%s-tmp.ac3", p_out->p_filename);
			unlink(p_final->s);

			vs_print(p_final, "%s.norm", p_out->p_filename);
			p_file = fopen(p_final->s, "r");
			if(!p_file)
			{
				printerror("unable to open %s.norm for autoaudio adjustment", p_out->p_filename);
				freestrings();
				return 1;
			}
			if(!fgets(p_buffer, sizeof(p_buffer) - 1, p_file))
			{
				printerror("%s.norm contains no relevant data", p_out->p_filename);
				fclose(p_file);
				freestrings();
				return 1;
			}
			p_out->volume = atof(p_buffer);
			fclose(p_file);
		}

		/* execute */
		print(ML_INFO, "starting CBR encode..\n");
		vs_print(p_final, "%s %s %s %s -Z %dx%d -y ffmpeg -F mpeg2video -o %s -m %s.ac3",
			p_generic->s, p_video->s, p_videoextra->s, p_audio->s,
			p_out->width, p_out->height, p_out->p_filename, p_out->p_filename);
		if(p_out->volume != 1.0)
			vs_cat(p_final, " -s %f", p_out->volume);
		if(forkit(cb_transcode, (void*)p_out, TRANSCODE_BIN, p_final->s))
		{
			print(ML_INFO, "\n");
			printerror("transcode failed");
			freestrings();
			return 2;
		}
		print(ML_INFO, "\n");
	}

	/* multiplex */
	if((options & SPANK_SECOND) || !(options & (SPANK_FIRST | SPANK_SECOND)) ||
		(options & SPANK_MULTIPLEX))
	{
		print(ML_INFO, "multiplexing..\n");
		if((options & SPANK_SECOND))
		{
			if(forkitf(NULL, "mplex |", MPLEX_BIN,
				"mplex -V -f 8 -o %s.mpg %s.m2v %s.ac3",
				p_out->p_filename, p_out->p_filename, p_out->p_filename))
				{
					print(ML_INFO, "\n");
					printerror("mplex failed");
					freestrings();
					return 3;
				}
		}
		else
		{
			if(forkitf(NULL, "mplex |", MPLEX_BIN,
				"mplex -f 8 -o %s.mpg %s.m2v %s.ac3",
				p_out->p_filename, p_out->p_filename, p_out->p_filename))
			{
				print(ML_INFO, "\n");
				printerror("mplex failed");
				freestrings();
				return 3;
			}
		}

		print(ML_INFO, "\n");
		vs_print(p_final, "%s.m2v", p_out->p_filename);
		unlink(p_final->s);
		vs_print(p_final, "%s.ac3", p_out->p_filename);
		unlink(p_final->s);
	}

	freestrings();
	return 0;
#undef freestrings
}

/* tcprobe identify callback */
int cb_tcprobe(const char *p_string, void *p_options)
{
	char *p_buffer;
	video_t *p_video;
	int numstrings, length, p_vector[20];
	int i;

	/* option is a video_t */
	p_video = (video_t*)p_options;

	/* initialisation command */
	if(!p_string)
	{
		/* pre set feedback as nothing found */
		p_video->feedback = 0;
		return 0;
	}

	pprint(ML_VERBOSE, "tcprobe", "%s\n", p_string);

	/* video regex */
	length = strlen(p_string);
	numstrings = pcre_exec(gp_tcvideoregex, NULL, p_string, length, 0, 0,
		p_vector, sizeof(p_vector) / sizeof(int));

	if(numstrings > 0)
	{
		p_video->feedback = 1;

		/* allocate, copy string and terminate vectors */
		length = strlen(p_string);
		p_buffer = malloc(length + 1);
		memcpy(p_buffer, p_string, length + 1);
		for(i = 1; i < 6; i++)
			p_buffer[p_vector[i * 2 + 1]] = 0;

		/* video settings found, update any missing */
		if(!p_video->fps)
			p_video->fps = atof(&p_buffer[p_vector[2]]);

		if(!p_video->p_vcodec[0])
			strncpy(p_video->p_vcodec, &p_buffer[p_vector[4]], sizeof(p_video->p_vcodec));

		if(!p_video->frames)
			p_video->frames = atoi(&p_buffer[p_vector[6]]);

		if(!p_video->width)
			p_video->width = atoi(&p_buffer[p_vector[8]]);

		if(!p_video->height)
			p_video->width = atoi(&p_buffer[p_vector[10]]);

		if(p_video->width && p_video->height)
			p_video->aspect = (double)p_video->width / (double)p_video->height;

		free(p_buffer);

		return 0;
	}

	/* audio regex */
	length = strlen(p_string);
	numstrings = pcre_exec(gp_tcaudioregex, NULL, p_string, length, 0, 0,
		p_vector, sizeof(p_vector) / sizeof(int));

	if(numstrings > 0)
	{
		p_video->feedback = 1;

		/* allocate, copy string and terminate vectors */
		length = strlen(p_string);
		p_buffer = malloc(length + 1);
		memcpy(p_buffer, p_string, length + 1);
		for(i = 1; i < 5; i++)
			p_buffer[p_vector[i * 2 + 1]] = 0;

		/* audio settings found, update any missing */
		if(!p_video->hz)
			p_video->hz = atoi(&p_buffer[p_vector[2]]);

		if(!p_video->abits)
			p_video->abits = atoi(&p_buffer[p_vector[4]]);

		if(!p_video->achans)
			p_video->achans = atoi(&p_buffer[p_vector[6]]);

		if(!p_video->arate)
			p_video->arate = atoi(&p_buffer[p_vector[8]]);

		free(p_buffer);

		return 0;
	}

	return 0;
}

/* transcode progress callback */
int cb_transcode(const char *p_string, void *p_options)
{
	char p_buffer[1024];
	int numstrings, length, p_vector[20];
	video_t *p_video;
	int i;

	/* initialisation command */
	if(!p_string)
		return 0;

	/* option is a video_t */
	p_video = (video_t*)p_options;

	/* regex string */
	length = strlen(p_string);
	numstrings = pcre_exec(gp_transregex, NULL, p_string, length, 0, 0,
		p_vector, sizeof(p_vector) / sizeof(int));

	/* make copy of string and terminate vectors with 0 */
	if(numstrings > 0)
	{
		if(length >= sizeof(p_buffer))
		{
			printerror("string exceeds buffer");
			return 1;
		}
		memcpy(p_buffer, p_string, length);
		for(i = 1; i < 6; i++)
			p_buffer[p_vector[i * 2 + 1]] = 0;
		if(p_video->frames)
			fprintf(stderr,"  %6.2f fps, %.2f%% complete, frame %d  \r", atof(&p_buffer[p_vector[4]]),
				atof(&p_buffer[p_vector[2]]) * 100.0 / (double)p_video->frames, atoi(&p_buffer[p_vector[2]]));
		else
			fprintf(stderr,"  %6.2f fps, frame %09d, %d:%02d:%02d  \r", atof(&p_buffer[p_vector[4]]),
				atoi(&p_buffer[2]), atoi(&p_buffer[p_vector[4]]), atoi(&p_buffer[p_vector[6]]),
				atoi(&p_buffer[p_vector[8]]));
	}
	else
	{
		pprint(ML_VERBOSE, "transcode", "%s\n", p_string);
	}

	return 0;
}

/* mplayer identify callback */
int cb_mplayer(const char *p_string, void *p_options)
{
	const char *p_id;
	const char *p_value;
	video_t *p_video;
	int p_vector[20];
	int numstrings;

	/* initialisation command */
	if(!p_string)
		return 0;

	/* option is output to a video_t */
	p_video = (video_t*)p_options;

	pprint(ML_VERBOSE, "mplayer", "%s\n", p_string);

	/* examine the string, return if nothing */
	numstrings = pcre_exec(gp_idregex, NULL, p_string, strlen(p_string), 0, 0, p_vector, 20);
	if(numstrings < 0)
		return 0;

	/* create temp strings and feed data into video_t, speed is not needed here */
	pcre_get_substring(p_string, p_vector, numstrings, 1, &p_id);
	pcre_get_substring(p_string, p_vector, numstrings, 2, &p_value);

	if(!strcmp(p_id, "VIDEO_FORMAT"))
		strncpy(p_video->p_vcodec, p_value, sizeof(p_video->p_vcodec) - 1);
	if(!strcmp(p_id, "VIDEO_BITRATE"))
		p_video->vrate = atoi(p_value) / 1000;
	if(!strcmp(p_id, "VIDEO_WIDTH"))
		p_video->width = atoi(p_value);
	if(!strcmp(p_id, "VIDEO_HEIGHT"))
	{
		p_video->height = atoi(p_value);
		p_video->aspect = (float)p_video->width / (float)p_video->height;
	}
	if(!strcmp(p_id, "VIDEO_FPS"))
		p_video->fps = atof(p_value);
	if(!strcmp(p_id, "AUDIO_CODEC"))
		strncpy(p_video->p_acodec, p_value, sizeof(p_video->p_acodec) - 1);
	if(!strcmp(p_id, "AUDIO_BITRATE"))
		p_video->arate = atoi(p_value) / 1000;
	if(!strcmp(p_id, "AUDIO_RATE"))
		p_video->hz = atoi(p_value);
	if(!strcmp(p_id, "AUDIO_NCH"))
		p_video->achans = atoi(p_value);
	if(!strcmp(p_id, "LENGTH"))
	{
		p_video->length = atoi(p_value);
		p_video->hours   = p_video->length / 3600;
		p_video->minutes = p_video->length % 3600 / 60;
		p_video->seconds = p_video->length % 60;
	}

	pcre_free_substring(p_id);
	pcre_free_substring(p_value);

	return 0;
}

/* generic callback */
int cb_default(const char *p_string, void *p_options)
{

	/* initialisation command */
	if(!p_string)
		return 0;

	/* output le shiz */
	pprint(ML_VERBOSE, (char*)p_options, "%s\n", p_string);

	return 0;
}

/* fork another program */
int forkit(forkitcallback callback, void *p_options, const char *p_path, const char *p_cmd)
{
	char *pp_cmdopts[100], *p_cmdline, p_buffer[4096];
	int cmdlength;
	int p_pipe[2], pid;
	int status = 0;
	int retval = 0;
	int count;
	int i, pos, start, copy;

	cmdlength = strlen(p_cmd);
	p_cmdline = malloc(cmdlength + 1);

	if(!callback)
		callback = cb_default;

	/* create parameter index */
	strncpy(p_cmdline, p_cmd, cmdlength + 1);
	pos = start = 0;
	print(ML_VERBOSE, "fork %s, %s\n", p_path,p_cmd);

	for(i = 0; i < sizeof(pp_cmdopts) / sizeof(char**); i++)
	{
		while(p_cmdline[pos] != ' ' && p_cmdline[pos] != 0)
			pos++;

		pp_cmdopts[i] = &p_cmdline[start];
		if(p_cmdline[pos] == 0)
		{
			pp_cmdopts[i + 1] = 0;
			break;
		}
		p_cmdline[pos] = 0;
		pos++;
		start = pos;
	}

	/* open comminication pipe */
	if(pipe(p_pipe) == -1)
	{
		printerror("open pipe failed");
		return 1;
	}

	/* fork */
	pid = fork();
	switch(pid)
	{
		case -1:
			/* error */
			close(p_pipe[0]);
			close(p_pipe[1]);
			printerror("fork %s failed", p_path);
			retval = -2;
			break;

		case 0:
			/* child, run program */
			close(p_pipe[0]);
			dup2(p_pipe[1], fileno(stdout));
			dup2(p_pipe[1], fileno(stderr));
			execv(p_path, pp_cmdopts);
			printerror("fork %s, %s failed", p_path, p_cmd);
			free(p_cmdline);
			exit(1);

		default:
			/* parent, send output to callback */
			close(p_pipe[1]);

			/* reset the callback */
			if(callback)
				callback(NULL, p_options);

			/* read input while waiting for child to term */
			start = 0;
			while(waitpid(pid, &status, WNOHANG) != pid)
			{
				count = read(p_pipe[0], &p_buffer[start], sizeof(p_buffer) - 1 - start);
				if(count)
				{
					/* seperate lines and send each to callback */
					p_buffer[start + count] = 0;
					for(start = 0, pos = 0; pos < sizeof(p_buffer) - 1; pos++)
					{
						if(p_buffer[pos] == '\n' || p_buffer[pos] == '\r')
						{
							/* send line to callback */
							p_buffer[pos] = 0;
							if(callback && callback(&p_buffer[start], p_options))
								retval = -3;
							start = pos + 1;
						}
						else if(p_buffer[pos] == 0)
						{
							/* copy unread buffer to beginning */
							for(copy = 0; copy < pos - start; copy++)
								p_buffer[copy] = p_buffer[start + copy];

							/* set position ready for continued reading */
							start = pos - start;

							break;
						}
					}
				}
			}
			/* return value same as exit status or -1 if terminated by signal */
			if(!retval)
			{
				if(WIFEXITED(status))
					retval = WEXITSTATUS(status);
				else if(WIFSIGNALED(status))
					retval = -1;
			}

			close(p_pipe[0]);
			break;
	}

	free(p_cmdline);
	return retval;
}

/* formatted forking of another program */
int forkitf(forkitcallback callback, void *p_options, const char *p_path, const char *p_cmd, ...)
{
	va_list argp;
	char p_buffer[4096];

	va_start(argp, p_cmd);
	vsnprintf(p_buffer, sizeof(p_buffer) - 1, p_cmd, argp);
	va_end(argp);

	return forkit(callback, p_options, p_path, p_buffer);
}

/* entry point */
int main(int argc, char **argv)
{
	const char *p_error;
	int erroffset;
	int opt;
	char p_filters[512];
	int vrate = 0, arate = 0, aspect = 0, vcrop = 0, hcrop = 0, channels = 0;
	int calcsize = 0, calcborder = 0, calcclip = 0;
	char vidmode = 'a';
	int options = 0;
	video_t invid, outvid;

	p_filters[0] = 0;

	/* clear video data */
	memset(&invid, '\0', sizeof(video_t));
	memset(&outvid, '\0', sizeof(video_t));
	outvid.volume = outvid.gamma = 1.0;

	atexit(cleanup);

	/* show title */
	print(ML_INFO, gp_title);

	/* compile regex */
	if(!(gp_transregex = pcre_compile(
		"^encoding\\sframes\\s\\[\\d+\\-(\\d+)\\],\\s*(\\d+[.]\\d+)\\sfps.*?EMT:\\s*(\\d+):(\\d+):(\\d+),.*",
		0, &p_error, &erroffset, NULL)))
	{
		printerror("gp_transregex compile failed - %s", p_error);
		exit(1);
	}
	if(!(gp_idregex = pcre_compile("^ID_([^=]+)=(.*)$", 0, &p_error, &erroffset, NULL)))
	{
		printerror("gp_idregex compile failed - %s", p_error);
		exit(1);
	}
	if(!(gp_tcvideoregex = pcre_compile("^\\[.*?\\]\\sV:\\s([^ ]*)\\sfps,\\scodec=([^,]*),"
		"\\sframes=([^,]*),\\swidth=([^,]*),\\sheight=(.*)$", 0, &p_error, &erroffset, NULL)))
	{
		printerror("error: gp_tcvideoregex compile failed - %s", p_error);
		exit(1);
	}
	if(!(gp_tcaudioregex = pcre_compile("^\\[.*?\\]\\sA:\\s([^ ]*)\\sHz,\\sformat=[^,]*,"
		"\\sbits=([^,]*),\\schannels=([^,]*),\\sbitrate=(.*),$", 0, &p_error, &erroffset, NULL)))
	{
		printerror("error: gp_tcaudioregex compile failed - %s", p_error);
		exit(1);
	}

	/* parse options */
	while( (opt = getopt_long(argc, argv, gp_shortopts, gp_longopts, NULL)) != -1 )
	{
		switch(opt)
		{
			case '0':
				options |= SPANK_AVPASS;
				break;
			case '1':
				options &= 0xffff ^ SPANK_SECOND;
				options |= SPANK_FIRST;
				break;
			case '2':
				options &= 0xffff ^ SPANK_FIRST;
				options |= SPANK_SECOND;
				break;
			case 'a':
				aspect = atoi(optarg);
				if(aspect != 2 && aspect != 3)
				{
					printerror("aspect not 2 or 3");
					exit(1);
				}
				break;
			case 'A':
				options |= SPANK_AVOLUME;
				break;
			case 'b':
				calcborder = 1;
				break;
			case 'B':
				calcclip = 1;
				break;
			case 'c':
				channels = atoi(optarg);
				if(!channels || channels > 6)
				{
					printerror("channels not 1-6");
					exit(1);
				}
				break;
			case 'f':
				strncpy(p_filters, optarg, sizeof(p_filters) - 1);
				break;
			case 'F':
				/* frameadjust, now automatic */
				break;
			case 'g':
				outvid.volume = atof(optarg);
				break;
			case 'G':
				outvid.gamma = atof(optarg);
				break;
			case 'h':
				printf(gp_help, argv[0]);
				exit(0);
				break;
			case 'H':
				hcrop = atoi(optarg);
				break;
			case 'i':
				strncpy(invid.p_filename, optarg, sizeof(invid.p_filename) - 1);
				break;
			case 'I':
				options |= SPANK_NONICE;
				break;
			case 'L':
				printf(gp_licence);
				exit(0);
				break;
			case 'm':
				options |= SPANK_FIRST | SPANK_SECOND;
				break;
			case 'M':
				options |= SPANK_MULTIPLEX;
				break;
			case 'n':
				vidmode = 'n';
				break;
			case 'N':
				options |= SPANK_FIXNTSC;
				break;
			case 'p':
				vidmode = 'p';
				break;
			case 'P':
				options |= SPANK_PREVIEW;
				break;
			case 'o':
				strncpy(outvid.p_filename, optarg, sizeof(outvid.p_filename) - 1);
				break;
			case 'O':
				options |= SPANK_NORMALISE;
				break;
			case 'r':
				vrate = atoi(optarg);
				if(vrate < 600 || vrate > 7500)
				{
					printerror("vrate not 600 - 7500");
					exit(1);
				}
				break;
			case 'R':
				arate = atoi(optarg);
				if(arate < 64 || arate > 448)
				{
					printerror("arate not 64 - 488");
					exit(1);
				}
				break;
			case 's':
				calcsize = atoi(optarg);
				break;
			case 'T':
				options |= SPANK_PPROCESS;
				break;
			case 'v':
				g_msglevel = ML_VERBOSE;
				break;
			case 'V':
				vcrop = atoi(optarg);
				break;
			case '?':
				exit(1);
				break;
			default:
				printerror("unhandled option %c", opt);
				exit(1);
				break;
		}
	}

	/* input file */
	if(!invid.p_filename[0])
	{
		printerror("no input file");
		exit(1);
	}
	{
		struct stat sdata;
		if(stat(invid.p_filename, &sdata))
		{
			printerror("cannot open input file %s", invid.p_filename);
			exit(1);
		}
		invid.bytes = sdata.st_size;
	}


	/* create output filename based on input filename if none given */
	if(!outvid.p_filename[0])
	{
		pcre *p_regex;
		int p_vector[10];
		int found;

		p_regex = pcre_compile("^.*?([^/.]+)(?:\\.|)[^/.]*$", 0, &p_error, &erroffset, NULL);
		if(!p_regex)
		{
			printerror("failed to compile filename regex");
			exit(1);
		}
		found = pcre_exec(p_regex, NULL, invid.p_filename, strlen(invid.p_filename), 0, 0,
			p_vector, sizeof(p_vector) / sizeof(int));
		if(found > 0)
		{
			strncpy(outvid.p_filename, &invid.p_filename[p_vector[2]], p_vector[3] - p_vector[2]);
			outvid.p_filename[p_vector[3] - p_vector[2]] = 0;
		}
		else
		{
			printerror("unable to fathom output filename from input");
			pcre_free(p_regex);
			exit(1);
		}
		pcre_free(p_regex);
	}

	/* start log */
	{
		vstr *p_str;

		if(!(p_str = vs_new()))
		{
			printerror("unable to create string");
			exit(1);
		}
		vs_print(p_str, "%s.log", outvid.p_filename);
		if(!(gp_logfile = fopen(p_str->s, "w")))
		{
			printerror("unable creating log file");
			vs_free(p_str);
			exit(1);
		}
		vs_free(p_str);
	}

	/* get source video details */
	forkitf(cb_mplayer, &invid, MPLAYER_BIN,
		"mplayer %s -identify -frames 0 -ao null -vo null", invid.p_filename);
	forkitf(cb_tcprobe, &invid, TCPROBE_BIN, "tcprobe -i %s", invid.p_filename);
	if(!invid.feedback) /* if tcprobe unable to detect, flag use of mplayer for input */
	{
		if(invid.p_vcodec[0] == 0)
		{
			printerror("unknown input format. do you have the codec installed?");
			exit(1);
		}
		options |= SPANK_MPLAYER;
	}

	/* set audio bitrate / codec and scale */
	strcpy(outvid.p_acodec, "AC3");
	if(!arate)
		arate = invid.arate;
	for(outvid.arate = 64; outvid.arate < arate; outvid.arate += 32)
		;
	arate = outvid.arate;

	/* set audio channels */
	if(channels)
		outvid.achans = channels;
	else
		outvid.achans = invid.achans;

	/* set audio junk */
	outvid.abits = 16;
	outvid.hz = 48000;

	/* approximate missing data */
	if(!invid.frames && invid.length)
		invid.frames = (int)((double)invid.length * invid.fps);
	if(!invid.vrate && invid.frames)
		invid.vrate = (double)invid.bytes * 8.4 / ((double)invid.frames / invid.fps) / 1000.0 -
			invid.arate;

	/* calculate video bitrate for final size allowing 5% overhead */
	if(calcsize)
	{
		vrate = (calcsize - ((arate + arate / 20) / 8 * invid.length)) * 8
			/ invid.length;
		vrate -= vrate / 20;

		if(vrate < 600 || vrate > 7500)
		{
			printerror("calculated video rate %d not 600 - 7500 for size %d KB", vrate, calcsize);
			printerror("suggested minimum size setting of %d KB",
				((631 + arate + arate / 20) / 8 * invid.length) / 1000 * 1000);
			printerror("suggested maximum size setting of %d KB",
				((7875 + arate + arate / 20) / 8 * invid.length) / 1000 * 1000);
			exit(1);
		}
	}

	/* adjust minimum automatic bitrate */
	if(!vrate && invid.vrate < 600)
		vrate = 600; /* TODO: set rate based on input filesize if rate == 0 */

	/* set video rate */
	if(vrate)
	{
		outvid.vrate = vrate;
	}
	else
	{
		outvid.vrate = invid.vrate;
		vrate = outvid.vrate;
	}

	/* set aspect ratio */
	if(aspect == 2)
	{
		outvid.aspect = 1.333333;
	}
	else if(aspect == 3)
	{
		outvid.aspect = 1.777777;
	}
	else
	{
		if(invid.aspect < 1.555555)
			outvid.aspect = 1.333333;
		else
			outvid.aspect = 1.777777;
	}

	/* set doublefps if needed */
	if(invid.fps <= 15)
		options |= SPANK_DOUBLEFPS;

	/* set pal/ntsc */
	if(vidmode == 'a')
	{
		if(invid.fps == 25)
			vidmode = 'p';
		else
			vidmode = 'n';
	}
	outvid.width = 720;
	if(vidmode == 'p')
	{
		outvid.height = 576;
		strcpy(outvid.p_vcodec, "PAL");
	}
	else
	{
		outvid.height = 480;
		strcpy(outvid.p_vcodec, "NTSC");
	}

	/* set length */
	outvid.length  = invid.length;
	outvid.hours   = invid.hours;
	outvid.minutes = invid.minutes;
	outvid.seconds = invid.seconds;
	outvid.frames  = invid.frames;

	/* set fps */
	if(vidmode == 'p' && invid.fps == 25)
	{
		outvid.fps = 25;
	}
	else if(vidmode =='n' && (invid.fps == 23.976 || invid.fps == 29.97))
	{
		outvid.fps = invid.fps;
	}
	else if(vidmode == 'p')
	{
		options |= SPANK_NEWFPS;
		outvid.fps = 25;
	}
	else
	{
		options |= SPANK_NEWFPS;
		outvid.fps = 23.976;
	}

	/* set border/crop */
	if(calcborder)
	{
		hcrop = 0;
		if(outvid.aspect < 1.5) /* 4:3 */
		{
			vcrop = -(invid.width * 10 / 4 * 3 - invid.height * 10) / 40 * 2;
			if(vcrop > 0)
			{
				hcrop = (invid.width * 10 - invid.height * 10 / 3 * 4) / 40 * 2;
				vcrop = 0;
			}
		}
		else /* 16:9 */
		{
			vcrop = -(invid.width * 10 / 16 * 9 - invid.height * 10) / 40 * 2;
			if(vcrop > 0)
			{
				hcrop = (invid.width * 10 - invid.height * 10 / 9 * 16) / 40 * 2;
				vcrop = 0;
			}
		}
	}
	if(calcclip)
	{
		hcrop = 0;
		if(outvid.aspect < 1.5) /* 4:3 */
		{
			vcrop = -(invid.width * 10 / 4 * 3 - invid.height * 10) / 40 * 2;
			if(vcrop < 0)
			{
				hcrop = (invid.width * 10 - invid.height * 10 / 3 * 4) / 40 * 2;
				vcrop = 0;
			}
		}
		else /* 16:9 */
		{
			vcrop = -(invid.width * 10 / 16 * 9 - invid.height * 10) / 40 * 2;
			if(vcrop < 0)
			{
				hcrop = (invid.width * 10 - invid.height * 10 / 9 * 16) / 40 * 2;
				vcrop = 0;
			}
		}
	}
	outvid.vcrop = vcrop;
	outvid.hcrop = hcrop;

	outvid.bytes = (vrate + vrate / 20 + arate + arate / 20) / 8 * invid.length * 1000;

	/* print source and dest video encoding info */
	print(ML_INFO, " source | %s\n", invid.p_filename);
	print(ML_INFO, "  video | %s, %dx%d, %d kbps, %.3f fps\n",
		invid.p_vcodec, invid.width, invid.height, invid.vrate, invid.fps);
	print(ML_INFO, "        | %.4f aspect, %d frames, %d:%02d:%02d\n",
		invid.aspect, invid.frames,	invid.hours, invid.minutes, invid.seconds);
	print(ML_INFO, "  audio | %s, %d kbps, %d Hz, %d bit, %d channel\n",
		invid.p_acodec, invid.arate, invid.hz, invid.abits, invid.achans);
	print(ML_INFO, "   file | %d KB\n", invid.bytes / 1000);
	if(options & SPANK_MPLAYER)
		print(ML_INFO, "   info | using mplayer decoder\n");
	print(ML_INFO, "\n");

	print(ML_INFO, "   dest | %s.mpg\n", outvid.p_filename);
	print(ML_INFO, "  video | %s, %dx%d, %d kbps, %.3f fps\n",
		outvid.p_vcodec, outvid.width, outvid.height, outvid.vrate, outvid.fps);
	print(ML_INFO, "        | %.4f aspect, %d frames, %d:%02d:%02d\n",
		outvid.aspect, outvid.frames, outvid.hours, outvid.minutes, outvid.seconds);
	print(ML_INFO, "        | %d(v):%d(h) crop, %s\n",
		outvid.vcrop, outvid.hcrop, (options & SPANK_MULTI) ? "VBR" : "CBR");
	print(ML_INFO, "  audio | %s, %d kbps, %d Hz, %d bit, %d channel\n",
		outvid.p_acodec, outvid.arate, outvid.hz, outvid.abits, outvid.achans);
	print(ML_INFO, "   file | ~%d KB\n", outvid.bytes / 1000);
	print(ML_INFO, "\n");

	exit(encode(&invid, &outvid, options, p_filters));
}
