/*
 * forkit - a piped console feedback forker
 * Copyright (C) 2007  Jeffrey Grembecki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <fcntl.h>
#include <string.h>
#include "forkit.h"
#include "vstr.h"

/* this was a learning experience with unplaned
 * expansion to pipes and a deadline to meet hence the mess
 */

int forkitf(forkit_cb callback, forkitprint_cb print, void *p_options, const char *p_path,
	const char *p_format, ...)
{
	char *p_cmdstr;
	char **pp_args;

	/* prepare command */
	{
		va_list list;
		int cmdlen, pathlen;
		vstr *p_out = NULL;
		int i, start, argc, isarg, pos;

		/* calculate sizes */
		pathlen = strlen(p_path);
		va_start(list, p_format);
		cmdlen = vsnprintf(NULL, 0, p_format, list);
		va_end(list);

		/* create command string */
		if(!(p_cmdstr = malloc(cmdlen + 1)))
			return 1;
		va_start(list, p_format);
		vsprintf(p_cmdstr, p_format, list);
		va_end(list);

		/* calc # args and allocate */
		argc = 0;
		isarg = 0;
		for(i = 0; i < cmdlen; i++)
		{
			if(p_cmdstr[i] == '\n')
			{
				argc++;
				isarg = 0;
			}
			else
			{
				isarg = 1;
			}
		}
		if(isarg)
			argc++;
		if(!(pp_args = malloc(sizeof(char**) * argc + 1)))
		{
			free(p_cmdstr);
			return 1;
		}

		/* copy args */
		if(print)
		{
			p_out = vs_new();
			vs_print(p_out, "forkit: [%s]", p_path);
		}
		start = 0;
		pos = 0;
		isarg = 0;
		for(i = 0; i < cmdlen; i++)
		{
			if(p_cmdstr[i] == '\n')
			{
				pp_args[pos++] = &p_cmdstr[start];
				p_cmdstr[i] = 0;
				if(p_out)
					vs_cat(p_out, " [%s]", &p_cmdstr[start]);
				start = i + 1;
				isarg = 0;
			}
			else
			{
				isarg = 1;
			}
		}
		if(isarg)
		{
			pp_args[pos++] = &p_cmdstr[start];
			p_cmdstr[i] = 0;
			if(p_out)
				vs_cat(p_out, " [%s]", &p_cmdstr[start]);
		}
		pp_args[pos] = NULL;

		if(p_out)
		{
			print(p_out->s);
			vs_free(p_out);
		}
	}

	/* forkit */
	{
		char p_buffer[1024];
		int p_pipe[2], pid;
		int status = 0;
		int retval = 0;
		int count;
		int pos, start, copy;

		/* open comminication pipe */
		if(pipe(p_pipe) == -1)
		{
			free(p_cmdstr);
			free(pp_args);
			return 1;
		}

		/* fork */
		pid = fork();
		switch(pid)
		{
			case -1:
				/* error */
				close(p_pipe[0]);
				close(p_pipe[1]);
				retval = -2;
				break;

			case 0:
				/* child, run program */
				close(p_pipe[0]);
				dup2(p_pipe[1], fileno(stdout));
				dup2(p_pipe[1], fileno(stderr));
				execvp(p_path, pp_args);
				free(p_cmdstr);
				free(pp_args);
				exit(1);

			default:
				/* parent, send output to callback */
				close(p_pipe[1]);

				/* reset the callback */
				if(callback)
					callback(NULL, p_options);

				/* read input while waiting for child to term */
				start = 0;

				while(waitpid(pid, &status, WNOHANG) != pid)
				{
					count = read(p_pipe[0], &(p_buffer[start]), sizeof(p_buffer) - 1 - start);
					if(count > 0)
					{
						p_buffer[start + count] = 0;

						/* seperate lines and send each to callback */
						p_buffer[start + count] = 0;
						for(start = 0, pos = 0; pos < sizeof(p_buffer) - 1; pos++)
						{
							if(p_buffer[pos] == '\n' || p_buffer[pos] == '\r')
							{
								/* send line to callback */
								p_buffer[pos] = 0;
								if(callback && callback(&(p_buffer[start]), p_options))
									retval = -3;
								start = pos + 1;
							}
							else if(p_buffer[pos] == 0)
							{
								/* copy unread buffer to beginning */
								for(copy = 0; copy < pos - start; copy++)
									p_buffer[copy] = p_buffer[start + copy];

								/* set position ready for continued reading */
								start = pos - start;

								break;
							}
						}
					}
				}
				if(start && callback && callback(&(p_buffer[start]), p_options))
					retval = -3;
				/* return value same as exit status or -1 if terminated by signal */
				if(!retval)
				{
					if(WIFEXITED(status))
						retval = WEXITSTATUS(status);
					else if(WIFSIGNALED(status))
						retval = -1;
				}

				close(p_pipe[0]);
				break;
		}

		free(p_cmdstr);
		free(pp_args);
		return retval;
	}
}

int pipeit(int fd, forkit_cb callback, forkitprint_cb print, void *p_options, const char *p_path,
	const char *p_format, ...)
{
	char *p_cmdstr;
	char **pp_args;

	/* prepare command */
	{
		va_list list;
		int cmdlen, pathlen;
		vstr *p_out = NULL;
		int i, start, argc, isarg, pos;

		/* calculate sizes */
		pathlen = strlen(p_path);
		va_start(list, p_format);
		cmdlen = vsnprintf(NULL, 0, p_format, list);
		va_end(list);

		/* create command string */
		if(!(p_cmdstr = malloc(cmdlen + 1)))
			return 1;
		va_start(list, p_format);
		vsprintf(p_cmdstr, p_format, list);
		va_end(list);

		/* calc # args and allocate */
		argc = 0;
		isarg = 0;
		for(i = 0; i < cmdlen; i++)
		{
			if(p_cmdstr[i] == '\n')
			{
				argc++;
				isarg = 0;
			}
			else
			{
				isarg = 1;
			}
		}
		if(isarg)
			argc++;
		if(!(pp_args = malloc(sizeof(char**) * argc + 1)))
		{
			free(p_cmdstr);
			return 1;
		}

		/* copy args */
		if(print)
		{
			p_out = vs_new();
			vs_print(p_out, "pipeit: [%s]", p_path);
		}
		start = 0;
		pos = 0;
		isarg = 0;
		for(i = 0; i < cmdlen; i++)
		{
			if(p_cmdstr[i] == '\n')
			{
				pp_args[pos++] = &p_cmdstr[start];
				p_cmdstr[i] = 0;
				if(p_out)
					vs_cat(p_out, " [%s]", &p_cmdstr[start]);
				start = i + 1;
				isarg = 0;
			}
			else
			{
				isarg = 1;
			}
		}
		if(isarg)
		{
			pp_args[pos++] = &p_cmdstr[start];
			p_cmdstr[i] = 0;
			if(p_out)
				vs_cat(p_out, " [%s]", &p_cmdstr[start]);
		}
		pp_args[pos] = NULL;

		if(p_out)
		{
			print(p_out->s);
			vs_free(p_out);
		}
	}

	/* forkit */
	{
		char p_buffer[1024];
		int p_pipe[2], pid;
		int status = 0;
		int retval = 0;
		int count;
		int pos, start, copy;

		/* open comminication pipe */
		if(pipe(p_pipe) == -1)
		{
			free(p_cmdstr);
			free(pp_args);
			return 1;
		}

		/* fork */
		pid = fork();
		switch(pid)
		{
			case -1:
				/* error */
				close(p_pipe[0]);
				close(p_pipe[1]);
				retval = -2;
				break;

			case 0:
				/* child, run program */
				close(p_pipe[0]);
				dup2(fd, fileno(stdout));
				dup2(p_pipe[1], fileno(stderr));
				execvp(p_path, pp_args);
				free(p_cmdstr);
				free(pp_args);
				exit(1);

			default:
				/* parent, send output to callback */
				close(p_pipe[1]);

				/* reset the callback */
				if(callback)
					callback(NULL, p_options);

				/* read input while waiting for child to term */
				start = 0;

				while(waitpid(pid, &status, WNOHANG) != pid)
				{
					count = read(p_pipe[0], &(p_buffer[start]), sizeof(p_buffer) - 1 - start);
					if(count > 0)
					{
						p_buffer[start + count] = 0;

						/* seperate lines and send each to callback */
						p_buffer[start + count] = 0;
						for(start = 0, pos = 0; pos < sizeof(p_buffer) - 1; pos++)
						{
							if(p_buffer[pos] == '\n' || p_buffer[pos] == '\r')
							{
								/* send line to callback */
								p_buffer[pos] = 0;
								if(callback && callback(&(p_buffer[start]), p_options))
									retval = -3;
								start = pos + 1;
							}
							else if(p_buffer[pos] == 0)
							{
								/* copy unread buffer to beginning */
								for(copy = 0; copy < pos - start; copy++)
									p_buffer[copy] = p_buffer[start + copy];

								/* set position ready for continued reading */
								start = pos - start;

								break;
							}
						}
					}
				}
				if(start && callback && callback(&(p_buffer[start]), p_options))
					retval = -3;
				/* return value same as exit status or -1 if terminated by signal */
				if(!retval)
				{
					if(WIFEXITED(status))
						retval = WEXITSTATUS(status);
					else if(WIFSIGNALED(status))
						retval = -1;
				}

				close(p_pipe[0]);
				break;
		}

		free(p_cmdstr);
		free(pp_args);
		return retval;
	}
}

int pipefile(const char *p_filename, forkit_cb callback, forkitprint_cb print, void *p_options,
	const char *p_path, const char *p_format, ...)
{
	char *p_cmdstr;
	char **pp_args;
	int fd;

	/* prepare command */
	{
		va_list list;
		int cmdlen, pathlen;
		vstr *p_out = NULL;
		int i, start, argc, isarg, pos;

		/* calculate sizes */
		pathlen = strlen(p_path);
		va_start(list, p_format);
		cmdlen = vsnprintf(NULL, 0, p_format, list);
		va_end(list);

		/* create command string */
		if(!(p_cmdstr = malloc(cmdlen + 1)))
			return 1;
		va_start(list, p_format);
		vsprintf(p_cmdstr, p_format, list);
		va_end(list);

		/* calc # args and allocate */
		argc = 0;
		isarg = 0;
		for(i = 0; i < cmdlen; i++)
		{
			if(p_cmdstr[i] == '\n')
			{
				argc++;
				isarg = 0;
			}
			else
			{
				isarg = 1;
			}
		}
		if(isarg)
			argc++;
		if(!(pp_args = malloc(sizeof(char**) * argc + 1)))
		{
			free(p_cmdstr);
			return 1;
		}

		/* copy args */
		if(print)
		{
			p_out = vs_new();
			vs_print(p_out, "pipefile: (%s) [%s]", p_filename, p_path);
		}
		start = 0;
		pos = 0;
		isarg = 0;
		for(i = 0; i < cmdlen; i++)
		{
			if(p_cmdstr[i] == '\n')
			{
				pp_args[pos++] = &p_cmdstr[start];
				p_cmdstr[i] = 0;
				if(p_out)
					vs_cat(p_out, " [%s]", &p_cmdstr[start]);
				start = i + 1;
				isarg = 0;
			}
			else
			{
				isarg = 1;
			}
		}
		if(isarg)
		{
			pp_args[pos++] = &p_cmdstr[start];
			p_cmdstr[i] = 0;
			if(p_out)
				vs_cat(p_out, " [%s]", &p_cmdstr[start]);
		}
		pp_args[pos] = NULL;

		if(p_out)
		{
			if(print)
				print(p_out->s);
			vs_free(p_out);
		}
	}

	/* open file */
	if((fd = open(p_filename, O_RDONLY)) == -1)
	{
		if(print)
			print("pipefile: unable to open file");
		return -5;
	}

	/* forkit */
	{
		char p_buffer[1024];
		int p_pipe[2], p_revpipe[2], pid;
		int status = 0;
		int retval = 0;
		int count;

		/* open comminication pipe */
		if(pipe(p_pipe) == -1 || pipe(p_revpipe) == -1)
		{
			free(p_cmdstr);
			free(pp_args);
			return 1;
		}

		/* fork */
		pid = fork();
		switch(pid)
		{
			case -1:
				/* error */
				close(p_pipe[0]);
				close(p_pipe[1]);
				close(p_revpipe[0]);
				close(p_revpipe[1]);
				retval = -2;
				break;

			case 0:
				/* child, run program */
				close(p_pipe[0]);
				close(p_revpipe[1]);
				dup2(p_revpipe[0], fileno(stdin));
				dup2(p_pipe[1], fileno(stdout));
				dup2(p_pipe[1], fileno(stderr));
				execvp(p_path, pp_args);
				free(p_cmdstr);
				free(pp_args);
				exit(1);

			default:
				/* parent, send output to callback */
				close(p_pipe[1]);
				close(p_revpipe[0]);
				fcntl(p_pipe[0], F_SETFL, O_NONBLOCK);

				/* reset the callback */
				if(callback)
					callback(NULL, p_options);

				while((count = read(fd, p_buffer, sizeof(p_buffer) - 1)) > 0)
					write(p_revpipe[1], p_buffer, count);

				close(p_revpipe[1]);
				close(fd);

				while(waitpid(pid, &status, WNOHANG) != pid)
				{
					count = read(p_pipe[0], p_buffer, sizeof(p_buffer));
					if(count > 0)
					{
						p_buffer[count] = 0;
						if(callback && callback(p_buffer, p_options))
							retval = -3;
					}
					else
					{
						usleep(500);
					}
				}

				/* return value same as exit status or -1 if terminated by signal */
				if(!retval)
				{
					if(WIFEXITED(status))
						retval = WEXITSTATUS(status);
					else if(WIFSIGNALED(status))
						retval = -1;
				}

				close(p_pipe[0]);
				break;
		}

		free(p_cmdstr);
		free(pp_args);
		return retval;
	}
}

int omfgwriteapipelib(const char *p_source, const char *p_dest, forkit_cb callback,
	forkitprint_cb print, void *p_options, const char *p_path, const char *p_format, ...)
{
	char *p_cmdstr;
	char **pp_args;
	FILE *p_filein, *p_fileout;

	/* prepare command */
	{
		va_list list;
		int cmdlen, pathlen;
		vstr *p_out = NULL;
		int i, start, argc, isarg, pos;

		/* calculate sizes */
		pathlen = strlen(p_path);
		va_start(list, p_format);
		cmdlen = vsnprintf(NULL, 0, p_format, list);
		va_end(list);

		/* create command string */
		if(!(p_cmdstr = malloc(cmdlen + 1)))
			return 1;
		va_start(list, p_format);
		vsprintf(p_cmdstr, p_format, list);
		va_end(list);

		/* calc # args and allocate */
		argc = 0;
		isarg = 0;
		for(i = 0; i < cmdlen; i++)
		{
			if(p_cmdstr[i] == '\n')
			{
				argc++;
				isarg = 0;
			}
			else
			{
				isarg = 1;
			}
		}
		if(isarg)
			argc++;
		if(!(pp_args = malloc(sizeof(char**) * argc + 1)))
		{
			free(p_cmdstr);
			return 1;
		}

		/* copy args */
		if(print)
		{
			p_out = vs_new();
			vs_print(p_out, "pipefile: < (%s) > (%s) [%s]", p_source, p_dest, p_path);
		}
		start = 0;
		pos = 0;
		isarg = 0;
		for(i = 0; i < cmdlen; i++)
		{
			if(p_cmdstr[i] == '\n')
			{
				pp_args[pos++] = &p_cmdstr[start];
				p_cmdstr[i] = 0;
				if(p_out)
					vs_cat(p_out, " [%s]", &p_cmdstr[start]);
				start = i + 1;
				isarg = 0;
			}
			else
			{
				isarg = 1;
			}
		}
		if(isarg)
		{
			pp_args[pos++] = &p_cmdstr[start];
			p_cmdstr[i] = 0;
			if(p_out)
				vs_cat(p_out, " [%s]", &p_cmdstr[start]);
		}
		pp_args[pos] = NULL;

		if(p_out)
		{
			if(print)
				print(p_out->s);
			vs_free(p_out);
		}
	}

	/* open files */
	if(!(p_filein = fopen(p_source, "rb")))
	{
		if(print)
			print("pipefile: unable to open file");
		return -5;
	}
	if(!(p_fileout = fopen(p_dest, "wb")))
	{
		if(print)
			print("pipefile: unable to open file");
		return -5;
	}

	/* forkit */
	{
		char p_buffer[1024];
		int p_pipe[2], p_revpipe[2], pid;
		int status = 0;
		int retval = 0;
		int count;

		/* open comminication pipe */
		if(pipe(p_pipe) == -1 || pipe(p_revpipe) == -1)
		{
			free(p_cmdstr);
			free(pp_args);
			return 1;
		}

		/* fork */
		pid = fork();
		switch(pid)
		{
			case -1:
				/* error */
				close(p_pipe[0]);
				close(p_pipe[1]);
				close(p_revpipe[0]);
				close(p_revpipe[1]);
				retval = -2;
				break;

			case 0:
				/* child, run program */
				close(p_pipe[0]);
				close(p_revpipe[1]);
				dup2(p_revpipe[0], fileno(stdin));
				dup2(fileno(p_fileout), fileno(stdout));
				dup2(p_pipe[1], fileno(stderr));
				execvp(p_path, pp_args);
				free(p_cmdstr);
				free(pp_args);
				exit(1);

			default:
				/* parent, send output to callback */
				close(p_pipe[1]);
				close(p_revpipe[0]);
				fcntl(p_pipe[0], F_SETFL, O_NONBLOCK);

				/* reset the callback */
				if(callback)
					callback(NULL, p_options);

				while((count = read(fileno(p_filein), p_buffer, sizeof(p_buffer) - 1)) > 0)
					write(p_revpipe[1], p_buffer, count);

				close(p_revpipe[1]);
				fclose(p_filein);

				while(waitpid(pid, &status, WNOHANG) != pid)
				{
					count = read(p_pipe[0], p_buffer, sizeof(p_buffer));
					if(count > 0)
					{
						p_buffer[count] = 0;
						if(callback && callback(p_buffer, p_options))
							retval = -3;
					}
					else
					{
						usleep(500);
					}
				}

				/* return value same as exit status or -1 if terminated by signal */
				if(!retval)
				{
					if(WIFEXITED(status))
						retval = WEXITSTATUS(status);
					else if(WIFSIGNALED(status))
						retval = -1;
				}

				close(p_pipe[0]);
				break;
		}

		fclose(p_fileout);
		free(p_cmdstr);
		free(pp_args);
		return retval;
	}
}


