/*
 * dvdspanky - a video to DVD MPEG conversion front-end for transcode
 * Copyright (C) 2007  Jeffrey Grembecki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <string.h>
#include <pcre.h>
#include <sys/stat.h>
#include <stdarg.h>
#include <dirent.h>
#include <fnmatch.h>
#include "dvdspanky.h"
#include "vstr.h"
#include "forkit.h"
#include "../config.h"

static char gp_transcodebin[] = "transcode";
static char gp_ffmpegbin[] = "ffmpeg";
static char gp_mplayerbin[] = "mplayer";
static char gp_mplexbin[] = "mplex";
static char gp_tcprobebin[] = "tcprobe";
static char gp_fehbin[] = "feh";
static char gp_soxbin[] = "sox";

static char gp_licence[] =
"This program is free software; you can redistribute it and/or modify\n"
"it under the terms of the GNU General Public License as published by\n"
"the Free Software Foundation; either version 2 of the License, or\n"
"(at your option) any later version.\n"
"\n"
"This program is distributed in the hope that it will be useful,\n"
"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
"GNU General Public License for more details.\n"
"\n"
"You should have received a copy of the GNU General Public License\n"
"along with this program; if not, write to the Free Software\n"
"Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,\n"
"MA 02110-1301, USA.\n";

static char gp_help[] =
"%s -i file [OPTIONS]\n"
"\n"
"following; [ ] = default, { } = requirements, * = experimental\n"
"\n"
"-0, --avpass           auto volume pass only [off]\n"
"-1, --firstpass        1st pass only [off]\n"
"-2, --secondpass       2nd pass only including multiplex [off]\n"
"-a, --aspect=MODE      aspect ratio 2=4:3, 3=16:9 [auto] {2,3}\n"
"-A, --autovolume       automatically adjust volume [off]\n"
"-b, --border           automatic black border [off]\n"
"-B, --clip             clips edges to fit aspect [off]\n"
"-c, --channels=CHANS   number of audio channels [auto] {1 - 5}\n"
"-C, --inaspect=ASPECT  force the input aspect\n"
"-d, --dvd=TITLE        title to read from the source dvd\n"
"-D, --deinterlace      deinterlace video [off]\n"
"-f, --filters=FILTERS  specify transcode filters [none]\n"
"-d, --dvd=TITLE        take input from dvd\n"
"-g, --gain=GAIN        multiply volume by GAIN [off/1.0]\n"
"-G, --gamma=GAMMA      set gamma, < 1.0 is brighter, > is darker [1.0]\n"
"-h, --help             this help text\n"
"-H, --hcrop=PIXELS     as above [off]\n"
"-i, --input=FILE       input file\n"
"-I, --nonice           do not run at nice priority 19 [off]\n"
"-L, --licence          show the program licence\n"
"-m, --multi            2 pass mode [off]\n"
"-M, --mplex            multiplex only [off]\n"
"-n, --ntsc             set ntsc [auto]\n"
"-N, --ntscfix          fix variable framerate ntsc video [off] *\n"
"-o, --output=FILE      output filename prefix (w/o extension)\n"
"-O, --normalise        dynamically normalise audio volume [off]\n"
"-r, --vrate=RATE       video bitrate [auto] {600 - 7500}\n"
"-p, --pal              set pal [auto]\n"
"-P, --preview          previews 5 frames using feh then exits [off]\n"
"-R, --arate=RATE       audio bitrate [auto] {64 - 448, 192 is good}\n"
"-s, --size=KB          final mpg size in kilobytes\n"
"-S, --speed            change movie speed and pitch to match [off]\n"
"-T, --postprocess      enable post processing filter [off]\n"
"-U, --noremove         disable removal of temporary files [off]\n"
"-v, --verbose          be verbose [off]\n"
"-V, --vcrop=PIXELS     pos for crop, neg for black border [off]\n"
"\nsee man 1 dvdspanky for more information\n";

static struct option gp_longopts[] =
{
	{"avpass", 0, 0, '0'},
	{"firstpass", 0, 0, '1'},
	{"secondpass", 0, 0, '2'},
	{"aspect", 1, 0, 'a'},
	{"autovolume", 0, 0, 'A'},
	{"border", 0, 0, 'b'},
	{"clip", 0, 0, 'B'},
	{"channels", 1, 0, 'c'},
	{"inaspect", 1, 0, 'c'},
	{"dvd", 1, 0, 'd'},
	{"deinterlace", 0, 0, 'D'},
	{"filters", 1, 0, 'f'},
	{"frameadjust", 0, 0, 'F'},
	{"gain", 1, 0, 'g'},
	{"gamma", 1, 0, 'G'},
	{"hcrop", 1, 0, 'H'},
	{"help", 0, 0, 'h'},
	{"input", 1, 0, 'i'},
	{"nonice", 0, 0, 'I'},
	{"licence", 0, 0, 'L'},
	{"mplex", 0, 0, 'M'},
	{"multi", 0, 0, 'm'},
	{"multipass", 0, 0, 'm'},
	{"ntsc", 0, 0, 'n' },
	{"ntscfix", 0, 0, 'N'},
	{"output", 1, 0, 'o'},
	{"normalise", 0, 0, 'O'},
	{"normalize", 0, 0, 'O'},
	{"pal", 0, 0, 'p' },
	{"preview", 0, 0, 'P'},
	{"vrate", 1, 0, 'r'},
	{"arate", 1, 0, 'R'},
	{"size", 1, 0, 's'},
	{"speed", 0, 0, 'S'},
	{"postprocess", 0, 0, 'T'},
	{"noremove", 0, 0, 'U'},
	{"verbose", 0, 0, 'v'},
	{"vcrop", 1, 0, 'V'},
};

static char gp_shortopts[] = "012a:AbBc:C:d:Df:Fg:G:hH:i:ILmMnNo:OpPr:R:s:STUvV:";

static pcre *gp_idregex = NULL;
static pcre *gp_idregex2 = NULL;
static pcre *gp_transregex = NULL;
static pcre *gp_tcregex = NULL;
static pcre *gp_tcregex2 = NULL;
static pcre *gp_tcregex3 = NULL;
static pcre *gp_tcregex4 = NULL;
static pcre *gp_warnerrregex = NULL;
static pcre *gp_soxregex = NULL;
static pcre *gp_ffmpegregex = NULL;

#define ML_ERROR 4
#define ML_WARNING 3
#define ML_INFO 2
#define ML_VERBOSE 1
#define ML_DEBUG 0

static int g_msglevel = ML_INFO;

static FILE *gp_logfile = NULL;
static char gp_printstr[2048];

static void print(int msglevel, const char *p_prefix, const char *p_format, ...)
{
	va_list argp;

	va_start(argp, p_format);
	vsnprintf(gp_printstr, sizeof(gp_printstr), p_format, argp);
	gp_printstr[sizeof(gp_printstr) - 1] = 0;
	va_end(argp);

	if(msglevel >= g_msglevel)
		fprintf(stderr, "%s\n", gp_printstr);

	if(gp_logfile)
	{
		if(p_prefix)
			fprintf(gp_logfile, "%10s | %s\n", p_prefix, gp_printstr);
		else
			fprintf(gp_logfile, " dvdspanky | %s\n", gp_printstr);
		fflush(gp_logfile);
	}
}

static void printerror(const char *p_format, ...)
{
	va_list argp;

	va_start(argp, p_format);
	vsnprintf(gp_printstr, sizeof(gp_printstr), p_format, argp);
	gp_printstr[sizeof(gp_printstr) - 1] = 0;
	va_end(argp);

	if(ML_ERROR >= g_msglevel)
		fprintf(stderr, "error: %s\n", gp_printstr);

	if(gp_logfile)
	{
		fprintf(gp_logfile, " dvdspanky | error: %s\n", gp_printstr);
		fflush(gp_logfile);
	}
}

void cb_print(const char *p_string)
{
	if(ML_VERBOSE >= g_msglevel)
		fprintf(stderr, "%s\n", p_string);

	if(gp_logfile)
	{
		fprintf(gp_logfile, " dvdspanky | %s\n", p_string);
		fflush(gp_logfile);
	}
}

/* termination cleanup */
static void cleanup()
{
#define freepcre(_x) { if(_x) free(_x); _x=NULL; }
	freepcre(gp_idregex);
	freepcre(gp_idregex2);
	freepcre(gp_transregex);
	freepcre(gp_tcregex);
	freepcre(gp_tcregex2);
	freepcre(gp_tcregex3);
	freepcre(gp_tcregex4);
	freepcre(gp_warnerrregex);
	freepcre(gp_soxregex);
	freepcre(gp_ffmpegregex);
#undef freepcre

	if(gp_logfile) { fclose(gp_logfile); gp_logfile = NULL; }
}

int oneshotregex(const char *p_pattern, const char *p_string, int *p_vector, int nvect)
{
	pcre *p_regex;
	const char *p_err;
	int erroff;
	int count;

	if(!(p_regex = pcre_compile(p_pattern, 0, &p_err, &erroff, NULL)))
	{
		printerror("regex compile \"%s\" failed: %s @ %d", p_pattern, p_err, erroff);
		return -2;
	}
	count = pcre_exec(p_regex, NULL, p_string, strlen(p_string), 0, 0, p_vector, nvect);
	free(p_regex);

	return count;
}

double parseaspect(const char *p_str)
{
	int p_vector[10];
	int nvect = 10;

	if(oneshotregex("^(\\d+)(?::|/)(\\d+)(?:\\D.*|)$", p_str, p_vector, nvect) > 0)
		return atof(&p_str[p_vector[2]]) / atof(&p_str[p_vector[4]]);

	if(oneshotregex("^(\\d+\\.\\d+)(?:\\D.*|)$", p_str, p_vector, nvect) > 0)
		return atof(&p_str[p_vector[2]]) / atof(&p_str[p_vector[4]]);

	if(oneshotregex("^(\\d)(?:\\D.*|[\\)].*|)$", p_str, p_vector, nvect) > 0)
	{
		switch(p_str[p_vector[2]])
		{
			case '2':
				return 1.33333333;
			case '3':
				return 1.77777777;
		}
	}

	return 0;
}

/* encode / preview */
int encode(video_t *p_in, video_t *p_out, uint32_t options, const char *p_filters)
{
	vstr *p_generic, *p_video, *p_videoextra, *p_audio, *p_speedout;
	vstr *p_final, *p_outcodec, *p_preview, *p_fpschange;

	p_generic = vs_new();
	p_video = vs_new();
	p_videoextra = vs_new();
	p_audio = vs_new();
	p_final = vs_new();
	p_outcodec = vs_new();
	p_preview = vs_new();
	p_fpschange = vs_new();
	p_speedout = vs_new();

	if(!p_generic || !p_video || !p_videoextra || !p_audio || !p_final || 
		!p_outcodec || !p_preview || !p_speedout || !p_fpschange)
	{
		if(p_generic) vs_free(p_generic);
		if(p_video) vs_free(p_video);
		if(p_videoextra) vs_free(p_videoextra);
		if(p_audio) vs_free(p_audio);
		if(p_final) vs_free(p_final);
		if(p_outcodec) vs_free(p_outcodec);
		if(p_preview) vs_free(p_preview);
		printerror("unable to allocate strings");
		return 1;
	}

#define freestrings() { vs_free(p_generic); vs_free(p_video); vs_free(p_videoextra); \
	vs_free(p_audio); vs_free(p_final); vs_free(p_preview);}

	/* generic */
	vs_cat(p_generic, "transcode\n-i\n%s", p_in->p_filename);
	if(options & SPANK_MPLAYER)
	{
		unlink("stream.yuv");
		vs_cat(p_generic, "\n-H\n0\n-x\nmplayer,mplayer\n-g\n%dx%d\n-n\n0x1\n-e\n%d,%d,%d", p_in->width,
			p_in->height, p_in->hz,	(p_in->abits) ? p_in->abits : 16, p_in->achans);
	}
	if(options & SPANK_DVD)
		vs_cat(p_generic, "\n-T\n%d,-1", p_in->title);

	/* audio */
	vs_cat(p_audio, "\n-E\n48000,%d,%d\n-b\n%d",
		(p_out->abits) ? p_out->abits : 16, p_out->achans, p_out->arate);

	/* video */
	{
		int asr = (p_out->aspect < 1.5) ? 2 : 3;
		vs_cat(p_video, "\n-w\n%d\n-f\n%.3f\n--import_asr\n%d\n--export_asr\n%d\n--encode_fields\n%c",	
			p_out->vrate, p_in->fps, asr, asr, (p_out->p_vcodec[0] == 'P') ? 't' : 'b');
	}
	
	/* video speed change */
	vs_cat(p_fpschange, "\n--export_fps\n%.3f", p_out->fps);

	/* video extra & some preview options */
	if(options & SPANK_DOUBLEFPS)
		vs_cat(p_videoextra, "\n-J\ndoublefps");
	if(options & SPANK_HARDFPS)
		vs_cat(p_videoextra, "\n-J\nmodfps=clonetype=1");
	if(options & SPANK_FIXNTSC)
		vs_cat(p_videoextra, "\n-M\n2\n-J\nivtc,32detect=force_mode=5,decimate\n--hard_fps");
	if(p_out->vcrop || p_out->hcrop)
		vs_cat(p_videoextra, "\n-j\n%d,%d,%d,%d",
			p_out->vcrop, p_out->hcrop, p_out->vcrop, p_out->hcrop);
	if(p_out->gamma != 1.0)
		vs_cat(p_videoextra, "\n-G\n%f", p_out->gamma);
	if(options & SPANK_PPROCESS)
	{
		if(options & SPANK_DEINTER)
		{
			vs_cat(p_videoextra, "\n-J\npp=pre:ci:ha:va:dr:tn:fq");
			vs_cat(p_preview, "\n-J\npp=pre:ci:ha:va:dr:tn:fq");
		}
		else
		{
			vs_cat(p_videoextra, "\n-J\npp=pre:ha:va:dr:tn:fq");
			vs_cat(p_preview, "\n-J\npp=pre:ha:va:dr:tn:fq");
		}
	}
	else if(options & SPANK_DEINTER)
	{
		vs_cat(p_videoextra, "\n-J\npp=pre:ci");
		vs_cat(p_preview, "\n-J\npp=pre:ci");
	}
	if(options & SPANK_NORMALISE)
		vs_cat(p_videoextra, "\n-J\nnormalize=algo=2\n-s\n0.5");
	if(p_filters[0])
		vs_cat(p_videoextra, "\n-J\n%s", p_filters);

	/* preview options */
	if(p_out->vcrop || p_out->hcrop)
		vs_cat(p_preview, "\n-j\n%d,%d,%d,%d",
			p_out->vcrop, p_out->hcrop, p_out->vcrop, p_out->hcrop);
	if(p_out->gamma != 1.0)
		vs_cat(p_preview, "\n-G\n%f", p_out->gamma);

	/* export codecs */
	if(options & SPANK_SPEED)
	{
		vs_cat(p_outcodec, "\n-y\nffmpeg,null\n-F\nmpeg2video\n", p_out->p_filename);
		vs_cat(p_speedout, "\n-y\nnull,wav\n-m\n%s_full.wav", p_out->p_filename);
	}
	else
		vs_cat(p_outcodec, "\n-N\n0x2000\n-y\nffmpeg\n-F\nmpeg2video\n-m\n%s.ac3", p_out->p_filename);

	/* create ffmpeg.cfg */
	system("echo \"[mpeg2video]\nvrc_minrate=0\nvrc_maxrate=7500\nvrc_buf_size=1792\n\""
		" > ffmpeg.cfg");

	/* preview settings */
	if(options & SPANK_PREVIEW)
	{
		struct dirent **pp_dirent;
		int count;

		print(ML_INFO, NULL, "generating previews.."); /* generate previews */
		if(forkitf(cb_default, cb_print, "transcode", gp_transcodebin, "%s%s%s\n-y\njpg,null\n-F\n100\n-Z\n%dx%d"
			"\n-c\n300-301,600-601,900-901,1200-1201,1500-1501\n-o\npre",
			p_generic->s, p_video->s, p_preview->s,
			(int)((double)p_out->height * p_out->aspect), p_out->height, p_out->p_filename))
		{
			printerror("failed generating previews");
			freestrings();
			return 1;
		}

		count = scandir(".", &pp_dirent, 0, 0);
		if(count >= 0) /* display then remove previews */
		{
			int found = count;

			print(ML_INFO, NULL, "displaying previews..");

			vs_print(p_final, "feh");
			while(count--)
			{
				if(!fnmatch("pre*.jpg", pp_dirent[count]->d_name, 0))
					vs_cat(p_final, "\n%s", pp_dirent[count]->d_name);
			}
			forkitf(cb_default, cb_print, "feh", gp_fehbin, "%s", p_final->s);

			count = found;
			while(count--)
			{
				if(!(options & SPANK_NOUNLINK) && 
					!fnmatch("pre*.jpg", pp_dirent[count]->d_name, 0))
					unlink(pp_dirent[count]->d_name);
			}

			free(pp_dirent);
		}
		freestrings();
		return 0;
	}

	/* auto volume only pass */
	if(options & SPANK_AVPASS)
	{
		print(ML_INFO, NULL, "scanning volume..");
		vs_print(p_final, "%s-tmp.ac3", p_out->p_filename);
		if(forkitf(cb_transcode, cb_print, (void*)p_out, gp_transcodebin,
			"%s%s\n-y\nnull,ac3\n-m\n%s-tmp\n-J\nastat=%s.norm",
			p_generic->s, p_audio->s, p_out->p_filename, p_out->p_filename))
		{
			print(ML_INFO, NULL, "");
			printerror("error scanning volume");
			if(!(options & SPANK_NOUNLINK))
				unlink(p_final->s);
			freestrings();
			return 1;
		}
		print(ML_INFO, NULL, "");
		if(!(options & SPANK_NOUNLINK))
			unlink(p_final->s);
		return 0;
	}

	/* execute 1st pass */
	if(options & SPANK_FIRST && !(options & SPANK_MULTIPLEX))
	{
		print(ML_INFO, NULL, "starting 1st VBR pass..");

		vs_print(p_final, "%s%s%s%s", p_generic->s, p_video->s, p_audio->s, p_fpschange->s);
		if(options & SPANK_AVOLUME)
			vs_cat(p_final,
				"\n-y\nffmpeg\n-F\nmpeg2video\n-R\n1,%s.pass1\n-m\n%s.tmp\n-J\nastat=%s.norm",
				p_out->p_filename, p_out->p_filename, p_out->p_filename);
		else
			vs_cat(p_final, "\n-y\nffmpeg,null\n-F\nmpeg2video\n-R\n1,%s.pass1", p_out->p_filename);

		if(forkitf(cb_transcode, cb_print, (void*)p_out, gp_transcodebin, "%s", p_final->s))
		{
			vs_print(p_final, "%s.tmp", p_out->p_filename);
			if(!(options & SPANK_NOUNLINK))
				unlink(p_final->s);
			print(ML_INFO, NULL, "");
			printerror("transcode failed");
			freestrings();
			return 1;
		}
		vs_print(p_final, "%s.tmp", p_out->p_filename);
		if(!(options & SPANK_NOUNLINK))
			unlink(p_final->s);

		print(ML_INFO, NULL, "");
	}

	/* execute 2nd pass */
	if(options & SPANK_SECOND && !(options & SPANK_MULTIPLEX))
	{
		/* volume auto adjust */
		if(options & SPANK_AVOLUME)
		{
			FILE *p_file;
			char p_buffer[320];

			vs_print(p_final, "%s.norm", p_out->p_filename);
			p_file = fopen(p_final->s, "r");
			if(!p_file)
			{
				printerror("unable to open %s.norm for autoaudio adjustment", p_out->p_filename);
				freestrings();
				return 1;
			}
			if(!fgets(p_buffer, sizeof(p_buffer) - 1, p_file))
			{
				printerror("%s.norm contains no relevant data", p_out->p_filename);
				fclose(p_file);
				freestrings();
				return 1;
			}
			p_out->volume = atof(p_buffer);
			if(p_out->volume >= 1.1)
				p_out->volume -= 0.1;
			else
				p_out->volume = 1.0;
			fclose(p_file);
		}

		/* check pass1 file exists */
		{
			struct stat sdata;
			vs_print(p_final, "%s.pass1", p_out->p_filename);
			if(stat(p_final->s, &sdata))
			{
				printerror("unable to open %s.pass1", p_out->p_filename);
				freestrings();
				return 1;
			}
		}

		/* transcode */
		if(options & SPANK_SPEED)
			print(ML_INFO, NULL, "starting 2nd VBR pass (video)..");
		else
			print(ML_INFO, NULL, "starting 2nd VBR pass..");
			
		vs_print(p_final,
			"%s%s%s%s%s%s\n-Z\n%dx%d\n-R\n2,%s.pass1\n-o\n%s",
			p_generic->s, p_audio->s, p_video->s, p_videoextra->s, p_outcodec->s, 
			p_fpschange->s,	p_out->width, p_out->height, p_out->p_filename, p_out->p_filename);
		if(p_out->volume != 1.0)
			vs_cat(p_final, "\n-s\n%f", p_out->volume);
		if(forkitf(cb_transcode, cb_print, (void*)p_out, gp_transcodebin, "%s", p_final->s))
		{
			print(ML_INFO, NULL, "");
			printerror("transcode failed");
			freestrings();
			return 1;
		}
		if(options & SPANK_SPEED)
		{
			print(ML_INFO, NULL, "");
			print(ML_INFO, NULL, "starting 2nd VBR pass (audio)..");
			if(forkitf(cb_transcode, cb_print, (void*)p_out, gp_transcodebin, 
				"%s%s%s", p_generic->s, p_audio->s, p_speedout->s))
			{
				print(ML_INFO, NULL, "");
				printerror("transcode audio failed");
				freestrings();
				return 1;
			}
		}
		print(ML_INFO, NULL, "");
	}

	/* CBR encoding */
	if(!(options & (SPANK_FIRST | SPANK_SECOND)) && !(options & SPANK_MULTIPLEX))
	{
		/* volume auto adjust */
		if(options & SPANK_AVOLUME)
		{
			char p_buffer[320];
			FILE *p_file;

			print(ML_INFO, NULL, "scanning volume..");
			if(forkitf(cb_transcode, cb_print, (void*)p_out, gp_transcodebin,
				"%s%s\n-y\nnull,ac3\n-m\n%s-tmp\n-J\nastat=%s.norm",
				p_generic->s, p_audio->s, p_out->p_filename, p_out->p_filename))
			{
				print(ML_INFO, NULL, "");
				printerror("error scanning volume");
			}
			print(ML_INFO, NULL, "");

			vs_print(p_final, "%s-tmp.ac3", p_out->p_filename);
			if(!(options & SPANK_NOUNLINK))
				unlink(p_final->s);

			vs_print(p_final, "%s.norm", p_out->p_filename);
			p_file = fopen(p_final->s, "r");
			if(!p_file)
			{
				printerror("unable to open %s.norm for autoaudio adjustment", p_out->p_filename);
				freestrings();
				return 1;
			}
			if(!fgets(p_buffer, sizeof(p_buffer) - 1, p_file))
			{
				printerror("%s.norm contains no relevant data", p_out->p_filename);
				fclose(p_file);
				freestrings();
				return 1;
			}
			p_out->volume = atof(p_buffer);
			fclose(p_file);
		}

		/* execute */
		if(options & SPANK_SPEED)
			print(ML_INFO, NULL, "starting CBR encode (video)..");
		else
			print(ML_INFO, NULL, "starting CBR encode..");
			
		vs_print(p_final, "%s%s%s%s%s%s\n-Z\n%dx%d\n-o\n%s",
			p_generic->s, p_video->s, p_videoextra->s, p_audio->s, p_outcodec->s,
			p_fpschange->s, p_out->width, p_out->height, p_out->p_filename);
		if(p_out->volume != 1.0)
			vs_cat(p_final, "\n-s\n%f", p_out->volume);
		if(forkitf(cb_transcode, cb_print, (void*)p_out, gp_transcodebin, "%s", p_final->s))
		{
			print(ML_INFO, NULL, "");
			printerror("transcode failed");
			freestrings();
			return 2;
		}
		
		if(options & SPANK_SPEED)
		{
			print(ML_INFO, NULL, "");
			print(ML_INFO, NULL, "starting CBR encode (audio)..");
			
			vs_print(p_final, "%s%s%s", p_generic->s, p_audio->s, p_speedout->s);
			if(p_out->volume != 1.0)
				vs_cat(p_final, "\n-s\n%f", p_out->volume);

			if(forkitf(cb_transcode, cb_print, (void*)p_out, gp_transcodebin, "%s", p_final->s))
			{
				print(ML_INFO, NULL, "");
				printerror("transcode failed");
				freestrings();
				return 2;
			}
		}
		print(ML_INFO, NULL, "");
	}

	if(options & SPANK_MPLAYER)
		unlink("stream.yuv");

	/* multiplex / audioconvert */
	if((options & SPANK_SECOND) || !(options & (SPANK_FIRST | SPANK_SECOND)) ||
		(options & SPANK_MULTIPLEX))
	{
		if(options & SPANK_SPEED)
		{
			double secs;
			
			secs = (double) p_out->length / (double) p_in->length
				* (double) p_out->length * 1.1 - (double) p_out->length;
			
			if(secs < 0)
				secs = -secs;
			
			print(ML_INFO, NULL, "changing audio speed..");
			if(forkitf(cb_sox, cb_print, p_out, gp_soxbin, 
				"sox\n-S\n-t\nwav\n%s_full.wav\n-t\nwav\n%s.wav\nspeed\n%.9f",
				p_out->p_filename, p_out->p_filename, p_out->fps / p_in->fps))
			{
				print(ML_INFO, NULL, "");
				printerror("sox failed");
				freestrings();
				return 3;
			}
			print(ML_INFO, NULL, "");

			vs_print(p_final, "%s.ac3", p_out->p_filename);
			unlink(p_final->s);
				
			print(ML_INFO, NULL, "converting audio to AC3..");
			if(forkitf(cb_ffmpeg, cb_print, p_out, gp_ffmpegbin,
				"ffmpeg\n-y\n-acodec\nadpcm_ima_wav\n-i\n%s.wav\n-acodec\nac3\n-ab\n%d\n-ac\n%d\n%s.ac3",
				p_out->p_filename, p_out->arate, p_out->achans, p_out->p_filename))
			{
				print(ML_DEBUG, NULL, "ffmpeg failed, attempting version workaround");
				if(forkitf(cb_ffmpeg, cb_print, p_out, gp_ffmpegbin,
					"ffmpeg\n-y\n-acodec\nadpcm_ima_wav\n-i\n%s.wav\n-acodec\nac3\n-ab\n%dk\n-ac\n%d\n%s.ac3",
					p_out->p_filename, p_out->arate, p_out->achans, p_out->p_filename))
				{
					print(ML_INFO, NULL, "");
					printerror("ffmpeg failed");
					freestrings();
					return 3;
				}
			}
			print(ML_INFO, NULL, "");
			vs_print(p_final, "%s.wav", p_out->p_filename);
			if(!(options & SPANK_NOUNLINK))
				unlink(p_final->s);
		}
		print(ML_INFO, NULL, "multiplexing..");
		if((options & SPANK_SECOND))
		{
			if(forkitf(cb_default, cb_print, "mplex", gp_mplexbin,
				"mplex\n-V\n-f\n8\n-o\n%s.mpg\n%s.m2v\n%s.ac3",
				p_out->p_filename, p_out->p_filename, p_out->p_filename))
				{
					print(ML_INFO, NULL, "");
					printerror("mplex failed");
					freestrings();
					return 3;
				}
		}
		else
		{
			if(forkitf(cb_default, cb_print, "mplex", gp_mplexbin,
				"mplex\n-f\n8\n-o\n%s.mpg\n%s.m2v\n%s.ac3",
				p_out->p_filename, p_out->p_filename, p_out->p_filename))
			{
				print(ML_INFO, NULL, "");
				printerror("mplex failed");
				freestrings();
				return 3;
			}
		}

		print(ML_INFO, NULL, "");
		if(options & SPANK_SPEED)
		{
			vs_print(p_final, "%s_full.wav", p_out->p_filename);
			if(!(options & SPANK_NOUNLINK))
				unlink(p_final->s);
		}
		if(!(options & SPANK_NOUNLINK))
		{
			vs_print(p_final, "%s.m2v", p_out->p_filename);
			unlink(p_final->s);
			vs_print(p_final, "%s.ac3", p_out->p_filename);
			unlink(p_final->s);
			vs_print(p_final, "%s.wav", p_out->p_filename);
			unlink(p_final->s);
		}
	}

	freestrings();
	return 0;
#undef freestrings
}

/* generic callback */
int cb_default(const char *p_string, void *p_options)
{
	/* initialisation command */
	if(!p_string)
		return 0;

	/* check line against error|warning filter */
	if(pcre_exec(gp_warnerrregex, NULL, p_string, strlen(p_string), 0, 0, NULL, 0) >= 1)
		print(ML_WARNING, (char*)p_options, "%s", p_string);
	else
		print(ML_VERBOSE, (char*)p_options, "%s", p_string);

	return 0;
}

/* tcprobe identify callback */
int cb_tcprobe(const char *p_string, void *p_options)
{
	video_t *p_video;
	int numstrings, length, p_vector[20];
	int bpos, i;

	/* option is a video_t */
	p_video = (video_t*)p_options;

	/* initialisation command */
	if(!p_string)
	{
		/* pre set feedback as nothing found */
		p_video->feedback = 0;
		return 0;
	}

	length = strlen(p_string);

	if(pcre_exec(gp_warnerrregex, NULL, p_string, length, 0, 0, NULL, 0) >= 1)
		print(ML_WARNING, "tcprobe", "%s", p_string);
	else
		print(ML_VERBOSE, "tcprobe", "%s", p_string);

	for(bpos = 0; bpos < length; )
	{
		if((numstrings = pcre_exec(gp_tcregex, NULL, &p_string[bpos], length - bpos, 0, 0,
			p_vector, sizeof(p_vector) / sizeof(int))) > 0)
		{
			p_video->feedback = 1;

			switch(p_string[bpos + p_vector[2]])
			{
				case 'g':
					{
						int next = 0;
						int _w, _h;

						for(i = p_vector[4]; i < p_vector[5]; i++)
						{
							if(p_string[bpos + i] == 'x')
							{
								next = i + 1;
								break;
							}
						}

						if(!next)
							break;

						_w = atoi(&p_string[bpos + p_vector[4]]);
						_h = atoi(&p_string[bpos + next]);
						if(!p_video->width || !p_video->height)
						{
							p_video->width = _w;
							p_video->height = _h;
						}
						print(ML_VERBOSE, NULL, "detect video width and height: %dx%d", _w, _h);
					}
					break;

				case 'f':
					{
						double _fps = atof(&p_string[bpos + p_vector[4]]);
						if(!p_video->fps)
							p_video->fps = _fps;
						print(ML_VERBOSE, NULL, "detect video fps: %f", _fps);
					}
					break;

				case 'e':
					{
						int n1 = 0, n2 = 0;

						for(i = p_vector[4]; i < p_vector[5]; i++)
						{
							if(!n1)
							{
								if(p_string[bpos + i] == ',')
									n1 = i + 1;
							}
							else
							{
								if(p_string[bpos + i] == ',')
								{
									n2 = i + 1;
									break;
								}
							}
						}

						if(!n2)
							break;

						if(!p_video->hz)
							p_video->hz = atoi(&p_string[bpos + p_vector[4]]);
						if(!p_video->abits)
							p_video->abits = atoi(&p_string[bpos + n1]);
						if(!p_video->achans)
							p_video->achans = atoi(&p_string[bpos + n2]);

						print(ML_VERBOSE, NULL, "detect audio hz,bits,chans: %d,%d,%d",
							p_video->hz, p_video->abits, p_video->achans);
					}
					break;

				default:
					break;
			}
			bpos += p_vector[7] + 1;
		}
		else if(pcre_exec(gp_tcregex4, NULL, &p_string[bpos], length - bpos, 0, 0,
			p_vector, sizeof(p_vector) / sizeof(int)) > 0)
		{
			double _aspect = parseaspect(&p_string[p_vector[2]]);
			if(!p_video->aspect)
				p_video->aspect = _aspect;
			print(ML_VERBOSE, NULL, "detect video aspect: %f", _aspect);
			bpos += p_vector[3] + 1;
		}
		else if((numstrings = pcre_exec(gp_tcregex2, NULL, &p_string[bpos], length - bpos, 0, 0,
			p_vector, sizeof(p_vector) / sizeof(int))) > 0)
		{
			if(strncmp("bitrate", &p_string[bpos + p_vector[2]], p_vector[3] - p_vector[2]) == 0)
			{
				int _rate = atoi(&p_string[bpos + p_vector[4]]);
				if(!p_video->arate)
					p_video->arate = _rate;
				print(ML_VERBOSE, NULL, "detect audio rate: %f", _rate);
			}
			else if(strncmp("duration", &p_string[bpos + p_vector[2]],
				p_vector[3] - p_vector[2]) == 0)
			{
				int n1 = 0, n2 = 0;

				for(i = p_vector[4]; i < p_vector[5]; i++)
				{
					if(!n1)
					{
						if(p_string[bpos + i] == ':')
							n1 = i + 1;
					}
					else
					{
						if(p_string[bpos + i] == ':')
						{
							n2 = i + 1;
							break;
						}
					}
				}

				if(n2)
				{
					if(!p_video->length)
					{
						p_video->hours = atoi(&p_string[bpos + p_vector[4]]);
						p_video->minutes = atoi(&p_string[bpos + n1]);
						p_video->seconds = atoi(&p_string[bpos + n2]);
						p_video->length = p_video->seconds + p_video->minutes * 60
							+ p_video->hours * 3600;
					}
					print(ML_VERBOSE, NULL, "detect video duration");
				}
			}
			bpos += p_vector[5] + 1;
		}
		else if((numstrings = pcre_exec(gp_tcregex3, NULL, &p_string[bpos], length - bpos, 0, 0,
			p_vector, sizeof(p_vector) / sizeof(int))) > 0)
		{
			if(strncmp("frames", &p_string[bpos + p_vector[4]],
				p_vector[5] - p_vector[4]) == 0)
			{
				int _frames = atoi(&p_string[bpos + p_vector[2]]);
				if(!p_video->frames)
					p_video->frames = _frames;
				print(ML_VERBOSE, NULL, "detect video frames: %d", _frames);
			}
			bpos += p_vector[5] + 1;
		}  // end if tcregex
		else
		{
			break;
		}
	}

	return 0;
}

/* transcode progress callback */
int cb_transcode(const char *p_string, void *p_options)
{
	char p_buffer[1024];
	int numstrings, length, p_vector[20];
	video_t *p_video;
	int i;

	/* initialisation command */
	if(!p_string)
		return 0;

	/* option is a video_t */
	p_video = (video_t*)p_options;

	/* regex string */
	length = strlen(p_string);
	numstrings = pcre_exec(gp_transregex, NULL, p_string, length, 0, 0,
		p_vector, sizeof(p_vector) / sizeof(int));

	/* make copy of string and terminate vectors with 0 */
	if(numstrings > 0)
	{
		if(length >= sizeof(p_buffer))
		{
			printerror("string exceeds buffer");
			return 1;
		}
		memcpy(p_buffer, p_string, length);
		for(i = 1; i < 6; i++)
			p_buffer[p_vector[i * 2 + 1]] = 0;
		if(p_video->frames)
			fprintf(stderr,"  %6.2f fps, %.2f%% complete, frame %d  \r", atof(&(p_buffer[p_vector[4]])),
				atof(&(p_buffer[p_vector[2]])) * 100.0 / (double)p_video->frames, atoi(&(p_buffer[p_vector[2]])));
		else
			fprintf(stderr,"  %6.2f fps, frame %09d, %d:%02d:%02d  \r", atof(&(p_buffer[p_vector[4]])),
				atoi(&(p_buffer[2])), atoi(&(p_buffer[p_vector[4]])), atoi(&(p_buffer[p_vector[6]])),
				atoi(&(p_buffer[p_vector[8]])));
	}
	else
	{
		if(pcre_exec(gp_warnerrregex, NULL, p_string, strlen(p_string), 0, 0, NULL, 0) >= 1)
			print(ML_WARNING, "transcode", "%s", p_string);
		else
			print(ML_VERBOSE, "transcode", "%s", p_string);
	}

	return 0;
}

/* sox callback */
int cb_sox(const char *p_string, void *p_options)
{
	int length;
	int i;
	double t[2];
	int p_vector[30];
	char p_buffer[1024];
	video_t *p_video;

	if(!p_string)
		return 0;
		
	p_video = p_options;

	length = strlen(p_string);
	if(length >= sizeof(p_buffer) - 1)
		length = 1023;

	if(pcre_exec(gp_soxregex, NULL, p_string, length, 0, 0, p_vector,
		sizeof(p_vector) / sizeof(int)) > 0)
	{
		memcpy(p_buffer, p_string, length + 1);
		p_buffer[length] = 0;
		for(i = 1; i < 5; i++)
			p_buffer[p_vector[i * 2 + 1]] = 0;

		t[0] = atof(&p_buffer[p_vector[2]]) * 60.0 + atof(&p_buffer[p_vector[4]]);
		// for raw input sox doesn't display destination time
		//t[1] = atof(&p_buffer[p_vector[6]]) * 60.0 + atof(&p_buffer[p_vector[8]]);
		//fprintf(stderr, "  %.2f%% complete  \r", t[0] * 100.0 / t[1]);
		fprintf(stderr, "  %.2f%% complete  \r", t[0] * 100.0 / (double)p_video->length);
	}
	else if(pcre_exec(gp_warnerrregex, NULL, p_string, length, 0, 0, NULL, 0) >= 1)
	{
		print(ML_WARNING, "sox", "%s", p_string);
	}
	else
	{
		print(ML_VERBOSE, "sox", "%s", p_string);
	}
	return 0;
}

/* ffmpeg callback */
int cb_ffmpeg(const char *p_string, void *p_options)
{
	int length;
	video_t *p_video;
	int p_vector[10];

	char p_buffer[1024];

	if(!p_string)
		return 0;

	p_video = (video_t*)p_options;

	length = strlen(p_string);
	if(length >= sizeof(p_buffer) - 1)
		length = 1023;

	if(pcre_exec(gp_ffmpegregex, NULL, p_string, length, 0, 0, p_vector,
		sizeof(p_vector) / sizeof(int)) > 0)
	{
		memcpy(p_buffer, p_string, length);
		p_buffer[length] = 0;
		p_buffer[p_vector[3]] = 0;

		fprintf(stderr, "  %.2f%% complete  \r", atof(&p_buffer[p_vector[2]]) * 100.0 / p_video->length);
	}
	else if(pcre_exec(gp_warnerrregex, NULL, p_string, strlen(p_string), 0, 0, NULL, 0) >= 1)
	{
		print(ML_WARNING, "ffmpeg", "%s", p_string);
	}
	else
	{
		print(ML_VERBOSE, "ffmpeg", "%s", p_string);
	}
	return 0;
}

/* mplayer identify callback */
int cb_mplayer(const char *p_string, void *p_options)
{
	const char *p_id;
	const char *p_value;
	video_t *p_video;
	int p_vector[20];
	int numstrings;

	/* initialisation command */
	if(!p_string)
		return 0;

	/* option is output to a video_t */
	p_video = (video_t*)p_options;

	if(pcre_exec(gp_warnerrregex, NULL, p_string, strlen(p_string), 0, 0, NULL, 0) >= 1)
		print(ML_WARNING, "mplayer", "%s", p_string);
	else
		print(ML_VERBOSE, "mplayer", "%s", p_string);

	/* has aspect? */
	if(!p_video->aspect && pcre_exec(gp_idregex2, NULL, p_string, strlen(p_string), 0, 0,
		p_vector, sizeof(p_vector) / sizeof(int)) > 0)
	{
		p_video->aspect = parseaspect(&p_string[p_vector[2]]);
		print(ML_VERBOSE, NULL, "detect aspect: %f", p_video->aspect);
		return 0;
	}

	/* examine the string, return if nothing */
	numstrings = pcre_exec(gp_idregex, NULL, p_string, strlen(p_string), 0, 0,
		p_vector, sizeof(p_vector) / sizeof(int));
	if(numstrings < 0)
		return 0;

	/* create temp strings and feed data into video_t, speed is not needed here */
	pcre_get_substring(p_string, p_vector, numstrings, 1, &p_id);
	pcre_get_substring(p_string, p_vector, numstrings, 2, &p_value);

	if(!strcmp(p_id, "VIDEO_FORMAT"))
	{
		strncpy(p_video->p_vcodec, p_value, sizeof(p_video->p_vcodec) - 1);
		print(ML_VERBOSE, NULL, "detect video format: %s", p_video->p_vcodec);
	}
	if(!strcmp(p_id, "VIDEO_BITRATE"))
	{
		p_video->vrate = atoi(p_value) / 1000;
		print(ML_VERBOSE, NULL, "detect video rate: %d kbps", p_video->vrate);
	}
	if(!strcmp(p_id, "VIDEO_WIDTH"))
	{
		p_video->width = atoi(p_value);
		print(ML_VERBOSE, NULL, "detect video width: %d", p_video->width);
	}
	if(!strcmp(p_id, "VIDEO_HEIGHT"))
	{
		p_video->height = atoi(p_value);
		if(!p_video->aspect)
			p_video->aspect = (double)p_video->width / (double)p_video->height;
		print(ML_VERBOSE, NULL, "detect video height: %d", p_video->height);
	}
	if(!strcmp(p_id, "VIDEO_FPS"))
	{
		p_video->fps = atof(p_value);
		print(ML_VERBOSE, NULL, "detect video fps: %f", p_video->fps);
	}
	if(!strcmp(p_id, "AUDIO_CODEC"))
	{
		strncpy(p_video->p_acodec, p_value, sizeof(p_video->p_acodec) - 1);
		print(ML_VERBOSE, NULL, "detect audio format: %s", p_video->p_acodec);
	}
	if(!strcmp(p_id, "AUDIO_BITRATE"))
	{
		p_video->arate = atoi(p_value) / 1000;
		print(ML_VERBOSE, NULL, "detect audio rate: %d kbps", p_video->arate);
	}
	if(!strcmp(p_id, "AUDIO_RATE"))
	{
		p_video->hz = atoi(p_value);
		print(ML_VERBOSE, NULL, "detect audio hz: %d", p_video->hz);
	}
	if(!strcmp(p_id, "AUDIO_NCH"))
	{
		p_video->achans = atoi(p_value);
		print(ML_VERBOSE, NULL, "detect audio chans: %d", p_video->achans);
	}
	if(!strcmp(p_id, "LENGTH"))
	{
		p_video->length = atoi(p_value);
		p_video->hours   = p_video->length / 3600;
		p_video->minutes = p_video->length % 3600 / 60;
		p_video->seconds = p_video->length % 60;
		print(ML_VERBOSE, NULL, "detect video length: %d", p_video->length);
	}

	pcre_free_substring(p_id);
	pcre_free_substring(p_value);

	return 0;
}

/* entry point */
int main(int argc, char **argv)
{
	const char *p_error;
	int erroffset;
	int opt;
	char p_filters[512];
	int vrate = 0, arate = 0, vcrop = 0, hcrop = 0, channels = 0;
	int calcsize = 0, calcborder = 0, calcclip = 0;
	char vidmode = 'a';
	uint32_t options = 0;
	video_t invid, outvid;

	/* setup */
	p_filters[0] = 0;

	/* clear video data */
	memset(&invid, '\0', sizeof(video_t));
	memset(&outvid, '\0', sizeof(video_t));
	outvid.volume = outvid.gamma = 1.0;

	atexit(cleanup);

	/* show title */
	printf("dvdspanky - %s - (c) 2007 Jeffrey Grembecki\n\n", VERSION);

	/* compile regex */
	if(!(gp_transregex = pcre_compile(
		"^encoding\\sframes\\s\\[\\d+\\-(\\d+)\\],\\s*(\\d+[.]\\d+)\\sfps.*?EMT:\\s*(\\d+):(\\d+):(\\d+),.*",
		0, &p_error, &erroffset, NULL)))
	{
		printerror("gp_transregex compile failed - %s", p_error);
		exit(1);
	}
	if(!(gp_idregex = pcre_compile("^ID_([^=]+)=(.*)$", 0, &p_error, &erroffset, NULL)))
	{
		printerror("gp_idregex compile failed - %s", p_error);
		exit(1);
	}
	if(!(gp_idregex2 = pcre_compile("^.*\\(aspect (\\S+)\\).*$", 0, &p_error, &erroffset, NULL)))
	{
		printerror("gp_idregex2 compile failed - %s", p_error);
		exit(1);
	}

	if(!(gp_tcregex = pcre_compile(".*?\\-(\\w) (.+?) \\[(.*?)\\].*", 0, &p_error, &erroffset, NULL)))
	{ printerror("gp_tcsizeregex compile failed - %s", p_error); exit(1); }

	if(!(gp_tcregex2 = pcre_compile("^.* (\\w+)=([^ ,]+)(?:,.*| .*|)$", 0, &p_error, &erroffset, NULL)))
	{ printerror("gp_tcregex2 compile failed - %s", p_error); exit(1); }

	if(!(gp_tcregex3 = pcre_compile("^.* (\\S+) (\\S+)(?:,.*| .*|)$", 0, &p_error, &erroffset, NULL)))
	{ printerror("gp_tcregex3 compile failed - %s", p_error); exit(1); }

	if(!(gp_tcregex4 = pcre_compile("^.*aspect ratio: ([^ ]+)(?: .*|)$", 0, &p_error, &erroffset, NULL)))
	{ printerror("gp_tcregex4 compile failed - %s", p_error); exit(1); }

	if(!(gp_warnerrregex = pcre_compile("^.*(?:warning22|error22).*$",
		PCRE_CASELESS, &p_error, &erroffset, NULL)))
	{
		printerror("gp_warnerrregex compile failed - %s", p_error);
		exit(1);
	}
	if(!(gp_soxregex = pcre_compile("^.*Time:[ ]*(\\d+):(\\d+\\.\\d+) .* of (\\d+):(\\d+\\.\\d+) .*$",
		0, &p_error, &erroffset, NULL)))
	{
		printerror("gp_soxregex compile failed - %s", p_error);
		exit(1);
	}
	if(!(gp_ffmpegregex = pcre_compile("^.* time=(\\d+\\.\\d+) .*$",
		0, &p_error, &erroffset, NULL)))
	{
		printerror("gp_ffmpegregex compile failed - %s", p_error);
		exit(1);
	}

	/* parse options */
	while( (opt = getopt_long(argc, argv, gp_shortopts, gp_longopts, NULL)) != -1 )
	{
		switch(opt)
		{
			case '0':
				options |= SPANK_AVPASS;
				break;
			case '1':
				options &= SPANK_MASK ^ SPANK_SECOND;
				options |= SPANK_FIRST;
				break;
			case '2':
				options &= SPANK_MASK ^ SPANK_FIRST;
				options |= SPANK_SECOND;
				break;
			case 'a':
				outvid.aspect = parseaspect(optarg);
				if(!outvid.aspect)
				{
					printerror("aspect not 2, 3, *:*, */* or *.*");
					exit(1);
				}
				if(outvid.aspect < 1.5)
					outvid.aspect = 1.33333333;
				else
					outvid.aspect = 1.77777777;
				break;
			case 'A':
				options |= SPANK_AVOLUME;
				break;
			case 'b':
				calcborder = 1;
				break;
			case 'B':
				calcclip = 1;
				break;
			case 'c':
				channels = atoi(optarg);
				if(!channels || channels > 6)
				{
					printerror("channels not 1-6");
					exit(1);
				}
				break;
			case 'C':
				invid.aspect = parseaspect(optarg);
				break;
			case 'd':
				options |= SPANK_DVD;
				invid.title = atoi(optarg);
				break;
			case 'D':
				options |= SPANK_DEINTER;
				break;
			case 'f':
				strncpy(p_filters, optarg, sizeof(p_filters) - 1);
				break;
			case 'F':
				/* frameadjust, now automatic */
				break;
			case 'g':
				outvid.volume = atof(optarg);
				break;
			case 'G':
				outvid.gamma = atof(optarg);
				break;
			case 'h':
				fprintf(stderr, gp_help, argv[0]);
				exit(0);
				break;
			case 'H':
				hcrop = atoi(optarg);
				break;
			case 'i':
				strncpy(invid.p_filename, optarg, sizeof(invid.p_filename) - 1);
				break;
			case 'I':
				options |= SPANK_NONICE;
				break;
			case 'L':
				fprintf(stderr, gp_licence);
				exit(0);
				break;
			case 'm':
				options |= SPANK_FIRST | SPANK_SECOND;
				break;
			case 'M':
				options |= SPANK_MULTIPLEX;
				break;
			case 'n':
				vidmode = 'n';
				break;
			case 'N':
				options |= SPANK_FIXNTSC;
				break;
			case 'p':
				vidmode = 'p';
				break;
			case 'P':
				options |= SPANK_PREVIEW;
				break;
			case 'o':
				strncpy(outvid.p_filename, optarg, sizeof(outvid.p_filename) - 1);
				break;
			case 'O':
				options |= SPANK_NORMALISE;
				break;
			case 'r':
				vrate = atoi(optarg);
				if(vrate < 600 || vrate > 7500)
				{
					printerror("vrate not 600 - 7500");
					exit(1);
				}
				break;
			case 'R':
				arate = atoi(optarg);
				if(arate < 64 || arate > 448)
				{
					printerror("arate not 64 - 488");
					exit(1);
				}
				break;
			case 's':
				calcsize = atoi(optarg);
				break;
			case 'S':
				options |= SPANK_SPEED;
				break;
			case 'T':
				options |= SPANK_PPROCESS;
				break;
			case 'U':
				options |= SPANK_NOUNLINK;
				break;
			case 'v':
				g_msglevel = ML_VERBOSE;
				break;
			case 'V':
				vcrop = atoi(optarg);
				break;
			case '?':
				exit(1);
				break;
			default:
				printerror("unhandled option %c", opt);
				exit(1);
				break;
		}
	}

	if(!(options & SPANK_NONICE))
		nice(19);

	/* input file */
	if(!invid.p_filename[0])
	{
		printerror("no input file");
		exit(1);
	}
	if(!(options & SPANK_DVD))
	{
		struct stat sdata;
		if(stat(invid.p_filename, &sdata))
		{
			printerror("cannot open input file %s", invid.p_filename);
			exit(1);
		}
		invid.bytes = sdata.st_size;
	}

	/* create output filename based on input filename if none given */
	if(!outvid.p_filename[0])
	{
		pcre *p_regex;
		int p_vector[10];
		int found;

		p_regex = pcre_compile("^(?:.*[/]|)(.*?)(?:[.][^.]+|)$", 0, &p_error, &erroffset, NULL);
		if(!p_regex)
		{
			printerror("failed to compile filename regex - %s", perror);
			exit(1);
		}
		found = pcre_exec(p_regex, NULL, invid.p_filename, strlen(invid.p_filename), 0, 0,
			p_vector, sizeof(p_vector) / sizeof(int));
		if(found > 0)
		{
			strncpy(outvid.p_filename, &(invid.p_filename[p_vector[2]]), p_vector[3] - p_vector[2]);
			outvid.p_filename[p_vector[3] - p_vector[2]] = 0;
		}
		else
		{
			printerror("unable to fathom output filename from input");
			pcre_free(p_regex);
			exit(1);
		}
		pcre_free(p_regex);
	}

	/* start log */
	{
		vstr *p_str;

		if(!(p_str = vs_new()))
		{
			printerror("unable to create string");
			exit(1);
		}
		vs_print(p_str, "%s.log", outvid.p_filename);
		if(!(gp_logfile = fopen(p_str->s, "w")))
		{
			printerror("unable creating log file");
			vs_free(p_str);
			exit(1);
		}
		vs_free(p_str);
	}

	/* get source video details */
	if(options & SPANK_DVD)
	{
		forkitf(cb_mplayer, cb_print, &invid, gp_mplayerbin,
			"mplayer\n-dvd-device\n%s\ndvd://%d\n-identify\n-frames\n0\n-ao\nnull\n-vo\nnull",
			invid.p_filename, invid.title);
		forkitf(cb_tcprobe, cb_print, &invid, gp_tcprobebin, "tcprobe\n-i\n%s\n-T\n%d",
			invid.p_filename, invid.title);
		strcpy(invid.p_vcodec, "DVD");
	}
	else
	{
		forkitf(cb_mplayer, cb_print, &invid, gp_mplayerbin,
			"mplayer\n%s\n-identify\n-frames\n0\n-ao\nnull\n-vo\nnull", invid.p_filename);
		forkitf(cb_tcprobe, cb_print, &invid, gp_tcprobebin, "tcprobe\n-i\n%s", invid.p_filename);
	}

	if(!invid.feedback) /* if tcprobe unable to detect, flag use of mplayer for input */
	{
		if(invid.p_vcodec[0] == 0)
		{
			printerror("unknown input format. do you have the codec installed?");
			exit(1);
		}
		options |= SPANK_MPLAYER;
	}
	if(options & SPANK_DVD && !invid.hz)
		invid.hz = 48000;


	/* calc aspect */
	if(!invid.aspect)
		invid.aspect = (double)invid.width / (double)invid.height;
	if(!outvid.aspect)
	{
		if(invid.aspect < 1.55555555)
			outvid.aspect = 1.33333333;
		else
			outvid.aspect = 1.77777777;
	}

	/* set audio bitrate / codec and scale */
	strcpy(outvid.p_acodec, "AC3");
	if(!arate)
		arate = invid.arate;
	for(outvid.arate = 64; outvid.arate < arate; outvid.arate += 32)
		;
	arate = outvid.arate;

	/* set audio channels */
	if(channels)
		outvid.achans = channels;
	else
		outvid.achans = invid.achans;

	/* set audio junk */
	outvid.abits = 16;
	outvid.hz = 48000;

	/* approximate missing data */
	if(!invid.frames && invid.length)
		invid.frames = (int)((double)invid.length * invid.fps);
	if(!invid.vrate && invid.frames)
		invid.vrate = (double)invid.bytes * 8.4 / ((double)invid.frames / invid.fps) / 1000.0 -
			invid.arate;

	/* calculate video bitrate for final size allowing 5% overhead */
	if(calcsize)
	{
		vrate = (calcsize - ((arate + arate / 20) / 8 * invid.length)) * 8
			/ invid.length;
		vrate -= vrate / 20;

		if(vrate < 600 || vrate > 7500)
		{
			printerror("calculated video rate %d not 600 - 7500 for size %d KB", vrate, calcsize);
			printerror("suggested minimum size setting of %d KB",
				((631 + arate + arate / 20) / 8 * invid.length) / 1000 * 1000);
			printerror("suggested maximum size setting of %d KB",
				((7875 + arate + arate / 20) / 8 * invid.length) / 1000 * 1000);
			exit(1);
		}
	}

	/* adjust minimum automatic bitrate */
	if(!vrate && invid.vrate < 600)
		vrate = 600; /* TODO: set rate based on input filesize if rate == 0 */

	/* set video rate */
	if(vrate)
	{
		outvid.vrate = vrate;
	}
	else
	{
		outvid.vrate = invid.vrate;
		vrate = outvid.vrate;
	}

	/* set doublefps if needed */
	if(invid.fps <= 15)
		options |= SPANK_DOUBLEFPS;

	/* set pal/ntsc */
	if(vidmode == 'a')
	{
		if(invid.fps == 25)
			vidmode = 'p';
		else
			vidmode = 'n';
	}
	outvid.width = 720;
	if(vidmode == 'p')
	{
		outvid.height = 576;
		strcpy(outvid.p_vcodec, "PAL");
	}
	else
	{
		outvid.height = 480;
		strcpy(outvid.p_vcodec, "NTSC");
	}

	/* set fps */
	if(vidmode == 'p' && invid.fps == 25)
	{
		outvid.fps = 25;
		options &= SPANK_MASK ^ SPANK_SPEED;
	}
	else if(vidmode =='n' && (invid.fps == 23.976 || invid.fps == 29.97))
	{
		outvid.fps = invid.fps;
		options &= SPANK_MASK ^ SPANK_SPEED;
	}
	else if(vidmode == 'p')
	{
		if(!(options & SPANK_SPEED))
			options |= SPANK_HARDFPS;
		outvid.fps = 25;
	}
	else
	{
		if(!(options & SPANK_SPEED))
			options |= SPANK_HARDFPS;
		outvid.fps = 23.976;
	}

	/* set length */
	outvid.frames  = invid.frames;
	if(outvid.fps == invid.fps || options & SPANK_HARDFPS)
	{
		outvid.length = invid.length;
	}
	else
	{
		outvid.length = ((double)invid.length * invid.fps / outvid.fps);
	}
	outvid.hours   = outvid.length / 3600;
	outvid.minutes = outvid.length % 3600 / 60;
	outvid.seconds = outvid.length % 60;

	/* set border/crop */
	if(calcborder)
	{
		hcrop = 0;
		vcrop = (int)((double)invid.height - (double)invid.height / outvid.aspect * invid.aspect) / 4 * 2;
		if(vcrop > 0)
		{
			hcrop = (int)((double)invid.width - (double)invid.width / outvid.aspect * invid.aspect) / 4 * 2;
			vcrop = 0;
		}
	}
	if(calcclip)
	{
		hcrop = 0;
		printf("%dx%d %f -> %f\n", invid.width, invid.height, invid.aspect,outvid.aspect);
		vcrop = (int)((double)invid.height - (double)invid.height / outvid.aspect * invid.aspect) / 4 * 2;
		if(vcrop < 0)
		{
		printf("%dx%d %f -> %f\n", invid.width, invid.height, invid.aspect,outvid.aspect);
			hcrop = (int)((double)invid.width - (double)invid.width / outvid.aspect * invid.aspect) / 4 * 2;
			vcrop = 0;
		}
	}
	outvid.vcrop = vcrop;
	outvid.hcrop = hcrop;

	outvid.bytes = (vrate + vrate / 20 + arate + arate / 20) / 8 * invid.length * 1000;

	/* print source and dest video encoding info */
	print(ML_INFO, NULL, " source | %s", invid.p_filename);
	print(ML_INFO, NULL, "  video | %s, %dx%d, %d kbps, %.3f fps",
		invid.p_vcodec, invid.width, invid.height, invid.vrate, invid.fps);
	print(ML_INFO, NULL, "        | %.4f aspect, %d frames, %d:%02d:%02d",
		invid.aspect, invid.frames,	invid.hours, invid.minutes, invid.seconds);
	print(ML_INFO, NULL, "  audio | %s, %d kbps, %d Hz, %d bit, %d channel",
		invid.p_acodec, invid.arate, invid.hz, invid.abits, invid.achans);
	print(ML_INFO, NULL, "   file | %d KB", invid.bytes / 1000);
	if(options & SPANK_MPLAYER)
		print(ML_INFO, NULL, "   info | using mplayer decoder");
	print(ML_INFO, NULL, "");

	print(ML_INFO, NULL, "   dest | %s.mpg", outvid.p_filename);
	print(ML_INFO, NULL, "  video | %s, %dx%d, %d kbps, %.3f fps",
		outvid.p_vcodec, outvid.width, outvid.height, outvid.vrate, outvid.fps);
	print(ML_INFO, NULL, "        | %.4f aspect, %d frames, %d:%02d:%02d",
		outvid.aspect, outvid.frames, outvid.hours, outvid.minutes, outvid.seconds);
	print(ML_INFO, NULL, "        | %d(v):%d(h) crop, %s",
		outvid.vcrop, outvid.hcrop, (options & SPANK_MULTI) ? "VBR" : "CBR");
	print(ML_INFO, NULL, "  audio | %s, %d kbps, %d Hz, %d bit, %d channel",
		outvid.p_acodec, outvid.arate, outvid.hz, outvid.abits, outvid.achans);
	print(ML_INFO, NULL, "   file | ~%d KB", outvid.bytes / 1000);
	print(ML_INFO, NULL, "");

	exit(encode(&invid, &outvid, options, p_filters));
}
