//
// PublisherIdentityPermissionAttributeTest.cs -
//	NUnit Test Cases for PublisherIdentityPermissionAttribute
//
// Author:
//	Sebastien Pouliot  <sebastien@ximian.com>
//
// (C) 2003 Motus Technologies Inc. (http://www.motus.com)
// Copyright (C) 2004 Novell, Inc (http://www.novell.com)
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

using NUnit.Framework;
using System;
using System.IO;
using System.Security;
using System.Security.Permissions;
using System.Security.Cryptography.X509Certificates;

namespace MonoTests.System.Security.Permissions {

	[TestFixture]
	public class PublisherIdentityPermissionAttributeTest {

		private static byte[] cert1 = { 0x30,0x82,0x05,0x0F,0x30,0x82,0x03,0xF7,0xA0,0x03,0x02,0x01,0x02,0x02,0x0A,0x61,0x07,0x11,0x43,0x00,0x00,0x00,0x00,0x00,0x34,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x05,0x05,0x00,0x30,0x81,0xA6,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0A,0x57,0x61,0x73,0x68,0x69,0x6E,0x67,0x74,0x6F,0x6E,0x31,0x10,0x30,0x0E,0x06,0x03,0x55,0x04,0x07,0x13,0x07,0x52,0x65,0x64,0x6D,0x6F,0x6E,0x64,0x31,0x1E,0x30,0x1C,0x06,0x03,
			0x55,0x04,0x0A,0x13,0x15,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x69,0x6F,0x6E,0x31,0x2B,0x30,0x29,0x06,0x03,0x55,0x04,0x0B,0x13,0x22,0x43,0x6F,0x70,0x79,0x72,0x69,0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x20,0x32,0x30,0x30,0x30,0x20,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x2E,0x31,0x23,0x30,0x21,0x06,0x03,0x55,0x04,0x03,0x13,0x1A,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x64,0x65,0x20,0x53,0x69,0x67,
			0x6E,0x69,0x6E,0x67,0x20,0x50,0x43,0x41,0x30,0x1E,0x17,0x0D,0x30,0x32,0x30,0x35,0x32,0x35,0x30,0x30,0x35,0x35,0x34,0x38,0x5A,0x17,0x0D,0x30,0x33,0x31,0x31,0x32,0x35,0x30,0x31,0x30,0x35,0x34,0x38,0x5A,0x30,0x81,0xA1,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0A,0x57,0x61,0x73,0x68,0x69,0x6E,0x67,0x74,0x6F,0x6E,0x31,0x10,0x30,0x0E,0x06,0x03,0x55,0x04,0x07,0x13,0x07,0x52,0x65,0x64,0x6D,0x6F,0x6E,0x64,0x31,0x1E,0x30,0x1C,0x06,
			0x03,0x55,0x04,0x0A,0x13,0x15,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x69,0x6F,0x6E,0x31,0x2B,0x30,0x29,0x06,0x03,0x55,0x04,0x0B,0x13,0x22,0x43,0x6F,0x70,0x79,0x72,0x69,0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x20,0x32,0x30,0x30,0x32,0x20,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x2E,0x31,0x1E,0x30,0x1C,0x06,0x03,0x55,0x04,0x03,0x13,0x15,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x6F,0x72,0x61,
			0x74,0x69,0x6F,0x6E,0x30,0x82,0x01,0x22,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0F,0x00,0x30,0x82,0x01,0x0A,0x02,0x82,0x01,0x01,0x00,0xAA,0x99,0xBD,0x39,0xA8,0x18,0x27,0xF4,0x2B,0x3D,0x0B,0x4C,0x3F,0x7C,0x77,0x2E,0xA7,0xCB,0xB5,0xD1,0x8C,0x0D,0xC2,0x3A,0x74,0xD7,0x93,0xB5,0xE0,0xA0,0x4B,0x3F,0x59,0x5E,0xCE,0x45,0x4F,0x9A,0x79,0x29,0xF1,0x49,0xCC,0x1A,0x47,0xEE,0x55,0xC2,0x08,0x3E,0x12,0x20,0xF8,0x55,0xF2,0xEE,0x5F,0xD3,0xE0,0xCA,0x96,0xBC,0x30,
			0xDE,0xFE,0x58,0xC8,0x27,0x32,0xD0,0x85,0x54,0xE8,0xF0,0x91,0x10,0xBB,0xF3,0x2B,0xBE,0x19,0xE5,0x03,0x9B,0x0B,0x86,0x1D,0xF3,0xB0,0x39,0x8C,0xB8,0xFD,0x0B,0x1D,0x3C,0x73,0x26,0xAC,0x57,0x2B,0xCA,0x29,0xA2,0x15,0x90,0x82,0x15,0xE2,0x77,0xA3,0x40,0x52,0x03,0x8B,0x9D,0xC2,0x70,0xBA,0x1F,0xE9,0x34,0xF6,0xF3,0x35,0x92,0x4E,0x55,0x83,0xF8,0xDA,0x30,0xB6,0x20,0xDE,0x57,0x06,0xB5,0x5A,0x42,0x06,0xDE,0x59,0xCB,0xF2,0xDF,0xA6,0xBD,0x15,0x47,0x71,0x19,0x25,0x23,0xD2,0xCB,0x6F,0x9B,0x19,0x79,0xDF,0x6A,0x5B,
			0xF1,0x76,0x05,0x79,0x29,0xFC,0xC3,0x56,0xCA,0x8F,0x44,0x08,0x85,0x55,0x8A,0xCB,0xC8,0x0F,0x46,0x4B,0x55,0xCB,0x8C,0x96,0x77,0x4A,0x87,0xE8,0xA9,0x41,0x06,0xC7,0xFF,0x0D,0xE9,0x68,0x57,0x63,0x72,0xC3,0x69,0x57,0xB4,0x43,0xCF,0x32,0x3A,0x30,0xDC,0x1B,0xE9,0xD5,0x43,0x26,0x2A,0x79,0xFE,0x95,0xDB,0x22,0x67,0x24,0xC9,0x2F,0xD0,0x34,0xE3,0xE6,0xFB,0x51,0x49,0x86,0xB8,0x3C,0xD0,0x25,0x5F,0xD6,0xEC,0x9E,0x03,0x61,0x87,0xA9,0x68,0x40,0xC7,0xF8,0xE2,0x03,0xE6,0xCF,0x05,0x02,0x03,0x01,0x00,0x01,0xA3,0x82,
			0x01,0x40,0x30,0x82,0x01,0x3C,0x30,0x0E,0x06,0x03,0x55,0x1D,0x0F,0x01,0x01,0xFF,0x04,0x04,0x03,0x02,0x06,0xC0,0x30,0x13,0x06,0x03,0x55,0x1D,0x25,0x04,0x0C,0x30,0x0A,0x06,0x08,0x2B,0x06,0x01,0x05,0x05,0x07,0x03,0x03,0x30,0x1D,0x06,0x03,0x55,0x1D,0x0E,0x04,0x16,0x04,0x14,0x6B,0xC8,0xC6,0x51,0x20,0xF0,0xB4,0x2F,0xD3,0xA0,0xB6,0xAE,0x7F,0x5E,0x26,0xB2,0xB8,0x87,0x52,0x29,0x30,0x81,0xA9,0x06,0x03,0x55,0x1D,0x23,0x04,0x81,0xA1,0x30,0x81,0x9E,0x80,0x14,0x29,0x5C,0xB9,0x1B,0xB6,0xCD,0x33,0xEE,0xBB,0x9E,
			0x59,0x7D,0xF7,0xE5,0xCA,0x2E,0xC4,0x0D,0x34,0x28,0xA1,0x74,0xA4,0x72,0x30,0x70,0x31,0x2B,0x30,0x29,0x06,0x03,0x55,0x04,0x0B,0x13,0x22,0x43,0x6F,0x70,0x79,0x72,0x69,0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x20,0x31,0x39,0x39,0x37,0x20,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x2E,0x31,0x1E,0x30,0x1C,0x06,0x03,0x55,0x04,0x0B,0x13,0x15,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x69,0x6F,0x6E,0x31,0x21,0x30,0x1F,0x06,0x03,0x55,
			0x04,0x03,0x13,0x18,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x52,0x6F,0x6F,0x74,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x82,0x10,0x6A,0x0B,0x99,0x4F,0xC0,0x00,0xDE,0xAA,0x11,0xD4,0xD8,0x40,0x9A,0xA8,0xBE,0xE6,0x30,0x4A,0x06,0x03,0x55,0x1D,0x1F,0x04,0x43,0x30,0x41,0x30,0x3F,0xA0,0x3D,0xA0,0x3B,0x86,0x39,0x68,0x74,0x74,0x70,0x3A,0x2F,0x2F,0x63,0x72,0x6C,0x2E,0x6D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x2E,0x63,0x6F,0x6D,0x2F,0x70,0x6B,0x69,0x2F,0x63,0x72,0x6C,0x2F,0x70,0x72,
			0x6F,0x64,0x75,0x63,0x74,0x73,0x2F,0x43,0x6F,0x64,0x65,0x53,0x69,0x67,0x6E,0x50,0x43,0x41,0x2E,0x63,0x72,0x6C,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x35,0x23,0xFD,0x13,0x54,0xFC,0xE9,0xDC,0xF0,0xDD,0x0C,0x14,0x7A,0xFA,0xA7,0xB3,0xCE,0xFD,0xA7,0x3A,0xC8,0xBA,0xE5,0xE7,0xF6,0x03,0xFB,0x53,0xDB,0xA7,0x99,0xA9,0xA0,0x9B,0x36,0x9C,0x03,0xEB,0x82,0x47,0x1C,0x21,0xBD,0x14,0xCB,0xE7,0x67,0x40,0x09,0xC7,0x16,0x91,0x02,0x55,0xCE,0x43,0x42,0xB4,
			0xCD,0x1B,0x5D,0xB0,0xF3,0x32,0x04,0x3D,0x12,0xE5,0x1D,0xA7,0x07,0xA7,0x8F,0xA3,0x7E,0x45,0x55,0x76,0x1B,0x96,0x95,0x91,0x69,0xF0,0xDD,0x38,0xF3,0x48,0x89,0xEF,0x70,0x40,0xB7,0xDB,0xB5,0x55,0x80,0xC0,0x03,0xC4,0x2E,0xB6,0x28,0xDC,0x0A,0x82,0x0E,0xC7,0x43,0xE3,0x7A,0x48,0x5D,0xB8,0x06,0x89,0x92,0x40,0x6C,0x6E,0xC5,0xDC,0xF8,0x9A,0xEF,0x0B,0xBE,0x21,0x0A,0x8C,0x2F,0x3A,0xB5,0xED,0xA7,0xCE,0x71,0x87,0x68,0x23,0xE1,0xB3,0xE4,0x18,0x7D,0xB8,0x47,0x01,0xA5,0x2B,0xC4,0x58,0xCB,0xB2,0x89,0x6C,0x5F,0xFD,
			0xD3,0x2C,0xC4,0x6F,0xB8,0x23,0xB2,0x0D,0xFF,0x3C,0xF2,0x11,0x45,0x74,0xF2,0x09,0x06,0x99,0x18,0xDD,0x6F,0xC0,0x86,0x01,0x18,0x12,0x1D,0x2B,0x16,0xAF,0x56,0xEF,0x65,0x33,0xA1,0xEA,0x67,0x4E,0xF4,0x4B,0x82,0xAB,0xE9,0x0F,0xDC,0x01,0xFA,0xDF,0x60,0x7F,0x66,0x47,0x5D,0xCB,0x2C,0x70,0xCC,0x7B,0x4E,0xD9,0x06,0xB8,0x6E,0x8C,0x0C,0xFE,0x62,0x1E,0x42,0xF9,0x93,0x7C,0xA2,0xAB,0x0A,0x9E,0xD0,0x23,0x10,0xAE,0x4D,0x7B,0x27,0x91,0x6F,0x26,0xBE,0x68,0xFA,0xA6,0x3F,0x9F,0x23,0xEB,0xC8,0x9D,0xBB,0x87 };

		private static byte[] cert2 = { 0x30,0x82,0x01,0xDF,0x30,0x82,0x01,0x48,0x02,0x01,0x00,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x30,0x39,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x0D,0x30,0x0B,0x06,0x03,0x55,0x04,0x0A,0x14,0x04,0x41,0x54,0x26,0x54,0x31,0x1B,0x30,0x19,0x06,0x03,0x55,0x04,0x0B,0x14,0x12,0x44,0x69,0x72,0x65,0x63,0x74,0x6F,0x72,0x79,0x20,0x53,0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x30,0x1E,0x17,0x0D,0x39,0x36,0x30,0x31,0x31,0x38,0x32,0x31,0x30,0x33,0x35,0x32,
			0x5A,0x17,0x0D,0x30,0x31,0x30,0x31,0x31,0x36,0x32,0x31,0x30,0x33,0x35,0x32,0x5A,0x30,0x39,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x0D,0x30,0x0B,0x06,0x03,0x55,0x04,0x0A,0x14,0x04,0x41,0x54,0x26,0x54,0x31,0x1B,0x30,0x19,0x06,0x03,0x55,0x04,0x0B,0x14,0x12,0x44,0x69,0x72,0x65,0x63,0x74,0x6F,0x72,0x79,0x20,0x53,0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x30,0x81,0x9D,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8B,0x00,0x30,0x81,0x87,
			0x02,0x81,0x81,0x00,0x87,0x64,0x72,0x89,0x0B,0x20,0x8F,0x87,0x27,0xAC,0xC6,0x22,0xFE,0x00,0x40,0x69,0x48,0xAF,0xC6,0x86,0xCD,0x23,0x33,0xE3,0x11,0xC5,0x31,0x1A,0x1F,0x7E,0x9E,0x92,0x13,0xB6,0xA2,0xAC,0xE3,0xB0,0x1F,0x2A,0x07,0x6C,0xB6,0xD4,0xDE,0x4B,0xFA,0xF1,0xA2,0xA0,0x7D,0xCE,0x4B,0xBE,0xBE,0x26,0x48,0x09,0x8C,0x85,0x11,0xDE,0xCB,0x22,0xE7,0xC2,0xEE,0x44,0x51,0xFE,0x67,0xD5,0x5B,0x5A,0xE0,0x16,0x37,0x54,0x04,0xB8,0x3B,0x32,0x12,0x94,0x83,0x9E,0xB1,0x4D,0x80,0x6C,0xA4,0xA9,0x76,0xAC,0xB8,0xA4,
			0x97,0xF7,0xAB,0x0B,0x6C,0xA5,0x43,0xBA,0x6E,0x4F,0xC5,0x4E,0x00,0x30,0x16,0x3C,0x3F,0x99,0x14,0xDA,0xA2,0x20,0x08,0x8B,0xBA,0xED,0x76,0xAC,0x97,0x00,0xD5,0x6D,0x02,0x01,0x0F,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x03,0x81,0x81,0x00,0x38,0x50,0x1D,0x0A,0xD3,0x1B,0xBB,0xA2,0x9F,0x6C,0x8D,0x10,0xAA,0x42,0x1B,0x05,0x8D,0xE4,0x25,0xAB,0xFB,0x55,0xAE,0x6D,0xBA,0x53,0x67,0x15,0x07,0x9A,0xEC,0x55,0x9F,0x72,0x89,0x5F,0x24,0xB0,0xDB,0xCA,0x64,0xBD,0x64,0xAA,0xC2,0x8C,
			0xD9,0x3D,0xA2,0x45,0xB7,0xC6,0x92,0x71,0x51,0xEF,0xED,0xE1,0x51,0x54,0x97,0x56,0x35,0xA1,0xCE,0xE4,0x44,0xC4,0x47,0x66,0xFF,0x91,0xDA,0x88,0x9C,0x23,0xC2,0xB3,0xD4,0x62,0x4A,0xBC,0x94,0x55,0x9C,0x80,0x8E,0xB3,0xDD,0x4F,0x1A,0xED,0x12,0x5A,0xB5,0x2E,0xBC,0xF8,0x4B,0xCE,0xC6,0xD4,0x70,0xB3,0xB3,0x22,0xF8,0x5E,0x5C,0x36,0x7A,0xA6,0xB8,0x39,0x73,0x46,0x43,0x5C,0x9B,0x9A,0xBD,0x1E,0x7E,0xA7,0x04,0xCF,0x25,0x36 };

		[TestFixtureSetUp]
		public void FixtureSetUp ()
		{
			using (FileStream fs1 = File.OpenWrite ("cert1.cer")) {
				fs1.Write (cert1, 0, cert1.Length);
				fs1.Close ();
			}
			using (FileStream fs2 = File.OpenWrite ("cert2.cer")) {
				fs2.Write (cert2, 0, cert2.Length);
				fs2.Close ();
			}
		}

		[TestFixtureTearDown]
		public void FixtureTearDown ()
		{
			try {
				File.Delete ("cert1.cer");
				File.Delete ("cert2.cer");
			}
			catch {} // don't mess up results over that
		}

		[Test]
		public void Default () 
		{
			PublisherIdentityPermissionAttribute a = new PublisherIdentityPermissionAttribute (SecurityAction.Assert);
			Assert.IsNull (a.CertFile, "CertFile");
			Assert.IsNull (a.SignedFile, "SignedFile");
			Assert.IsNull (a.X509Certificate, "X509Certificate");
			Assert.AreEqual (a.ToString (), a.TypeId.ToString (), "TypeId");
			Assert.IsFalse (a.Unrestricted, "Unrestricted");

			PublisherIdentityPermission perm = (PublisherIdentityPermission) a.CreatePermission ();
			Assert.IsNull (perm.Certificate, "CreatePermission.Certificate");
		}

		[Test]
		public void Action () 
		{
			PublisherIdentityPermissionAttribute a = new PublisherIdentityPermissionAttribute (SecurityAction.Assert);
			Assert.AreEqual (SecurityAction.Assert, a.Action, "Action=Assert");
			a.Action = SecurityAction.Demand;
			Assert.AreEqual (SecurityAction.Demand, a.Action, "Action=Demand");
			a.Action = SecurityAction.Deny;
			Assert.AreEqual (SecurityAction.Deny, a.Action, "Action=Deny");
			a.Action = SecurityAction.InheritanceDemand;
			Assert.AreEqual (SecurityAction.InheritanceDemand, a.Action, "Action=InheritanceDemand");
			a.Action = SecurityAction.LinkDemand;
			Assert.AreEqual (SecurityAction.LinkDemand, a.Action, "Action=LinkDemand");
			a.Action = SecurityAction.PermitOnly;
			Assert.AreEqual (SecurityAction.PermitOnly, a.Action, "Action=PermitOnly");
			a.Action = SecurityAction.RequestMinimum;
			Assert.AreEqual (SecurityAction.RequestMinimum, a.Action, "Action=RequestMinimum");
			a.Action = SecurityAction.RequestOptional;
			Assert.AreEqual (SecurityAction.RequestOptional, a.Action, "Action=RequestOptional");
			a.Action = SecurityAction.RequestRefuse;
			Assert.AreEqual (SecurityAction.RequestRefuse, a.Action, "Action=RequestRefuse");
#if NET_2_0
			a.Action = SecurityAction.DemandChoice;
			Assert.AreEqual (SecurityAction.DemandChoice, a.Action, "Action=DemandChoice");
			a.Action = SecurityAction.InheritanceDemandChoice;
			Assert.AreEqual (SecurityAction.InheritanceDemandChoice, a.Action, "Action=InheritanceDemandChoice");
			a.Action = SecurityAction.LinkDemandChoice;
			Assert.AreEqual (SecurityAction.LinkDemandChoice, a.Action, "Action=LinkDemandChoice");
#endif
		}

		[Test]
		public void Action_Invalid ()
		{
			PublisherIdentityPermissionAttribute a = new PublisherIdentityPermissionAttribute ((SecurityAction)Int32.MinValue);
			// no validation in attribute
		}

		[Test]
		public void None () 
		{
			PublisherIdentityPermissionAttribute attr = new PublisherIdentityPermissionAttribute (SecurityAction.Assert);
			Assert.IsNull (attr.CertFile, "None.CertFile");
			Assert.IsNull (attr.SignedFile, "None.SignedFile");
			Assert.IsNull (attr.X509Certificate, "None.X509Certificate");
			PublisherIdentityPermission p = (PublisherIdentityPermission) attr.CreatePermission ();
			Assert.IsNull (p.Certificate, "None.CreatePermission");
		}

		[Test]
		public void CertFile () 
		{
			X509Certificate x509 = new X509Certificate (cert1);
			PublisherIdentityPermissionAttribute attr = new PublisherIdentityPermissionAttribute (SecurityAction.Assert);
			attr.CertFile = "cert1.cer";
			attr.SignedFile = null;
			attr.X509Certificate = null;
			Assert.AreEqual ("cert1.cer", attr.CertFile, "CertFile");
			Assert.IsNull (attr.SignedFile, "SignedFile");
			Assert.IsNull (attr.X509Certificate, "X509Certificate");
			PublisherIdentityPermission p = (PublisherIdentityPermission) attr.CreatePermission ();
			Assert.AreEqual (x509.GetRawCertDataString (), p.Certificate.GetRawCertDataString (), "CertFile.Certificate");
		}

		[Test]
		[Ignore ("CreateFromSignedFile isn't finished yet")]
		public void SignedFile () 
		{
			X509Certificate x509 = new X509Certificate (cert1);
			PublisherIdentityPermissionAttribute attr = new PublisherIdentityPermissionAttribute (SecurityAction.Assert);
			attr.CertFile = null;
			attr.SignedFile = "cert1.cer";
			attr.X509Certificate = null;
			Assert.IsNull (attr.CertFile, "CertFile");
			Assert.AreEqual ("cert1.cer", attr.SignedFile, "SignedFile");
			Assert.IsNull (attr.X509Certificate, "X509Certificate");
			PublisherIdentityPermission p = (PublisherIdentityPermission) attr.CreatePermission ();
			Assert.AreEqual (x509.GetRawCertDataString (), p.Certificate.GetRawCertDataString (), "SignedFile.Certificate");
		}

		[Test]
		public void X509CertificateProperty () 
		{
			X509Certificate x509 = new X509Certificate (cert2);
			PublisherIdentityPermissionAttribute attr = new PublisherIdentityPermissionAttribute (SecurityAction.Assert);
			attr.CertFile = null;
			attr.SignedFile = null;
			attr.X509Certificate = x509.GetRawCertDataString ();
			Assert.IsNull (attr.CertFile, "CertFile");
			Assert.IsNull (attr.SignedFile, "SignedFile");
			Assert.AreEqual (x509.GetRawCertDataString (), attr.X509Certificate, "X509Certificate");
			PublisherIdentityPermission p = (PublisherIdentityPermission) attr.CreatePermission ();
			Assert.AreEqual (x509.GetRawCertDataString (), p.Certificate.GetRawCertDataString (), "X509CertificateProperty.Certificate");
		}

		[Test]
		public void Priority () 
		{
			X509Certificate x509 = new X509Certificate (cert1);
			PublisherIdentityPermissionAttribute attr = new PublisherIdentityPermissionAttribute (SecurityAction.Assert);
			attr.CertFile = "cert1.cer";
			attr.SignedFile = "cert2.cer";
			attr.X509Certificate = x509.GetRawCertDataString ();
			Assert.AreEqual ("cert1.cer", attr.CertFile, "CertFile");
			Assert.AreEqual ("cert2.cer", attr.SignedFile, "SignedFile");
			Assert.AreEqual (x509.GetRawCertDataString (), attr.X509Certificate, "X509Certificate");
			// from documentation X509Certificate has priority over CertFile and SignedFile
			PublisherIdentityPermission p = (PublisherIdentityPermission) attr.CreatePermission ();
			Assert.AreEqual (x509.GetRawCertDataString (), p.Certificate.GetRawCertDataString (), "p.Certificate");
			// but no word about CertFile and SignedFile priority
			attr.CertFile = "cert1.cer";
			attr.SignedFile = "cert2.cer";
			attr.X509Certificate = null;
			p = (PublisherIdentityPermission) attr.CreatePermission ();
			Assert.AreEqual (x509.GetRawCertDataString (), p.Certificate.GetRawCertDataString (), "p.Certificate");
			// CertFile !
		}

		[Test]
#if !NET_2_0
		[ExpectedException (typeof (ArgumentException))]
#else
		[Category ("NotWorking")]
#endif
		public void Unrestricted () 
		{
			PublisherIdentityPermissionAttribute a = new PublisherIdentityPermissionAttribute (SecurityAction.Assert);
			a.Unrestricted = true;
			IPermission perm = a.CreatePermission ();
		}

		[Test]
		public void Attributes ()
		{
			Type t = typeof (PublisherIdentityPermissionAttribute);
			Assert.IsTrue (t.IsSerializable, "IsSerializable");

			object [] attrs = t.GetCustomAttributes (typeof (AttributeUsageAttribute), false);
			Assert.AreEqual (1, attrs.Length, "AttributeUsage");
			AttributeUsageAttribute aua = (AttributeUsageAttribute)attrs [0];
			Assert.IsTrue (aua.AllowMultiple, "AllowMultiple");
			Assert.IsFalse (aua.Inherited, "Inherited");
			AttributeTargets at = (AttributeTargets.Assembly | AttributeTargets.Class | AttributeTargets.Struct | AttributeTargets.Constructor | AttributeTargets.Method);
			Assert.AreEqual (at, aua.ValidOn, "ValidOn");
		}
	}
}
