/*
 * ViewBoard.cs - Main view widget for a Mahjongg board.
 *
 * Copyright (C) 2003  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace Mahjongg
{

using System;
using Qt;

// This class draws the board on-screen.

public sealed class ViewBoard : QFrame
{
    // Internal state.
    private Board board;
    private TileSet tiles;
    private QBrush background;
    private int layer1, x1, y1;
    private int layer2, x2, y2;
	private Tile tile1, tile2;
    private int boardWidth, boardHeight;
	private QPixmap offScreen;

    // Constructor.
    public ViewBoard(QWidget parent, Board board, TileSet tiles,
		     		 QBrush background)
	    	: base(parent)
	    	{
				// Initialize the board information.
				this.board = board;
				this.tiles = tiles;
				this.background = background;
				this.layer1 = -1;
				this.layer2 = -1;
				this.boardWidth = 600;
				this.boardHeight = 500;
				this.offScreen = new QPixmap(boardWidth, boardHeight);

				// Hook up interesting events and signals.
				paintEvent += new PaintEvent (PerformPaint);
				mousePressEvent += new MousePressEvent (MousePressed);
	    	}

	// Draw the tiles along a specific row.
	private void DrawRow(QPainter painter, int layer, int y,
						 int topx, int topy, int tileWidth,
						 int tileHeight, int tileOverlap,
						 int minx, int maxx)
			{
				int x;
				int showx, showy;
				Tile tile;
				bool highlight;
				for(x = maxx - 1; x >= minx; --x)
				{
					showx = topx + (x / 2) * tileWidth + tileOverlap * layer;
					if((x % 2) != 0)
					{
						showx += tileWidth / 2;
					}
	    			showy = topy + (y / 2) * tileHeight - tileOverlap * layer;
					if((y % 2) != 0)
					{
						showy += tileHeight / 2;
					}
	    			tile = board.GetTileDirect(layer, x, y);
	    			if(tile >= Tile.Min && tile <= Tile.Max)
	    			{
						highlight = IsHighlighted(layer, x, y);
						tiles.DrawTile(painter, showx, showy,
			       					   tile, highlight);
	    			}
				}
			}

    // Draw the entire board to a painter.
    public void Draw(QPainter painter)
	    	{
				// Fill the background.
				painter.FillRect(0, 0, boardWidth, boardHeight, background);

				// Get the tile sizes.
				int tileWidth = tiles.Width - tiles.Overlap;
				int tileHeight = tiles.Height - tiles.Overlap;
				int tileOverlap = tiles.Overlap;

				// Determine the top-left corner of the board.
				int topx = (boardWidth - tileWidth * 15) / 2 - tileOverlap;
				int topy = (boardHeight - tileHeight * 8 - tileOverlap * 2) / 2;

				// Draw the board from the bottom layer upwards.
				int layer, y;
				for(layer = 0; layer < 5; ++layer)
				{
					for(y = 0; y < 6; ++y)
					{
						DrawRow(painter, layer, y, topx, topy,
								tileWidth, tileHeight, tileOverlap, 0, 30);
					}
					DrawRow(painter, layer, 7, topx, topy,
							tileWidth, tileHeight, tileOverlap, 16, 30);
					DrawRow(painter, layer, 6, topx, topy,
							tileWidth, tileHeight, tileOverlap, 0, 30);
					DrawRow(painter, layer, 8, topx, topy,
							tileWidth, tileHeight, tileOverlap, 0, 30);
					DrawRow(painter, layer, 7, topx, topy,
							tileWidth, tileHeight, tileOverlap, 0, 16);
					for(y = 9; y < 16; ++y)
					{
						DrawRow(painter, layer, y, topx, topy,
								tileWidth, tileHeight, tileOverlap, 0, 30);
					}
				}
	    	}

    // Add a highlighted tile position (maximum of 2).
    public void AddHighlight(int layer, int x, int y, Tile tile)
	    	{
				if(layer1 == -1)
				{
		    		layer1 = layer;
		    		x1 = x;
		    		y1 = y;
					tile1 = tile;
				}
				else
				{
		    		layer2 = layer;
		    		x2 = x;
		    		y2 = y;
					tile2 = tile;
				}
	    	}

    // Remove highlighted tiles.
    public void RemoveHighlight()
	    	{
				layer1 = -1;
				layer2 = -1;
	    	}

    // Determine if a tile position is highlighted.
    public bool IsHighlighted(int layer, int x, int y)
	    	{
				if(layer1 == layer && x1 == x && y1 == y)
				{
		    		return true;
				}
				if(layer2 == layer && x2 == x && y2 == y)
				{
		    		return true;
				}
				return false;
	    	}

    // Convert a mouse position into a tile (layer, x, y) position.
    // Returns false if not on a tile.
    public bool MouseToTile(int x, int y, ref int tilelayer,
						    ref int tilex, ref int tiley)
	    	{
				// Get the tile sizes.
				int tileWidth = tiles.Width - tiles.Overlap;
				int tileHeight = tiles.Height - tiles.Overlap;
				int tileOverlap = tiles.Overlap;

				// Determine the top-left corner of the board.
				int topx = (boardWidth - tileWidth * 15) / 2 - tileOverlap;
				int topy = (boardHeight - tileHeight * 8 - tileOverlap * 2) / 2;

				// Iterate through the layers until we find a match.
				int tlayer, tx, ty;
				int showx, showy;
				Tile tile;
				for(tlayer = 4; tlayer >= 0; --tlayer)
				{
					for(ty = 0; ty < 16; ++ty)
					{
						for(tx = 0; tx < 30; ++tx)
						{
							tile = board.GetTileDirect(tlayer, tx, ty);
							if(tile >= Tile.Min && tile <= Tile.Max)
							{
								showx = topx + (tx / 2) * tileWidth +
										tileOverlap * tlayer;
								if((tx % 2) != 0)
								{
									showx += tileWidth / 2;
								}
								showx += tileOverlap;
								showy = topy + (ty / 2) * tileHeight -
										tileOverlap * tlayer;
								if((ty % 2) != 0)
								{
									showy += tileHeight / 2;
								}
								if(x >= showx && x < (showx + tileWidth) &&
								   y >= showy && y < (showy + tileHeight))
								{
									tilelayer = tlayer;
									tilex = tx;
									tiley = ty;
									return true;
								}
							}
						}
					}
				}

				// Was not positioned over a tile.
				return false;
	    	}

    // Paint the widget in response to a paint event.
    public void PerformPaint(QPaintEvent e)
	    	{
				QPainter painter = new QPainter(offScreen);
				Draw(painter);
				painter.Dispose();
				BitBlt(this, 0, 0, offScreen, 0, 0,
				       boardWidth, boardHeight, RasterOp.CopyROP, true);
	    	}

	// Determine if two tiles are the same, taking seasons and flowers
	// into account.
	private bool SameTile(Tile tile1, Tile tile2)
			{
				if(tile1 >= Tile.Season_1 && tile2 <= Tile.Season_4)
				{
					return (tile2 >= Tile.Season_1 && tile2 <= Tile.Season_4);
				}
				if(tile1 >= Tile.Flower_1 && tile2 <= Tile.Flower_4)
				{
					return (tile2 >= Tile.Flower_1 && tile2 <= Tile.Flower_4);
				}
				return (tile1 == tile2);
			}

	// Select a specific tile, based on a mouse event.
	private void SelectTile(int layer, int x, int y, Tile tile)
			{
				if(layer1 != -1)
				{
					// One tile already selected, so see if they match.
					if(layer1 == layer && x1 == x && y1 == y)
					{
						// Clicked on the same tile, so deselect it.
						RemoveHighlight();
					}
					else if(SameTile(tile1, tile))
					{
						// Remove both tiles.
						board.Remove(layer1, x1, y1);
						board.Remove(layer, x, y);
						RemoveHighlight();
					}
					else
					{
						// Tiles did not match, so beep and reset.
						QApplication.Beep();
						RemoveHighlight();
					}
				}
				else
				{
					// Select the first tile in a pair.
					AddHighlight(layer, x, y, tile);
				}

				// Redraw the board after the modification.
				PerformPaint(null);
			}

	// Handle a mouse press event.
	protected void MousePressed (QMouseEvent e)
			{
				if(e.Button() == ButtonState.LeftButton)
				{
					QPoint pt = e.Pos();
					int layer = 0;
					int x = 0;
					int y = 0;
					Tile tile;
					if(MouseToTile(pt.X(), pt.Y(), ref layer,
								   ref x, ref y))
					{
						tile = board.CanSelect(layer, x, y);
						if(tile != Tile.None)
						{
							SelectTile(layer, x, y, tile);
						}
					}
				}
			}

}; // class ViewBoard

}; // namespace Mahjongg
