/*  Copyright (C) 2011 Ben Asselstine
 *  This file originates in the doodlehash project.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope tmoustache it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
 *  02110-1301, USA.
 */
#include <config.h>
#include "moustache.h"
#include <libxml/tree.h>
#include <glib.h>
#include <string.h>
#include "util.h"
#include "object.h"

DoodleHashMoustache* doodlehash_moustache_new()
{
  DoodleHashMoustache *m = g_malloc (sizeof (DoodleHashMoustache));
  if (!m)
    return NULL;
  memset (m, 0, sizeof (DoodleHashMoustache));
  m->base = doodlehash_object_new();
  return m;
}
DoodleHashMoustache* doodlehash_moustache_copy(const DoodleHashMoustache*orig)
{
  DoodleHashMoustache *moustache = doodlehash_moustache_new();
  g_free (moustache->base);
  moustache->base = doodlehash_object_copy(orig->base);
  moustache->connect_to_mouth_x = orig->connect_to_mouth_x;
  moustache->connect_to_mouth_y = orig->connect_to_mouth_y;
  return moustache;
}

int doodlehash_moustache_load_from_svg(DoodleHashMoustache *m, const char *file)
{
  if (doodlehash_object_load_from_svg_file(m->base, file) != 0)
    return -1;
  RsvgHandle *handle = rsvg_handle_new_from_data((const guint8*)m->base->data, 
                                                 m->base->data_len, NULL);
  if (find_connect_point(handle, "#connect-to-mouth", 
                         &m->connect_to_mouth_x, &m->connect_to_mouth_y) != 0)
    return -2;
  rsvg_handle_close(handle, NULL);
  return 0;
}

DoodleHashMoustache* doodlehash_moustache_load_from_svg_file(const char *file)
{
  DoodleHashMoustache *moustache = doodlehash_moustache_new();
  if (doodlehash_moustache_load_from_svg(moustache, file) == 0)
    return moustache;
  else
    {
      doodlehash_moustache_free(moustache);
      return NULL;
    }
}

void doodlehash_moustache_get_width_and_height(DoodleHashMoustache* moustache, guint32* width, guint32* height)
{
  doodlehash_object_get_width_and_height(moustache->base, width, height);
}

guint32 doodlehash_moustache_get_width(DoodleHashMoustache* moustache)
{
  return doodlehash_object_get_width(moustache->base);
}

guint32 doodlehash_moustache_get_height(DoodleHashMoustache* moustache)
{
  return doodlehash_object_get_height(moustache->base);
}

void doodlehash_moustache_set_width(DoodleHashMoustache* moustache, guint32 width)
{
  double scale = (double) width / (double)moustache->base->width;
  moustache->connect_to_mouth_x *= scale;
  doodlehash_object_set_width(moustache->base, moustache->base->width * scale);
}

void doodlehash_moustache_set_height(DoodleHashMoustache* moustache, guint32 height)
{
  double scale = (double)moustache->base->height / (double)height;
  moustache->connect_to_mouth_y *= scale;
  doodlehash_object_set_height(moustache->base, moustache->base->height *scale);
}

void doodlehash_moustache_get_mouth_connection_point(DoodleHashMoustache* moustache, guint32* x, guint32* y)
{
  *x = moustache->connect_to_mouth_x;
  *y = moustache->connect_to_mouth_y;
}

guint32 doodlehash_moustache_get_mouth_connection_point_x(DoodleHashMoustache* moustache)
{
  return moustache->connect_to_mouth_x;
}

guint32 doodlehash_moustache_get_mouth_connection_point_y(DoodleHashMoustache* moustache)
{
  return moustache->connect_to_mouth_y;
}

void doodlehash_moustache_set_mouth_connection_point(DoodleHashMoustache* moustache, guint32 x, guint32 y)
{
  moustache->connect_to_mouth_x = x;
  moustache->connect_to_mouth_y = y;
}

void doodlehash_moustache_set_random_colours(DoodleHashMoustache* moustache, guint code)
{
  doodlehash_object_set_random_colours(moustache->base, code);
}
void doodlehash_moustache_set_colours(DoodleHashMoustache* moustache, guint code, char *s, char *f)
{
  doodlehash_object_set_colours(moustache->base, code, s, f);
}

gchar *doodlehash_moustache_get_fill_colour(DoodleHashMoustache* moustache, guint code)
{
  return doodlehash_object_get_fill_colour(moustache->base, code);
}

gchar *doodlehash_moustache_get_stroke_colour(DoodleHashMoustache* moustache, guint code)
{
  return doodlehash_object_get_stroke_colour(moustache->base, code);
}

void doodlehash_moustache_scale(DoodleHashMoustache *m, double scale)
{
  m->connect_to_mouth_x *= scale;
  m->connect_to_mouth_y *= scale;
  doodlehash_object_scale(m->base, scale);
}

void doodlehash_moustache_free(DoodleHashMoustache* m)
{
  if (m && m->base)
    doodlehash_object_free (m->base);
  g_free (m);
}
