#!/bin/bash

# Copyright (C) 2012 Charles Atkinson
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

# Purpose: removes a docoll instance

# Usage: details in the usage function below

# Function call tree
#    +
#    |
#    +-- initialise
#    |   |
#    |   +-- parse_cmdline
#    |   |   |
#    |   |   +-- usage
#    |   |
#    |   +-- ck_file
#    |
#    +-- remove_instance
#    |
#    +-- finalise
#
# Utility functions called from various places:
#    msg

# Function definitions in alphabetical order.  Execution begins after the last function definition.

#--------------------------
# Name: ck_file
# Purpose: for each file listed in the argument list: checks that it is 
#   reachable, exists and that the user has the listed permissions
# Usage: ck_file [ file_name <file_type>:<permission>...:[a] ]
#   where 
#       file_name is a file name
#       file_type is b (block special file), f (file) or d (directoy)
#       permission is r, w and x
#       a requires that file_name must be absolute (begin with /)
#   Example: ck_file foo d:rwx:
# Output: writes any errors to stdout
# Return: non-zero on error
#--------------------------
function ck_file {

    local absolute_flag buf file_name file_type perm perms retval

    # For each file ...
    # ~~~~~~~~~~~~~~~~~
    retval=0
    while [[ $# -gt 0 ]]
    do  
        file_name=$1
        file_type=${2%%:*}
        buf=${2#$file_type:}
        perms=${buf%%:*}
        absolute=${buf#$perms:}
		case $absolute in 
			'' )
            	absolute_flag=$false
				;;
			a )
            	absolute_flag=$true
				;;
			* )
                echo "$my_nam: ck_file: invalid absoluteness flag in '$2' specified for file '$file_name'" >&2
                let retval=retval+1
            	absolute_flag=$false
		esac
        shift 2

        # Is the file reachable and does it exist?
        # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        case $file_type in
            b ) 
                if [[ ! -b $file_name ]]; then
                    echo "file '$file_name' is unreachable, does not exist or is not a block special file" >&2
                    let retval=retval+1
                    continue
                fi  
                ;;  
            f ) 
                if [[ ! -f $file_name ]]; then
                    echo "file '$file_name' is unreachable, does not exist or is not an ordinary file" >&2
                    let retval=retval+1
                    continue
                fi  
                ;;  
            d ) 
                if [[ ! -d $file_name ]]; then
                    echo "directory '$file_name' is unreachable, does not exist or is not a directory" >&2
                    let retval=retval+1
                    continue
                fi
                ;;
            * )
                echo "$my_nam: ck_file: invalid file type '$file_type' specified for file '$file_name'" >&2
                return 1
        esac

        # Does the file have the requested permissions?
        # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        buf="$perms"
        while [[ $buf ]]
        do
            perm="${buf:0:1}"
            buf="${buf:1}"
            case $perm in
                r )
                    if [[ ! -r $file_name ]]; then
                        echo "file '$file_name' does not have requested read permission" >&2
                        let retval=retval+1
                        continue
                    fi
                    ;;
                w )
                    if [[ ! -w $file_name ]]; then
                        echo "file '$file_name' does not have requested write permission" >&2
                        let retval=retval+1
                        continue
                    fi
                    ;;
                x )
                    if [[ ! -x $file_name ]]; then
                        echo "file '$file_name' does not have requested execute permission" >&2
                        let retval=retval+1
                        continue
                    fi
                    ;;
                * )
                    echo "$my_nam: ck_file: unexpected permisssion '$perm' requested for file '$file_name'" >&2
                    return 1
            esac
        done

        # Does the file have the requested absoluteness?
        # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        if [[ $absolute_flag && ${file_name:0:1} != / ]]; then
          	echo "file '$file_name' does not begin with /" >&2
            let retval++
		fi

    done

return $retval

}  #  end of function ck_file

#--------------------------
# Name: finalise
# Purpose: final logging and get out of here
#--------------------------
function finalise {

    local msg rc

    # Final logging
    # ~~~~~~~~~~~~~
    msg=
    rc=$1
    case $rc in 
        129 )
            msg I "$my_nam: finalising on SIGHUP"
            ;;
        130 )
            msg I "$my_nam: finalising on SIGINT"
            ;;
        131 )
            msg I "$my_nam: finalising on SIGQUIT"
            ;;
        143 )
            msg I "$my_nam: finalising on SIGTERM"
            ;;
    esac

    exit $rc

}  # end of function finalise

#--------------------------
# Name: initialise
# Purpose: sets up environment, parses command line, sets up logging and parses the config file
#--------------------------
function initialise {

    local dir buf regex

    # Configure shell
    # ~~~~~~~~~~~~~~~
    export PATH=/usr/sbin:/sbin:/usr/bin:/bin
    set -o nounset
    shopt -s extglob nullglob
    umask 0022
    
    # Utility variables
    # ~~~~~~~~~~~~~~~~~
    readonly false=
    readonly my_nam=${0##*/}
    readonly true=true
    
    # Set traps
    # ~~~~~~~~~
    trap 'finalise 129' 'HUP'
    trap 'finalise 130' 'INT'
    trap 'finalise 131' 'QUIT'
    trap 'finalise 143' 'TERM'

    # Ensure being run by user docoll
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    if [[ $( id --name --user ) != 'docoll' ]]; then
        echo "This script must be run by docoll" >&2
        exit 1
    fi

    # Parse command line
    # ~~~~~~~~~~~~~~~~~~
    parse_cmdline "${@:-}"

    # Error trap command line data
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    if [[ $instance =~ / ]]; then
       msg E "Instance name may not contain a /: $instance"
       exit 1
    fi
    regex=' '
    if [[ $instance =~ $regex ]]; then
       msg E "Instance name may not contain a space: '$instance'"
       exit 1
    fi

    # Ask for user confirmation
    # ~~~~~~~~~~~~~~~~~~~~~~~~~
    prompt="Are you sure you want to remove instance $instance?"
    prompt=$prompt$'\n'"You might want to save a copy of the config files."
    prompt=$prompt$'\n'"Y to continue.  Any other reponse quits > "
    read -p "$prompt"
    [[ ${REPLY^^} != Y ]] && exit 0
        
}  # end of function initialise

#--------------------------
# Name: msg
# Purpose: generalised messaging interface
# Usage: msg class msg_text
#    class: must be one of D, E, I or W indicating Debug, Error, Information or Warning
#    msg_text: is the text of the message
# Output: information messages to stdout; the rest to stderr
# Return code: always zero
#--------------------------
function msg {

    local class message_text prefix

    class="${1:-}"
    case "$class" in 
    	D ) 
			prefix='DEBUG: '
	    	;;
    	E ) 
			prefix='ERROR: '
	    	;;
    	I ) 
			prefix=
	    	;;
    	W ) 
			prefix='WARN: '
	    	;;
	    * )
	    	msg E "$my_nam: msg: invalid class '$class': '$*'"
			class=I
    esac
    message_text="$prefix${2:-}"

	if [[ $class = I ]]; then
		echo "$message_text"
	else
		echo "$message_text" >&2
	fi

    return 0

}  #  end of function msg

#--------------------------
# Name: parse_cmdline
# Purpose: parses the command line
#--------------------------
function parse_cmdline {

    while getopts hi: opt 2>/dev/null
    do
        case $opt in
            h )
                usage verbose
                exit 0
                ;;
            i )
                instance=$OPTARG
                ;;
            * )
                emsg="$emsg"$'\n'"  Invalid option '$opt'"
        esac
    done
    
    # Error trap the option arguments
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    emsg=
    if [[ ${instance:-} = '' ]]; then
        emsg="$emsg"$'\n'"  Mandatory option -i not given or has empty argument"
    fi

    # Test for non-option arguments
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    shift $(( $OPTIND-1 ))
    if [[ $* != '' ]]; then
        emsg="$emsg"$'\n'"  Invalid non-option argument(s) '$*'"
    fi

    # Report any command line errors
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    if [[ $emsg != '' ]]; then
        msg E "Command line error(s):$emsg"
        usage
        exit 1
    fi
    
}  # end of function parse_cmdline

#--------------------------
# Name: remove_instance
# Purpose: creates the instance
#--------------------------
function remove_instance {

    local buf dir emsg i instance_fs_items instance_parent_dirs regex

    # Check the instance parent directories
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    instance_parent_dirs=(
        /etc/opt/docoll
        /opt/docoll
        /srv/docoll/
        /srv/rsync/docoll
        /var/log/docoll
        /var/opt/docoll
        /var/www/docoll
    )
    emsg=
    for (( i=0; i<${#instance_parent_dirs[*]}; i++ ))
    do
        dir=${instance_parent_dirs[i]}
        buf=$( ck_file "$dir" d:rwx: 2>&1 )
        [[ $buf != '' ]] && emsg="$emsg"$'\n'"  $dir: $buf"
    done
    if [[ $emsg != '' ]]; then
        msg E "Directory problem(s):$emsg"
        finalise 1
    fi

    # Remove the instance
    # ~~~~~~~~~~~~~~~~~~~
    emsg=
    buf=$( rm /opt/docoll/$instance 2>&1 )
    [[ $buf != '' ]] && emsg=$emsg$'\n'"  $buf"
    instance_dirs=(
        /etc/opt/docoll/$instance
        /srv/docoll/$instance
        /var/log/docoll/$instance
        /var/opt/docoll/$instance
        /var/www/docoll/$instance
    ) 
    for (( i=0; i<${#instance_dirs[*]}; i++ ))
    do  
        dir=${instance_dirs[i]}
        buf=$( rm -r "$dir" 2>&1 )
        [[ $buf != '' ]] && emsg=$emsg$'\n'"  $buf"
    done
    buf=$( dropdb docoll_$instance 2>&1 )
    [[ $buf != '' ]] && emsg=$emsg$'\n'"  $buf"
    [[ $emsg != '' ]] && msg E "During removal:$emsg"$'\n'

    msg I "Done.  You may want to manually remove /srv/rsync/docoll/$instance (it has intentionally not been removed)"

    finalise 0

}  # end of function remove_instance

#--------------------------
# Name: usage
# Purpose: prints usage message
#--------------------------
function usage {

    echo "usage: ${0##*/} [-h] -i instance_name" >&2    
    if [[ ${1:-} != 'verbose' ]]
    then
        echo "(use -h for help)" >&2
    else
        echo "  where:
    -h prints this help and exits
    -i names the instance to be created
" >&2
    fi

}  # end of function usage

#--------------------------
# Name: main
# Purpose: where it all happens
#--------------------------
initialise "${@:-}"
remove_instance    # calls finalise
