#! /usr/bin/ruby -wEUTF-8:UTF-8

# Copyright (C) 2012 Charles Atkinson
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

# Purpose: parses the docoll common configuration file for the docoll
#   bash scripts.

# Usage:
#   * See usage function or call with --help
#   * Current working directory must be the directory this script is in

require 'English'
require 'getoptlong'

require './ruby_lib'
require './Log'


# Method definitions
# ~~~~~~~~~~~~~~~~~~
# (in alphabetical order)

def Finalise( exitcode, *msg )
  # Log any optional message
  if msg.length > 0; $log.write( Log::INFO, msg[ 0 ] ) end

  # Final logging
  if $log.n_warnings > 0
    if $log.n_warnings == 1
      $log.write( Log::WARN, "There was one warning" )
    else
      $log.write( Log::WARN, "There were #{ $log.n_warnings } warnings" )
    end
    if exitcode == 0; exitcode = 1 end
  end
  if $log.n_errors > 0
    if $log.n_errors == 1
      $log.write( Log::ERROR, "There was one error" )
    else
      $log.write( Log::ERROR, "There were #{ $log.n_errors } errors" )
    end
    if exitcode == 0; exitcode = 1 end
  end
  $log.write( Log::INFO, "#{ File.basename( $0 ) }: exiting with exitcode #{ exitcode }" )
  $log.close

  # Bye!
  exit exitcode
end


def Initialise
  # Disable common traps until logging is initialised
  trap( "INT" ) { }
  trap( "HUP" ) { }
  trap( "QUIT" ) { }
  trap( "TERM" ) { }

  # Set default parameters
  InitialiseParameters( )

  # Parse command line
  # Must do now in case "--help" is given or there are any logging options
  # Save the options and arguments because GetoptsLong in ParseCommandLine will empty ARGV :-(
  opts_and_args = ARGV.join( ' ' )
  cmd_line_error_msg = ParseCommandLine( )

  # Set up logging
  now = "#{ Time.now.strftime( '%y-%m-%d@%H:%M:%S' ) }"
  if $parameters[ "LogToFile" ]
    # TODO: pass path to $Log.new when it can accept
    # TODO: error trap the File.open (which would be better in the Log class anyway)
    log_fd = File.open( $parameters[ "LogPath" ], 'a' )
    timestamps = true
  else
    log_fd = $stdout
    timestamps = false
  end
  $log = Log.new( log_fd, $parameters[ "LogLevel" ], timestamps )

  # Set common traps
  trap( "HUP" ) { Finalise( 129, "Received signal HUP" ) }
  trap( "INT" ) { Finalise( 130, "Received signal INT" ) }
  trap( "QUIT" ) { Finalise( 131, "Received signal QUIT" ) }
  trap( "TERM" ) { Finalise( 143, "Received signal TERM" ) }

  # Log startup message
  $log.write( Log::INFO, \
    "#{ File.basename( $0 ) } started at #{ now } by " + \
    "#{ $0 } #{ opts_and_args }" \
  )

  # Report any command line or config errors
  if cmd_line_error_msg != ''
    $log.write( Log::ERROR, cmd_line_error_msg )
    Usage( "not verbose" )
  end

  # Parse the config file
  config_file_error_msg = ParseConfigFile( $parameters[ "ConfigFile" ], $parameters.keys )

  if config_file_error_msg != ''
    $log.write( Log::FATAL, config_file_error_msg )
  end

  # Normalise, log and check $parameters
  NormaliseParameters( )
  LogParameters( )
  parameter_error_msg = CheckParameters( )
  if parameter_error_msg != '' 
    $log.write( Log::FATAL, "Parameter error(s):" + parameter_error_msg )
  end

  # Exit on error
  if ( cmd_line_error_msg + config_file_error_msg + parameter_error_msg ) != ''
    Finalise( 1 )
  end
end


def ParseCommandLine( )
  # Options that require an argument are marked OPTIONAL_ARGUMENT so this
  #   script can handle missing arguments itself
  opts = GetoptLong.new(
    [ '--config', GetoptLong::OPTIONAL_ARGUMENT ],
    [ '--help', '-h', GetoptLong::NO_ARGUMENT ],
    [ '--log', GetoptLong::OPTIONAL_ARGUMENT ],
    [ '--loglevel', GetoptLong::OPTIONAL_ARGUMENT ]
  )
  # Option arguments that may later be changed are duplicated
  # (strings in ARGV[ ] are frozen)
  config_option_given = false
  error_msg = ''
  opts.each \
  do |opt, arg|
    case opt
      when "--config"
        config_option_given = true
        if arg != ''
          $parameters[ "ConfigFile" ] = arg.dup
        else
          error_msg += "\n  '--config' argument missing"
        end
      when "--help"
        Usage( "verbose" )
        exit( 0 )
      when "--log"
        $parameters[ "LogToFile" ] = true
        if arg != ''
          $parameters[ "LogPath" ] = arg.dup
        end
      when "--loglevel"
        case arg
          when 'D' 
            $parameters[ "LogLevel" ] = "DEBUG"
          when 'I' 
            $parameters[ "LogLevel" ] = "INFO"
          when 'W' 
            $parameters[ "LogLevel" ] = "WARN"
          when 'E' 
            $parameters[ "LogLevel" ] = "ERROR"
          when 'F' 
            $parameters[ "LogLevel" ] = "FATAL"
          when ''
            error_msg += "\n  '--loglevel' argument missing"
          else
            error_msg += "\n  Invalid '--loglevel' argument: '#{ arg }'"
        end
      else
        error_msg += "\n  Invalid option, '#{ opt }'"
    end
  end
  if ! config_option_given
    error_msg += "\n  Mandatory option --config not given"
  end
  if ARGV.length != 0
    error_msg += "\n  Invalid argument(s) after options and their arguments: '#{ ARGV.join( ' ' ) }'"
  end
  if error_msg != ''
    error_msg = "Command line error(s):" + error_msg
  end
  return error_msg
end
  

def Usage( verbosity )
  # If logging not set up, set up default logging
  # This is required when "--help" is given on the command line
  if $log == nil
    log_fd = $stdout
    timestamps = false
    $log = Log.new( log_fd, $parameters[ "LogLevel" ], timestamps )
  end

  # Display usage
  $log.write( Log::INFO, "Usage: #{File.basename( $0 )} " + \
    "--config config_file [--help] [--log [log_file]] [--loglevel level]" \
  )
  if verbosity == "verbose"
    $log.write( Log::INFO, \
      "  --config: names the configuration file" + \
      "\n  --help: print this help message and exit" + \
      "\n  --log" + \
      "\n    log_file given: specify the log_file" + \
      "\n    log_file not given: log to the default log file" + \
      "\n  --loglevel: set lowest log level messages to log.  In order:" + \
      "\n    D for debug" + \
      "\n    I for informtion" + \
      "\n    W for warning" + \
      "\n    E for error" + \
      "\n    F for fatal" \
  )
  end
end


def WriteParametersForBash( )
  info_msg = "Parameters:"
  $parameters.each \
  do |parameter|
    info_msg += "\n  #{ parameter[ 0 ] }: #{ parameter[ 1 ] }"
  end
  puts( info_msg )
end


# Execute
# ~~~~~~~
Initialise( )
WriteParametersForBash( )
Finalise( 0 )
