#! /bin/bash

# Copyright (C) 2012 Charles Atkinson
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

# Purpose: backs up the collation and Xapian databases

# Usage: 
#    * The current working directory must be this script's directory
#    * Arguments
#      1: configuration directory.  Required
#      2: backup directory.  Required
#      3: log file.  Required when logging (no tty or $SET_HAVE_TTY_FALSE is true)
#    * Automatically outputs to log if there is no tty to log to
#    * To force output to log: export SET_HAVE_TTY_FALSE=true

# Programmers' notes: function call tree
#    +
#    |
#    +-- initialise
#    |   |
#    |   +-- parse_cfg
#    |   |
#    |   +-- parse_omindex_sh_cfg
#    |
#    +-- backup
#    |
#    +-- finalise
#
# Utility functions called from various places:
#    ck_file msg

# Function definitions in alphabetical order.  Execution begins after the last function definition.

#--------------------------
# Name: backup
# Purpose: backs up the databases
#--------------------------
function backup {
    local file
    ck_file $backup_dir d:rw: || finalise 1
    msg I "$my_nam: starting database '$db_name' backup"
    file="${backup_dir}$db_name.$( date +'%Y-%m-%d@%H:%M' )"
    export PGPASSWORD=$password
    pg_dump \
        --file="$file" \
        --format=c \
        --host=$host \
        --port=$port \
        -U $user \
        $db_name \
      2>> "$log_fn" >&2
    rc=$?
    if [[ $rc -eq 0 ]]; then
        msg I "$my_nam: database '$db_name' backup file created:"$'\n'"$( ls -l --human-readable "$file" )"
    else
        msg E "$my_nam: return code $rc from pg_dump"
    fi
    file="${backup_dir}Xapian_index.$( date +'%Y-%m-%d@%H:%M' ).tgz"
    msg I "$my_nam: starting Xapian index database backup"
    cd "$index_db_dir"
    tar -czf "$file" .
    rc=$?
    if [[ $rc -eq 0 ]]; then
        msg I "$my_nam: Xapian index backup file created from $index_db_dir:"$'\n'"$( ls -l --human-readable "$file" )"
    else
        msg E "$my_nam: return code $rc from tar"
    fi

}  # end of function backup

#--------------------------
# Name: finalise
# Purpose: final logging and gets out of here
#--------------------------
function finalise {
    local buf msg rc

    # Remove temporary file
    [[ ${parse_cfg_for_bash_rb_log_fn:-} != '' ]] && rm -f "$parse_cfg_for_bash_rb_log_fn"

    # Final logging
    # ~~~~~~~~~~~~~
    msg=
    rc=$1
    case $rc in 
        129 | 130 | 131 | 143 )
            case $rc in
                129 )
                    buf='SIGHUP'
                    ;;
                130 )
                    buf='SIGINT'
                    ;;
                131 )
                    buf='SIGQUIT'
                    ;;
                143 )
                    buf='SIGTERM'
                    ;;
            esac
            msg I "my_nam: finalising on $buf"
    esac
    if [[ $global_warning_flag ]]; then
        msg="$msg"$'\n'"  There was at least one WARNING"
    fi
    if [[ $global_error_flag ]]; then
        msg="$msg"$'\n'"  There was at least one ERROR"
    fi
    if [[ "$msg" != '' ]]; then
        [[ $rc -lt 1 ]] && rc=1
        msg I "Error and warning summary:$msg"
    fi
    msg I "$my_nam: exiting with return code $rc"
    exit $rc
}  # end of function finalise

#--------------------------
# Name: initialise
# Purpose: sets up environment, parses command line, sets up logging and parses the config file
#--------------------------
function initialise {

    local bash_lib cfg_fn emsg my_cfg_fn my_cfg_emsg omindex_cfg_fn omidex_cfg_emsg

    # Source the bash library
    # ~~~~~~~~~~~~~~~~~~~~~~~
    bash_lib=./bash_lib.sh
    source $bash_lib
    if [[ $? -ne 0 ]]; then
        echo "Unable to read the bash library, '$bash_lib'. Exiting" >&2
        exit 1
    fi
    
    # Override tty status
    # ~~~~~~~~~~~~~~~~~~~
    [[ ${SET_HAVE_TTY_FALSE:-$false} ]] && have_tty=$false
    
    # Parse command line
    # ~~~~~~~~~~~~~~~~~~
    # Has to be done now to determine log directory
    if [[ ${1:-} = '' ]]; then
        msg E "$my_nam: mandatory configuration directory argument missing or empty"
        finalise 1
    fi
    cfg_dir=${1%/}/
    if [[ ${2:-} = '' ]]; then
        msg E "$my_nam: mandatory output directory argument missing or empty"
        finalise 1
    fi
    backup_dir=${2%/}/
    if [[ ${3:-} = '' ]]; then
        msg E "$my_nam: mandatory log file name argument missing or empty"
        finalise 1
    fi
    log_fn=$3
    log_dir=${log_fn%/*}/
    [[ $log_dir = / ]] && log_dir=./
    ck_file $log_dir d:rwx: || finalise 1
    
    # Set up output redirection and logging
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    if [[ ! $have_tty ]]; then
        exec 1>>"$log_fn"
        exec 2>>"$log_fn"
    else
        exec 1>/dev/tty
        exec 2>/dev/tty
    fi
    msg I "$my_nam: started by: $0 $*"
    
    # Parse own configuration file
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    emsg=''
    my_cfg_fn=${cfg_dir}${my_nam%.*}.cfg
    parse_cfg "$my_cfg_fn"
    if [[ ${dump_file_retention:-} != '' ]]; then
    	if [[ ! $dump_file_retention =~ ^[0-9]+$ ]]; then
            emsg="$emsg"$'\n'"  'dump file retention' value '$dump_file_retention' is not an unsigned integer"
	fi
    else
        emsg="$emsg"$'\n'"  keyword 'dump file retention' not found or has no data"
    fi
    if [[ $emsg != '' ]]; then
        emsg="Problems with configuration file $my_cfg_fn:$emsg"
    fi
    my_cfg_emsg=$emsg

    # Parse omindex.sh configuration file
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    emsg=''
    omindex_cfg_fn=${cfg_dir}omindex.sh.cfg
    parse_omindex_sh_cfg $omindex_cfg_fn
    if [[ $index_db_dir = '' ]]; then
        emsg="$emsg"$'\n'"  $my_cfg_fn: keyword 'omega index database directory' has no value"
    fi
    omindex_cfg_emsg=$emsg

    # Parse the common docoll configuration file
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    emsg=
    cfg_fn=${cfg_dir}collate.cfg
    ck_file $cfg_fn f:r: || finalise 1
    now=$( date +'%y-%m-%d@%H:%M' )
    parse_cfg_for_bash_rb_log_fn=${log_dir}parse_cfg_for_bash.rb.$now.log
    buf=$( ./parse_cfg_for_bash.rb --config $cfg_fn --log $parse_cfg_for_bash_rb_log_fn 2>&1 )
    if [[ ! $buf =~ ^Parameters ]]; then
        cat $parse_cfg_for_bash_rb_log_fn
        finalise 1
    fi
    buf=$( echo "$buf" | grep 'Database: ' | sed -e 's/  Database: {"//' -e 's/[",>]//g' -e 's/}//' )
    while [[ $buf != '' ]]
    do
        case $buf in
            host=* )
                buf=${buf#host=}
                host=${buf%% *}
                ;;
            db_name=* )
                buf=${buf#db_name=}
                db_name=${buf%% *}
                ;;
            password=* )
                buf=${buf#password=}
                password=${buf%% *}
                ;;
            port=* )
                buf=${buf#port=}
                port=${buf%% *}
                ;;
            user=* )
                buf=${buf#user=}
                user=${buf%% *}
                ;;
        esac
        buf=${buf##*([^ ])*( )}
    done
    if [[ ${host:-} = '' ]]; then
        emsg="$emsg"$'\n'"  keyword 'host' not found or has no data"
    fi
    if [[ ${db_name:-} = '' ]]; then
        emsg="$emsg"$'\n'"  keyword 'db_name' not found or has no data"
    fi
    if [[ ${password:-} = '' ]]; then
        emsg="$emsg"$'\n'"  keyword 'password' not found or has no data"
    fi
    if [[ ${port:-} = '' ]]; then
        emsg="$emsg"$'\n'"  keyword 'port' not found or has no data"
    else
    	if [[ ! $port =~ ^[0-9]+$ ]]; then
            emsg="$emsg"$'\n'"  'port' is not an unsigned integer"
	fi
    fi
    if [[ ${user:-} = '' ]]; then
        emsg="$emsg"$'\n'"  keyword 'user' not found or has no data"
    fi
    if [[ $emsg != '' ]]; then
        emsg="Problems with configuration file $cfg_fn:$emsg"
    fi

    # Check directory permissions
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~
    buf=$( ck_file $index_db_dir d:rx: 2>&1 )
    if [[ $buf != '' ]]; then
        omindex_cfg_emsg="$omindex_cfg_emsg"$'\n'"  $omindex_cfg_fn: keyword 'omega index database directory': "$'\n'"    $buf"
    fi

    # Exit if any configuration errors
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    if [[ $my_cfg_emsg != '' ]]; then
        if [[ $emsg != '' ]]; then
            emsg=$'\n'$emsg
        fi
        emsg=$my_cfg_emsg$emsg
    fi
    if [[ $omindex_cfg_emsg != '' ]]; then
        if [[ $emsg != '' ]]; then
            emsg=$'\n'$emsg
        fi
        emsg=$omindex_cfg_emsg$emsg
    fi
    if [[ $emsg != '' ]]; then
        echo "$my_nam: $emsg" >&2
        exit 1
    fi

    # Log configuration values
    # ~~~~~~~~~~~~~~~~~~~~~~~~
    msg I "$my_nam: configuration values:
  host: $host
  db_name: $db_name
  password: <not logged>
  port: $port
  user: $user
  dump_file_retention: $dump_file_retention
  index_db_dir: $index_db_dir"

}  # end of function initialise

#--------------------------
# Name: parse_cfg
# Purpose: parses the configuration file
# $1 - pathname of file to parse
#--------------------------
function parse_cfg {

    local cfg_fn data keyword

    # Does the file exist?
    # ~~~~~~~~~~~~~~~~~~~~
    cfg_fn=$1
    if [[ ! -f $cfg_fn || ! -r $cfg_fn ]]; then
        echo "Configuration file '$cfg_fn' does not exist, cannot be read or is unreachable" >&2
        exit 1
    fi

    exec 3< $cfg_fn                                 # set up the config file for reading on file descriptor 3
    while read -u 3 buf                             # for each line of the config file
    do
        buf="${buf%%#*}"                            # strip any comment
        buf="${buf%%*(  )}"                         # strip any trailing spaces and tabs
        buf="${buf##*(  )}"                         # strip any leading spaces and tabs
        if [[ $buf = '' ]]; then
            continue                                # empty line
        fi
        keyword="${buf%%=*}"
        keyword="${keyword%%*([[:space:]])}"        # strip any trailing spaces and tabs
        data="${buf#*=}"
        data="${data##*([[:space:]])}"              # strip any leading spaces and tabs
        case "$keyword" in
            'dump file retention' )
                dump_file_retention=$data
                ;;
            * )
                emsg="$emsg"$'\n'"  Unrecognised keyword. '$keyword'"
                ;;
        esac

    done
    exec 3<&- # free file descriptor 3

}  # end of function parse_cfg

#--------------------------
# Name: main
# Purpose: where it all happens
#--------------------------
initialise "${@:-}"
backup
find "$backup_dir" -type f -name 'docoll*' -mtime +$dump_file_retention -delete
finalise 0
