#! /bin/bash

# rsync script created for the docoll system

# Copyright (C) 2011 Charles Atkinson
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

# Usage
#   * Details in the usage function below.
#   * To force logging to file rather than controlling terminal:
#         export SET_HAVE_TTY_FALSE=true
#   * To cancel logging to file rather than controlling terminal:
#         unset SET_HAVE_TTY_FALSE

# Function call tree
#    +
#    |
#    +-- initialise
#    |   |
#    |   +-- parse_cmdline
#    |   |   |
#    |   |   +-- usage
#    |   |
#    |   +-- parse_cfg
#    |
#    +-- ck_server
#    |
#    +-- synchronise
#    |
#    +-- finalise
#
# Utility functions called from various places: 
#    ck_file

# Functions (in alphabetical order) followed by main sequence

#--------------------------
# Name: ck_file
# Purpose: for each file listed in the argument list: checks that it is 
#   reachable, exists and that the user has the listed permissions
# Usage: ck_file [ file_name:<filetype><permission>...[a] ]
#   where 
#       file_name is a file name
#       type is b (block special file), f (file) or d (directoy)
#       permissions are one or more of r, w and x.  For example rx
#       [a] if specified, requires that file_name must be absolute (begin with /)
# Return: non-zero on error
#--------------------------
function ck_file {

    local absolute_flag buf perm perms retval filetype

    # For each file ...
    # ~~~~~~~~~~~~~~~~~
    retval=0
    while [[ $# -gt 0 ]]
    do  
        file=${1%:*}
        buf=${1#*:}
        filetype="${buf:0:1}"
        perms="${buf:1}"
        if [[ $perms =~ a$ ]]; then
            absolute_flag=$true
            perms="${perms%a}"
        else
            absolute_flag=$false
        fi  
        shift

        if [[ $file = $perms ]]; then
            echo "$my_nam: ck_file: no permisssions requested for file '$file'" >&2 
            return 1
        fi  

        # Is the file reachable and does it exist?
        # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        case $filetype in
            b ) 
                filetype='block special'
                if [[ ! -b $file ]]; then
                    echo "$filetype '$file' is unreachable, does not exist or is not a block special file" >&2
                    : $(( retval=retval+1 ))
                    continue
                fi  
                ;;  
            f ) 
                filetype=file
                if [[ ! -f $file ]]; then
                    echo "$filetype '$file' is unreachable, does not exist or is not an ordinary file" >&2
                    : $(( retval=retval+1 ))
                    continue
                fi  
                ;;  
            d ) 
                filetype=directory
                if [[ ! -d $file ]]; then
                    echo "$filetype '$file' is unreachable, does not exist or is not a directory" >&2
                    : $(( retval=retval+1 ))
                    continue
                fi
                ;;
            * )
                echo "$my_nam: ck_file: invalid filetype '$filetype' specified for '$file'" >&2
                return 1
        esac

        # Does the file have the requested permissions?
        # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        buf="$perms"
        while [[ $buf ]]
        do
            perm="${buf:0:1}"
            buf="${buf:1}"
            case $perm in
                r )
                    if [[ ! -r $file ]]; then
                        echo "$filetype '$file': no read permission" >&2
                        let retval=retval+1
                        continue
                    fi
                    ;;
                w )
                    if [[ ! -w $file ]]; then
                        echo "$filetype '$file': no write permission" >&2
                        let retval=retval+1
                        continue
                    fi
                    ;;
                x )
                    if [[ ! -x $file ]]; then
                        echo "$filetype '$file': no execute permission" >&2
                        let retval=retval+1
                        continue
                    fi
                    ;;
                * )
                    echo "$my_nam: ck_file: unexpected permisssion '$perm' requested for file '$file'" >&2
                    return 1
            esac
        done

        # Does the file have the requested absoluteness?
        # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        [[ $absolute_flag && ${file:0:1} != / ]] && { echo "$filetype '$file' does not begin with /" >&2; let retval++; }

    done

return $retval

}  #  end of function ck_file

#--------------------------
# Name: ck_server
# Purpose: checks that the rsync server is available
#--------------------------
function ck_server 
{
    local i
    for (( i=0; i<server_wait; i++ ))
    do
        ping -c1 $server_ID >/dev/null 2>&1 && return
        sleep 1
    done
    echo "Timed out waiting for rsync server to become available; exiting" >&2
    finalise 1
}

#--------------------------
# Name: finalise
# Purpose: cleans up and gets out of here
#--------------------------
function finalise 
{
    local my_rc
    my_rc=$1
    if [[ -e "$rsync_log_fn" ]]; then
        # The grep is to suppress unimportant messages
        cat "$rsync_log_fn" | grep -E -v ' skipping non-regular file '
        rm_file "$rsync_log_fn" "temporary rsync log" "$continue"
    fi
    [[ $pid_fn_written ]] && rm_file "$pid_fn" "PID file" "$continue"
    find "$log_dir" -mtime "+$log_retention" -name '*.log' -exec rm {} \;
    echo "$my_nam: Exiting at $( date +'%y-%m-%d@%H:%M' ) with return code $my_rc"
    exit $my_rc
}

#--------------------------
# Name: initialise
# Purpose: sets up environment and parses command line
#--------------------------
function initialise {

    local description lock_fn pid uint_regex

    # Configure process environment
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    export PATH=/usr/bin:/bin
    
    # Configure script environment
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    set -o nounset
    shopt -s extglob
    
    # Set utility variables
    # ~~~~~~~~~~~~~~~~~~~~~
    readonly false=
    readonly true=true

    readonly my_nam=${0##*/}
    readonly my_name=${0}
    finalise=$true
    continue=$false

    # Set configuration variables
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~
    readonly lock_wait=5        # Seconds
    cfg_fn=/etc/opt/docoll/rsync_client/rsync_client.cfg
    exclude_from=
    files_from=/etc/opt/docoll/rsync_client/files_from
    include_from=
    log_dir=/var/log/docoll/rsync_client
    log_retention=28
    password=
    pid_dir=/var/run
    server_ID=
    server_wait=300    # Cycles of ping -c1 then sleep 1
    verbose=0
    lock_fn=$pid_dir/$my_nam.lock
    
    # Determine whether there is a terminal available for output
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    # The "standard" test is to check $PS1 but this test is more reliable
    have_tty=$false
    case "$( /bin/ps -p $$ -o tty 2>&1 )" in
        *TT*-* | *TT*\?* ) 
            ;;  
        *TT* )
            have_tty=$true
    esac
    # Override
    [[ ${SET_HAVE_TTY_FALSE:-$false} ]] && have_tty=$false
    
    # Set up output redirection and logging
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    # This logs output when run without a controlling terminal, for example when 
    # run from boot script or cron, but sends it to terminal when there is one, for
    # example during interactive development
    ck_file $log_dir:drwxa || exit 1 
    timestamp=$( date '+%y-%m-%d@%H-%M-%S' )
    my_log_fn=$log_dir/$my_nam.$timestamp.log
    rsync_log_fn=$log_dir/rsync_client.$timestamp.log
    if [[ ! $have_tty ]]; then
        exec 1>>"$my_log_fn"
        exec 2>>"$my_log_fn"
    else
        exec 1>/dev/tty
        exec 2>/dev/tty
    fi
    echo "$my_nam: Starting at $timestamp"
    
    # Read configuration file
    # ~~~~~~~~~~~~~~~~~~~~~~~
    pid_fn_written=$false    # Referenced by finalise function
    ck_file "$cfg_fn:fra" || finalise 1 
    emsg=
    parse_cfg "$cfg_fn"
    
    # Error trap configuration data
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    uint_regex='^[0-9]+$'
    [[ ! $log_retention =~ $uint_regex ]] && emsg="$emsg\n  log retention is not an unsigned integer: $log_retention"
    [[ ! $password ]] && emsg="$emsg\n  password not specified"
    [[ ! $server_ID ]] && emsg="$emsg\n  server id not specified"
    [[ ! $server_wait =~ $uint_regex ]] && emsg="$emsg\n   server wait is not an unsigned integer: $server_wait"
    verbose_reported=$verbose
    case $verbose in
      0 )
        verbose=
        ;;
      1 )
        verbose=--verbose
        ;;
      2 )
        verbose='--verbose --verbose'
        ;;
      3 )
        verbose='--verbose --verbose --verbose'
        ;;
      * )
        emsg="$emsg\n  invalid verbose value '$verbose'; must be 0 to 3"
    esac
    if [[ $emsg != '' ]]; then
        echo -e "Errors in configuration file '$cfg_fn':$emsg" >&2
        finalise 1
    fi

    # Log configuration data
    # ~~~~~~~~~~~~~~~~~~~~~~
    echo "Configuration file: $cfg_fn"
    echo 'Effective configuration values:'
    echo "  exclude_from file (optional): $exclude_from"
    echo "  files_from file: $files_from"
    echo "  Lock file: $lock_fn"
    echo "  Log directory: $log_dir"
    echo "  Log retention (days): $log_retention"
    echo "  PID file directory: $pid_dir"
    echo "  Server ID: $server_ID"
    echo "  Server wait (seconds, approximate): $server_wait"
    echo "  Verbosity of rsync logging: $verbose_reported"
    
    # Check permissions
    # ~~~~~~~~~~~~~~~~~
    if [[ $exclude_from != '' ]]; then
        buf=$( ck_file "$$exclude_from:fra" 2>&1 )
        [[ $buf != '' ]] && emsg=$emsg$'\n'"  $buf"
    fi
    buf=$( ck_file "$files_from:fra" 2>&1 )
    [[ $buf != '' ]] && emsg=$emsg$'\n'"  $buf"
    buf=$( ck_file "${lock_fn%/*}:drwxa" 2>&1 )
    [[ $buf != '' ]] && emsg=$emsg$'\n'"  $buf"
    buf=$( ck_file "$log_dir:drwxa" 2>&1 )
    [[ $buf != '' ]] && emsg=$emsg$'\n'"  $buf"
    buf=$( ck_file "$pid_dir:drwxa" 2>&1 )
    [[ $buf != '' ]] && emsg=$emsg$'\n'"  $buf"
    if [[ $emsg != '' ]]; then
        echo -e "Permissions problem(s):$emsg" >&2
        finalise 1
    fi

    # Ensure directory paths have no trailing /
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    log_dir=${log_dir%%/}
    pid_dir=${pid_dir%%/}

    # Create PID file
    # ~~~~~~~~~~~~~~~
    pid_fn=$pid_dir/$my_nam.pid
    i=0
    while ! mkfifo $lock_fn 2>/dev/null    # Take lock during PID file changes
    do
        sleep 1
        if [[ $(( i++ )) -gt $lock_wait ]]; then
            description='stale lock file'
            echo "Removing $description $lock_fn" >&2
            rm_file "$lock_fn" "$description" $finalise
        fi
    done
    if [[ -e $pid_fn ]]; then
        ck_file $pid_fn:fr || finalise 1    
        read pid < $pid_fn
        case $( ps --pid $pid -o command --no-heading ) in
            "/bin/bash $my_name"* )
                echo "Another instance of $my_name already running; exiting" >&2
                description='lock file'
                rm_file "$lock_fn" "$description" "$finalise"  # Release lock
                finalise 1
                ;;
            * )
                description='stale PID file'
                echo "Removing $description $pid_fn" >&2
                rm_file "$pid_fn" "$description" "$finalise"  # Release lock
        esac
    fi
    echo $$ > $pid_fn
    pid_fn_written=$true
    description='lock file'
    rm_file "$lock_fn" "$description" "$continue"  # Release lock

}  # end of function initialise

#--------------------------
# Name: parse_cfg
# Purpose: parses the configuration file
# Input:
#    $1 - pathname of file to parse
# Output: 
#    Envalues global values according to the case statement
#    Sets emsg on finding an invalid keyword
#--------------------------
function parse_cfg {

    local buf conf_file data keyword keyword_org tab
     tab=$'\t'

    conf_file="$1"
    exec 3< $conf_file                              # set up the config file for reading on file descriptor 3
    while read -u 3 buf                             # for each line of the config file
    do
        buf="${buf%%#*}"                            # strip any comment
        buf="${buf%%*([ $tab])}"                    # strip any trailing spaces and tabs
        if [[ $buf = '' ]]; then
            continue                                # empty line
        fi
        keyword="${buf%%=*}"
        keyword="${keyword##*([ $tab])}"            # strip any leading spaces and tabs
        keyword="${keyword%%*([ $tab])}"            # strip any trailing spaces and tabs
        keyword_org=$keyword
        keyword=${keyword,,}                        # ensure lower case
        data="${buf#*=}"
        data="${data##*([ $tab])}"                  # strip any leading spaces and tabs
        case "$keyword" in
            'include_from' )
                include_from=$data
                ;;
            'exclude_from' )
                exclude_from=$data
                ;;
            'files_from' )
                files_from=$data
                ;;
            'lock file' )
                lock_fn=$data
                ;;
            'log directory' )
                log_dir=$data
                ;;
            'log retention' )
                log_retention=$data
                ;;
            'password' )
                password=$data
                ;;
            'pid directory' )
                pid_dir=$data
                ;;
            'server id' )
                server_ID=$data
                ;;
            'server wait' )
                server_wait=$data
                ;;
            'verbose' )
                verbose=$data
                ;;
            * )
                emsg="\n  Unrecognised keyword: '$keyword_org'"
                ;;
        esac

    done
    exec 3<&- # free file descriptor 3

}  # end of function parse_cfg

#--------------------------
# Name: parse_cmdline
# Purpose: parses the command line
#--------------------------
function parse_cmdline {

    while getopts hc: opt 2>/dev/null
    do
        case $opt in
            h )
                usage verbose
                exit 0
                ;;
            c )
                cfg_fn=$OPTARG
                ;;
            * )
                emsg="$emsg"$'\n'"  Invalid option '$opt'"
        esac
    done
    
    # Test for non-option arguments
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    shift $(( $OPTIND-1 ))
    if [[ $* != '' ]]; then
        emsg="$emsg"$'\n'"  Invalid non-option argument(s) '$*'"
    fi

    # Report any command line errors
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    if [[ $emsg != '' ]]; then
        echo "Command line error(s):$emsg" >&2
        usage
        exit 1
    fi
    
}  # end of function parse_cmdline

#--------------------------
# Name: rm_file
# Purpose: removes a file
# Input:
#    $1 - pathname of file to remove
#    $2 - description of file
#    $3 - flag - call finalise on failure if true
# Output: none
#--------------------------
function rm_file {
    local buf description path finalise

    path=$1
    description=$2
    finalise=$3

    buf=$( rm "$path" 2>&1 )
    if [[ $buf != '' ]]; then
        if [[ $finalise ]]; then
            echo "Failed to remove $path $description; exiting" >&2
            finalise 1
        else
            echo "Failed to remove $path $description; continuing" >&2
        fi
    fi

}  # end of function rm_file

#--------------------------
# Name: synchronise
# Purpose: synchronises selected local files to the server
#--------------------------
function synchronise {


    # Build and log the rsync command
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    if [[ $exclude_from != '' ]]; then
        exclude_from="--exclude-from $exclude_from"
    fi
    if [[ $include_from != '' ]]; then
        include_from="--include-from $include_from"
    fi
    hostname=$( hostname )
    rsync_command=( \
        nice -n 19 ionice -c2 -n0 rsync \
            --backup \
            --backup-dir "$timestamp" \
            --compress \
            --bwlimit=2048 \
            $include_from \
            $exclude_from \
            --files-from "$files_from" \
            --log-file="$rsync_log_fn" \
            --partial \
            --partial-dir=.rsync-partial \
            --prune-empty-dirs \
            --recursive \
            --relative \
            --times \
            $verbose \
            / \
            "$hostname@$server_ID::$hostname/"
    )
    echo -e "rsync command:\n===command begins===\n${rsync_command[*]}\n===command ends==="

    # Run the rsync command
    # ~~~~~~~~~~~~~~~~~~~~~
    export RSYNC_PASSWORD="$password"
    # The grep suppresses routine non-error  messages 
    "${rsync_command[@]}" | grep -E -v '^skipping non-regular file'

}  # end of function synchronise

#--------------------------
# Name: usage
# Purpose: prints usage message
#--------------------------
function usage {

    echo "usage: ${0##*/} -c conf_file [-h]" >&2    
    if [[ ${1:-} != 'verbose' ]]
    then
        echo "(use -h for help)" >&2
    else
        echo "  where:
    -c names the configuration file (default /etc/opt/docoll/rsync_client/rsync_client.cfg)
    -h prints this help and exits
" >&2
    fi

}  # end of function usage

#--------------------------
# The main sequence
#--------------------------
initialise "${@:-}"
ck_server
synchronise
find "$log_dir" -name '*.log' -mtime +30 -exec rm {} \; # Age out log files
finalise 0
