# Class CollatedFile definition

# Copyright (C) 2011 Charles Atkinson
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

class CollatedFile
  attr_reader :in_db, :inode, :mime_type, :mtime
  attr_writer :inode, :md5, :mime_type, :mtime, :path, :sha1

  # Create a new CollatedFile
  # ~~~~~~~~~~~~~~~~~~~~~~~~~
  def initialize( md5, sha1, path )
    @md5 = md5
    @sha1 = sha1

    # Search DB for a matching collated file
    @in_db, rec_inode, rec_mime_type, rec_mtime = \
      LookupCollatedFileByChecksums( @md5, @sha1 )
    if @in_db
      $log.write( Log::DEBUG, "CollatedFile: found in database" ) 
      @inode = rec_inode
      @mime_type = rec_mime_type
      @mtime = rec_mtime
    else
      $log.write( Log::DEBUG, "CollatedFile: not found in database" ) 
      @inode, @mtime = GetInodeAndMtime( path )
      if @inode == nil
        $log.write( Log::ERROR, "CollatedFile.initialize: got nil inode" )
        raise
      end
      @mime_type = GetMIMEtype( path )
      if @mime_type == nil
        $log.write( Log::ERROR, "CollatedFile.initialize: got nil mime_type" )
        raise
      end
    end
  end

  # Insert this CollatedFile into the database
  # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  def insert_into_db
    InsertCollatedFileIntoDB( @inode, @md5, @mime_type, @mtime, @sha1 )
    @in_db = true
  end

  # Delete a CollatedFile from the database
  # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  def delete_from_db( inode )
    $log.write( Log::INFO, "CollatedFile.delete_from_db: called for inode #{ inode }" )
    DeleteCollatedFileFromDB( inode )
    @in_db = false
  end

  # Update this CollatedFile in the database
  # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  def update_db
    UpdateCollatedFileInDB( @inode, @md5, @mime_type, @mtime, @sha1 )
    @in_db = true
  end

end
