#! /bin/bash

# rsync script created for the docoll system

# Copyright (C) 2011 Charles Atkinson
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

# Usage
#     * No options or arguments
#     * Requires config file /etc/opt/rsync_script/rsync.conf
#   * To force logging to file rather than controlling terminal:
#        export SET_HAVE_TTY_FALSE=true
#   * To cancel logging to file rather than controlling terminal:
#         unset SET_HAVE_TTY_FALSE

# Function call tree
#    +
#    |
#    +-- initialise
#    |   |   
#    |   +-- parse_conf
#    |
#    +-- ck_server
#    |
#    +-- synchronise
#    |
#    +-- finalise
#
# Utility functions called from various places: 
#    ck_file

# Functions (in alphabetical order) followed by main sequence

#--------------------------
# Name: ck_file
# Purpose: for each file listed in the argument list: checks that it is 
#   reachable, exists and that the user has the listed permissions
# Usage: ck_file [ file_name:<filetype><permission>...[a] ]
#   where 
#       file_name is a file name
#       type is b (block special file), f (file) or d (directoy)
#       permissions are one or more of r, w and x.  For example rx
#       [a] if specified, requires that file_name must be absolute (begin with /)
# Return: non-zero on error
#--------------------------
function ck_file {

    local absolute_flag buf perm perms retval filetype

    # For each file ...
    # ~~~~~~~~~~~~~~~~~
    retval=0
    while [[ $# -gt 0 ]]
    do  
        file=${1%:*}
        buf=${1#*:}
        filetype="${buf:0:1}"
        perms="${buf:1}"
        if [[ $perms =~ a$ ]]; then
            absolute_flag=$true
            perms="${perms%a}"
        else
            absolute_flag=$false
        fi  
        shift

        if [[ $file = $perms ]]; then
            echo "$my_nam: ck_file: no permisssions requested for file '$file'" >&2 
            return 1
        fi  

        # Is the file reachable and does it exist?
        # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        case $filetype in
            b ) 
                if [[ ! -b $file ]]; then
                    echo "file '$file' is unreachable, does not exist or is not a block special file" >&2
                    : $(( retval=retval+1 ))
                    continue
                fi  
                ;;  
            f ) 
                if [[ ! -f $file ]]; then
                    echo "file '$file' is unreachable, does not exist or is not an ordinary file" >&2
                    : $(( retval=retval+1 ))
                    continue
                fi  
                ;;  
            d ) 
                if [[ ! -d $file ]]; then
                    echo "directory '$file' is unreachable, does not exist or is not a directory" >&2
                    : $(( retval=retval+1 ))
                    continue
                fi
                ;;
            * )
                echo "$my_nam: ck_file: invalid filetype '$filetype' specified for file '$file'" >&2
                return 1
        esac

        # Does the file have the requested permissions?
        # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        buf="$perms"
        while [[ $buf ]]
        do
            perm="${buf:0:1}"
            buf="${buf:1}"
            case $perm in
                r )
                    if [[ ! -r $file ]]; then
                        echo "file '$file' does not have requested read permission" >&2
                        let retval=retval+1
                        continue
                    fi
                    ;;
                w )
                    if [[ ! -w $file ]]; then
                        echo "file '$file' does not have requested write permission" >&2
                        let retval=retval+1
                        continue
                    fi
                    ;;
                x )
                    if [[ ! -x $file ]]; then
                        echo "file '$file' does not have requested execute permission" >&2
                        let retval=retval+1
                        continue
                    fi
                    ;;
                * )
                    echo "$my_nam: ck_file: unexpected permisssion '$perm' requested for file '$file'" >&2
                    return 1
            esac
        done

        # Does the file have the requested absoluteness?
        # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        [[ $absolute_flag && ${file:0:1} != / ]] && { echo "file '$file' does not begin with /" >&2; let retval++; }

    done

return $retval

}  #  end of function ck_file

#--------------------------
# Name: ck_server
# Purpose: checks that the rsync server is available
#--------------------------
function ck_server 
{
    local i
    for (( i=0; i<server_wait; i++ ))
    do
        ping -c1 $server_ID >/dev/null 2>&1 && return
        sleep 1
    done
    echo "Timed out waiting for rsync server to become available; exiting" >&2
    finalise 1
}

#--------------------------
# Name: finalise
# Purpose: cleans up and gets out of here
#--------------------------
function finalise 
{
    my_rc=$1
    if [[ -e $rsync_log_afn ]]; then
        # The grep is to suppress unimportant messages
        cat $rsync_log_afn | grep -E -v ' skipping non-regular file '
        rm $rsync_log_afn
    fi
    [[ $pid_afn_written ]] && rm -f $pid_afn
    echo "$my_nam: Exiting at $( date +'%y-%m-%d@%H:%M' ) with return code $my_rc"
    exit $my_rc
}

#--------------------------
# Name: initialise
# Purpose: sets up environment and parses command line
#--------------------------
function initialise {
    local lock_afn i pid pid_command

    # Configure process environment
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    export PATH=/usr/bin:/bin
    
    # Configure script environment
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    set -o nounset
    shopt -s extglob
    unalias -a
    
    # Initialise variables
    # ~~~~~~~~~~~~~~~~~~~~
    # Utility variables
    readonly false=
    readonly my_nam=${0##*/}
    readonly my_name=${0}
    readonly true=true

    # Configuration variables
    readonly conf_afn=/etc/opt/rsync_script/rsync.sh.conf
    readonly log_dir=/var/opt/rsync_script/log/
    readonly pid_dir=/var/run/
    readonly lock_afn=$log_dir$my_nam.lock
    readonly lock_wait=10        # Seconds
    readonly server_wait=300    # Cycles of ping -c1 then sleep 1
    exclude_from=
    files_from=
    include_from=
    password=
    server_ID=
    verbose=
    
    # The rest
    err_msg=
    pid_afn_written=false

    # Determine whether there is a terminal available for output
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    # The "standard" test is to check $PS1 but this test is more reliable
    have_tty=$false
    case "$( /bin/ps -p $$ -o tty 2>&1 )" in
        *TT*-* | *TT*\?* ) 
            ;;  
        *TT* )
            have_tty=$true
    esac
    # Override
    [[ ${SET_HAVE_TTY_FALSE:-$false} ]] && have_tty=$false
    
    # Set up output redirection and logging
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    # This logs output when run without a controlling terminal, for example when 
    # run from boot script or cron, but sends it to terminal when there is one, for
    # example during interactive development
    ck_file $log_dir:drwxa || exit 1 
    timestamp=$( date '+%y-%m-%d@%H-%M-%S' )
    my_log_afn=$log_dir$my_nam.$timestamp.log
    rsync_log_afn=${log_dir}rsync.$timestamp.log
    if [[ ! $have_tty ]]; then
        exec 1>>"$my_log_afn"
        exec 2>>"$my_log_afn"
    else
        exec 1>/dev/tty
        exec 2>/dev/tty
    fi
    echo "$my_nam: Starting at $timestamp"
    
    # Get configuration data
    # ~~~~~~~~~~~~~~~~~~~~~~
    ck_file $conf_afn:fra || finalise 1 
    parse_conf $conf_afn 
    
    # Error trap configuration data
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    [[ ! $files_from ]] && err_msg="$err_msg\n  files_from not specified"
    hostname=$( hostname )
    [[ ! $password ]] && err_msg="$err_msg\n  password not specified"
    [[ ! $server_ID ]] && err_msg="$err_msg\n  server_ID not specified"
    verbose_reported=$verbose
    case $verbose in
      0 )
        verbose=
        ;;
      1 )
        verbose=--verbose
        ;;
      2 )
        verbose='--verbose --verbose'
        ;;
      3 )
        verbose='--verbose --verbose --verbose'
        ;;
      * )
        err_msg="$err_msg\n  invalid verbose value '$verbose'; must be 0 to 3"
    esac
    if [[ $err_msg != '' ]]; then
        echo -e "Errors in configuration file '$conf_afn':$err_msg" >&2
        finalise 1
    fi
    ck_file "$files_from":fra || finalise 1
    if [[ $exclude_from != '' ]]; then
        ck_file "$exclude_from":fra || finalise 1
    fi
    if [[ $include_from != '' ]]; then
        ck_file "$include_from":fra || finalise 1
    fi

    # Log configuration data
    # ~~~~~~~~~~~~~~~~~~~~~~
    echo "Configuration file: $conf_afn"
    echo "Exclude from file (optional): $exclude_from"
    echo "Files from file: $files_from"
    echo "Include from file (optional): $include_from"
    echo "Server ID: $server_ID"
    echo "Verbosity (optional): $verbose_reported"
    
    # Post-process configuration data
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    if [[ $exclude_from != '' ]]; then
        exclude_from="--exclude-from $exclude_from"
    fi
    if [[ $include_from != '' ]]; then
        include_from="--include-from $include_from"
    fi

    # PID file actions
    # ~~~~~~~~~~~~~~~~
    ck_file $pid_dir:drwxa || finalise 1
    pid_afn=$pid_dir$my_nam.pid
    i=0
    while ! mkfifo $lock_afn 2>/dev/null    # Take lock  
    do
        sleep 1
        if [[ $(( i++ )) -gt $lock_wait ]]; then
            echo "Timed out waiting to create lock file $lock_afn; exiting" >&2
            finalise 1
        fi
    done
    if [[ -e $pid_afn ]]; then
        ck_file $pid_afn:fr || finalise 1    
        read pid < $pid_afn
        case $( ps --pid $pid -o command --no-heading ) in
            "/bin/bash $my_name"* )
                echo "Another instance of $my_name already running; exiting" >&2
                finalise 1
                ;;
            * )
                echo "Removing stale PID file" >&2
                rm -f $pid_afn
        esac
    fi
    echo $$ > $pid_afn
    rm -f $lock_afn                            # Release lock
    pid_afn_written=true
}  # end of function initialise

#--------------------------
# Name: parse_conf
# Purpose: parses the configuration file
# Input:
#    $1 - pathname of file to parse
# Output: 
#    Envalues global values according to the case statement
#    Sets err_msg on finding an invalid keyword
#--------------------------
function parse_conf {

    local buf conf_file data keyword keyword_org tab
     tab=$'\t'

    conf_file="$1"
    exec 3< $conf_file                              # set up the config file for reading on file descriptor 3
    while read -u 3 buf                             # for each line of the config file
    do
        buf="${buf%%#*}"                            # strip any comment
        buf="${buf%%*([ $tab])}"                    # strip any trailing spaces and tabs
        if [[ $buf = '' ]]; then
            continue                                # empty line
        fi
        keyword="${buf%%=*}"
        keyword="${keyword##*([ $tab])}"            # strip any leading spaces and tabs
        keyword="${keyword%%*([ $tab])}"            # strip any trailing spaces and tabs
        keyword_org=$keyword
        keyword="${keyword//[ $tab]/}"              # remove any spaces and tabs
        keyword="$( echo -n "$keyword" | /usr/bin/tr '[:upper:]' '[:lower:]' )"
        data="${buf#*=}"
        data="${data##*([ $tab])}"                  # strip any leading spaces and tabs
        case "$keyword" in
            'include_from' )
                include_from=$data
                ;;
            'exclude_from' )
                exclude_from=$data
                ;;
            'files_from' )
                files_from=$data
                ;;
            'password' )
                password=$data
                ;;
            'server_id' )
                server_ID=$data
                ;;
            'verbose' )
                verbose=$data
                ;;
            * )
                err_msg="\n  Unrecognised keyword: '$keyword_org'"
                ;;
        esac

    done
    exec 3<&- # free file descriptor 3

}  # end of function parse_conf

#--------------------------
# Name: synchronise
# Purpose: synchronises selected local files to the server
#--------------------------
function synchronise {
    rsync_command=( \
        nice -n 19 ionice -c2 -n0 rsync \
            --backup \
            --backup-dir "$timestamp" \
            --compress \
            --bwlimit=2048 \
            $include_from \
            $exclude_from \
            --files-from "$files_from" \
            --log-file="$rsync_log_afn" \
            --partial \
            --partial-dir=.rsync-partial \
            --prune-empty-dirs \
            --recursive \
            --relative \
            --times \
            $verbose \
            / \
            "$hostname@$server_ID::$hostname/"
    )
    echo -e "rsync command:\n===command begins===\n${rsync_command[*]}\n===command ends==="

    # The grep is to suppress unimportant messages 
    export RSYNC_PASSWORD="$password"
    "${rsync_command[@]}" | grep -E -v '^skipping non-regular file'

}  # end of function synchronise

#--------------------------
# The main sequence
#--------------------------
initialise "${@:-}"
ck_server
synchronise
find "$log_dir" -name '*.log' -mtime +30 -exec rm {} \; # Age out log files
finalise 0
