' docoll Windows rsync client script

' Copyright (C) 2011 Charles Atkinson
'
' This program is free software; you can redistribute it and/or modify
' it under the terms of the GNU General Public License as published by
' the Free Software Foundation; either version 2 of the License, or
' (at your option) any later version.
'
' This program is distributed in the hope that it will be useful,
' but WITHOUT ANY WARRANTY; without even the implied warranty of
' MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
' GNU General Public License for more details.
'
' You should have received a copy of the GNU General Public License
' along with this program; if not, write to the Free Software
' Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

' Purpose: synchronises configured file trees to an rsync server

' Usage: described in "docoll Windows rsync client sysadmin guide".

' Developed and tested using Windows Script Host Version 5.6

' Notes for developers:
'	 * TODO: add MyMsgBox to check Wscript.Interactive; if True use MsgBox
'      else use Wscript.echo with only the first argument.
' 	 * Pro-forma function and subroutine header:
'      '*********************************************************
'      ' Purpose: What the procedure does (not how).
'      ' Assumptions: 
'      '     List of any external variable, control, or other element whose
'      '     state affects this procedure.
'      ' Effects:
'      '     List of the procedure's effect on each external variable, control,
'      '     or other element.
'      ' Inputs: 
'      '     Explanation of each argument that is not obvious. Each argument
'      '     should be on a separate line with inline comments.
'      ' Return Values: Explanation of the value returned (functions only).
'      '*********************************************************
' 	* Variable naming conventions
' 	  > Prefixes:
'       Script scope: s (before other prefixes)
'       Boolean: bln
'       Integer: int
'       Object: obj
'       String: str
'   * Error handling: the script tries to continue to maximise chance of
'     synchronisation

' Function and Subroutine call tree
'    +
'    |
'    +-- Initialise
'    |   |
'    |   +-- ParseArguments
'    |   |
'    |   +-- ParseCfg
'    |   |   |
'    |   |   +-- ParseSection
'    |   |   |
'    |   |   +-- ParseKey
'    |   |
'    |   +-- ChkPath
'    |
'    +-- Synchronise
'    |   |
'    |   +-- ConvertPathToCygdrive
'    |
'    +-- Finalise
'
' Functions and subroutines called from several places: FormatErrDescription, Log, MakeTimestamp, MyTrim 

Option Explicit

' Declare script scope constants
Const ALPHABET="ABCDEFGHIJKLMNOPQRSTUVWXYZ"
Const CONFIGS_EXCLUDE_FROM = 2
Const CONFIGS_FILES_FROM = 1
Const CONFIGS_GLOBAL = -1
Const CONFIGS_INVALID = -2
Const DATE_PART_ABBREVIATE = True
Const DEFAULT_CONFIG_FILE = "\Program Files\cwRsync\docoll_cwrsync.ini"
Const DEFAULT_LOG_DIR = "\Program Files\cwRsync"
Const DRIVE_LETTER = 0
Const DQUOTE = """"
Const FSO_OPEN_TEXT_FILE_FOR_READING = 1
Const R = 0

' Declare script scope variables
Private sblnDebug
Private sblnError
Private sblnWarn
Private sintLogRetention
Private sobjFSO
Private sobjLogFile
Private sobjWshShell
Private sstrComputerName
Private sstrLogFile
Private sstrLogDir
Private sstrPassword
Private sstrMyName
Private sstrSystemDrive
Private sstrServerID
Private sstrConfigs(25, 2)

' Function and Sub definitions in alphabetical order.  Execution begins after
' the last one.

'*********************************************************
' Purpose: Checks the given path exists and has the requested access
' Effects: 
'   * Adds any error mesage to strErrorMsg
' Return:
'   True if the given path exists and has the requested access
'   False otherwise
'*********************************************************
Function ChkPath(strPath, strAccess, ByRef strErrorMsg)

    Dim objFile

	' Could use FSO' File object's Attributes method but this is more direct 
	' and simpler.
    Select Case strAccess
		Case R
			On Error Resume Next
			Err.Clear
			Set objFile = sobjFSO.OpenTextFile(strPath, FSO_OPEN_TEXT_FILE_FOR_READING)
			If Err.Number = 0 Then
				ChkPath = True
				objFile.Close
			Else
				FormatErrDescription
				strErrorMsg = strErrorMsg & vbCrLf & "  Could not open " & strPath & " for reading: " & Err.Number & Err.Description
				ChkPath = False
			End If
			On Error GoTo 0
		Case Else
			Log "E", "Function Log called with invalid access of " & strClass
			ChkPath = False
    End Select

End Function

'*********************************************************
' Purpose: Converts the path to cygdrive format
' Input: path must begin with "<drive letter>:"
' Return: the converted path
'*********************************************************
Function ConvertPathToCygdrive (strPath)

	Dim strDriveLetter
	
	strDriveLetter = Ucase(Mid(strPath, 1, 1))
	strPath = Mid(strPath, 3)
	strPath = "/cygdrive/" & strDriveLetter & Replace(strPath, "\", "/")
	ConvertPathToCygdrive = strPath
	
End Function

'*********************************************************
' Purpose: cleans up and quits
'*********************************************************
Sub Finalise (intReturnValue)

	Dim objFile, objFiles, objFolder

    If sblnError Then
        Log "E", "There was at least one error"
        intReturnValue = 1
    ElseIf sblnWarn Then
        Log "W", "There was at least one warning"
        intReturnValue = 1
    End If

    ' Final message
    Log "I", "Quitting with return value " & intReturnValue 

	' Delete old log files
	If Not IsEmpty(sstrLogDir) Then
		Set objFolder = sobjFSO.GetFolder(sstrLogDir) 
		Set objFiles = objFolder.Files
		For each objFile in objFiles	
			If ( _
					LCase(Right(Cstr(objFile.Name), 3)) = "log" _
					Or LCase(Right(Cstr(objFile.Name), 7)) = "log.rtf" _ 
				) _
				And objFile.DateLastModified < (Date() - sintLogRetention) Then
				On Error Resume Next
				Err.Clear
				objFile.Delete
				If Err.Number <> 0 Then
					FormatErrDescription
					Log "E", "Could not delete old log file " & Cstr(objFile.Name) & Err.Number & Err.Description
				End If
				On Error GoTo 0
			End If 
		Next
	End If

    ' Close log file
    On Error Resume Next
    sobjLogFile.Close
    On Error GoTo 0

    WScript.Quit(intReturnValue)

End Sub

'*********************************************************
' Purpose: formats Err.Description for use in error messages
'*********************************************************
Sub FormatErrDescription ()
    If Err.Description <> Empty And Err.Description <> Null Then
        Err.Description = ": " & Err.Description
    Else
		Select Case Err.Number
			Case 52 Err.Description = ": Bad file name"
			Case 53 Err.Description = ": File not found"
			Case 70 Err.Description = ": Permission denied"
		End Select
    End If
End Sub

'*********************************************************
' Purpose: initialises the script in preparation for running Synchronise
' Actions: 
'   * Creates access objects
'   * Parses any command line arguments 
'   * Parses the configuration file
'   * Sets up logging
'*********************************************************
Sub Initialise ()

    ' Declare variables
    Dim blnChkConfig
	Dim intIdx
    Dim strBuf
    Dim strCfgFile
    Dim strDriveLetter
    Dim strErrorMsg
    Dim strFilesFrom
    Dim strExcludeFrom
    Dim strInfoMsg
    Dim strTimestamp

    ' Create access objects
    Set sobjWshShell = WScript.CreateObject("WScript.Shell")
    Set sobjFSO = CreateObject("Scripting.FileSystemObject")

    ' Set unchanging configuration variables
	sstrMyName = WScript.ScriptName
    sstrSystemDrive = sobjWshShell.ExpandEnvironmentStrings("%systemdrive%")
	
	' Parse command line
	blnChkConfig = False
	sblnDebug = False
    strCfgFile = sstrSystemDrive & DEFAULT_CONFIG_FILE
	ParseArguments strCfgFile, blnChkConfig

    ' Parse configuration file
    sintLogRetention = 28
    sstrComputerName = sobjWshShell.ExpandEnvironmentStrings("%computername%")
    sstrLogDir = sstrSystemDrive & DEFAULT_LOG_DIR
    strErrorMsg = ""
    ParseCfg strCfgFile, strErrorMsg
	If IsEmpty(sstrPassword) Then
	    strErrorMsg = strErrorMsg & vbCrLf & "  Password not set"
	End If
	If IsEmpty(sstrServerID) Then
	    strErrorMsg = strErrorMsg & vbCrLf & "  Server-ID not set"
	End If

    ' Set up logging
    sblnError = False
    sblnWarn = False
    ' TODO: if folder exists, ensure it is writeable.  Saner choide of alternative folder?
	If Not sobjFSO.FolderExists(sstrLogDir) Then
        sstrLogDir = sstrSystemDrive & "\"
        Wscript.Echo "Creating log file in " & sstrLogDir
    End If
    strTimestamp = MakeTimestamp
    sstrLogFile = sstrLogDir & "\docoll_cwrsync." & strTimestamp & ".log"
    On Error Resume Next
    Err.Clear
    Set sobjLogFile = sobjFSO.CreateTextFile(sstrLogFile)
    If Err.Number <> 0 Then
        ' Can't log anything so at least help any interactive problem investigator
        FormatErrDescription
        Wscript.Echo "Could not create log file " & sstrLogFile & ". Error " & Err.Number & Err.Description
    End If

    ' Error trap configuration data and build configuration report
    For intIdx = LBound(sstrConfigs, 1) To UBound(sstrConfigs, 1)
        strDriveLetter= sstrConfigs(intIdx, DRIVE_LETTER)
        If Not IsEMpty(strDriveLetter) Then
			strInfoMsg = strInfoMsg & vbCrLf & "  Drive " & strDriveLetter
            strFilesFrom = sstrConfigs(intIdx, CONFIGS_FILES_FROM)
            If Not IsEmpty(strFilesFrom) Then
                strInfoMsg = strInfoMsg & vbCrLf & "    files-from: " & strFilesFrom
                If Not ChkPath(strFilesFrom, R, strErrorMsg) Then
                    ' File not readable; disable synchronising this drive
                    sstrConfigs(intIdx, DRIVE_LETTER) = ""
                End If
            Else
               strErrorMsg = strErrorMsg & vbCrLf & "  No files-from keyword for drive " & strDriveLetter
            End If
            strExcludeFrom = sstrConfigs(intIdx, CONFIGS_EXCLUDE_FROM)
            If Not IsEmpty(strExcludeFrom) Then
                strInfoMsg = strInfoMsg & vbCrLf & "    exclude-from: " & strExcludeFrom
                If Not ChkPath(strExcludeFrom, R, strErrorMsg) Then
                    ' File not readable; disable synchronising this drive
                    sstrConfigs(intIdx, DRIVE_LETTER) = ""
                End If
            End If
        End If
    Next

    ' Report any configuration errors
    If strErrorMsg <> "" Then
       Log "E", "Configuration file " & strCfgFile & strErrorMsg
       Wscript.Echo "Configuration file " & strCfgFile & strErrorMsg
    End If
	
	' Quit if /V verification switch was used
	If blnChkConfig Then
		If strErrorMsg = "" Then
			Wscript.Echo "Configuration file " & strCfgFile & " is OK"
		End If
		Finalise 0
	End If

    ' Log configuration values
    Log "I", "Configuration (from " & strCfgFile & "):" & strInfoMsg

	' Debug: log contents of sstrConfigs
    If sblnDebug Then
		For intIdx = LBound(sstrConfigs, 1) To UBound(sstrConfigs, 1)
			Log "D", "sstrConfigs(" & intIdx & "): " & sstrConfigs(intIdx, 0) & ", " & sstrConfigs(intIdx, 1) & ", " & sstrConfigs(intIdx, 2) 
		Next
	End If

End Sub

'*********************************************************
' Purpose: logs messages
' Assumptions: log file is open for reading as sobjLogFile
' Inputs:
'   strClass: D, E, F, I or W for Debug, Error, Fatal, Information or Warning
'   strMsg: message text
'*********************************************************
Sub Log (strClass, strMsg)

    Dim strPrefix

    Select Case strClass
        Case "D" strPrefix = "DEBUG: " 
        Case "E" 
            strPrefix = "ERROR: " 
            sblnError = True
        Case "F" 
            strPrefix = "FATAL ERROR: " 
            sblnError = True
        Case "I" 
        Case "W" 
            strPrefix = "WARN: " 
            sblnWarn = True
        Case Else
            Log "E", "Sub Log called with invalid strClass of " & strClass
    End Select

    On Error Resume Next
    sobjLogFile.WriteLine(strPrefix & strMsg)    
    On Error GoTo 0

    If strClass = "F" Then
        Wscript.Echo strPrefix & strMsg
        Finalise 1
    End If

End Sub

'*********************************************************
' Purpose: returns a timestamp
'*********************************************************
Function MakeTimestamp ()
	MakeTimestamp = DatePart( "d", Now) _
        & "-" & MonthName(DatePart("m", Now), DATE_PART_ABBREVIATE ) _
        & "-" & DatePart( "yyyy", Now) _
        & "@" & DatePart( "h", Now) _
        & "-" & DatePart( "n", Now) _
        & "-" & DatePart( "s", Now)	
End Function

'*********************************************************
' Purpose: trims aribtrary characters
' Inputs:
'   strString: string to be trimmed
'   strStr: string of characters to trim
'*********************************************************
Function MyTrim (strString, strStr)

    Dim blnTrimmed, intIdx, strArray(), strChar
    
    ' Convert strStr to an array of single characters
    ReDim strArray(Len(strStr) - 1)
    intIdx = 1
    Do 
        strArray(intIdx - 1) = Mid(strStr, intIdx, 1)
        intIdx = intIdx + 1
    Loop While intIDx <= Len(strStr)

    ' Trim
    Do
        If strString = "" Then
		    Exit Do
		End If
		blnTrimmed = False
        For intIdx = LBound(strArray) To UBound(strArray)
            strChar = strArray(intIdx)
            If Mid(strString, 1, 1) = strChar Then
                strString = Mid(strString, 2)
                blnTrimmed = True
            End If
			If Mid(strString, Len(strString), 1) = strChar Then
                strString = Mid(strString, 1, Len(strString) - 1)
                blnTrimmed = True
            End If
        Next
    Loop While blnTrimmed

    MyTrim = strString

End Function

'*********************************************************
' Purpose: parses the command line arguments
' Effects: may set Initialise's strCfgFile and/or blnChkConfig
'*********************************************************
Sub ParseArguments(ByRef strCfgFile, ByRef blnChkConfig)

	Dim intIdx, strArgLetter, strArgValue, strErrorMsg
	
	If Wscript.Arguments.Count = 0 Then
		Exit Sub
	End If
	
	For intIdx = 0 to Wscript.Arguments.Count - 1
		strArgLetter = Mid(LCase(Wscript.Arguments(intIdx)), 1, 2)
		If Len(Wscript.Arguments(intIdx)) > 2 Then
			strArgValue = Mid(Wscript.Arguments(intIdx), 4)
		Else
		    strArgValue = ""
		End If
		Select Case strArgLetter
			Case "/v" blnChkConfig = True
			Case "/c" 
				If strCfgFile <> "" Then
					strCfgFile = strArgValue
				Else
					strErrorMsg = strErrorMsg & vbCrLf & "Switch /C requires a configuration file path"
				End If
			Case "/d"
				sblnDebug = True
			Case "/h"
				Wscript.Echo "Usage:" _
				    & vbCrLf & sstrMyName & " [/C:<configuration file>] [/H] [/V] " _
					& vbCrLf & "  /C names the configuration file (default " & sstrSystemDrive & DEFAULT_CONFIG_FILE & ")." _
					& vbCrLf & "  /H prints this help and exits." _
					& vbCrLf & "  /V validates the configuration file."
				 WScript.Quit(0)
		    Case Else
				strErrorMsg = strErrorMsg & vbCrLf & "Invalid switch " & Wscript.Arguments(intIdx) & " (switch /H prints help)"
	    End Select
    Next

    ' Report any errors
    If strErrorMsg <> "" Then
		Wscript.Echo "Command line error(s) " & strErrorMsg
		Finalise 1
	End If

End Sub

'*********************************************************
' Purpose: parses the configuration file
'*********************************************************
Sub ParseCfg (strCfgFile, ByRef strErrorMsg)

    Dim objCfgFile
    Dim strLine
    Dim iIndex

    ' Open the configuration file
    On Error Resume Next
    Err.Clear
    Set objCfgFile = sobjFSO.OpenTextFile(strCfgFile, FSO_OPEN_TEXT_FILE_FOR_READING)
    If Err.Number <> 0 Then
        FormatErrDescription
        Log "F", "Could not open configuration file " & strCfgFile & Err.Description
    End If
    On Error GoTo 0

    ' For each line ...
    iIndex = CONFIGS_INVALID
    Do Until objCfgFile.AtEndOfStream
        strLine = objCfgFile.Readline
        strLine = MyTrim(strLine, " " & vbTab)
        Select Case Left(strLine, 1)
            Case "[" iIndex = ParseSection(strLine, strErrorMsg)
            Case "", ";"  
            Case Else ParseKey iIndex, strLine, strErrorMsg 
        End Select
    Loop

    objCfgFile.Close
End Sub

'*********************************************************
' Purpose: parses a keyword line from the configuration file
' Effects:
'    Loads array sstrConfigs, member (iIndex, *) with the keyword value
'    where:
'        iIndex is 0 for A, 1 for 2 ... 25 for Z
'        *=CONFIGS_FILES_FROM: for files-from value
'        *=CONFIGS_EXCLUDE_FROM: for exclude-from value
'*********************************************************
Sub ParseKey (intIndex, strLine, ByRef strErrorMsg)

    Dim intMyIndex, strBuf, strKeyword, strValue

    ' Get the keyword and the value
    intMyIndex = InStr(strLine, "=" )
    If intMyIndex = 0 Then
        strErrorMsg = strErrorMsg & vbCrLf & "  Keyword line has no '=': " & strLine
        Exit Sub
    End If
	strBuf = Mid(strLine, 1, intMyIndex - 1)
    strKeyword = Lcase(MyTrim(strBuf, " " & vbTab))
    strBuf = Mid(strLine, intMyIndex + 1)
    strValue = MyTrim(strBuf, " " & vbTab)

    ' Store the value
    Select Case strKeyword
       Case "client-name", "client_name"
 			If intIndex <> CONFIGS_GLOBAL Then
				strErrorMsg = strErrorMsg & vbCrLf & "  Keyword Client-name found outside the Global section"
				Exit Sub
			End If
            sstrComputerName = strValue
        Case "exclude-from", "exclude_from"
 			If intIndex < 0 Then
				strErrorMsg = strErrorMsg & vbCrLf & "  Keyword Exclude-from found outside a drive letter section"
				Exit Sub
			End If
            sstrConfigs(intIndex, CONFIGS_EXCLUDE_FROM) = strValue
        Case "files-from", "files_from"
			If intIndex < 0 Then
				strErrorMsg = strErrorMsg & vbCrLf & "  Keyword Files-from found outside a drive letter section"
				Exit Sub
			End If
			sstrConfigs(intIndex, CONFIGS_FILES_FROM) = strValue
       Case "log-dir", "log_dir"
 			If intIndex <> CONFIGS_GLOBAL Then
				strErrorMsg = strErrorMsg & vbCrLf & "  Keyword Log-dir found outside the Global section"
				Exit Sub
			End If
            sstrLogDir = strValue
       Case "log-retention", "log_retention"
 			If intIndex <> CONFIGS_GLOBAL Then
				strErrorMsg = strErrorMsg & vbCrLf & "  Keyword Log-retention found outside the Global section"
				Exit Sub
			End If
            sintLogRetention = Cint(strValue)
       Case "password"
 			If intIndex <> CONFIGS_GLOBAL Then
				strErrorMsg = strErrorMsg & vbCrLf & "  Keyword Password found outside the Global section"
				Exit Sub
			End If
            sstrPassword = strValue
       Case "server-id", "server_id"
 			If intIndex <> CONFIGS_GLOBAL Then
				strErrorMsg = strErrorMsg & vbCrLf & "  Keyword Server-ID found outside the Global section"
				Exit Sub
			End If
            sstrServerID = strValue
	    Case Else
            strErrorMsg = strErrorMsg & vbCrLf & "  Keyword invalid: " & strLine
    End Select

End Sub

'*********************************************************
' Purpose: parses a section line from the configuration file
' Effects:
'    Loads array sstrConfigs, member (iIndex, DRIVE_LETTER) with the drive letter
' Return:
'    -2 on error
'    -1 on finding the [Global] section
'    Index of drive letter on finding [<drive letter>] section
'    with 0 for A, 1 for 2 ... 25 for Z
'*********************************************************
Function ParseSection (strLine, ByRef strErrorMsg)

    Dim intIndex, strBuf, strSection

    ' Get the section name
    If Mid(strLine, Len(strLine)) <> "]" Then
        strErrorMsg = strErrorMsg & vbCrLf & "  No closing ] on section " & strLine
        ParseSection = CONFIGS_INVALID
        Exit Function
    End if
    strBuf = Mid(strLine, 2)
    strBuf = Mid(strBuf, 1, Len(strBuf) - 1)
    strSection = Ucase(MyTrim(strBuf, " " & vbTab))
	If Len(strSection) = 0 Then
        strErrorMsg = strErrorMsg & vbCrLf & "  Empty section " & strLine
        ParseSection = CONFIGS_INVALID
        Exit Function
    End If

    ' Look for the [Global] section
    If strSection = "GLOBAL" Then
        ParseSection = CONFIGS_GLOBAL
        Exit Function
    End If

    ' Look for a [<drive letter>] section
    intIndex = InStr(ALPHABET, strSection) - 1
    If Len(strSection) = 1 And intIndex >= 0  Then
        sstrConfigs(intIndex, DRIVE_LETTER) = strSection
        ParseSection = intIndex
        Exit Function
    End If

    strErrorMsg = strErrorMsg & vbCrLf & "  Invalid configuration section " & strLine
    ParseSection = CONFIGS_INVALID

End Function

'*********************************************************
' Purpose: synchronises files by running cwrsync 
'*********************************************************
Function Synchronise ()

	Const EXEC_STATUS_RUNNING = 0

    Dim intIdx, intReturn
    Dim objExec
	Dim strCommand, strCwrsyncBin, strCygdriveLetter, strDriveLetter, strExcludeFrom
	Dim strExcludeFromOpt, strFilesFrom, strLogClass, strLogFile, strPasswordLine, strProcessPath, strProgramFiles
	Dim strRsyncExe, strStderr, strStdout, strTimestamp, strUserPath	
	
	' Set "constant"
	strPasswordLine = "Password: " & vbLf
	
	' Set log file path
	' .rtf is used because rsync logs with Linux line ends
	strLogFile = ConvertPathToCygdrive(sstrLogDir & "\rsync." & MakeTimestamp() & ".log.rtf")
	
	' Get environment variables
    strProgramFiles = sobjWshShell.ExpandEnvironmentStrings("%ProgramFiles%")
    strProcessPath = sobjWshShell.ExpandEnvironmentStrings("%Path%")
	strUserPath = sobjWshShell.RegRead("HKCU\Environment\PATH") 
	
	' Set environent variables for rsync.exe
	sobjWshShell.Environment("USER").Item("CWRSYNCHOME") = strProgramFiles & "\cwRsync"
	sobjWshShell.Environment("USER").Item("CWOLDPATH") = strProcessPath
	sobjWshShell.Environment("USER").Item("CYGWIN") = "nontsec"
	strCwrsyncBin = strProgramFiles & "\cwRsync\bin"
	If Instr(strProcessPath, strCwrsyncBin ) = 0 Then
		sobjWshShell.Environment("USER").Item("PATH") = strProgramFiles & "\cwRsync\bin" & ";" & strUserPath
    End If	
	' RSYNC_PASSWORD is used because the rsync --password-file option
	' cannot be used because this script is intended to be run by SYSTEM and 
	' rsync requires that the password file is only readable by "root".
	' 
	' Experimentation showed different behaviour regards passing the password 
	' to rsync, depending on how this script was run:
	' * For a Scheduled Task running as SYSTEM, the SYSTEM environment
	'   variable worked; the USER one did not.
	' * Running docoll_cwrsync.vbs from the command line, neither worked but
	'   sending the password plus CRLF on stdin worked.
	' * Running 
	'   %ComSpec% /c CScript //NoLogo "C:\Program Files\cwRsync\docoll_cwrsync.vbs"
	'   from the command line, neither worked but sending the password plus
	'   CRLF on stdin worked.
    ' Therfore, for robustness, this script both sets the SYSTEM RSYNC_PASSWORD
	' and writes the password to the rsync process' stdin.
	sobjWshShell.Environment("SYSTEM").Item("RSYNC_PASSWORD") = sstrPassword
	'sobjWshShell.Environment("USER").Item("RSYNC_PASSWORD") = sstrPassword

	' Debug: log environment variables
	If sblnDebug Then
		Dim objEnv, strBuf, strMsg
		Set objEnv = sobjWshShell.Environment("USER")
		For Each strBuf In objEnv
			strMsg = strMsg & vbCrLf & "  " & strBuf
		Next
		Log "D", "USER environment variables:" & strMsg
		strMsg = ""
		Set objEnv = sobjWshShell.Environment("SYSTEM")
		For Each strBuf In objEnv
			strMsg = strMsg & vbCrLf & "  " & strBuf
		Next
		Log "D", "SYSTEM environment variables:" & strMsg
	End If
	
	' Run rsync for each configured drive
	strRsyncExe = strProgramFiles & "\cwRsync\bin\rsync.exe"
    For intIdx = LBound(sstrConfigs, 1) To UBound(sstrConfigs, 1)
        strDriveLetter = sstrConfigs(intIdx, DRIVE_LETTER)
        If Not IsEMpty(strDriveLetter) Then
		
			' Build command string
            strCygdriveLetter = "/cygdrive/./" & strDriveLetter
			strFilesFrom = ConvertPathToCygdrive(sstrConfigs(intIdx, CONFIGS_FILES_FROM))
            strExcludeFrom = sstrConfigs(intIdx, CONFIGS_EXCLUDE_FROM)
            If IsEmpty(strExcludeFrom) Then
                strExcludeFromOpt = ""
			Else
				strExcludeFromOpt = "--exclude-from " & DQUOTE & ConvertPathToCygdrive(strExcludeFrom) & DQUOTE
            End If
			strTimestamp =  _
				DatePart( "yyyy", Now) _
				& "-" & DatePart("m", Now) _
				& "-" & DatePart("d", Now) _
				& "@" & DatePart( "h", Now) _
				& "-" & DatePart( "n", Now)			
			strCommand = DQUOTE & strRsyncExe & DQUOTE _
			    & " --backup" _ 
				& " --backup-dir=" & strTimestamp _
				& " --compress" _
				& strExcludeFromOpt _
				& " --files-from=" & DQUOTE & strFilesFrom & DQUOTE _
				& " --log-file=" & DQUOTE & strLogFile & DQUOTE _
				& " --partial" _
				& " --partial-dir=.rsync-partial" _
				& " --prune-empty-dirs" _
				& " --recursive" _
				& " --relative" _
				& " --times" _
				& " " & strCygdriveLetter _
				& " " & sstrComputerName & "@" & sstrServerID & "::" & sstrComputerName & "/" & strDriveLetter
			Log "I", strDriveLetter & " drive rsync command: " & strCommand
			
			' Start process with command
			On Error Resume Next
			Err.Clear
			Set objExec = sobjWshShell.Exec(strCommand)
			If Err.Number <> 0 Then
				On Error GoTo 0
				FormatErrDescription
				Log "E", "Error running rsync: " & Err.Number & Err.Description
				Continue
			End If
			On Error GoTo 0
			
			' Send password and read past the password prompt
			objExec.StdIn.Write sstrPassword & vbCrLf
			
			' Monitor process
			Do While objExec.Status = EXEC_STATUS_RUNNING
				Wscript.Sleep 1000    ' milli-seconds
			    strStderr = objExec.StdErr.ReadAll
				If strStderr = strPasswordLine Then
					strStderr = ""
				End If
				If (Not objExec.StdOut.AtEndOfStream) Or (strStderr <> "") Then
					Exit Do	
				End If
			Loop
			Wscript.Sleep 1000    ' Allow time for more output
			strStdout =  objExec.StdErr.ReadAll
			strStderr = strStderr & objExec.StdErr.ReadAll
			If strStdout <> "" Then
				Log "E", "Unexpected stdout from rsync: " _
					& vbCrLf & strStdOut
			End If
			If strStderr <> "" Then
			    ' Sometimes the "Password: " line is found again
				If Instr(strSTderr, strPasswordLine) > 0 Then
					strStderr = Mid(strStderr, Len(strPasswordLine) + 1) 
				End If
				Log "E", "Unexpected stderr from rsync: " _
					& vbCrLf & strStderr
			End If
			If objExec.Status = EXEC_STATUS_RUNNING Then
				objExec.Terminate
			End If
			If objExec.ExitCode = 0 Then
				strLogClass = "I"
			Else
				strLogClass = "E"
			End If
			Log strLogClass, "Exit code from rsync for " & strDriveLetter & " drive: " & objExec.ExitCode
        End If
    Next
	
	' Remove and restore environment variables
	sobjWshShell.Environment("SYSTEM").Remove("RSYNC_PASSWORD")
	sobjWshShell.Environment("USER").Remove("CWOLDPATH")
	sobjWshShell.Environment("USER").Remove("CWRSYNCHOME")
	sobjWshShell.Environment("USER").Remove("CYGWIN")
	sobjWshShell.Environment("USER").Item("PATH") = strUserPath

End Function

'*********************************************************
' Main sequence.  Execution begins here
'*********************************************************
Initialise
Synchronise
Finalise 0 
