/*****************************************************************************************/
/* Copyright 2008,2009,2010,2011,2012 Elias Potapov. */
/* Copyright 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007
   2008, 2009, 2010, 2011 The GSL Team. */
/*****************************************************************************************/
/* This file is part of DINAMICA. */

/* DINAMICA is free software: you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation, either version 3 of the License, or */
/* (at your option) any later version. */

/* DINAMICA is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */

/* You should have received a copy of the GNU General Public License */
/* along with DINAMICA.  If not, see <http://www.gnu.org/licenses/>. */
/****************************************************************************************/
/****************************************************************************************/
/* Original author is Elias Potapov <elias.potapov@gmail.com>
   Lomonosov Moscow State University, Biophysics Dep..
   Tampere University of Technology, Department of Signal Processing
   Moscow, Russia / Tampere, Finland */
/****************************************************************************************/

#include "init.h"
#include "random.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gsl/gsl_randist.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_statistics_double.h>
#include <math.h>
#define FRAME_LINE_LEN 2000
#define FRAME_N_LINES 100

void write_autocorr_file(double autocorr[],const int frame_size,
			 char *out_fname,const char *mode)
{
  int i;
  FILE *out = fopen(out_fname,mode);
  if(out == NULL)
    fprintf(stderr,"Could not open file: %s\n",out_fname);
  else{
    for(i=0;i<frame_size;i++)
      fprintf(out,"%G\n",autocorr[i]);
    fprintf(out,"\n\n");
  }
  fclose(out);
}

double *compute_autocorrelation(double **frame, const int frame_size, \
				double *autocorr, const int var_ind)
{/* This function computes the auto-correlation function for a single frame, i.e. for
    a single stochastic trajectory and for a specified variable(defined with var_ind,
    which must be in C style, i.e. starting with 0; note also that frame contains
    time in the first column). It returns the pointer to 2d array of autocorrelation
    function values for all DIM variables. The memory allocation for autocorr is done
    within the function. */
  int i;
  int eff_frame_size;/*Effective frame size, the one that the frame was allocated for*/
  if(frame_size <= FRAME_N_LINES)/* No realloc-ion took place when frame was read */
    eff_frame_size = FRAME_N_LINES;
  else
    eff_frame_size = frame_size;/* For the reallocated frame's memory */
  autocorr = (double *)malloc(frame_size*sizeof(double));
  for(i=0;i<frame_size-1;i++){/* Ignore the last value to avoid NaN */
    autocorr[i] = 
      gsl_stats_covariance(*(frame+var_ind+1)+i,1,	\
      			   *(frame+var_ind+1),1,frame_size-i);
  }
  for(i=1;i<frame_size-1;i++)/* Normalize all values except the first */
    autocorr[i] = autocorr[i] /	autocorr[0];
  //      gsl_stats_covariance(*(frame+var_ind+1),1,*(frame+var_ind+1),1,frame_size);
  autocorr[0] = autocorr[0] / autocorr[0];/* Formally... */
  return autocorr;
}

double ** load_frame(FILE *data, double **frame, int *n_pts)
{
  /*Loads the frame containing the timeseries from data. The 'data' must be
    initialized elsewhere, because this function load only segment from the file,
    separated from other segments by double '\n' indicator. This indicator is the
    convention in DINAMICA. This function should be used inside caller function which
    load the whole data file in some kind of for/while loop. It return 2d array of
    doubles and number of lines/points read.*/
  int i,k,end_frame_count = 0;
  int realloc_count = 1;
  char *symbuf,*start_symbuf,*tmp;
  char delim = ' ';
  *n_pts = 0;
  frame = (double **)malloc((DIM+1)*sizeof(double *));
  if(frame == NULL){
    fprintf(stderr,"Error: could not malloc the frame.\n");
    return NULL;
  }
  for(i=0;i<(DIM+1);i++){
    frame[i] = (double *)malloc(FRAME_N_LINES*sizeof(double));
    if(frame[i] == NULL){
      fprintf(stderr,"Error: could not malloc the frame[%d].\n",i);
      return NULL;
    }
  }
  symbuf = (char *)malloc(FRAME_LINE_LEN*sizeof(char));
  start_symbuf = symbuf;/* Store the position of symbuf */
  /* Start reading line by line */
  while((symbuf=fgets(symbuf,FRAME_LINE_LEN,data)) != NULL){
    if((*symbuf) == '#')/* Technical line in the beginning of the data */
      continue;
    if((*n_pts) == (realloc_count*FRAME_N_LINES)){
      /* We need reallocation of memory */
      realloc_count++;
      for(i=0;i<(DIM+1);i++){
	frame[i] = (double *)realloc(frame[i],
				     realloc_count*FRAME_N_LINES*sizeof(double));
	if(frame[i] == NULL){
	  fprintf(stderr,"Error: could not realloc the frame[%d].\n",i);
	  return NULL;
	}
      }
    }
    k = 0;/* Number of values read in a line */
    tmp = start_symbuf;/* Difference between symbuf and tmp determines the value */
    if(strchr(symbuf,'\n') == NULL){/* No newline means too many symbols in the line*/
      fprintf(stdout,"Warning: you have more than %d bytes in the line.\n",FRAME_LINE_LEN);
      fprintf(stdout,"Exit.\n");
      break;
    }
    *(strchr(symbuf,'\n')) = 0;/* Remove newline */
    if(*symbuf == 0){
      end_frame_count++;/* one '\n' in the symbuf */
      if(end_frame_count == 2){
	/* End of a frame, i.e. 2 '\n's occured in a row, this is assured by restoring
	   the value of end_frame_count to 0, after this if statement(see below) */
	return frame;/* Return success */
      }
      continue;/* No need to go on down the code */
    }
    end_frame_count = 0;/* Restore the value of end_frame_count */
    /* Analyze the read line */
    while(1){
      symbuf = strchr(symbuf,delim);/* Find next delimiter, end of the value */
      /* Even for the very last value in the string we have the delimiter following
	 that value (see integrator.c). This should NOT be changed then. */
      if(symbuf == NULL)/* No delimiter is found, we break */
	break;
      *symbuf = 0;/* Remove the delimiter and put "end of the string" */
      frame[k++][*n_pts] = atof(tmp);/* Get the value into the frame */
      tmp = ++symbuf;/* Put the start of the next value */
    }
    symbuf = start_symbuf;/* Restore the start position of symbuf */
    (*n_pts)++;/* Increase number of points(=lines) read */
  }
  fprintf(stdout,"Error: End of file reached. ");
  fprintf(stdout,"Frame was not terminated.\n");
  return NULL;/* not success */
}

void free_frame(double **frame)
{
  int i;
  for(i=0;i<DIM+1;i++)
    free(frame[i]);
  free(frame);
}

void write_frame(FILE *data, double **frame,const int frame_size)
{
  /* Writes the frame to the output file, the descriptor for which must be allocated
     elsewhere. frame_size is the number of points in the frame, actual points
     containing real values(returned by load_frame()). The number of values
     for each of the point is DIM+1. */
  int i,j;
  if(data == NULL)
    fprintf(stderr,"Error: file descriptor is bad\n");
  else{
    for(i=0;i<frame_size;i++){
      for(j=0;j<DIM+1;j++)
	fprintf(data,"%.5lf ",*(*(frame+j)+i));//frame[j][i]);
      fprintf(data,"\n");
    }
    fprintf(data,"\n\n");/* Separating consequent frames in the file */
  }
}

int init_rand(const long int seed_num)
{
  memset(rng_type_new,'\0',(SEED_SIZE+20));
  memset(rng_seed_new,'\0',(SEED_SIZE+20));
  memset(seed,'\0',SEED_SIZE);

  sprintf(seed,"%ld",seed_num);
  strcat(rng_seed_new,rng_seed);
  strcat(rng_seed_new,seed);
  putenv(rng_seed_new);

  strcat(rng_type_new,rng_type);
  strcat(rng_type_new,type_rnd);
  putenv(rng_type_new);

  gsl_rng_env_setup();

  T=gsl_rng_default;
  r=gsl_rng_alloc(T);
  
  return 0;
}

int free_rand()
{
  gsl_rng_free(r);
  
  return 0;
}
int random_d(const int num_points, const long int seed_num)
{
  int i,j;

  init_rand(seed_num);
  
  /*Reallocating memory...*/
  if(num_points>DIM*NUM_POI)
    u=(double *)realloc(u,num_points*sizeof(double));
  
  for(i=0;i<num_points;i++)
    u[i] = gsl_rng_uniform_pos(r);

  free_rand();
  
  return 0;
}

int random_dist(const int num_points, const long int seed_num)
{
  int i,j;

  if(distribution_type==NULL){
    fprintf(stderr,"You should provide valid string for the generator.\n");
    return 10;
  }

  if(num_points>DIM*NUM_POI)
    u=(double *)realloc(u,num_points*sizeof(double));

  for(i=0;i<num_points;i++){
    if(strcmp(distribution_type,"gauss")==0){
      u[i] = gsl_ran_gaussian(r,distribution_par[1]);
      u[i] = u[i] + distribution_par[0];/*Mean transformation.*/
    }
    else{
      u[i] = gsl_ran_ugaussian(r);/*Default behaviour.*/
    }
  }
  
  return 0;
}

int generate_seed()
{
  int i,m;
  FILE *src;
  src=fopen("/proc/uptime","r");/* We use this file for seed */
  /* File opening error */
  if(src == NULL){
    fprintf(stdout,"Error: cannot open the file.\n");
    return -12;/* This is wrong file error number */
  }
    
  i=0;
  while((m=getc(src))!=' '){
    seed[i]=(char)m;
    i++;
  }
  seed[i]='\0';/*trailing null char.*/
  for(i=0;i<strlen(seed);i++){
    if(seed[i]=='.'){/*According to the format of the /proc/uptime*/
      seed[i]=seed[i+1];
      seed[i+1]=seed[i+2];
      seed[i+2]='\0';
    }
  }
  sd=atoi(seed);/* Converting first number to integer */
  memset(seed,'\0',SEED_SIZE);/* Clean the seed string */

  fclose(src);
  
  return 0;
}

int random_init()
{
  type_rnd="ranlxd2";/*Default random generator.*/
  num_poin=10;

  seed_flag=2;/*Default: automatically computed seed.*/
  //lgv_flag=0;/*Not Langevin set of ODEs*/
  rng_seed="GSL_RNG_SEED=";
  rng_type="GSL_RNG_TYPE=";
  seed=(char *)malloc(SEED_SIZE*sizeof(char));
  rng_seed_new=(char *)malloc((SEED_SIZE+20)*sizeof(char));
  rng_type_new=(char *)malloc((SEED_SIZE+20)*sizeof(char));

  int j;

  u=(double *)malloc(DIM*NUM_POI*sizeof(double));
  /*Distribution*/
  distribution_type="gauss";/*Default*/
  distribution_n_par=2;
  distribution_par=(double *)realloc(distribution_par,distribution_n_par*sizeof(double));
  distribution_par[0]=0;/*Mean*/
  distribution_par[1]=1;/*Std. dev.*/
  /******/
  MAX_RND=(double *)malloc(DIM*sizeof(double));
  MIN_RND=(double *)malloc(DIM*sizeof(double));

  for(j=0;j<DIM;j++){
    MAX_RND[j]=100.0;
    MIN_RND[j]=0.0;
  }
  
  return 0;
	
}
