/*****************************************************************************************/
/* Copyright 2008,2009,2010,2011,2012,2013 Elias Potapov. */
/* Copyright 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007,
   2008, 2009, 2010, 2011 The GSL Team. */
/*****************************************************************************************/
/* This file is part of DINAMICA. */

/* DINAMICA is free software: you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation, either version 3 of the License, or */
/* (at your option) any later version. */

/* DINAMICA is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */

/* You should have received a copy of the GNU General Public License */
/* along with DINAMICA.  If not, see <http://www.gnu.org/licenses/>. */
/****************************************************************************************/
/****************************************************************************************/
/* Original author is Elias Potapov <elias.potapov@gmail.com>
   Lomonosov Moscow State University, Biophysics Dep..
   Tampere University of Technology, Dep. of Signal Processing.
   Moscow, Russia / Tampere, Finland */
/****************************************************************************************/
/* input_interpreter.c is the file for interpreting any command of DINAMICA typed in
   menu or any submenu. The main function here is read_menu(...) that reads any input
   from user in any menu/submenu. Other functions here are of the form
   MENU_interp(...), where MENU is appropriate menu of the program.  */

#include "init.h"
#include "continue.h"
#include "errors.h"
#include "random.h"
#include "singularity.h"
#include "lyapunov.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
//#include <plot.h>
#include <gsl/gsl_statistics_double.h>

/*This file describes interpretation of the commands*/

int read_menu(char **cmd,const char *prompt)
{
  int i,k;
  char c;
  /* Free the memory of the old calls */
  for(i=0;i<MAX_N_ARG;i++)
    memset(cmd[i],'\0',MAX_ARG_LEN);
  fprintf(stdout,"%s>",prompt);
  k = 0;/*Number of commands read*/
  while((c=getchar()) == ' ' || c == '\t');/* Ignore first spaces and tabs */
  i = 0;
  while(c!='\n'){
    if(c==' ' || c=='\t'){/*Separate words*/
      cmd[k][i++]='\0';
      k++;i=0;
      while((c=getchar()) == ' ' || c == '\t');/* Ignore additional spaces and
      tabs before end '\n' of input */
      if((k == MAX_N_ARG) && (c != '\n')){
	fprintf(stderr,"Error: max # argin exceeded. Use only max # argin=%d\n",MAX_N_ARG);
	break;
      }
      continue;
    }
    cmd[k][i++] = c;
    c = getchar();
    if(c == '\n'){/* Commands DO NOT contain trailing newlines */
        cmd[k][i] = '\0';/* Trailing zero */
	k++;/* Increase number of read arguments */
    }
  }
  return 0;
}

int main_interp(char **buffer,const int depth_level)
{
  int i,j,k;
  int Len = 50;
  char *symbuf = (char *)malloc(Len*sizeof(char));
  if(strcasecmp(buffer[depth_level],"exit") == 0 || \
     strcasecmp(buffer[depth_level],"quit") == 0 || \
     strcasecmp(buffer[depth_level],"q") == 0) {
    printf("Quiting normally...\n");
    return 1000;
  }
  else if(strcasecmp(buffer[depth_level],"ls") == 0 || \
	  strcasecmp(buffer[depth_level],"l") == 0) {
    fprintf(stdout,"MAIN menu:\n");
    fprintf(stdout,"(N)umerics/\n");
    fprintf(stdout,"(R)un*\n");
    fprintf(stdout,"(R)un(t)*\n");
    fprintf(stdout,"(R)un(i)*\n");
    fprintf(stdout,"(P)arameters*\n");
    fprintf(stdout,"(I)nitials*\n");
    fprintf(stdout,"P(e)riodics/\n");
    fprintf(stdout,"(T)rajectory/\n");
    fprintf(stdout,"(F)ile/\n");
    fprintf(stdout,"C(o)ntinue/\n");
    fprintf(stdout,"R(a)ndom/\n");
    fprintf(stdout,"(Er)rors/\n");
    fprintf(stdout,"(S)ingularity/\n");
    fprintf(stdout,"(Ly)apunov/\n");
    fprintf(stdout,"(R)un (l)inear\n");
    fprintf(stdout,"(G)raphics\n");
    return 0;
  }
  else if((strcasecmp(buffer[depth_level],"graphics") == 0) ||	\
	  (strcasecmp(buffer[depth_level],"g") == 0)){
    if(strlen(buffer[depth_level+1])!=0)
      graphics_interp(buffer,depth_level+1);
    else{
      while(1){
	read_menu(buffer,graphics_prompt);
	if((graphics_interp(buffer,0)) == 1000)
	  break; 
      }
    }
    return 0;
  }
  else if((strcasecmp(buffer[depth_level],"numerics") == 0) ||	\
	  (strcasecmp(buffer[depth_level],"n") == 0)){
    if(strlen(buffer[depth_level+1])!=0)
      numerics_interp(buffer,depth_level+1);
    else{
      while(1){
	read_menu(buffer,numerics_prompt);
	if((numerics_interp(buffer,0)) == 1000)
	  break; 
      }
    }
    return 0;
  }
  else if((strcasecmp(buffer[depth_level],"run") == 0) || \
	  (strcasecmp(buffer[depth_level],"r") == 0)){
    /*Second argument of `run' must be number of simulations for the
    same init conditions, which is useful only for stochastic
    simulations*/
    if(strlen(cmd[1])==0)
      run(mynum.method,mynum.total_time,write_flag,0,1);
    else
      run(mynum.method,mynum.total_time,write_flag,0,atoi(cmd[1]));
    return 0;
  }
  else if((strcasecmp(buffer[depth_level],"runt") == 0) || \
	  (strcasecmp(buffer[depth_level],"rt") == 0)){
    run(mynum.method,mynum.trans_time,0,1,1);
				
    return 0;
  }
  else if((strcasecmp(buffer[depth_level],"runi") == 0) || \
	  (strcasecmp(buffer[depth_level],"ri") == 0)){
    for(i=0;i<DIM;i++){
      if(xin_par[i] >= 0){/* Check the init-param link */
	xin[i] = mu.P[xin_par[i]];
	yin[i] = mu.P[xin_par[i]];
      }
      x[i] = xin[i];
      y[i] = yin[i];
    }
    if(strlen(cmd[1])==0)
      run(mynum.method,mynum.total_time,write_flag,0,1);
    else
      run(mynum.method,mynum.total_time,write_flag,0,atoi(cmd[1]));
				
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"parameters") == 0) || \
	   (strcasecmp(buffer[depth_level],"p") == 0) ) {
    get_parameter(buffer,depth_level+1);
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"initials") == 0) || \
	   (strcasecmp(buffer[depth_level],"i") == 0) ) {
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Enter filename for initials:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if((*symbuf) != 0){
      if((load_initials(symbuf)) != 0){
	fprintf(stderr,"Error occurred.\n");
	return 10;
      }
    }
    printf("Init conditions:\n");
    for(i=0;i<DIM;i++)
      printf("%s(0) = %G\n",var_name[i],xin[i]);
    printf("\n");
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"periodics") == 0) || \
	   (strcasecmp(buffer[depth_level],"e") == 0) ) {
    if(strlen(buffer[depth_level+1])!=0)
      periods_interp(buffer,depth_level+1);
    else{
      while(1){
	read_menu(cmd,periodics_prompt);
	if((periods_interp(cmd,0)) == 1000)
	  break; 
      }
    }
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"trajectory") == 0) || \
	   (strcasecmp(buffer[depth_level],"t") == 0) ) {
    if(strlen(buffer[depth_level+1])!=0)
      traj_interp(buffer,depth_level+1);
    else{
      while(1){
	read_menu(cmd,traj_prompt);
	if((traj_interp(cmd,0)) == 1000)
	  break; 
      }
    }
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"file") == 0) || \
	   (strcasecmp(buffer[depth_level],"f") == 0) ) {
    ret_num=0;
    if(strlen(buffer[depth_level+1])!=0)
      file_interp(buffer,depth_level+1);
    else{
      while(!ret_num){
	read_menu(cmd,file_prompt);
	ret_num=file_interp(cmd,0);
	if(ret_num == 1000)
	  break; 
      }
    }
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"continue") == 0) || \
	   (strcasecmp(buffer[depth_level],"o") == 0) ){
    ret_num=0;
    while(!ret_num){
      read_menu(cmd,cont_prompt);
      ret_num=cont_interp(cmd[0]);
      if(ret_num == 1000)
	break; 
    }
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"random") == 0) || \
	   (strcasecmp(buffer[depth_level],"a") == 0) ){
    ret_num=0;
    if(strlen(cmd[1])!=0)
      rand_interp(cmd[1]);
    else{
      while(!ret_num){
	read_menu(cmd,rand_prompt);
	ret_num=rand_interp(cmd[0]);
	if(ret_num == 1000)
	  break; 
      }
    }
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"errors")==0) || \
	   (strcasecmp(buffer[depth_level],"errors")==0) || \
	   (strcasecmp(buffer[depth_level],"er")==0)){
    ret_num=0;
    if(strlen(buffer[depth_level+1])!=0)
      errors_interp(buffer,depth_level+1);
    else{
      while(!ret_num){
	read_menu(cmd,errors_prompt);
	ret_num = errors_interp(cmd,0);
	if(ret_num == 1000)
	  break; 
      }
    }
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"sing")==0) || \
	   (strcasecmp(buffer[depth_level],"s")==0)){
    ret_num=0;
    if(strlen(cmd[1])!=0)
      sing_interp(cmd[1]);
    else{
      while(!ret_num){
	read_menu(cmd,sing_prompt);
	ret_num=sing_interp(cmd[0]);
	if(ret_num==1000)
	  break;
      }
    }
    return 0;
  }
  else if ( (strcasecmp(buffer[depth_level],"lyap")==0) || \
	    (strcasecmp(buffer[depth_level],"ly")==0) ){
    ret_num=0;
    while(!ret_num){
      read_menu(cmd,"lyapunov");
      ret_num=lyap_interp(cmd[0]);
      if(ret_num==1000)
	break;
    }  
    return 0;
  }
  else if (strcasecmp(buffer[depth_level],"rl")==0){
    run_lin();
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"warranty")==0) || (strcasecmp(buffer[depth_level],"w")==0) ){
    fprintf(stdout,"THERE IS NO WARRANTY FOR THE PROGRAM, TO THE EXTENT PERMITTED BY\n");
    fprintf(stdout,"APPLICABLE LAW.  EXCEPT WHEN OTHERWISE STATED IN WRITING THE COPYRIGHT\n");
    fprintf(stdout,"HOLDERS AND/OR OTHER PARTIES PROVIDE THE PROGRAM \"AS IS\" WITHOUT WARRANTY\n");
    fprintf(stdout,"OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO,\n");
    fprintf(stdout,"THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR\n");
    fprintf(stdout,"PURPOSE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE PROGRAM\n");
    fprintf(stdout,"IS WITH YOU.  SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF\n"); 
    fprintf(stdout,"ALL NECESSARY SERVICING, REPAIR OR CORRECTION.\n");
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"")==0)
    return 0;
  else {
    fprintf(stdout,"No such command\n");
    return 0;
  }

}

int traj_interp(char **buffer,const int depth_level)
{
  int i,j;
  int Len = 20;
  char *symbuf = (char *)malloc(Len*sizeof(char));
  if(strcasecmp(buffer[depth_level],"exit") == 0 ||   \
     strcasecmp(buffer[depth_level],"quit") == 0 || \
     strcasecmp(buffer[depth_level],"q") == 0){
    printf("Leaving TRAJECTORY menu...\n");
    return 1000;
  }
  if(strcasecmp(buffer[depth_level],"ls") == 0 ||	\
     strcasecmp(buffer[depth_level],"l") == 0){
    fprintf(stdout,"Trajectory menu:\n");
    fprintf(stdout,"(D)ynamical regime\n");
    fprintf(stdout,"(M)olecular distribution\n");
    return 0;
  }
  if(strcasecmp(buffer[depth_level],"regime") == 0 ||	\
     strcasecmp(buffer[depth_level],"d") == 0){
    slope(data_name);
    //traj();
    return 0;
  }
  if(strcasecmp(buffer[depth_level],"m") == 0 || \
     strcasecmp(buffer[depth_level],"mol") == 0){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      i = 10;/* Number of bins for the histogram */
    }
    else{
      i = atoi(symbuf);
      if(i<=0){
	printf("Error(# frames): set to default.\n");
	i = 10;
      }
    }
    if((check_next_arg(buffer,depth_level+1,symbuf,Len) != 0)){
      j = 1;/* Number of frames to read */
    }
    else{
      j = atoi(symbuf);
      if(j<=0){
	printf("Error(# bins): set to default.\n");
	j = 1;
      }
    }
    printf("# bins: %d\n",i);
    printf("# frames: %d\n",j);
    mol_dist(j,i);
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"") == 0)
    return 0;
  else {
    fprintf(stdout,"No such command\n");
    return 0;
  }
}

int graphics_interp(char **buffer,const int depth_level)
{
  int i,j;
  int Len = 20;
  char *symbuf = (char *)malloc(Len*sizeof(char));
  char *p;
  if(strcasecmp(buffer[depth_level], "exit") == 0 || \
     strcasecmp(buffer[depth_level],"quit") == 0 || \
     strcasecmp(buffer[depth_level],"q") == 0) {
    printf("Leaving GRAPHICS menu...\n");
    return 1000;
  }
  else if( (strcasecmp(buffer[depth_level],"ls") == 0) || \
	   (strcasecmp(buffer[depth_level],"l") == 0) ){
    fprintf(stdout,"Graphics menu:\n");
    fprintf(stdout,"(Sh)ow summary\n");
    fprintf(stdout,"(G)raphics toggle\n");
    fprintf(stdout,"Gnuplot (p)rompt\n");
    fprintf(stdout,"(X)-axis\n");
    fprintf(stdout,"(Y)-axis\n");
    fprintf(stdout,"G(r)id toggle\n");
    fprintf(stdout,"(S)end to eps\n");
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"send") == 0 ||	\
	  strcasecmp(buffer[depth_level],"s") == 0){
    fprintf(stdout,"Dump the current graph to eps.\n");
    send_to_eps();
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"show") == 0 ||	\
	  strcasecmp(buffer[depth_level],"sh") == 0){
    fprintf(stdout,"Graphics is ");
    if(graph_flag)
      fprintf(stdout,"on\n");
    else
      fprintf(stdout,"off\n");
    fprintf(stdout,"Graphics output: X11\n");
    if(graph.xInd == -1)
      fprintf(stdout,"X-axis: Time, t\n");
    else
      fprintf(stdout,"X-axis: U(%d) = %s\n",graph.xInd+1,var_name[graph.xInd]);
    i = 0;
    fprintf(stdout,"Y-axis: ");
    while(i<3){
      if(graph.yInd[i] > -1)
	fprintf(stdout,"U(%d) = %s; ",graph.yInd[i]+1,var_name[graph.yInd[i]]);
      i++;
    }
    fprintf(stdout,"\n");
    fprintf(stdout,"Grid: ");
    if(graph.grid_flag)
      fprintf(stdout,"on\n");
    else
      fprintf(stdout,"off\n");
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"g") == 0 ||	\
	  strcasecmp(buffer[depth_level],"graph") == 0){
    if(graph_flag){
      graph_flag = 0;
      fprintf(stdout,"Graphics is off\n");
    }
    else{
      graph_flag = 1;
      fprintf(stdout,"Graphics is on\n");
    }
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"grid") == 0 ||	\
	  strcasecmp(buffer[depth_level],"r") == 0){
    if(graph.grid_flag){
      graph.grid_flag = 0;
      fprintf(stdout,"Grid is off\n");
    }
    else{
      graph.grid_flag = 1;
      fprintf(stdout,"Grid is on\n");
    }
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"x") == 0 ||		\
	  strcasecmp(buffer[depth_level],"xaxis") == 0 ||	\
	  strcasecmp(buffer[depth_level],"xax") == 0){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Type var name or index (0 = t, 1 = U(1), etc.)\n");
      fprintf(stdout,"X-axis:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(strlen(symbuf) == 0){
      fprintf(stdout,"No changes.\n");
    }else {
      if(isalpha(symbuf[0])){
	if(isVar(symbuf,var_name,DIM) > -1)
	  graph.xInd = isVar(symbuf,var_name,DIM);
	else
	  fprintf(stdout,"Do not know the variable\n");
      }
      else{
	if((atoi(symbuf) > DIM) || (atoi(symbuf) < 0))
	  fprintf(stdout,"Error: out of the range\n");
	else
	  graph.xInd = atoi(symbuf) - 1;
      }
    }
    if(graph.xInd == -1)
      fprintf(stdout,"X-axis: Time, t\n");
    else
      fprintf(stdout,"X-axis: U(%d) = %s\n",graph.xInd+1,var_name[graph.xInd]);
    graph_set_labels(graph,"tsphase");
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"y") == 0 ||		\
	  strcasecmp(buffer[depth_level],"yaxis") == 0 ||	\
	  strcasecmp(buffer[depth_level],"yax") == 0){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Type var name or index (0 = disable, 1 = U(1), etc.)\n");
      fprintf(stdout,"Multiple choices can be separated by space.\n");
      fprintf(stdout,"Y-axis:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(strlen(symbuf) == 0){
      fprintf(stdout,"No changes.\n");
    }else {
      i = 0;
      while(i<3){
	if((p = strchr(symbuf,' ')) != 0){/* We can find space */
	  *p = 0;/* then terminate at space */
	}
	if(isalpha(*symbuf)){
	  if(isVar(symbuf,var_name,DIM) > -1)
	    graph.yInd[i] = isVar(symbuf,var_name,DIM);
	  else
	    fprintf(stdout,"Do not know the variable\n");
	}
	else{
	  if((atoi(symbuf) > DIM) || (atoi(symbuf) < 0))
	    fprintf(stdout,"Error: out of the range\n");
	  else{
	    /* Y-axis is not allowed to be time */
	    if(atoi(symbuf) == 0)
	      graph.yInd[i] = -2;/* Disabled */
	    else
	      graph.yInd[i] = atoi(symbuf) - 1;
	  }
	}
	if(p)/* p is not NULL */
	  symbuf = p + 1;/* Move position right after the last found space */
	else /* p is NULL */
	  break;
	i++;
      }
    }
    i = 0;
    fprintf(stdout,"Y-axis: ");
    while(i<3){
      if(graph.yInd[i] > -1)
	fprintf(stdout,"U(%d) = %s; ",graph.yInd[i]+1,var_name[graph.yInd[i]]);
      i++;
    }
    fprintf(stdout,"\n");
    graph_set_labels(graph,"tsphase");
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"p") == 0 ||		\
	  strcasecmp(buffer[depth_level],"gnuplot") == 0 ||	\
	  strcasecmp(buffer[depth_level],"prompt") == 0){
    gnuplot_interp();
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"")==0)
    return 0;
  else {
    fprintf(stdout,"No such command\n");
    return 0;
  }
}

int numerics_interp(char **buffer,const int depth_level)
{
  int i,j;
  int Len=20;
  char *symbuf=(char *)malloc(Len*sizeof(char));
  if(strcasecmp(buffer[depth_level], "exit") == 0 || \
     strcasecmp(buffer[depth_level],"quit") == 0 || \
     strcasecmp(buffer[depth_level],"q") == 0) {
    printf("Leaving NUMERICS menu...\n");
    return 1000;
  }
  else if( (strcasecmp(buffer[depth_level],"ls") == 0) || \
	   (strcasecmp(buffer[depth_level],"l") == 0) ){
    fprintf(stdout,"Numerics menu:\n");
    fprintf(stdout,"(Tt)ime\n");
    fprintf(stdout,"(Tr)time\n");
    fprintf(stdout,"(T)ra(j) transient time\n");
    fprintf(stdout,"(S)tep\n");
    fprintf(stdout,"N(p)ar\n");
    fprintf(stdout,"(B)uffer\n");
    fprintf(stdout,"Buffer (I)ncrement\n");
    fprintf(stdout,"(W)riting frequency\n");
    fprintf(stdout,"(S)ampling (f)requency\n");
    fprintf(stdout,"(M)ethod\n");

    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"ttime") == 0 || \
	  strcasecmp(buffer[depth_level],"tt") == 0) {
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Enter total time:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(strlen(symbuf) == 0){
      fprintf(stdout,"No value was stored. Total time unchanged.\n");
    }else {
      if((atof(symbuf)) < 0)
	fprintf(stdout,"Time must be positive.\n");
      else if ((atof(symbuf)) == 0)
	fprintf(stdout,"Error: must not be zero.\n");
      else
	p_mynum->total_time = atof(symbuf);
      fprintf(stdout,"Total time of integration now is %G\n",p_mynum->total_time);}
    if((p_mynum->total_time/(p_mynum->step*p_mynum->write_step)>p_mynum->global_buffer\
	&& p_mynum->write_step!=0) \
       && (strcasecmp(p_mynum->method,"run-kut4")==0)){
      fprintf(stdout,"Warning: You cannot get all points to storage array! Some of data will be lost\n");
      fprintf(stdout,"Lessen total time,increase step\n");
    }
    return 0;
  }
  else if((strcasecmp(buffer[depth_level],"trtime")==0) || \
	  (strcasecmp(buffer[depth_level],"tr") == 0)) {
   if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Enter transient time:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(strlen(symbuf) == 0){
      fprintf(stdout,"No value was stored.Transient time unchanged.\n");
    }else{
      if((atof(symbuf)) < 0)
	fprintf(stdout,"Trans time must be positive.\n");
      else if ((atof(symbuf)) == 0)
	fprintf(stdout,"Error: must not be zero.\n");
      else
	p_mynum->trans_time = atof(symbuf);
      fprintf(stdout,"Transient time of integration now is %G\n",p_mynum->trans_time);}
    if((p_mynum->trans_time/(p_mynum->step*p_mynum->write_step) > p_mynum->global_buffer \
  && p_mynum->write_step != 0) \
       && (strcasecmp(p_mynum->method,"run-kut4") == 0)){
      fprintf(stdout,"Warning: You cannot get all points to storage array! Some of data will be lost\n");
      fprintf(stdout,"Lessen transition time,increase step\n");
    }
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"tj")==0){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Enter new value:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(strlen(symbuf) == 0)
      fprintf(stdout,"No changes.\n");
    else{
      if((atoi(symbuf)) < 0)
	fprintf(stdout,"Error: must be positive.\n");
      else if ((atoi(symbuf)) == 0)
	fprintf(stdout,"Error: must not be zero.\n");
      else
	traj_trans = atoi(symbuf);
      fprintf(stdout,"Trajectory trans is %d\n",traj_trans);
    }
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"step") == 0 || \
	  strcasecmp(buffer[depth_level],"s") == 0) {
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Enter step:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(strlen(symbuf) == 0){
      fprintf(stdout,"No value was stored.Step unchanged.\n");
    }else{
      if((atof(symbuf)) < 0){
	fprintf(stdout,"Step must be positive.Sign changed.\n");
	p_mynum->step = -atof(symbuf);}
      else if((atof(symbuf)) == 0)
	fprintf(stdout,"Step must not be equal zero.Left unchanged.\n");
      else
	p_mynum->step = atof(symbuf);
      fprintf(stdout,"Step of integration now is %G\n",p_mynum->step);}
    if((p_mynum->total_time/(p_mynum->step*p_mynum->write_step) > p_mynum->global_buffer \
  && p_mynum->write_step!=0) \
       && (strcasecmp(p_mynum->method,"run-kut4") == 0)){
      fprintf(stdout,"Warning: You cannot get all points to storage array! Some of data will be lost\n");
      fprintf(stdout,"Lessen total time,increase step\n");
    }
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"buffer") == 0 || \
	  strcasecmp(buffer[depth_level],"b") == 0) {
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Enter number of points to be stored(global buffer):\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(strlen(symbuf) == 0){
      fprintf(stdout,"No value was stored.Global buffer unchanged.\n");
    }else{
      if((atoi(symbuf)) < 0)
	fprintf(stdout,"Error: less than zero.Left unchanged.\n");
      else if ((atoi(symbuf)) == 0)
	fprintf(stdout,"Error: must not be zero.\n");
      else{
	BUFFER = atoi(symbuf);
	p_mynum->global_buffer = BUFFER;}
      fprintf(stdout,"Global buffer now is %d\n",BUFFER);}
    /*Reallocating memory for storage arrays*/
    ts = (double *)realloc(ts,mynum.global_buffer*sizeof(double));
    xs = (double **)realloc(xs,mynum.global_buffer*sizeof(double *));
    for(i=0;i<mynum.global_buffer;i++)
      xs[i] = (double *)realloc(xs[i],DIM*sizeof(double));
    if((p_mynum->total_time/(p_mynum->step*p_mynum->write_step) > p_mynum->global_buffer \
  && p_mynum->write_step != 0)\
       && (strcasecmp(p_mynum->method,"run-kut4") == 0)){
      fprintf(stdout,"Warning: You cannot get all points to storage array! Some of data will be lost\n");
      fprintf(stdout,"Lessen total time,increase step\n");
    }
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"i")==0 || \
	  strcasecmp(buffer[depth_level],"increment")==0){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Enter number of points to be incremented:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(strlen(symbuf)==0){
      fprintf(stdout,"No value was stored. Buffer Increment unchanged.\n");
    }else{
      if((atoi(symbuf)) < 0)
	fprintf(stdout,"Error: less than zero.Left unchanged.\n");
      else if((atoi(symbuf)) == 0)
	fprintf(stdout,"Error: must not be zero\n");
      else
	BUF_INCR = atoi(symbuf);
      fprintf(stdout,"Buffer increment now is %d\n",BUF_INCR);
    }
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"w") == 0 || \
	  strcasecmp(buffer[depth_level],"wf") == 0) {
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Enter step for writing to the file:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(strlen(symbuf) == 0){
      fprintf(stdout,"No value was stored.\n");
    }else{
      if((atoi(symbuf)) < 0)
	fprintf(stdout,"Error: cannot be less than zero.Left unchanged.\n");
      else if((atoi(symbuf)) == 0)
	fprintf(stdout,"Error: must not be zero\n");
      else
	p_mynum->write_step = atoi(symbuf);
      fprintf(stdout,"Writing step now is %d\n",p_mynum->write_step);
    }
    if((p_mynum->total_time/(p_mynum->step*p_mynum->write_step) > p_mynum->global_buffer) \
  && p_mynum->write_step != 0 \
       && (strcasecmp(p_mynum->method,"run-kut4") == 0)){
      fprintf(stdout,"Warning: You cannot get all points to storage array! Some of data will be lost\n");
      fprintf(stdout,"Lessen total time,increase step\n");
    }
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"sf") == 0){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      printf("Enter sampling frequency of discrete simulations:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(strlen(symbuf) == 0){
      fprintf(stdout,"No value was stored. Sampling unchanged.\n");
    }else{
      if((atof(symbuf)) < 0){
	fprintf(stdout,"Cannot be less than zero.Invert.\n");
	p_mynum->smp_frq = -atof(symbuf);
      }
      else if((atof(symbuf)) == 0)
	fprintf(stdout,"Error: must not be zero\n");
      else
	p_mynum->smp_frq = atof(symbuf);
      fprintf(stdout,"Sampling frequency now is %.2lf\n",p_mynum->smp_frq);
    }
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"method") == 0 || \
	  strcasecmp(buffer[depth_level],"m") == 0) {
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"(0) Euler 2-nd order\n");
      fprintf(stdout,"(1) Runge-Kutta 4-th order\n");
      fprintf(stdout,"(2) Embedded Runge-Kutta-Fehlberg (4,5) method\n");
      fprintf(stdout,"(3) Embedded Runge-Kutta Prince-Dormand (8,9) method\n");
      fprintf(stdout,"(4) Embedded Runge-Kutta Cash-Karp (4,5) method\n");
      fprintf(stdout,"(5) Discrete\n");
      fprintf(stdout,"(6) Toggle Langevin flag\n");
      fprintf(stdout,"(7) Complex\n");
      fprintf(stdout,"(8) Implicit Bulirsch-Stoer method\n");
      fprintf(stdout,"Choose one.\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(strlen(symbuf) == 0){
      fprintf(stdout,"No value was stored. Method unchanged.\n");
    }
    else if(isalpha(*symbuf)){
      fprintf(stdout,"Should be numeric. Method unchanged.\n");
    }
    else{
      j = atoi(symbuf);
      if(j < 0){
	fprintf(stdout,"Error: less than zero. Left unchanged.\n");
      }
      if(j == 0) strcpy(p_mynum->method,"eu");
      if(j == 1) strcpy(p_mynum->method,"run-kut4");
      if(j == 2) strcpy(p_mynum->method,"rkf45");
      if(j == 3) strcpy(p_mynum->method,"rk8pd");
      if(j == 4) strcpy(p_mynum->method,"rkck");
      if(j == 5) strcpy(p_mynum->method,"discrete");
      if(j == 6) {
	if(strcmp(p_mynum->method,"discrete")==0){
	  printf("Langevin is not valid for discrete method. Left unchanged.\n");
	  return 0;
	}
	if(lang_flag==0){
	  lang_flag=1; printf("Langevin = true\n");}
	else{
	  lang_flag=0; printf("Langevin = false\n");}
      }
      if(j == 7){
	if(lang_flag==0 && (strcmp(p_mynum->method,"discrete")==0)){
	  /* Ask about deterministic method to use along with discrete
	     one */
	  printf("I will use default rkf45 method along with discrete one in this case\n");
	  printf("For other deterministic method:\n");
	  printf("1) Choose that method.\n");
	  printf("2) Toggle complex (langevin must be false).\n");
	  strcpy(p_mynum->method2,"rkf45");
	  strcpy(p_mynum->method,"complex");
	  /* Complex + lang_flag==0 => rkf45 + discrete */
	}
	else if(strcmp(p_mynum->method,"complex")==0){
	  printf("Error: method is already complex\n");
	}
	else if(lang_flag==0 && (strcmp(p_mynum->method,"discrete")!=0)){
	  /* Deterministic method is chosen, set up the discrete */
	  strcpy(p_mynum->method2,p_mynum->method);
	  strcpy(p_mynum->method,"complex");
	  /* Complex + lang_flag==0 => `current determ. method' + discrete */
	}
	else if(lang_flag!=0 && (strcmp(p_mynum->method,"discrete")==0)){
	  /* Error */
	  printf("Error: no complex option for two stochastic methods\n");
	}
	else if(lang_flag!=0 && (strcmp(p_mynum->method,"discrete")!=0)){
	  strcpy(p_mynum->method2,p_mynum->method);
	  strcpy(p_mynum->method,"complex");
	}
      }
      if(j == 8) strcpy(p_mynum->method,"bsimp");
      if(j > 8){
	fprintf(stdout,"Error: more than we have.Left unchanged.\n");
      }
      if(strcmp(p_mynum->method,"complex")!=0)
	fprintf(stdout,"Method is %s\n",p_mynum->method);
      else{
	if(lang_flag==0)
	  fprintf(stdout,"Method is %s:%s/discrete\n",p_mynum->method,p_mynum->method2);
	else
	  fprintf(stdout,"Method is %s:%s/Langevin\n",p_mynum->method,p_mynum->method2);
      }
    }
    if((p_mynum->total_time/(p_mynum->step*p_mynum->write_step) > p_mynum->global_buffer \
	&& p_mynum->write_step != 0)					\
       && (strcasecmp(p_mynum->method,"run-kut4") == 0)){
      fprintf(stdout,"Warning: You cannot get all points to storage array! Some of data will be lost\n");
      fprintf(stdout,"Lessen total time,increase step or write step,change algorithm.\n");
    }
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"show") == 0 || \
	  strcasecmp(buffer[depth_level],"sh") == 0){
    printf("Total time: %G\n",p_mynum->total_time);
    printf("Transient time: %G\n",p_mynum->trans_time);
    printf("Trajectory trans: %d\n",traj_trans);
    printf("Step: %G\n",p_mynum->step);
    printf("Dimension: %d\n",DIM);
    printf("Number of systems: %d\n",LDIM);
    printf("Number of parameters: %d\n", p_mynum->num_par);
    printf("Global buffer: %d\n",BUFFER);
    printf("Buffer increment: %d\n",BUF_INCR);
    printf("Writing frequency: %d\n",p_mynum->write_step);
    printf("Sampling frequency: %.2lf\n",p_mynum->smp_frq);
    printf("Method: %s\n",p_mynum->method);
    if(strcmp(mynum.method,"complex")==0)
      printf("Method2: %s\n",mynum.method2);
    if(lang_flag==0)
      printf("Langevin flag: false\n");
    else
      printf("Langevin flag: true\n");
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"")==0)
    return 0;
  else {
    fprintf(stdout,"No such command\n");
    return 0;
  }
}

int periods_interp(char **buffer, const int depth_level) 
{
  int i,j,k,l,m;
  double tmp;
  int Len = 20;
  char * symbuf = (char *)malloc(Len*sizeof(char));
  if(strcasecmp(buffer[depth_level],"exit") == 0 || \
     strcasecmp(buffer[depth_level],"quit") == 0 || \
     strcasecmp(buffer[depth_level],"q") == 0) {
    printf("Leaving PERIODICS menu...\n");
    return 1000;
  }
  else if( (strcasecmp(buffer[depth_level],"ls") == 0) || \
	   (strcasecmp(buffer[depth_level],"l") == 0) ){
    fprintf(stdout,"PERIODICS menu:\n");
    fprintf(stdout,"(R)e(c)ompute\n");
    fprintf(stdout,"(P)eriod\n");
    fprintf(stdout,"(A)veraged period\n");
    fprintf(stdout,"P(e)ak\n");
    fprintf(stdout,"(C)ross level\n");
    fprintf(stdout,"Period (v)ariable\n");
    fprintf(stdout,"(S)ubperiod\n");
    fprintf(stdout,"(R)atio\n");
    fprintf(stdout,"A(m)plitude\n");
    fprintf(stdout,"Me(t)hod\n");
    fprintf(stdout,"(Tm)ap\n");
    fprintf(stdout,"(Th)istogram\n");
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"rc") == 0) || \
	   (strcasecmp(buffer[depth_level],"recomp") == 0) ) {
    printf("%s will be analyzed...\n",data_name);
    if((analyze_traj(data_name)) != 0)/* Recompute common entities */
      return 10;
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"period") == 0) || \
	   (strcasecmp(buffer[depth_level],"p") == 0) ) {
    fprintf(stdout,"Periods by ");
    if(per_method == 0)
      fprintf(stdout,"Poincare sections (cross=%G) ",cross);
    if(per_method == 1)
      fprintf(stdout,"autocorrelation ");
    if(nPerDet || nPerStoch)
      fprintf(stdout,"(U(%d)=%s):\n",perVarInd+1,var_name[perVarInd]);
    else
      fprintf(stdout,"\n");
    /* Deterministic periods */
    if(nPerDet){
      fprintf(stdout,"Determ.: mean=%G, std=%G, n=%d\n",
	      gsl_stats_mean(perDet,1,nPerDet),
	      gsl_stats_sd(perDet,1,nPerDet),nPerDet);
      if(nPerDet > 30){/* Truncate the output */
	fprintf(stdout,"(Output is truncated to 30 values)\n");
	for(i=0;i<15;i++)
	  fprintf(stdout,"%G ",perDet[i]);
	fprintf(stdout,"... ");
	for(i=nPerDet-15;i<nPerDet;i++)
	  fprintf(stdout,"%G ",perDet[i]);
	fprintf(stdout,"\n");
      }
      else{
	for(i=0;i<nPerDet;i++)
	  fprintf(stdout,"%G ",perDet[i]);
	fprintf(stdout,"\n");
      }
    }
    /* Stochastic periods */
    if(nPerStoch){
      fprintf(stdout,"Stoch.: mean=%G, std=%G, n=%d\n",
	      gsl_stats_mean(perStoch,1,nPerStoch),
	      gsl_stats_sd(perStoch,1,nPerStoch),nPerStoch);
      if(nPerStoch > 30){/* Truncate the output */
	fprintf(stdout,"(Output is truncated to 30 values)\n");
	for(i=0;i<15;i++)
	  fprintf(stdout,"%G ",perStoch[i]);
	fprintf(stdout,"... ");
	for(i=nPerStoch-15;i<nPerStoch;i++)
	  fprintf(stdout,"%G ",perStoch[i]);
	fprintf(stdout,"\n");
      }
      else{
	for(i=0;i<nPerStoch;i++)
	  fprintf(stdout,"%G ",perStoch[i]);
	fprintf(stdout,"\n");
      }
    }
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"a") == 0) ) {
    printf("Obsolete for now. Maybe retrieved in the future.\n");
    /* if(numPerStoch){ */
    /*   fprintf(stdout,"Average stochastic period (from %s):\n",\ */
    /* 	      var_name[varForPerStoch]); */
    /*   tmp = 0.0; */
    /*   for(i=0;i<numPerStoch;i++) */
    /* 	tmp = tmp + perStoch[i]/numPerStoch; */
    /*   fprintf(stdout,"%G\n",tmp); */
    /* } */
    /* fprintf(stdout,"Averaged periods:\n"); */
    /* for(i=0; i < DIM; i++){ */
    /*   k=n_subperiods[i]; */
    /*   if(n_subperiods[i] == 0) k=k+1; */
    /*   fprintf(stdout,"U(%d)=%s: \n",i+1,var_name[i]); */
    /*   for(l=0;l<k;l++) */
    /* 	fprintf(stdout,"%G \n",per_aver[l][i]); */
    /*   if(k != 1) */
    /* 	fprintf(stdout,"Big period: %f \n", big_per[i]); */
    /* } */
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"subperiod") == 0) || \
	   (strcasecmp(buffer[depth_level],"s") == 0) ) {
    fprintf(stdout,"Spectra:\n");
    if(per_method == 0){
      for(k=1; k <= DIM; k++){
	fprintf(stdout,"U(%d)=%s\n",k,var_name[k-1]);
	fprintf(stdout,"Subperiods: %d\n",n_subperiods[k-1]);
	for(i=1; i <= n_isec[k-1]-2; i++){
	  fprintf(stdout,"%d ",spectrum_per[i-1][k-1]);
	  if(i == n_isec[k-1]-2) fprintf(stdout,"\n");
	}
      }
    }
    if(per_method == 1){
      for(k=1; k <= DIM; k++){
	fprintf(stdout,"U(%d)=%s\n",k,var_name[k-1]);
	fprintf(stdout,"Subperiods: %d\n",n_subperiods[k-1]);
	for(i=1; i <= n_peaks[k-1]-2; i++){
	  fprintf(stdout,"%d ",spectrum_per[i-1][k-1]);
	  if(i == n_peaks[k-1]-2) fprintf(stdout,"\n");
	}
      }
    }
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"peak") == 0) || \
	   (strcasecmp(buffer[depth_level],"e") == 0) ) {
    printf("Obsolete for now. Maybe retrieved in the future.\n");
    /* fprintf(stdout,"Peaks:\n"); */
    /* for(k=1; k <= DIM; k++){ */
    /*   fprintf(stdout,"U(%d)=%s (%d peaks):\n",k,var_name[k-1],n_peaks[k-1]); */
    /*   fprintf(stdout,"Time\t\tPeak height\n"); */
    /*   for(j=1;j <= n_peaks[k-1];j++){ */
    /* 	fprintf(stdout,"%G\t\t%G\n",t_peak[j-1][k-1],x_peak[j-1][k-1]); */
    /*   } */
    /* } */
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"cross") == 0) ||	\
	   (strcasecmp(buffer[depth_level],"c") == 0) ) {
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Cross level = %G\n",cross_level);
      fprintf(stdout,"[Poincare section = (max - min)/cross_level + min]\n");
      fprintf(stdout,"Enter cross level:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(*symbuf == 0){
      fprintf(stdout,"No change.\n");
    }
    else{
      cross_level = atof(symbuf);
      fprintf(stdout,"Cross level = %G\n",cross_level);
    }
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"v") == 0) ||	\
	   (strcasecmp(buffer[depth_level],"pv") == 0) ) {
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Periods are calculated for the variable.\n");
      fprintf(stdout,"Choose 0 for automatic detection.\n");
      fprintf(stdout,"Period variable = ");
      if(per_var == -1)
	fprintf(stdout,"automatic (max amplitude var)\n");
      else
	fprintf(stdout,"%s\n",var_name[per_var]);
      fprintf(stdout,"Enter period variable:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(*symbuf == 0){
      fprintf(stdout,"No change.\n");
    }
    else{
      if(isalpha(*symbuf))
	per_var = isVar(symbuf,var_name,DIM);
      else
	per_var = atoi(symbuf) - 1;
      if(per_var < -1 || per_var > (DIM-1)){
	fprintf(stderr,"Error: wrong input. Left default.\n");
	per_var = -1;
      }
      fprintf(stdout,"Period variable = ");
      if(per_var == -1)
	fprintf(stdout,"automatic(max amplitude chosen)\n");
      else
	fprintf(stdout,"%s\n",var_name[per_var]);
    }
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"ratio") == 0) || \
	   (strcasecmp(buffer[depth_level],"r") == 0) ) {
    fprintf(stdout,"Ratio:\n");
    for(i=0;i<DIM;i++){
      fprintf(stdout,"U(%d)=%s: ",i+1,var_name[i]);
      fprintf(stdout,"%lf\n",per_ratio[i]);
    }
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"amplitude") == 0) || \
	   (strcasecmp(buffer[depth_level],"m") == 0) ) {
    fprintf(stdout,"Var.\t\tAmpl.\t\t\tMax\t\t\tMin\n");
    for(i=0;i<DIM;i++){
      fprintf(stdout,"U(%d)=%s\t\t%G\t\t%G\t\t%G\n",i+1,var_name[i],ampl[i],max_x[i],min_x[i]);
    }
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"method") == 0) || \
	   (strcasecmp(buffer[depth_level],"t") == 0) ){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Choose the method for period calculation:\n");
      if(per_method == 0){
	fprintf(stdout,"(0)By Poincare section(default)*\n");
	fprintf(stdout,"(1)By auto-correlation\n");
      }
      if(per_method == 1){
	fprintf(stdout,"(0)By Poincare section(default)\n");
	fprintf(stdout,"(1)By auto-correlation*\n");
      }
      symbuf = read_input_line(symbuf,Len);
    }
    if(*symbuf == 0){
      fprintf(stdout,"No changes.\n");
    }else{
      i = per_method;
      per_method = atoi(symbuf);
      if((per_method != 0) && (per_method != 1)){
	fprintf(stdout,"Wrong choice! Unchanged.\n");
	per_method = i;
      }
      printf("Method is: ");
      if(per_method == 0) printf("section\n");
      if(per_method == 1) printf("auto-correlation\n");
    }
    return 0;
  }
  else if ((strcasecmp(buffer[depth_level],"tmap")==0) || \
	   (strcasecmp(buffer[depth_level],"tm")==0)){
    int iter[DIM];
    fprintf(stdout,"Total number of periods are:\n");
    if(per_method == 0){
      for(k=0;k<DIM;k++){
	fprintf(stdout,"%d ",n_isec[k]-1);
	iter[k]=n_isec[k]-1;
      }
      fprintf(stdout,"\n");
    }
    if(per_method == 1){
      for(k=0;k<DIM;k++){
	fprintf(stdout,"%d ",n_peaks[k]-1);
	iter[k]=n_peaks[k]-1;
      }
      fprintf(stdout,"\n");
    }
    fprintf(stdout,"Output file name is `tmap'\n");
    FILE *ftmap;
    ftmap=fopen("tmap","w");
    for(i=0;i<DIM;i++){
      for(k=0;k<iter[i]-1;k++)
	fprintf(ftmap,"%f %f\n",per[k][i],per[k+1][i]);
      fprintf(ftmap,"\n\n");
    }
    fclose(ftmap);
    return 0;
  }
  else if ((strcasecmp(buffer[depth_level],"th")==0) || \
	   (strcasecmp(buffer[depth_level],"thist")==0) || \
	   (strcasecmp(buffer[depth_level],"thistogram")==0)){
    if(strlen(buffer[depth_level+1]) != 0){
      periodics_hist_interp(buffer,depth_level+1);
    }
    else{
      while(1){
	read_menu(buffer,periodics_hist_prompt);
	if(periodics_hist_interp(buffer,0) == 1000)
	  break;
      }
    }
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"")==0)
    return 0;
  else {
    fprintf(stdout,"No such command\n");
    return 0;
  }
}

int file_interp(char **buffer, const int depth_level)
{
  int i,j,*info;
  FILE *fname;
  int Len = 100;//filename can be long with directory specification
  char *symbuf = (char *)malloc(Len*sizeof(char));
  if(strcasecmp(buffer[depth_level], "exit") == 0 || strcasecmp(buffer[depth_level],"quit") == 0 || strcasecmp(buffer[depth_level],"q") == 0) {
    printf("Leaving FILE menu...\n");
    return 1000;
  }
  else if( (strcasecmp(buffer[depth_level],"ls") == 0) || \
	   (strcasecmp(buffer[depth_level],"l") == 0) ){
    fprintf(stdout,"FILE menu:\n");
    fprintf(stdout,"(S)av(i)n - save current point to initials\n");
    fprintf(stdout,"(S)av(d)at - write current data,trajectory file\n");
    fprintf(stdout,"(L)oad (d)ata - load data file, recompute common entities\n");
    fprintf(stdout,"(C)onvert - convert data file\n");
    fprintf(stdout,"(D)ata file - change data file to write for next runs\n");
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"lf")==0){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Current file to look into: %s\n",data_name);
      fprintf(stdout,"How many frames to load:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(*symbuf == 0){/* No input */
      fprintf(stdout,"No changes\n");
    }
    else{
      double **frame;
      int size_frame;
      FILE *out = fopen("frame","w");
      fname = fopen(data_name,"r");
      for(i=0;i<atoi(symbuf);i++){
	frame = load_frame(fname,frame,&size_frame);
	if(frame == NULL)
	  fprintf(stderr,"Perhaps you want more frames than there exist\n");
	write_frame(out,frame,size_frame);
      }
      free(frame);
      fclose(fname);
      fclose(out);
    }
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"d")==0 || \
	  strcasecmp(buffer[depth_level],"data")==0){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Current file name: %s\n",data_name);
      fprintf(stdout,"Enter new name:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(strlen(symbuf) == 0){
      fprintf(stdout,"No changes\n");
      fprintf(stdout,"Data file: %s\n",data_name);
    }
    else{
      if(Len < FNAME_SIZE)
      	strncpy(data_name,symbuf,Len);
      else
      	strncpy(data_name,symbuf,FNAME_SIZE);
      fprintf(stdout,"New data file: %s\n",data_name);
    }
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"c")==0 || \
	  strcasecmp(buffer[depth_level],"convert")==0){
    if(strlen(buffer[depth_level+1])!=0)
      convert_data_file(data_name,1);
    else
      convert_data_file(data_name,1);
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"savin")==0) || \
	   (strcasecmp(buffer[depth_level],"si") == 0)) {
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Filename for initials:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    fname = fopen(symbuf,"w");
    for(i=0;i<DIM;i++)
      fprintf(fname,"%lf\n",x[i]);
    fclose(fname);
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"savdat")==0) || \
	   (strcasecmp(buffer[depth_level],"sd")==0) ){
    fprintf(stdout,"Obsolete for now. Maybe retrieved in the future.\n");
    /* if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){ */
    /*   fprintf(stdout,"Filename for data:\n");
	 symbuf = read_input_line(symbuf,Len);*/
    /* } */
    /* if(strlen(symbuf) == 0){ */
    /*   fprintf(stdout,"Nothing was written\n"); */
    /* }else{ */
    /*   fname = fopen(symbuf,"w"); */
    /*   for(j=0;j<write_count;j++){ */
    /* 	fprintf(fname,"%.5lf ",ts[j]); */
    /* 	for(i=0;i<DIM;i++){ */
    /* 	  fprintf(fname,"%.5lf ",xs[j][i]); */
    /* 	} */
    /* 	fprintf(fname,"\n"); */
    /*   } */
    /*   fclose(fname); */
    /* } */
    return 0;
  }
  else if( (strcasecmp(buffer[depth_level],"ld")==0) || \
	   (strcasecmp(buffer[depth_level],"load")==0) ) {
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Data file to load:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(strlen(symbuf) == 0){
      fprintf(stdout,"Nothing was read\n");
    }else{
      if((analyze_traj(symbuf)) != 0)/* Recompute common entities */
	return 10;
      if(graph_flag){/* Plot the data */
	fname = fopen(symbuf,"r");
	if((info=get_info_data(info,fname)) == NULL){
	  fprintf(stderr,"Error: could not get data info\n");
	}
	fclose(fname);
	if(info[0]){//method is complex
	  /* First, plot determ trajectory */
	  gplot_results(symbuf,1,"rkf45");//any determ method, not "discrete"
	  /* Plot stoch trajectory */
	  if(!info[3])//lang_flag
	    gplot_results(symbuf,2,"discrete");
	  else
	    gplot_results(symbuf,2,"rkf45");//any determ method, not "discrete"
	}
	else{//method is not complex
	  if((!info[3]) && (!info[2]))//lang_flag and pure_det are false
	    gplot_results(symbuf,0,"discrete");//...then discrete method
	  else
	    gplot_results(symbuf,0,"rkf45");//determ or langevin alone...
	}
	fprintf(stdout,"Graph is updated.\n");
      }
    }
    return 0;
  }
  else if (strcasecmp(buffer[depth_level],"")==0)
    return 0;
  else{
    fprintf(stdout,"No such command\n");
    return 0;
  }
}

int cont_interp(char *buffer)
{
  int i,j,k;
  char *inpts_name;
  inpts_name=(char *)malloc(50*sizeof(char));
  if(strcasecmp(buffer, "exit") == 0 || strcasecmp(buffer,"quit") == 0 || strcasecmp(buffer,"q") == 0) {
    printf("Leaving CONTINUE menu...\n");
    return 1000;
  }
  else if( (strcasecmp(buffer,"ls") == 0) || (strcasecmp(buffer,"l") == 0) ){
    fprintf(stdout,"CONTINUE menu:\n");
    fprintf(stdout,"(P)arameters\n");
    fprintf(stdout,"(P)roblem (p)arameters\n");
    fprintf(stdout,"(I)nitial\n");
    fprintf(stdout,"(B)ounds\n");
    fprintf(stdout,"(S)tep\n");
    fprintf(stdout,"(Sh)ow\n");
    fprintf(stdout,"(R)un\n");
    fprintf(stdout,"S(o)lution\n");
    return 0;
  }
  else if( strcasecmp(buffer,"p") == 0 ) {
    fprintf(stdout,"Parameters:\n");
    for(i=0; i<mynum.num_par;i++)
      fprintf(stdout,"%s = %lf\n",par_name[i], mu.P[i]);

    return 0;
  }
  else if(strcasecmp(buf,"pp") == 0){
    // First problem parameter.
    k=0;
    fprintf(stdout,"Enter 1st problem parameter:\n");
    fgets(buf,10,stdin);
    for(j=0;j<10;j++)
      if((*(buf+j) == (char)'\n') || (*(buf+j) == (char)' ') || (*(buf+j) == (char)'\t'))
	*(buf+j) = (char)'\0';
    for(i=0; i<mynum.num_par; i++){
      if(strcasecmp(buf,par_name[i]) == 0){
	P1=mu.P[i]; par1=i;
	k+=1;}
    }
    if(k == 0)
      fprintf(stdout,"Error: no such parameter\n");
    if(k > 1)
      fprintf(stdout,"Many parameters for the same name?\n");
    // Second problem parameter.
    k=0;
    fprintf(stdout,"Enter 2nd problem parameter:\n");
    fgets(buf2,10,stdin);
    for(j=0;j<10;j++)
      if((*(buf2+j) == (char)'\n') || (*(buf2+j) == (char)' ') || (*(buf2+j) == (char)'\t'))
	*(buf2+j) = (char)'\0';
    for(i=0; i<mynum.num_par; i++){
      if(strcasecmp(buf2,par_name[i]) == 0){
	P2=mu.P[i]; par2=i;
	k+=1;}
    }
    if(k == 0)
      fprintf(stdout,"Error: no such parameter\n");
    if(k > 1)
      fprintf(stdout,"Many parameters for the same name?\n");
		
    return 0;
  }
  else if( strcasecmp(buffer,"i") == 0 ){
    fprintf(stdout,"Enter starting values for parameters:\n");
    fprintf(stdout,"Par. #1(%s):\n",par_name[par1]);
    fgets(buf,20,stdin);
    if(strlen(buf) == 1){
      fprintf(stdout,"No changes made.\n");
    }else{
      P1in=atof(buf);
    }
    //fscanf(stdin,"%lf",&P1);
    fprintf(stdout,"Par. #2(%s):\n",par_name[par2]);
    fgets(buf2,20,stdin);
    if(strlen(buf2) == 1){
      fprintf(stdout,"No changes made.\n");
    }else{
      P2in=atof(buf2);
    }
    //fscanf(stdin,"%lf",&P2);

    return 0;
  }
  else if( strcasecmp(buffer,"b") == 0 ) {
    // par #1
    fprintf(stdout,"Bounds for par. #1(%s)(newline are delimiters of min and max values):\n",
	    par_name[par1]);
    fgets(buf,20,stdin);
    if(strlen(buf) == 1){
      fprintf(stdout,"No changes made.\n");
    }else{
      P1min=atof(buf);}
    fgets(buf2,20,stdin);
    if(strlen(buf2) == 1){
      fprintf(stdout,"No changes made.\n");
    }else{
      P1max=atof(buf2);}
    //checking if P1min < P1max
    if(P1min > P1max){
      double tmp=P1max;
      P1max=P1min;
      P1min=tmp;
    }
			
    //par #2
    fprintf(stdout,"Bounds for par. #2(%s)(newline are delimiters of min and max values):\n",
	    par_name[par2]);
    fgets(buf,20,stdin);
    if(strlen(buf) == 1){
      fprintf(stdout,"No changes made.\n");
    }else{
      P2min=atof(buf);}
    fgets(buf2,20,stdin);
    if(strlen(buf2) == 1){
      fprintf(stdout,"No changes made.\n");
    }else{
      P2max=atof(buf2);}
    //checking if P2min < P2max
    if(P2min > P2max){
      double tmp=P2max;
      P2max=P2min;
      P2min=tmp;
    }

    return 0;
  }
  else if( strcasecmp(buffer,"s") == 0 ){
    fprintf(stdout,"Initial step for continuation.\n");

    fprintf(stdout,"Par. #1(%s):\n",par_name[par1]);
    fgets(buf,20,stdin);
    if(strlen(buf) == 1){
      fprintf(stdout,"Left unchanged.\n");
    }else{
      P1stepin=atof(buf);
    }

    fprintf(stdout,"Par. #2(%s):\n",par_name[par2]);
    fgets(buf,20,stdin);
    if(strlen(buf) == 1){
      fprintf(stdout,"Leaven unchanged.\n");
    }else{
      P2stepin=atof(buf);
    }

    return 0;
  }
  else if( strcasecmp(buffer,"sh") == 0 ){
    fprintf(stdout,"Continuation parameters:\n");
    fprintf(stdout,"%s\t%s\n",par_name[par1],par_name[par2]);
    fprintf(stdout,"Initials:\n");
    fprintf(stdout,"%G\t%G\n",P1in,P2in);

    fprintf(stdout,"Parameters range:\n");
    fprintf(stdout,"[%G %G],[%G %G]\n",P1min,P1max,P2min,P2max);

    fprintf(stdout,"Initial step for continuation:\n");
    fprintf(stdout,"%G  %G\n",P1stepin,P2stepin);

    fprintf(stdout,"Problem solution:\n");
    if(cont_sol==0)
      fprintf(stdout,"(Is not specified)\n");
    else
      fprintf(stdout,"%d\n",cont_sol);

    return 0;
  }
  else if( (strcasecmp(buffer,"run") == 0) || (strcasecmp(buffer,"r") == 0) ){
    continuation();

    return 0;
  }
  else if( strcasecmp(buffer,"o") == 0 ){
    cont_sol=SS+HSS;
    fprintf(stdout,"Currently not supported, while changing whole trajectory system\n");
    fprintf(stdout,"Default solution: %d\n",cont_sol);

    return 0;
  }
  else if (strcasecmp(buffer,"")==0)
    return 0;
  else {
    fprintf(stdout,"No such command\n");
    return 0;
  }
  
}

int rand_interp(char *buffer)
{
  int i,j,k,l,m;

  if(strcasecmp(buffer, "exit") == 0 || strcasecmp(buffer,"quit") == 0 || strcasecmp(buffer,"q") == 0) {
    printf("Leaving RANDOM menu...\n");
    return 1000;
  }
  else if(strcasecmp(buffer,"ls") == 0) {
    fprintf(stdout,"RANDOM menu:\n");
    fprintf(stdout,"(Sh)ow summary\n");
    fprintf(stdout,"(D)istribution\n");
    fprintf(stdout,"A(m)plitudes\n");
    fprintf(stdout,"(A)lgorithm\n");
    fprintf(stdout,"S(o)lution\n");
    fprintf(stdout,"(P)oints\n");
    fprintf(stdout,"(S)eries\n");
    fprintf(stdout,"S(e)ed\n");
    fprintf(stdout,"(Up)per bound(one for all)\n");
    fprintf(stdout,"(Lo)wer bound(one for all)\n");
    fprintf(stdout,"(U)pper bound(own for each)\n");
    fprintf(stdout,"(L)ower bound(own for each)\n");
    fprintf(stdout,"(R)un\n");

    return 0;
  }
  else if(strcasecmp(buffer,"sh") == 0){
    fprintf(stdout,"Distribution: %s: pars: ",distribution_type);
    fprintf(stdout,"[");
    for(j=0;j<distribution_n_par;j++){
      if(j==distribution_n_par-1){
	fprintf(stdout,"%G]\n",distribution_par[j]);break;
      }
      fprintf(stdout,"%G,",distribution_par[j]);
    }
    fprintf(stdout,"Amplitudes:\n");
    for(i=0;i<DIM;i++)
      fprintf(stdout,"%s\t",var_name[i]);
    fprintf(stdout,"\n");
    for(i=0;i<DIM;i++)
      fprintf(stdout,"%G\t",ksi[i]);
    fprintf(stdout,"\n");
    fprintf(stdout,"Algorithm: %s\n",type_rnd);
    if(rnd_solution==0)
      fprintf(stdout,"Solution: all(Is not specified)\n");
    else
      fprintf(stdout,"Solution:  %d\n",rnd_solution);
    fprintf(stdout,"Points:    %d\n",num_poin);
    fprintf(stdout,"Seed:      %ld",sd);
    if(seed_flag==1) fprintf(stdout,"(manual)\n");
    if(seed_flag==2) fprintf(stdout,"(automatic)\n");
    fprintf(stdout,"Upper\tLower\n");
    for(j=0;j<DIM;j++)
      fprintf(stdout,"%G\t%G\n",MAX_RND[j],MIN_RND[j]);
    
    return 0;
  }
  else if(strcasecmp(buffer,"d") == 0){
    fprintf(stdout,"Distribution: %s\n",distribution_type);
    fprintf(stdout,"# of pars: %d\n",distribution_n_par);
    fprintf(stdout,"=========\n");
    fprintf(stdout,"(1) Gaussian\n");
    fprintf(stdout,"(2) Exponential\n");
    fgets(buf,10,stdin);
    while(strlen(buf)==1){
      fprintf(stdout,"Cannot understand.\n");
      fgets(buf,10,stdin);
    }
    m=atoi(buf);
    while(m!=1 && m!=2){
      fprintf(stdout,"Cannot understand.\n");
      fgets(buf,10,stdin);
      m=atoi(buf);
    }
    if(m==1) {
      distribution_type="gauss";
      distribution_n_par=2;
      distribution_par=(double *)realloc(distribution_par,distribution_n_par*sizeof(double));
      distribution_par[0]=0;
      distribution_par[1]=1;
    }
    if(m==2) {
      printf("Sorry: it is not supported yet.\n");
      /* distribution_type="exp"; */
/*       distribution_n_par=1; */
/*       distribution_par=(double *)realloc(distribution_par,sizeof(double)); */
/*       distribution_par[0]=1; */
    }
    
    return 0;
  }
  else if(strcasecmp(buffer,"m") == 0){
    fprintf(stdout,"(1)Set one for all\n");
    fprintf(stdout,"(2)Set for each\n");
    fgets(buf,10,stdin);
    while(strlen(buf)==1){
      fprintf(stdout,"Cannot understand.\n");
      fgets(buf,10,stdin);
    }
    m=atoi(buf);
    while(m!=1 && m!=2){
      fprintf(stdout,"Cannot understand.\n");
      fgets(buf,10,stdin);
      m=atoi(buf);
    }
    if(m==1){
      fprintf(stdout,"Value:\n");
      fgets(buf,20,stdin);
      ksi[0]=atof(buf);
      for(i=1;i<DIM;i++)
	ksi[i]=ksi[0];
    }
    if(m==2){
      for(i=0;i<DIM;i++){
	fprintf(stdout,"%s:\n",var_name[i]);
	fgets(buf,20,stdin);
	ksi[i]=atof(buf);
      }
    }
    
    return 0;
  }
  else if(strcasecmp(buffer,"a") == 0){
    fprintf(stdout,"(1)MT19937\n");
    fprintf(stdout,"(2)Combined multiple recursive generator\n");
    fprintf(stdout,"(3)The fifth-order multiple recursive generator\n");
    fprintf(stdout,"(4)Tausworthe generator 1\n");
    fprintf(stdout,"(5)Tausworthe generator 2\n");
    fprintf(stdout,"(6)RANLXS generator (default)\n");
    fprintf(stdout,"Choose one...\n");
    fgets(buf,10,stdin);
    if(strlen(buf) == 1){
      fprintf(stdout,"No changes.\n");
    }else{
      k=atoi(buf);
      if(k <= 0){
	fprintf(stdout,"Error: must be positive.\n");
	k=6;}
      if(k > 6){
	fprintf(stdout,"Error: more than we have here.\n");
	k=6;}
      if(k==1)
	type_rnd="mt19937";
      if(k==2)
	type_rnd="cmrg";
      if(k==3)
	type_rnd="mrg";
      if(k==4)
	type_rnd="taus";
      if(k==5)
	type_rnd="taus2";
      if(k==6)
	type_rnd="ranlxd2";
    }

    return 0;
  }
  else if(strcasecmp(buffer,"o")==0 || strcasecmp(buffer,"sol")==0 || strcasecmp(buffer,"solution")==0){
    rnd_solution=SS+HSS;
    rnd_solution=LC+IHLC;
    //fprintf(stdout,"Currently not supported, while changing whole trajectory system\n");
    fprintf(stdout,"Solution: %d\n",rnd_solution);
    
    return 0;
  }
  else if(strcasecmp(buffer,"p") == 0){
    fprintf(stdout,"Points: %d\n",num_poin);
    fprintf(stdout,"How many points?\n");
    fgets(buf,20,stdin);
    if(strlen(buf) == 1){
      fprintf(stdout,"No changes.\n");
    }else{
      num_poin=atoi(buf);
      if(num_poin < 0){
	fprintf(stdout,"Error: must be positive.Sign change.\n");
	num_poin=-num_poin;
      }
    }

    return 0;
  }
  
  else if(strcasecmp(buffer,"up") == 0){
    fprintf(stdout,"Upper bound(one for all):\n");
    fprintf(stdout,"%G\n",MAX_RND[0]);
    fprintf(stdout,"New value:\n");
    fgets(buf,20,stdin);
    if(strlen(buf)==1)
      fprintf(stdout,"No changes.\n");
    else
      MAX_RND[0]=atof(buf);
    if(MAX_RND[0]<=MIN_RND[0]){
      fprintf(stdout,"MAX must be greater than MIN. Reverse.\n");
      MAX_RND[0]=MIN_RND[0];
      MIN_RND[0]=atof(buf);
      for(i=1;i<DIM;i++)
	MIN_RND[i]=MIN_RND[0];
    }
    for(i=1;i<DIM;i++)
      MAX_RND[i]=MAX_RND[0];
    
    return 0;
  }
  else if(strcasecmp(buffer,"lo") == 0){
    fprintf(stdout,"Lower bound(one for all):\n");
    fprintf(stdout,"%G\n",MIN_RND[0]);
    fprintf(stdout,"New value:\n");
    fgets(buf,20,stdin);
    if(strlen(buf)==1)
      fprintf(stdout,"No changes.\n");
    else
      MIN_RND[0]=atof(buf);
    if(MIN_RND[0]>=MAX_RND[0]){
      fprintf(stdout,"MIN must be less than MAX. Reverse.\n");
      MIN_RND[0]=MAX_RND[0];
      MAX_RND[0]=atof(buf);
      for(i=1;i<DIM;i++)
	MAX_RND[i]=MAX_RND[0];
    }
    for(i=1;i<DIM;i++)
      MIN_RND[i]=MIN_RND[0];
    
    return 0;
  }
  else if(strcasecmp(buffer,"u") == 0){
    for(i=0;i<DIM;i++){
      fprintf(stdout,"New value for var #%d(%s)\n",i+1,var_name[i]);
      fgets(buf2,20,stdin);
      if(strlen(buf2)==1)
	fprintf(stdout,"No changes.\n");
      else
	MAX_RND[i]=atof(buf2);
    }
    
    return 0;
  }
  else if(strcasecmp(buffer,"l") == 0){
    for(i=0;i<DIM;i++){
      fprintf(stdout,"New value for var #%d(%s)\n",i+1,var_name[i]);
      fgets(buf2,20,stdin);
      if(strlen(buf2)==1)
	fprintf(stdout,"No changes.\n");
      else
	MIN_RND[i]=atof(buf2);
    }
    
    return 0;
  }
  else if(strcasecmp(buffer,"e") == 0){
    fprintf(stdout,"How to get seed:\n");
    fprintf(stdout,"(1) manually\n");
    fprintf(stdout,"(2) automatically\n");
    fgets(buf,10,stdin);
    while(strlen(buf)==1){
      fprintf(stdout,"Cannot understand.\n");
      fgets(buf,10,stdin);
    }
    m=atoi(buf);
    while(m!=1 && m!=2){
      fprintf(stdout,"Cannot understand.\n");
      fgets(buf,10,stdin);
      m=atoi(buf);
    }
    if(m==1) {
      seed_flag=1;
      fprintf(stdout,"Enter your seed:\n");
      i=0;
      while((l=getchar())!='\n'){
	seed[i]=(char)l;
	i++;
      }
      seed[i]='\0';
      sd=atoi(seed);
      memset(seed,'\0',SEED_SIZE);
    }
    if(m==2) seed_flag=2;
    
    return 0;
  }
  else if(strcasecmp(buffer,"s") == 0){
    //Generating SEED for random generator.
    if(seed_flag==2)
      generate_seed();
    //=====
    random_d(num_poin*DIM,sd);
    for(j=0;j<num_poin*DIM;j++)
      printf("%lf\n",u[j]);
    printf("=====\n");
    printf("Total: %d\nDIM: %d\nPer variable: %d.\n",num_poin*DIM,DIM,num_poin);

    return 0;
  }
  else if(strcasecmp(buffer,"r") == 0  || strcasecmp(buffer,"run")==0) {
    fprintf(stdout,"Under reconstruction. Wait for future releases.\n");
/*     int count,count1=0; */

/*     if(seed_flag==2) */
/*       generate_seed(); */
/*     //===== */
/*     random_d(num_poin*DIM,sd); */
/*     FILE *rnd_out; */
/*     rnd_out=fopen("rand","w"); */

/*     for(count=0;count<num_poin;count++){ */
/*       for(j=0;j<DIM;j++){ */
/* 	if(xin_par[j] >= 0){/\* Check the init-param link *\/ */
/* 	  /\* For now we do not vary randomly initials that have parameter link. *\/ */
/* 	  xin[j] = mu.P[xin_par[j]]; */
/* 	  yin[j] = mu.P[xin_par[j]]; */
/* 	} */
/* 	else{ */
/* 	  xin[j] = MIN_RND[j] + (MAX_RND[j] - MIN_RND[j])*u[count*DIM+j]; */
/* 	} */
/* 	printf("%lf ",xin[j]); */
/* 	x[j] = xin[j]; */
/*       } */
/*       printf("\n"); */
/*       run(mynum.method,mynum.total_time,write_flag,0,1); */
/*       //Getting attractor by additional integrating. */
/*       /\* for(k=0;k<DIM;k++){ *\/ */
/* /\* 	l=1; *\/ */
/* /\* 	while(per_ratio[k] > eps_per_ratio){ *\/ */
/* /\* 	  run(mynum.method,mynum.trans_time,0,1); *\/ */
/* /\* 	  l++; *\/ */
/* /\* 	  if(l>10){//Integrating additional 10 times. Must be parameter. *\/ */
/* /\* 	    fprintf(stdout,"!Could not get attractor!\n"); *\/ */
/* /\* 	    break;} *\/ */
/* /\* 	} *\/ */
/* /\*       } *\/ */
/*       traj();/\* Regime *\/ */
/*       /\* Additional integrating as IHLC might be actually not totally *\/ */
/*       /\* convergent Steady State. So we integrate additional maximally */
/*       5 times for convergence to SS. This is because the algorithm for */
/*       finding SS and IHLC actually quite similar. They need to differ */
/*       from each other.*\/ */
/*      /\*  l=0; *\/ */
/* /\*       while((strcmp(regime,"IHLC")==0) && (l<5)){ *\/ */
/* /\* 	run(mynum.method,mynum.trans_time,0,1); *\/ */
/* /\* 	traj(); *\/ */
/* /\* 	l++; *\/ */
/* /\*       } *\/ */
/*       /\****************************************************************\/ */
/*       if(rnd_solution == 0){/\*All solutions to write*\/ */
/* 	//rnd_solution=SS+HSS;/\* Default solution to search *\/ */
/* 	printf("Step #%d\n",count+1); */
/* 	fprintf(rnd_out,"Step #%d\n",count+1); */
/* 	for(m=0;m<DIM;m++) */
/* 	  fprintf(rnd_out,"%lf ",xin[m]); */
/* 	fprintf(rnd_out,"\n"); */
/* 	fprintf(rnd_out,"%d\n",regime); */
/* 	count1++; */
/*       } */
/*       else if(regime==rnd_solution){ */
/* 	printf("Step #%d\n",count+1); */
/* 	fprintf(rnd_out,"Step #%d\n",count+1); */
/* 	for(m=0;m<DIM;m++) */
/* 	  fprintf(rnd_out,"%lf ",xin[m]); */
/* 	fprintf(rnd_out,"\n"); */
/* 	fprintf(rnd_out,"%d\n",regime); */
/* 	count1++; */
/*       } */
/*     } */
/*     if(rnd_solution==0){ */
/*       fprintf(rnd_out,"**********************************\n"); */
/*       fprintf(rnd_out,"All solutions.\n"); */
/*       fprintf(rnd_out,"%d total attempts\n",count); */
/*       fprintf(rnd_out,"**********************************\n"); */
/*     } */
/*     else{ */
/*       fprintf(rnd_out,"**********************************\n"); */
/*       fprintf(rnd_out,"%d occurrences of desired solution\n",count1); */
/*       fprintf(rnd_out,"%d total attempts\n",count); */
/*       fprintf(rnd_out,"**********************************\n"); */
/*     } */
/*     fclose(rnd_out); */
/*     if(rnd_solution==0){ */
/*       fprintf(stdout,"**********************************\n"); */
/*       fprintf(stdout,"All solutions.\n"); */
/*       fprintf(stdout,"%d total attempts\n",count); */
/*       fprintf(stdout,"**********************************\n"); */
/*     } */
/*     else{ */
/*       fprintf(stdout,"**********************************\n"); */
/*       fprintf(stdout,"%d occurrences of desired solution\n",count1); */
/*       fprintf(stdout,"%d total attempts\n",count); */
/*       fprintf(stdout,"**********************************\n"); */
/*     } */

    return 0;
  }
  else if (strcasecmp(buffer,"")==0)
    return 0;
  else {
    fprintf(stdout,"No such command\n");
    return 0;
  }
}

int errors_interp(char **buffer, const int depth_level)
{
  int Len = 20;
  char *symbuf = (char *)malloc(Len*sizeof(char));
  int k;
  if(strcasecmp(buffer[depth_level], "exit") == 0 || strcasecmp(buffer[depth_level],"quit") == 0 || strcasecmp(buffer[depth_level],"q") == 0) {
    printf("Leaving ERRORS menu...\n");
    return 1000;
  }
  else if( (strcasecmp(buffer[depth_level],"ls") == 0) || (strcasecmp(buffer[depth_level],"l") == 0) ){
    fprintf(stdout,"ERRORS menu:\n");
    fprintf(stdout,"(C)ross\n");
    fprintf(stdout,"(I)ntegrator\n");
    fprintf(stdout,"(P)eriod\n");
    fprintf(stdout,"P(e)ak\n");
    fprintf(stdout,"(A)mplitude\n");
    fprintf(stdout,"Pe(r)iod Ratio\n");
    fprintf(stdout,"Re(g)ime Ratio\n");
    fprintf(stdout,"(Sh)ow\n");
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"show")==0 || strcasecmp(buffer[depth_level],"sh")==0){
    fprintf(stdout,"cross:\n");
    fprintf(stdout,"\tAbs.:%G\n",eps_abs_tper);
    fprintf(stdout,"\tRel.:%G\n",eps_rel_tper);
    fprintf(stdout,"integrator:\n");
    fprintf(stdout,"\tAbs.:%G\n",eps_abs_int);
    fprintf(stdout,"\tRel.:%G\n",eps_rel_int);
    fprintf(stdout,"\tscaling:\n");
    fprintf(stdout,"\t\tvar.:%G\n",a_y);
    fprintf(stdout,"\t\tderiv.:%G\n",a_dydt);
    fprintf(stdout,"period:\n");
    fprintf(stdout,"\tAbs.:%G\n",eps_abs_per);
    fprintf(stdout,"\tRel.:%G\n",eps_rel_per);
    fprintf(stdout,"peak:\n");
    fprintf(stdout,"\tAbs.:%G\n",eps_abs_peak);
    fprintf(stdout,"\tRel.:%G\n",eps_rel_peak);
    fprintf(stdout,"amplitude:\n");
    fprintf(stdout,"\tAbs.:%G\n",eps_abs_am);
    fprintf(stdout,"\tRel.:%G\n",eps_rel_am);
    fprintf(stdout,"period ratio:%G\n",eps_per_ratio);
    fprintf(stdout,"regime ratio:%G\n",eps_inregime_ratio);
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"cross")==0 ||
	  strcasecmp(buffer[depth_level],"c")==0){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Cross errors:\n");
      fprintf(stdout,"(1) Abs.\n");
      fprintf(stdout,"(2) Rel.\n");
      symbuf = read_input_line(symbuf,Len);
      *(symbuf+1) = 0;/* Take just 1 symbol */
    }
    k = atoi(symbuf);
    if(k == 1){
      if((check_next_arg(buffer,depth_level+1,symbuf,Len) != 0)){
	fprintf(stdout,"Abs. error:\n");
	symbuf = read_input_line(symbuf,Len);
      }
      eps_abs_tper = atof(symbuf);
    }
    else if(k == 2){
      if((check_next_arg(buffer,depth_level+1,symbuf,Len) != 0)){
	fprintf(stdout,"Rel. error:\n");
	symbuf = read_input_line(symbuf,Len);
      }
      eps_rel_tper = atof(symbuf);      
    }
    else {
      fprintf(stdout,"No changes.\n");
    }
    return 0;
  }
  else if(strcasecmp(buffer[depth_level],"integrator")==0 || \
	  strcasecmp(buffer[depth_level],"int")==0 || \
	  strcasecmp(buffer[depth_level],"i")==0){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Integrator errors:\n");
      fprintf(stdout,"(1) Abs.\n");
      fprintf(stdout,"(2) Rel.\n");
      fprintf(stdout,"Scaling factors:\n");
      fprintf(stdout,"(3) Variable\n");
      fprintf(stdout,"(4) Derivative\n");
      symbuf = read_input_line(symbuf,Len);
      *(symbuf+1) = 0;/* Take just 1 symbol */
    }
    k = atoi(symbuf);
    if(k == 1){
      if((check_next_arg(buffer,depth_level+1,symbuf,Len) != 0)){
	fprintf(stdout,"Abs. error:\n");
	symbuf = read_input_line(symbuf,Len);
      }
      eps_abs_int=atof(symbuf);
    }
    else if(k == 2){
      if((check_next_arg(buffer,depth_level+1,symbuf,Len) != 0)){
	fprintf(stdout,"Rel. error:\n");
	symbuf = read_input_line(symbuf,Len);
      }
      eps_rel_int=atof(symbuf);
    }
    else if(k == 3){
      if((check_next_arg(buffer,depth_level+1,symbuf,Len) != 0)){
	fprintf(stdout,"Var. scaling:\n");
	symbuf = read_input_line(symbuf,Len);
      }
      a_y=atof(symbuf);
    }
    else if(k == 4){
      if((check_next_arg(buffer,depth_level+1,symbuf,Len) != 0)){
	fprintf(stdout,"Deriv. scaling:\n");
	symbuf = read_input_line(symbuf,Len);
      }
      a_dydt=atof(symbuf);
    }
    else {
      fprintf(stdout,"No changes.\n");
    }
    return 0;
  }
  else if (strcasecmp(buffer[depth_level],"period")==0 || \
	   strcasecmp(buffer[depth_level],"per")==0 || \
	   strcasecmp(buffer[depth_level],"p")==0){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Period errors:\n");
      fprintf(stdout,"(1) Abs.\n");
      fprintf(stdout,"(2) Rel.\n");
      symbuf = read_input_line(symbuf,Len);
      *(symbuf+1) = 0;/* Take just 1 symbol */
    }
    k = atoi(symbuf);
    if(k == 1){
      if((check_next_arg(buffer,depth_level+1,symbuf,Len) != 0)){
	fprintf(stdout,"Abs. error:\n");
	symbuf = read_input_line(symbuf,Len);
      }
      eps_abs_per=atof(symbuf);
    }
    else if(k == 2){
      if((check_next_arg(buffer,depth_level+1,symbuf,Len) != 0)){
	fprintf(stdout,"Rel. error:\n");
	symbuf = read_input_line(symbuf,Len);
      }
      eps_rel_per=atof(symbuf);
    }
    else {
      fprintf(stdout,"No changes.\n");
    }
    return 0;
  }
  else if (strcasecmp(buffer[depth_level],"peak")==0 || \
	   strcasecmp(buffer[depth_level],"e")==0){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Peak errors:\n");
      fprintf(stdout,"(1) Abs.\n");
      fprintf(stdout,"(2) Rel.\n");
      symbuf = read_input_line(symbuf,Len);
      *(symbuf+1) = 0;/* Take just 1 symbol */
    }
    k = atoi(symbuf);
    if(k == 1){
      if((check_next_arg(buffer,depth_level+1,symbuf,Len) != 0)){
	fprintf(stdout,"Abs. error:\n");
	symbuf = read_input_line(symbuf,Len);
      }
      eps_abs_peak = atof(symbuf);
    }
    else if(k == 2){
      if((check_next_arg(buffer,depth_level+1,symbuf,Len) != 0)){
	fprintf(stdout,"Rel. error:\n");
	symbuf = read_input_line(symbuf,Len);
      }
      eps_rel_peak = atof(symbuf);
    }
    else {
      fprintf(stdout,"No changes.\n");
    }
    return 0;
  }
  else if (strcasecmp(buffer[depth_level],"amplitude")==0 || \
	   strcasecmp(buffer[depth_level],"amp")==0 || \
	   strcasecmp(buffer[depth_level],"am")==0 || \
	   strcasecmp(buffer[depth_level],"a")==0){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Amplitude errors:\n");
      fprintf(stdout,"(1) Abs.\n");
      fprintf(stdout,"(2) Rel.\n");
      symbuf = read_input_line(symbuf,Len);
      *(symbuf+1) = 0;/* Take just 1 symbol */
    }
    k = atoi(symbuf);
    if(k == 1){
      if((check_next_arg(buffer,depth_level+1,symbuf,Len) != 0)){
	fprintf(stdout,"Abs. error:\n");
	symbuf = read_input_line(symbuf,Len);
      }
      eps_abs_am = atof(symbuf);
    }
    else if(k == 2){
      if((check_next_arg(buffer,depth_level+1,symbuf,Len) != 0)){
	fprintf(stdout,"Rel. error:\n");
	symbuf = read_input_line(symbuf,Len);
      }
      eps_rel_am = atof(symbuf);
    }
    else {
      fprintf(stdout,"No changes.\n");
    }
    return 0;
  }
  else if (strcasecmp(buffer[depth_level],"perratio")==0 || \
	   strcasecmp(buffer[depth_level],"pratio")==0 || \
	   strcasecmp(buffer[depth_level],"r")==0){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Period ratio:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(*symbuf == 0){
      fprintf(stdout,"No changes.\n");
    } else {
      eps_per_ratio = atof(symbuf);
    }
    return 0;
  }
  else if (strcasecmp(buffer[depth_level],"regratio")==0 || \
	   strcasecmp(buffer[depth_level],"rratio")==0 || \
	   strcasecmp(buffer[depth_level],"g")==0){
    if((check_next_arg(buffer,depth_level,symbuf,Len) != 0)){
      fprintf(stdout,"Regime ratio:\n");
      symbuf = read_input_line(symbuf,Len);
    }
    if(*symbuf == 0){
      fprintf(stdout,"No changes.\n");
    } else {
      eps_inregime_ratio = atof(symbuf);
    }
    return 0;
  }
  else if (strcasecmp(buffer[depth_level],"")==0)
    return 0;
  else {
    fprintf(stdout,"No such command\n");
    return 0;
  }
}

int sing_interp(char *buffer)
{
  int i;
  if(strcasecmp(buffer, "exit") == 0 || strcasecmp(buffer,"quit") == 0 || strcasecmp(buffer,"q") == 0) {
    printf("Leaving SINGULARITY menu...\n");
    return 1000;
  }
  else if( (strcasecmp(buffer,"ls") == 0) || (strcasecmp(buffer,"l") == 0) ){
    fprintf(stdout,"SINGULARITY menu:\n");
    fprintf(stdout,"(R)un\n");
    fprintf(stdout,"(F)ile\n");
    fprintf(stdout,"(R)andom (f)lag\n");
    fprintf(stdout,"(F)ile (f)lag\n");
    fprintf(stdout,"(P)oints\n");
    fprintf(stdout,"(Up)per\n");
    fprintf(stdout,"(Lo)wer\n");
    fprintf(stdout,"(Sh)ow\n");
    fprintf(stdout,"***Technical parameters***\n");
    fprintf(stdout,"(I)terations\n");
    fprintf(stdout,"(S)olutions\n");
    fprintf(stdout,"(T)est\n");
    fprintf(stdout,"(E)rror\n");
    return 0;
  }
  else if(strcasecmp(buffer,"sh")==0 || strcasecmp(buffer,"show")==0){
    fprintf(stdout,"File: %s\n",ss_name);
    if(RND_FLAG)
      fprintf(stdout,"Random flag: yes\n");
    else
      fprintf(stdout,"Random flag: no\n");
    if(FILE_FLAG)
      fprintf(stdout,"File flag: yes\n");
    else
      fprintf(stdout,"File flag: no\n");
    fprintf(stdout,"Points: %d\n",PTS);
    fprintf(stdout,"Upper bound: %G\n",MAX_SNG);
    fprintf(stdout,"Lower bound: %G\n",MIN_SNG);
    fprintf(stdout,"***Technical parameters***\n");
    fprintf(stdout,"Iterations: %d\n",MAX_ITER);
    fprintf(stdout,"Roots: %d\n",MAX_N_SOL);
    fprintf(stdout,"Testing error: %G\n",TEST);
    fprintf(stdout,"Solution error: %G\n",SOL_ERROR);
    return 0;
  }
  else if (strcasecmp(buffer,"run")==0 || strcasecmp(buffer,"r")==0){
    multiroot_find();

    return 0;
  }
  else if (strcasecmp(buffer,"file")==0 || strcasecmp(buffer,"f")==0){
    fprintf(stdout,"Current filename: %s\n",ss_name);
    fprintf(stdout,"New filename:\n");
    memset(buf,'\0',100);
    fgets(buf,20,stdin);
    buf_check(buf);
    if(strlen(buf) && strlen(buf)<20){
      if(ss_name == NULL)
	ss_name=(char *)calloc(strlen(buf),sizeof(char));
      else{
	free(ss_name);
	ss_name=(char *)calloc(strlen(buf),sizeof(char));
      }
    }
    else{
      fprintf(stdout,"No new name OR name is more than 20 characters long.\n");}
    for(i=0;i<strlen(buf);i++)
      *(ss_name+i)=*(buf+i);

    return 0;
  }
  else if (strcasecmp(buffer,"rf")==0){
    if(RND_FLAG)
      fprintf(stdout,"Current random flag: yes\n");
    else
      fprintf(stdout,"Current random flag: no\n");
    fprintf(stdout,"New choice:(yes or no, 0 or not 0)\n");
    memset(buf,'\0',100);
    fgets(buf,5,stdin);
    buf_check(buf);
    if(strlen(buf) && isalpha(buf[0])){
      if(strcasecmp(buf,"yes")==0)
	RND_FLAG=1;
      else if(strcasecmp(buf,"no")==0)
	RND_FLAG=0;
      else
	fprintf(stdout,"Did not get the word. No changes.\n");
    }
    else if(strlen(buf) && !isalpha(buf[0])){
      RND_FLAG=atoi(buf);
    }
    else
      fprintf(stdout,"No changes.\n");
    
    return 0;
  }
  else if (strcasecmp(buffer,"ff")==0){
    if(FILE_FLAG)
      fprintf(stdout,"Current file flag: yes\n");
    else
      fprintf(stdout,"Current file flag: no\n");
    fprintf(stdout,"New choice:(yes or no, 0 or not 0)\n");
    memset(buf,'\0',100);
    fgets(buf,5,stdin);
    buf_check(buf);
    if(strlen(buf) && isalpha(buf[0])){
      if(strcasecmp(buf,"yes")==0){
	if(ss_name == NULL)
	  fprintf(stdout,"Tell me what filename is.\n");
	FILE_FLAG=1;
      }
      else if(strcasecmp(buf,"no")==0)
	FILE_FLAG=0;
      else
	fprintf(stdout,"Did not get the word. No changes.\n");
    }
    else if(strlen(buf) && !isalpha(buf[0])){
      FILE_FLAG=atoi(buf);
      if(FILE_FLAG && ss_name == NULL)
	fprintf(stdout,"Tell me what filename is.\n");
    }
    else
      fprintf(stdout,"No changes.\n");
    
    return 0;
  }
  else if (strcasecmp(buffer,"p")==0 || strcasecmp(buffer,"point")==0\
	   || strcasecmp(buffer,"points")==0){
    fprintf(stdout,"Current number of points: %d\n",PTS);
    fprintf(stdout,"(Remember: it works only if random flag is YES)\n");
    memset(buf,'\0',100);
    fgets(buf,15,stdin);
    buf_check(buf);
    if(strlen(buf) && strlen(buf)<15)
      PTS=atoi(buf);
    else
      fprintf(stdout,"No changes.\n");

    return 0;
  }
  else if (strcasecmp(buffer,"up")==0 ||\
	   strcasecmp(buffer,"upper")==0){
    fprintf(stdout,"Current max value: %G\n",MAX_SNG);
    fprintf(stdout,"(Remember: it works only if random flag is YES)\n");
    fprintf(stdout,"New max value:\n");
    memset(buf,'\0',100);
    fgets(buf,15,stdin);
    buf_check(buf);
    if(strlen(buf) && strlen(buf)<15)
      MAX_SNG=atof(buf);
    else
      fprintf(stdout,"No changes.\n");

    return 0;
  }
  else if (strcasecmp(buffer,"lo")==0 ||\
	   strcasecmp(buffer,"lower")==0){
    fprintf(stdout,"Current min value: %G\n",MIN_SNG);
    fprintf(stdout,"(Remember: it works only if random flag is YES)\n");
    fprintf(stdout,"New min value:\n");
    memset(buf,'\0',100);
    fgets(buf,15,stdin);
    buf_check(buf);
    if(strlen(buf) && strlen(buf)<15)
      MIN_SNG=atof(buf);
    else
      fprintf(stdout,"No changes.\n");

    return 0;
  }
  else if (strcasecmp(buffer,"i")==0 || strcasecmp(buffer,"iter")==0){
    fprintf(stdout,"Max number of iterations for each set of \
initials.\n");
    fprintf(stdout,"Current value: %d\n",MAX_ITER);
    fprintf(stdout,"New value:\n");
    memset(buf,'\0',100);
    fgets(buf,15,stdin);
    buf_check(buf);
    if(strlen(buf) && strlen(buf)<15)
      MAX_ITER=atoi(buf);
    else
      fprintf(stdout,"No changes.\n");
    
    return 0;
  }
  else if (strcasecmp(buffer,"s")==0 || strcasecmp(buffer,"sol")==0){
    fprintf(stdout,"Max number of roots to find.\n");
    fprintf(stdout,"Current value: %d\n",MAX_N_SOL);
    fprintf(stdout,"New value:\n");
    memset(buf,'\0',100);
    fgets(buf,15,stdin);
    buf_check(buf);
    if(strlen(buf) && strlen(buf)<15)
      MAX_N_SOL=atoi(buf);
    else
      fprintf(stdout,"No changes.\n");

    return 0;
  }
  else if (strcasecmp(buffer,"t")==0 || strcasecmp(buffer,"test")==0){
    fprintf(stdout,"User-specified precision for successful iteration\
 process.\n");
    fprintf(stdout,"Current value: %G\n",TEST);
    fprintf(stdout,"New value:\n");
    memset(buf,'\0',100);
    fgets(buf,15,stdin);
    buf_check(buf);
    if(strlen(buf) && strlen(buf)<15)
      TEST=atof(buf);
    else
      fprintf(stdout,"No changes.\n");

    return 0;
  }
  else if (strcasecmp(buffer,"e")==0 || strcasecmp(buffer,"er")==0){
    fprintf(stdout,"Error to distinguish different solutions.\n");
    fprintf(stdout,"Current value: %G\n",SOL_ERROR);
    fprintf(stdout,"New value:\n");
    memset(buf,'\0',100);
    fgets(buf,15,stdin);
    buf_check(buf);
    if(strlen(buf) && strlen(buf)<15)
      SOL_ERROR=atof(buf);
    else
      fprintf(stdout,"No changes.\n");

    return 0;
  }
  else if (strcasecmp(buffer,"")==0)
    return 0;
  else {
    fprintf(stdout,"No such command\n");
    return 0;
  }
}

int lyap_interp(char *buffer)
{
  int i;
  double *tmp;
  if(strcasecmp(buffer, "exit") == 0 || strcasecmp(buffer,"quit") == 0 || strcasecmp(buffer,"q") == 0) {
    printf("Leaving LYAPUNOV menu...\n");
    return 1000;
  }
  else if( (strcasecmp(buffer,"ls") == 0) || (strcasecmp(buffer,"l") == 0) ){
    fprintf(stdout,"LYAPUNOV menu:\n");
    fprintf(stdout,"(R)un\n");
    fprintf(stdout,"(D)istance threshold\n");
    fprintf(stdout,"(I)nitial distance\n");
    fprintf(stdout,"(R)un (c)ontinuation\n");
    fprintf(stdout,"(P)arameter\n");
    fprintf(stdout,"Parameter ma(x)\n");
    fprintf(stdout,"Parameter mi(n)\n");
    fprintf(stdout,"Parameter (s)tep\n");
    fprintf(stdout,"(O)utput file\n");
    fprintf(stdout,"(R)un (s)pectrum\n");
    fprintf(stdout,"Run (s)pectrum (c)ontinuation\n");
    fprintf(stdout,"Output (f)ile\n");
    return 0;
  }
  else if ( (strcasecmp(buffer,"sh")==0) ){
    fprintf(stdout,"Threshold value (DTR):\t %G\n",DTR);
    fprintf(stdout,"Initial distance (DI):\t %G\n",DI);
    fprintf(stdout,"Parameter:\t %s=%G\n",par_name[LPI],mu.P[LPI]);
    fprintf(stdout,"Parameter max:\t %G\n",LPARMAX);
    fprintf(stdout,"Parameter min:\t %G\n",LPARMIN);
    fprintf(stdout,"Parameter step:\t %G\n",LDP);
    fprintf(stdout,"Compute time:\t %G(change from numerics)\n",mynum.total_time);
    fprintf(stdout,"MLE continuation output: %s\n",mle_file);
    fprintf(stdout,"LCE spectrum continuation output: %s\n",lyap_spec_file);
    return 0;
  }
  else if ( (strcasecmp(buffer,"r")==0) || \
	    (strcasecmp(buffer,"run")==0) ){
    tmp=(double *)malloc(sizeof(double));
    tmp=mle(tmp);
    return 0;
  }
  else if ( (strcasecmp(buffer,"d")==0) ){
    fprintf(stdout,"Enter threshold (DTR) value:\n");
    fgets(buf,20,stdin);
    buf_check(buf);
    if(isdigit(buf[0]))
      DTR=atof(buf);
    else
      printf("Left unchanged.\n");
    return 0;
  }
  else if ( (strcasecmp(buffer,"i")==0) ){
    fprintf(stdout,"Enter initial distance (DI) value:\n");
    fgets(buf,20,stdin);
    buf_check(buf);
    if(isdigit(buf[0]))
      DI=atof(buf);
    else
      printf("Left unchanged.\n");
    return 0;
  }
  else if (strcasecmp(buffer,"rc")==0){
    mle_cont();
    return 0;
  }
  else if (strcasecmp(buffer,"p")==0){
    fprintf(stdout,"Enter parameter name:\n");
    fgets(buf,20,stdin);
    if(strcmp(buf,"\n")==0){
      printf("No changes.\n");
      return 0;}
    buf_check(buf);
    for(i=0;i<PARDIM;i++){
      if(strcmp(buf,par_name[i])==0){
	LPI = i;break;}
      if(i+1==PARDIM)
	printf("No parameter with such a name (%s).\n",buf);
    }
    return 0;
  }
  else if (strcasecmp(buffer,"x")==0){
    fprintf(stdout,"Max parameter value:\n");
    fgets(buf,20,stdin);
    buf_check(buf);
    if(isdigit(buf[0]))
      LPARMAX=atof(buf);
    else
      printf("Left unchanged.\n");
    return 0;
  }
  else if (strcasecmp(buffer,"n")==0){
    fprintf(stdout,"Min parameter value:\n");
    fgets(buf,20,stdin);
    buf_check(buf);
    if(isdigit(buf[0]))
      LPARMIN=atof(buf);
    else
      printf("Left unchanged.\n");
    return 0;
  }
  else if (strcasecmp(buffer,"s")==0){
    fprintf(stdout,"Parameter step:\n");
    fgets(buf,20,stdin);
    buf_check(buf);
    if(strcmp(buf,"-")==0){
      printf("Changing the sign to opposite.\n");
      LDP=-LDP;
      return 0;
    }
    if(isdigit(buf[0]) || (char)buf[0]=='-')
      LDP=atof(buf);
    else
      printf("Left unchanged.\n");
    return 0;
  }
  else if (strcasecmp(buffer,"o")==0){
    fprintf(stdout,"File name for MLE continuation output:\n");
    fgets(buf,20,stdin);
    buf_check(buf);
    strcpy(mle_file,buf);
    return 0;
  }
  else if (strcasecmp(buffer,"rs")==0){
    tmp=(double *)malloc(DIM*sizeof(double));
    tmp=lyap_spec(tmp);
    return 0;
  }
  else if (strcasecmp(buffer,"sc")==0){
    lyap_spec_cont();
    return 0;
  }
  else if (strcasecmp(buffer,"f")==0){
    fprintf(stdout,"File name for LCE spectrum continuation output:\n");
    fgets(buf,20,stdin);
    buf_check(buf);
    strcpy(lyap_spec_file,buf);
    return 0;
  }
  else if (strcasecmp(buffer,"")==0)
    return 0;
  else {
    fprintf(stdout,"No such command\n");
    return 0;
  }
}

int get_parameter(char **buffer, const int depth_level){
  int i,j;
  int Len = 20;
  char * symbuf = (char *)malloc(Len*sizeof(char));
  /* Call with an argument */
  if(*(buffer[depth_level])){
    if(Len > MAX_ARG_LEN)
      strncpy(symbuf,buffer[depth_level],MAX_ARG_LEN);
    else
      strncpy(symbuf,buffer[depth_level],Len);
    for(i=0;i<PARDIM;i++){
      if(strcmp(symbuf,par_name[i]) == 0){
	if(check_next_arg(buffer,depth_level,symbuf,Len)){
	  fprintf(stdout,"value:\n");
	  symbuf = read_input_line(symbuf,Len);
	}
	/*Check for zero input*/
	if(*symbuf == 0){
	  fprintf(stdout,"No changes.\n");
	  break;
	}else{
	  if(isalpha(*symbuf)){
	    printf("Character! Left unchanged.\n");
	    break;
	  }
	  mu.P[i] = atof(symbuf);
	  for(j=0;j<DIM;j++){/* Check par-xin link */
	    if(xin_par[j] == i){
	      x[j] = mu.P[i];/* Set current var to the value */
	      y[j] = (long)mu.P[i];/* Set current var to the value */
	    }
	  }
	  break;
	}
      }
      if(i+1 == PARDIM)
	fprintf(stdout,"\nNo parameter with such a name.\n");
    }
  }
  /* Print parameters */
  fprintf(stdout,"Parameters:\n");
  for(i=0;i<PARDIM;i++)
    fprintf(stdout,"%s = %G\n",par_name[i],p_mu->P[i]);
  /* Call without an argument */
  if(!(*buffer[depth_level])){
    fprintf(stdout,"Enter parameter name:\n");
    symbuf = read_input_line(symbuf,Len);
    if(*symbuf == 0){
      fprintf(stdout,"No changes.\n");
    }else{
      for(i=0; i<PARDIM; i++){
	if( strcmp(symbuf,par_name[i]) == 0 ){
	  fprintf(stdout,"value:\n");
	  symbuf = read_input_line(symbuf,Len);
	  /*Check for zero input*/
	  if(*symbuf == 0){
	    fprintf(stdout,"No changes.\n");
	    break;
	  }else{
	    if(isalpha(*symbuf)){
	      printf("Character! Left unchanged.\n");
	      break;
	    }
	    mu.P[i] = atof(symbuf);
	    for(j=0;j<DIM;j++){/* Check par-xin link */
	      if(xin_par[j] == i){
		x[j] = mu.P[i];/* Set current var to the value */
		y[j] = (long)mu.P[i];/* Set current var to the value */
	      }
	    }
	    break;
	  }
	}
	if(i+1==PARDIM)
	  fprintf(stdout,"\nNo parameter with such a name.\n");
      }
    }
  }
  return 0;
}

int load_initials(char *init_name){
  /* The function loads the init conditions and stores them in the system-wide
     variables xin[]. If there are less than DIM numbers in init_name file then the
     function fills the rest values with zeros and makes a warning message. If there
     are more than DIM values in the source, the function takes only DIM values and
     ignores the rest. The function returns non-zero value when failed to open the
     file, otherwise, the return value is zero, even when not enough and too many
     points were found.*/
  int i,j;
  double tmp;
  FILE *init = fopen(init_name,"r");
  if(init == NULL){
    fprintf(stderr,"Error: cannot read the file.\n");
    return 10;
  }
  else{
    i = 0;
    while(i < DIM){
      j = fscanf(init,"%lf",(xin+i));
      if(j == EOF){
	fprintf(stderr,"Warning: not enough points.\n");
	break;
      }
      i++;
    }
    for(j=i;j<DIM;j++)/* Setting to zeros the rest */
      xin[j] = 0.0;
  }
  fclose(init);
  return 0;
}

int check_next_arg(char **buffer,const int depth_level, char *next_arg,
		   const size_t Len){
  /* If the next argument in the cmd exists, the function copies it to the *next_arg
  (a temporary array) and returns 0(true). Otherwise, it returns positive value(1)
  and does not change the *next_arg */
  if(strlen(buffer[depth_level+1]) != 0){
    strncpy(next_arg,buffer[depth_level+1],Len-1);
    *(next_arg+Len-1) = 0;/* Terminate string */
    if(strlen(buffer[depth_level+1])>Len-1){
      printf("Too many symbols in `%s'.",buffer[depth_level+1]);
      printf(" I read: `%s'\n",next_arg);
    }
    return 0;
  }
  else{
    return 1;
  }
}

char *read_input_line(char *symbuf, const int Len)
{
  /* This function reads a single input from stdin, removes newline
     and returns what it has read */
  fgets(symbuf,Len,stdin);
  if(strchr(symbuf,'\n') != NULL){
    *(strchr(symbuf,'\n')) = 0;
  }
  else{
    fprintf(stderr,"Input is too long: >%d char, read: %s\n",Len-1,symbuf);
  }
  return symbuf;
}
