/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi JRR <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "contactlist.h"

ContactList::ContactList(PumpController *pumpController,
                         GlobalObject *globalObject,
                         QString listType,
                         QWidget *parent) : QWidget(parent)
{
    this->pController = pumpController;
    this->globalObj = globalObject;


    QString filterNote = tr("Type a partial name or ID to find a contact...")
                         + " (Control+F)";
    this->filterLineEdit = new QLineEdit();
    filterLineEdit->setPlaceholderText(filterNote);
    filterLineEdit->setToolTip("<b></b>" + filterNote); // HTMLized for wordwrapping
    connect(filterLineEdit, SIGNAL(textChanged(QString)),
            this, SLOT(filterList(QString)));

    this->removeFilterButton = new QPushButton(QIcon::fromTheme("view-list-icons"),
                                               tr("F&ull List"));
    removeFilterButton->setDisabled(true); // Disabled initially, until a search happens
    connect(removeFilterButton, SIGNAL(clicked()),
            filterLineEdit, SLOT(clear()));

    this->filterLayout = new QHBoxLayout();
    filterLayout->addWidget(filterLineEdit);
    filterLayout->addWidget(removeFilterButton);

    goToFilterAction = new QAction(this);
    goToFilterAction->setShortcut(QKeySequence("Ctrl+F"));
    connect(goToFilterAction, SIGNAL(triggered()),
            filterLineEdit, SLOT(setFocus()));
    this->addAction(goToFilterAction);


    this->contactsLayout = new QVBoxLayout();
    contactsLayout->setAlignment(Qt::AlignTop);

    this->contactsWidget = new QWidget();
    contactsWidget->setLayout(contactsLayout);

    this->contactsScrollArea = new QScrollArea();
    contactsScrollArea->setWidget(contactsWidget);
    contactsScrollArea->setWidgetResizable(true);
    contactsScrollArea->setFrameStyle(QFrame::NoFrame);
    contactsScrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);


    this->mainLayout = new QVBoxLayout();
    mainLayout->addWidget(contactsScrollArea);
    mainLayout->addLayout(filterLayout);
    this->setLayout(mainLayout);


    // Add demo contacts
    QVariantMap demoContactData;
    QVariantMap demoContactHometown;
    QVariantMap demoContactFollowed;

    if (listType == "following")
    {
        demoContactData.insert("displayName",  "Demo Contact");
        demoContactData.insert("id",           "democontact@pumpserver.org");
        demoContactFollowed.insert("followed", "true");
    }
    else
    {
        demoContactData.insert("displayName",  "Demo Follower");
        demoContactData.insert("id",           "demofollower@pumpserver.org");
        demoContactFollowed.insert("followed", "false");
    }

    demoContactHometown.insert("displayName",  "Some city");
    demoContactData.insert("location", demoContactHometown);

    demoContactData.insert("pump_io",  demoContactFollowed);

    demoContactData.insert("published", "2013-05-01T00:00:00Z"); // Dianara's birthday


    ASPerson *demoContactPerson = new ASPerson(demoContactData);
    ContactCard *demoContactCard = new ContactCard(this->pController,
                                                   this->globalObj,
                                                   demoContactPerson, this);
    this->contactsLayout->addWidget(demoContactCard);
    this->contactsInList.append(demoContactCard);


    qDebug() << "ContactList created";
}

ContactList::~ContactList()
{
    qDebug() << "ContactList destroyed";
}



void ContactList::clearListContents()
{
    foreach (ContactCard *card, contactsInList)
    {
        this->contactsLayout->removeWidget(card);
        delete card;
    }
    this->contactsInList.clear();

    this->contactsListString.clear();
}


void ContactList::setListContents(QString listType,
                                  QVariantList contactList)
{
    qDebug() << "ContactList; Setting list contents";

    QString contactInfoLineString;
    QStringList followingIdStringList;

    foreach (QVariant contact, contactList)
    {
        ASPerson *person = new ASPerson(contact.toMap());
        ContactCard *contactCard = new ContactCard(this->pController,
                                                   this->globalObj,
                                                   person, this);

        contactInfoLineString = person->getName()
                              + "  <"
                              + person->getId()
                              + ">\n";

        this->contactsLayout->addWidget(contactCard);
        this->contactsInList.append(contactCard);
        contactsListString.append(contactInfoLineString);

        if (listType == "following")
        {
            followingIdStringList.append(person->getId());
        }
    }


    // Batch of contacts added to list, add them also to the internal list
    if (listType == "following")
    {
        this->pController->updateInternalFollowingIdList(followingIdStringList);
    }

    this->filterLineEdit->clear(); // Unfilter the list
}



void ContactList::adjustWidth()
{
    this->contactsWidget->setMaximumWidth(this->contactsScrollArea->viewport()->width());
}


QString ContactList::getContactsListString()
{
    return this->contactsListString;
}



/*****************************************************************************/
/*********************************** SLOTS ***********************************/
/*****************************************************************************/


void ContactList::filterList(QString filterText)
{
    qDebug() << "Filtering for contacts matching:" << filterText;

    if (!filterText.isEmpty())
    {
        foreach (ContactCard *card, contactsInList)
        {
            if (card->getNameAndIdString().contains(filterText, Qt::CaseInsensitive))
            {
                card->show();
            }
            else
            {
                card->hide();
            }
        }

        removeFilterButton->setEnabled(true);
    }
    else // If no filter at all, more optimized version showing all
    {
        foreach (ContactCard *card, contactsInList)
        {
            card->show();
        }

        removeFilterButton->setDisabled(true);
    }
}


void ContactList::addSingleContact(ASPerson *contact)
{
    ContactCard *card = new ContactCard(this->pController,
                                        this->globalObj,
                                        contact,
                                        this);

    this->contactsLayout->insertWidget(0, card);
    this->contactsInList.append(card);

    emit contactCountChanged(1);

    QStringList contactsToAdd;
    contactsToAdd.append(contact->getId());
    this->pController->updateInternalFollowingIdList(contactsToAdd);
}


void ContactList::removeSingleContact(ASPerson *contact)
{
    foreach (ContactCard *card, contactsInList)
    {
        if (card->getId() == contact->getId())
        {
            emit contactCountChanged(-1);

            this->pController->removeFromInternalFollowingList(contact->getId());

            card->setDisabled(true);
        }
    }

    contact->deleteLater();
}



/*****************************************************************************/
/********************************* PROTECTED *********************************/
/*****************************************************************************/



void ContactList::resizeEvent(QResizeEvent *event)
{
    this->adjustWidth();

    event->accept();
}

void ContactList::showEvent(QShowEvent *event)
{
    this->adjustWidth();

    event->accept();
}
