/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "post.h"

Post::Post(ASActivity *activity,
           bool highlightedByFilter,
           bool isStandalone,
           PumpController *pumpController,
           GlobalObject *globalObject,
           QWidget *parent)  :  QFrame(parent)
{
    this->pController = pumpController;
    this->globalObj = globalObject;
    this->standalone = isStandalone;

    this->seeFullImageString = tr("Click the image to see it in full size");
    this->downloadAttachmentString = tr("Click to download the attachment");
    this->postImageIsAnimated = false; // Initialize
    this->setSizePolicy(QSizePolicy::MinimumExpanding,
                        QSizePolicy::Minimum);
    this->setMinimumSize(30, 30); // Ensure something's visible at all times

    activity->setParent(this); // reparent the passed activity


    leftColumnFrame = new QFrame();

    ///////////////////////////////////////////////// Highlighting
    this->highlightType = NoHighlight;
    QStringList highlightColors = globalObj->getColorsList();
    QString highlightPostColor;

    // Post is adressed to us
    if (activity->getRecipientsIdList().contains("acct:"
                                                 + pController->currentUserId()))
    {
        highlightType = MessageForUserHighlight;
        highlightPostColor = highlightColors.at(0); // Color #1 in config
    }    
    else if (activity->author()->getId() == pController->currentUserId() // Post or activity is ours
          || activity->object()->author()->getId() == pController->currentUserId())
    {
        highlightType = OwnMessageHighlight;
        highlightPostColor = highlightColors.at(3); // Color #4 in the config
    }
    else if (highlightedByFilter) // Highlight by filtering rules
    {
        highlightType = FilterRulesHighlight;
        highlightPostColor = highlightColors.at(4); // Color #5
    }

    if (highlightType != NoHighlight
     && !standalone               // Don't use highlighting when opening as standalone post
     && !pController->currentUserId().isEmpty())
    {
        if (QColor::isValidColor(highlightPostColor))
        {
            // CSS for horizontal gradient from configured color to transparent
            QString css = QString("QFrame#LeftFrame "
                                  "{ background-color: "
                                  "  qlineargradient(spread:pad, "
                                  "  x1:0, y1:0, x2:1, y2:0, "
                                  "  stop:0 %1, stop:1 rgba(0, 0, 0, 0)); "
                                  "}")
                          .arg(highlightPostColor);

            leftColumnFrame->setObjectName("LeftFrame");
            leftColumnFrame->setStyleSheet(css);
        }
        else
        {
            this->leftColumnFrame->setFrameStyle(QFrame::Panel);
        }
    }

    this->unreadPostColor = highlightColors.at(5); // Color #6

    leftColumnLayout = new QVBoxLayout();


    this->postId = activity->object()->getId();
    this->postType = activity->object()->getType();
    QString postTypeString = activity->object()->getTranslatedType(postType);

    this->postUrl = activity->object()->getUrl();


    ASPerson *authorPerson;
    // Having ID means the object has proper author data
    if (!activity->object()->author()->getId().isEmpty())
    {
        authorPerson = activity->object()->author();
    }
    else // No ID means post author data is the activity's author data
    {    // This is a workaround, because pump.io doesn't give object author
         // if the activity is by the same user
        authorPerson = activity->author();
    }
    this->postAuthorId = authorPerson->getId();
    this->postAuthorName = authorPerson->getNameWithFallback();


    // Save post title for openClickedURL(), editPost() etc
    this->postTitle = activity->object()->getTitle();

    QString generator = activity->getGenerator();


    // If it's a standalone post, set window title and restore size
    if (standalone)
    {
        QString windowTitle = tr("Post", "Noun, not verb")
                              + ": " + postTypeString + " - Dianara";
        this->setWindowTitle(windowTitle);
        this->setWindowFlags(Qt::Dialog);
        this->setWindowModality(Qt::WindowModal);

        // Restore size
        this->setMinimumSize(420, 360);
        QSettings settings;
        this->resize(settings.value("Post/postSize",
                                    QSize(600, 440)).toSize());
    }

    postIsUnread = false;


    QFont detailsFont;
    detailsFont.setPointSize(detailsFont.pointSize() - 2); // FIXME: check size first


    /////////////////////////////////////////////////// Left column, post Meta info

    // Is the post a reshare? Indicate who shared it
    postIsSharedLabel = new QLabel(this); // With parent to avoid leaks
    postIsSharedLabel->hide();
    if (activity->isShared())
    {
        postIsSharedLabel->show();
        this->postSharedById = activity->getSharedById();

        QString sharedByName = MiscHelpers::fixLongName(activity->getSharedByName());
        postIsSharedLabel->setText("<p>"
                                   + QString::fromUtf8("\342\231\272  ") // Recycling symbol
                                   + tr("Via %1").arg(sharedByName)
                                   + QString::fromUtf8("  \342\231\272")
                                   + "</p> &nbsp;");
        postIsSharedLabel->setWordWrap(true);
        postIsSharedLabel->setAlignment(Qt::AlignCenter);
        postIsSharedLabel->setFont(detailsFont);
        postIsSharedLabel->setForegroundRole(QPalette::Text);
        postIsSharedLabel->setBackgroundRole(QPalette::Base);
        postIsSharedLabel->setAutoFillBackground(true);
        postIsSharedLabel->setFrameStyle(QFrame::Panel | QFrame::Raised);
        postIsSharedLabel->setSizePolicy(QSizePolicy::Ignored,
                                         QSizePolicy::Expanding);


        sharedByTooltipString = "<img align=\"left\" src=\""
                                + MiscHelpers::getCachedAvatarFilename(activity->getSharedByAvatar())
                                + "\" width=\"48\" />";
        sharedByTooltipString.append("&nbsp;&nbsp;<b>"
                                     + sharedByName
                                     + "</b><br>");
        sharedByTooltipString.append("&nbsp;&nbsp;<i>"
                                     + this->postSharedById
                                     + "</i>");

        shareTime = activity->getUpdatedAt();
        QString exactShareTime = Timestamp::localTimeDate(shareTime);
        sharedByTooltipString.append("<br><hr><br>"
                                     + tr("Shared on %1").arg(exactShareTime));
        // Fuzzy timestamp will be inserted here, in setFuzzyTimestamps()
        if (!generator.isEmpty())
        {
            sharedByGeneratorString = " " + tr("using %1").arg(generator);
            generator.clear(); // So it's not used in the post timestamp tooltip!
        }
        postIsSharedLabel->setToolTip(sharedByTooltipString);
        leftColumnLayout->addWidget(postIsSharedLabel);
        leftColumnLayout->addSpacing(2);
    }    



    // Different frame if post is ours
    if (postAuthorId == pController->currentUserId())
    {
        postIsOwn = true;
        qDebug() << "Post is our own!";

        this->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    }
    else
    {
        postIsOwn = false;
        this->setFrameStyle(QFrame::Box | QFrame::Raised);
    }


    // Author avatar
    postAuthorAvatarButton = new AvatarButton(authorPerson,
                                              this->pController,
                                              this->globalObj,
                                              QSize(48, 48),
                                              this);

    ///////////// Add extra options to the avatar menu
    postAuthorAvatarButton->addSeparatorToMenu();

    // Open post in browser
    openPostInBrowserAction = new QAction(QIcon::fromTheme("internet-web-browser"),
                                          tr("Open post in web browser"),
                                          this);
    connect(openPostInBrowserAction, SIGNAL(triggered()),
            this, SLOT(openPostInBrowser()));
    postAuthorAvatarButton->addActionToMenu(openPostInBrowserAction);

    // Copy URL
    copyPostUrlAction = new QAction(QIcon::fromTheme("edit-copy"),
                                    tr("Copy post link to clipboard"),
                                    this);
    connect(copyPostUrlAction, SIGNAL(triggered()),
            this, SLOT(copyPostUrlToClipboard()));
    postAuthorAvatarButton->addActionToMenu(copyPostUrlAction);

    // Disable Open and Copy URL if there's no URL
    if (this->postUrl.isEmpty())
    {
        openPostInBrowserAction->setDisabled(true);
        copyPostUrlAction->setDisabled(true);
    }


    // -----
    postAuthorAvatarButton->addSeparatorToMenu();

    // Normalize text colors
    normalizeTextAction = new QAction(QIcon::fromTheme("format-text-color"),
                                      tr("Normalize text colors"),
                                      this);
    connect(normalizeTextAction, SIGNAL(triggered()),
            this, SLOT(normalizeTextFormat()));
    postAuthorAvatarButton->addActionToMenu(normalizeTextAction);


    if (standalone) // own window, add close option to menu
    {
        // -----
        postAuthorAvatarButton->addSeparatorToMenu();

        this->closeAction = new QAction(QIcon::fromTheme("window-close"),
                                        tr("&Close"),
                                        this);
        connect(closeAction, SIGNAL(triggered()),
                this, SLOT(close()));
        postAuthorAvatarButton->addActionToMenu(closeAction);
    }


    // End avatar menu

    leftColumnLayout->addWidget(postAuthorAvatarButton, 0, Qt::AlignLeft);
    leftColumnLayout->addSpacing(2);


    QFont authorFont;
    authorFont.setBold(true);
    authorFont.setUnderline(true);
    authorFont.setPointSize(authorFont.pointSize()-1);
    if (postIsOwn) // Another visual hint when the post is our own
    {
        authorFont.setItalic(true);
    }

    // Author name
    QString authorName = MiscHelpers::fixLongName(postAuthorName);
    postAuthorNameLabel = new QLabel(authorName);
    postAuthorNameLabel->setTextFormat(Qt::PlainText);
    postAuthorNameLabel->setWordWrap(true);
    postAuthorNameLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postAuthorNameLabel->setFont(authorFont);
    postAuthorNameLabel->setToolTip(authorPerson->getTooltipInfo());
    // Ensure a certain width for the metadata column, by setting a minimum width for this label
    postAuthorNameLabel->setMinimumWidth(90); // FIXME: use font metrics for some chars instead


    leftColumnLayout->addWidget(postAuthorNameLabel);
    leftColumnLayout->addSpacing(2);


    // Post creation time
    postCreatedAtString = activity->object()->getCreatedAt();
    // and update time
    postUpdatedAtString = activity->object()->getUpdatedAt();
    if (postUpdatedAtString == postCreatedAtString)
    {
        postUpdatedAtString.clear();
    }
    // Format is "Combined date and time in UTC", like "2012-02-07T01:32:02Z"
    // as specified in ISO 8601 http://en.wikipedia.org/wiki/ISO_8601
    QString dateGeneratorTooltip = tr("Type", "As in: type of object") + ": "
                                   + postTypeString + "\n"
                                   + tr("Posted on %1", "1=Date")
                                     .arg(Timestamp::localTimeDate(postCreatedAtString));
    if (!generator.isEmpty())
    {
        dateGeneratorTooltip.append("\n" +
                                    tr("Using %1",
                                       "1=Program used for posting").arg(generator));
    }
    if (!postUpdatedAtString.isEmpty())
    {
        // Using \n instead of <br> because I don't want this tooltip to be HTML (wrapped)
        dateGeneratorTooltip.append("\n\n"
                                    + tr("Modified on %1")
                                      .arg(Timestamp::localTimeDate(postUpdatedAtString)));
    }

    postCreatedAtLabel = new HClabel();
    postCreatedAtLabel->setToolTip(dateGeneratorTooltip);
    postCreatedAtLabel->setTextFormat(Qt::PlainText);
    postCreatedAtLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postCreatedAtLabel->setFont(detailsFont);
    leftColumnLayout->addWidget(postCreatedAtLabel);
    leftColumnLayout->addSpacing(4);

    // Location information, if any
    QString location = activity->object()->getLocationName();
    if (!location.isEmpty())
    {
        this->postLocationLabel = new HClabel(tr("In")
                                              + ": <b>" + location + "</b>");
        postLocationLabel->setFont(detailsFont);
        postLocationLabel->setToolTip(activity->object()->getLocationTooltip());
        leftColumnLayout->addWidget(postLocationLabel);
        leftColumnLayout->addSpacing(2);
    }

    if (!activity->getToString().isEmpty())
    {
        this->postToLabel = new QLabel(tr("To") + ": "
                                       + activity->getToString());
        postToLabel->setWordWrap(true);
        postToLabel->setFont(detailsFont);
        postToLabel->setOpenExternalLinks(true);
        postToLabel->setSizePolicy(QSizePolicy::Ignored,
                                   QSizePolicy::Minimum);

        leftColumnLayout->addWidget(postToLabel);
        connect(postToLabel, SIGNAL(linkHovered(QString)),
                this, SLOT(showHighlightedUrl(QString)));
    }

    if (!activity->getCCString().isEmpty())
    {
        this->postCCLabel = new QLabel(tr("CC") + ": "
                                       + activity->getCCString());
        postCCLabel->setWordWrap(true);
        postCCLabel->setFont(detailsFont);
        postCCLabel->setOpenExternalLinks(true);
        postCCLabel->setSizePolicy(QSizePolicy::Ignored,
                                   QSizePolicy::Minimum);

        leftColumnLayout->addWidget(postCCLabel);
        connect(postCCLabel, SIGNAL(linkHovered(QString)),
                this, SLOT(showHighlightedUrl(QString)));
    }

    leftColumnLayout->addSpacing(4);


    this->postLikesUrl = activity->object()->getLikesUrl();
    this->postCommentsUrl = activity->object()->getCommentsUrl();
    this->postSharesUrl = activity->object()->getSharesUrl();


    // Set this QLabels with parent=this, since we're gonna hide them right away
    // They'll be reparented to the layout, but not having a parent before that
    // would cause visual glitches
    postLikesCountLabel = new HClabel("", this);
    postLikesCountLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postLikesCountLabel->setFont(detailsFont);
    leftColumnLayout->addWidget(postLikesCountLabel);
    postLikesCountLabel->hide();
    int likesCount = activity->object()->getLikesCount().toInt();
    this->setLikesLabel(likesCount);


    postCommentsCountLabel = new QLabel(this);
    postCommentsCountLabel->setWordWrap(true);
    postCommentsCountLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postCommentsCountLabel->setFont(detailsFont);
    leftColumnLayout->addWidget(postCommentsCountLabel);
    postCommentsCountLabel->hide();
    int commentsCount = activity->object()->getCommentsCount().toInt();
    this->setCommentsLabel(commentsCount);


    postResharesCountLabel = new HClabel("", this);
    postResharesCountLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postResharesCountLabel->setFont(detailsFont);
    leftColumnLayout->addWidget(postResharesCountLabel);
    postResharesCountLabel->hide();
    int sharesCount = activity->object()->getSharesCount().toInt();
    this->setSharesLabel(sharesCount);



    // Try to use all remaining space, aligning
    // all previous widgets nicely at the top
    // leftColumnLayout->setAlignment(Qt::AlignTop) caused glitches
    leftColumnLayout->addStretch(1);

    QFont buttonsFont;
    buttonsFont.setPointSize(buttonsFont.pointSize() - 1);


    // Check if the object is in reply to something (a shared comment, for instance)
    this->postParentMap = activity->object()->getInReplyTo();
    if (!postParentMap.isEmpty())
    {
        openParentPostButton = new QPushButton(QIcon::fromTheme("go-up-search",
                                                                QIcon(":/images/button-parent.png")),
                                               tr("Parent",
                                                  "As in 'Open the parent post'. "
                                                  "Try to use the shortest word!"));
        //openParentPostButton->setFlat(true);
        openParentPostButton->setFont(buttonsFont);
        openParentPostButton->setSizePolicy(QSizePolicy::Ignored,
                                            QSizePolicy::Maximum);
        openParentPostButton->setToolTip("<b></b>"
                                         + tr("Open the parent post, to which "
                                              "this one replies"));
        connect(openParentPostButton, SIGNAL(clicked()),
                this, SLOT(openParentPost()));
        leftColumnLayout->addWidget(openParentPostButton);
    }

    // If showing standalone, add a Close button; needed in some environments
    // Button disabled for now; added option to avatar menu
#if 0
    if (standalone)
    {
        this->closeButton = new QPushButton(QIcon::fromTheme("window-close"),
                                            tr("&Close"));
        closeButton->setFont(buttonsFont);
        closeButton->setSizePolicy(QSizePolicy::Ignored,
                                   QSizePolicy::Maximum);
        connect(closeButton, SIGNAL(clicked()),
                this, SLOT(close()));

        leftColumnLayout->addWidget(closeButton);
    }
#endif

    leftColumnFrame->setLayout(leftColumnLayout);


    //////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////// Right column, content

    rightColumnFrame = new QFrame();
    rightColumnFrame->setObjectName("RightFrame"); // For css changes later

    rightColumnLayout = new QVBoxLayout();
    rightColumnLayout->setAlignment(Qt::AlignTop);

    ///////////////////////////////////// Title
    if (!this->postTitle.isEmpty())
    {
        this->postTitleLabel = new QLabel();
        postTitleLabel->setWordWrap(true);
        QFont postTitleFont;
        postTitleFont.fromString(globalObj->getPostTitleFont());
        postTitleLabel->setFont(postTitleFont);
        postTitleLabel->setText(postTitle);
        rightColumnLayout->addWidget(postTitleLabel, 0);
    }

    ///////////////////////////////////// Summary
    QString postSummary = activity->object()->getSummary();
    if (!postSummary.isEmpty())
    {
        this->postSummaryLabel = new QLabel();
        postSummaryLabel->setWordWrap(true);
        postSummaryLabel->setFont(detailsFont);
        postSummaryLabel->setText(postSummary);
        rightColumnLayout->addWidget(postSummaryLabel, 0);
    }


    ///////////////////////////////////// Post text
    postText = new QTextBrowser();
    postText->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postText->setSizePolicy(QSizePolicy::MinimumExpanding,
                            QSizePolicy::MinimumExpanding);
    postText->setMinimumSize(10, 10);
    postText->setWordWrapMode(QTextOption::WrapAtWordBoundaryOrAnywhere);
    postText->setOpenLinks(false); // don't open links, manage in openClickedURL()
    postText->setReadOnly(true); // it's default with QTextBrowser, but still...

    QFont postContentsFont;
    postContentsFont.fromString(globalObj->getPostContentsFont());
    postText->setFont(postContentsFont);

    // To help screen readers, enable keyboard interaction here
    // This is disabled for now, since it doesn't really help; hopefully with Qt5
    // things will be better.
    //postText->setTextInteractionFlags(Qt::TextBrowserInteraction
    //                                  | Qt::TextSelectableByKeyboard);
    connect(postText, SIGNAL(anchorClicked(QUrl)),
            this, SLOT(openClickedURL(QUrl)));
    connect(postText, SIGNAL(highlighted(QString)),
            this, SLOT(showHighlightedUrl(QString)));
    rightColumnLayout->addWidget(postText, 5);


    // Buttons
    // Add like, comment, share and, if post is ours, edit and delete buttons
    buttonsLayout = new QHBoxLayout();
    buttonsLayout->setAlignment(Qt::AlignHCenter | Qt::AlignTop);
    buttonsLayout->setContentsMargins(0, 0, 0, 0);
    buttonsLayout->setMargin(0);
    buttonsLayout->setSpacing(0);


    // Like button
    likeButton = new QPushButton(QIcon::fromTheme("emblem-favorite",
                                                  QIcon(":/images/button-like.png")),
                                 "*like*");
    likeButton->setCheckable(true);
    this->fixLikeButton(activity->object()->isLiked());
    likeButton->setFlat(true);
    likeButton->setFont(buttonsFont);
    connect(likeButton, SIGNAL(clicked(bool)),
            this, SLOT(likePost(bool)));
    buttonsLayout->addWidget(likeButton,        0, Qt::AlignLeft);

    // Only add Comment and Share buttons if it's NOT a comment
    if (postType != "comment") // (can happen in the Favorites timeline or via shares)
    {
        // Comment (reply) button
        commentButton = new QPushButton(QIcon::fromTheme("mail-reply-sender",
                                                         QIcon(":/images/button-comment.png")),
                                        tr("Comment",
                                           "verb, for the comment button"));
        commentButton->setToolTip(tr("Reply to this post.")
                                  + "<br>"
                                  + tr("If you select some text, it will be quoted."));
        commentButton->setFlat(true);
        commentButton->setFont(buttonsFont);
        connect(commentButton, SIGNAL(clicked()),
                this, SLOT(commentOnPost()));
        buttonsLayout->addWidget(commentButton, 0, Qt::AlignLeft);


        // Share button
        shareButton = new QPushButton(QIcon::fromTheme("mail-forward",
                                                       QIcon(":/images/button-share.png")),
                                      "*share*");
        // Note: Gwenview includes 'document-share' icon
        shareButton->setFlat(true);
        shareButton->setFont(buttonsFont);
        if (postSharedById.isEmpty()
            || (pController->currentUserId() != this->postSharedById))
        {
            shareButton->setText(tr("Share"));
            shareButton->setToolTip("<b></b>"
                                    + tr("Share this post with your contacts"));
            connect(shareButton, SIGNAL(clicked()),
                    this, SLOT(sharePost()));
        }
        else // Shared by us!
        {
            shareButton->setText(tr("Unshare"));
            shareButton->setToolTip("<b></b>"
                                    + tr("Unshare this post"));
            connect(shareButton, SIGNAL(clicked()),
                    this, SLOT(unsharePost()));
            shareButton->setDisabled(true); // FIXME: disabled for 1.2.x, since it doesn't really work
        }
        buttonsLayout->addWidget(shareButton,   0, Qt::AlignLeft);
    }

    buttonsLayout->addStretch(1); // so the (optional) Edit and Delete buttons get separated

    if (postIsOwn)
    {
        editButton = new QPushButton(QIcon::fromTheme("document-edit",
                                                      QIcon(":/images/button-edit.png")),
                                     tr("Edit"));
        editButton->setToolTip("<b></b>"
                               + tr("Modify this post"));
        editButton->setFlat(true);
        editButton->setFont(buttonsFont);
        connect(editButton, SIGNAL(clicked()),
                this, SLOT(editPost()));
        buttonsLayout->addWidget(editButton, 0, Qt::AlignRight);

        deleteButton = new QPushButton(QIcon::fromTheme("edit-delete",
                                                        QIcon(":/images/button-delete.png")),
                                       tr("Delete"));
        deleteButton->setToolTip("<b></b>"
                                 + tr("Erase this post"));
        deleteButton->setFlat(true);
        deleteButton->setFont(buttonsFont);
        connect(deleteButton, SIGNAL(clicked()),
                this, SLOT(deletePost()));

        buttonsLayout->addWidget(deleteButton, 0, Qt::AlignRight);
    }



    /******************************************************************/

    this->pendingImagesList.clear(); // Will add the own "post image", and <img>-based ones

    // Get URL of post image, if it's "image" type of post
    if (this->postType == "image")
    {
        postImageUrl = activity->object()->getImageUrl();
        postAttachmentPureUrl = activity->object()->getAttachmentPureUrl();

        pendingImagesList.append(postImageUrl);
    }


    QString attachmentUrl;
    // Get URL of post audio file, if it's "audio" type of post
    if (this->postType == "audio")
    {
        postAudioUrl = activity->object()->getAudioUrl();
        attachmentUrl = postAudioUrl;
        postAttachmentPureUrl = activity->object()->getAttachmentPureUrl();

        qDebug() << "we got AUDIO:" << postAudioUrl;
    }

    // Get URL of post video file, if it's "video" type of post
    if (this->postType == "video")
    {
        postVideoUrl = activity->object()->getVideoUrl();
        attachmentUrl = postVideoUrl;
        postAttachmentPureUrl = activity->object()->getAttachmentPureUrl();

        qDebug() << "we got VIDEO:" << postVideoUrl;
    }

    // Get URL of post general file, if it's "file" type of post
    if (this->postType == "file")
    {
        postFileUrl = activity->object()->getFileUrl();
        attachmentUrl = postFileUrl;
        postAttachmentPureUrl = activity->object()->getAttachmentPureUrl();

        postFileMimeType = activity->object()->getMimeType();
        qDebug() << "we got FILE:" << postFileUrl << "; type:" << postFileMimeType;
    }


    this->postOriginalText = activity->object()->getContent(); // Save it for later...
    QStringList postTextImageList = MiscHelpers::htmlWithReplacedImages(postOriginalText,
                                                                        this->postText->width());
    postTextImageList.removeFirst(); // First is the HTML itself

    qDebug() << "Post has" << postTextImageList.size() << "images included...";

    pendingImagesList.append(postTextImageList);
    this->getPendingImages();


    // Button to join/leave the group, if the post is the creation of a group
    if (postType == "group") // FIXME: for now, only JOIN, as we can't know if we're members yet
    {
        this->joinLeaveButton = new QPushButton(QIcon::fromTheme("user-group-new"),
                                                tr("Join Group"));
        connect(joinLeaveButton, SIGNAL(clicked()),
                this, SLOT(joinGroup()));
        rightColumnLayout->addWidget(joinLeaveButton, 0, Qt::AlignCenter);

        this->groupInfoLabel = new QLabel(tr("%1 members in the group")
                                          .arg(activity->object()->getMemberCount()));
        groupInfoLabel->setWordWrap(true);
        groupInfoLabel->setFont(detailsFont);
        groupInfoLabel->setAlignment(Qt::AlignCenter);
        rightColumnLayout->addWidget(groupInfoLabel);
    }


    // Widget to download the attached media, if any
    if (!attachmentUrl.isEmpty())
    {
        QString suggestedFilename = MiscHelpers::getSuggestedFilename(this->postAuthorId,
                                                                      this->postType,
                                                                      this->postTitle,
                                                                      this->postAttachmentPureUrl);
        this->downloadWidget = new DownloadWidget(attachmentUrl,
                                                  suggestedFilename,
                                                  this->pController);

        rightColumnLayout->addWidget(downloadWidget, 0);
    }


    this->resizesCount = 0;
    // Post resizing takes place in resizeEvent()
    // which will also call setPostContents()


    // FIXME: button creation should be moved here
    rightColumnLayout->addLayout(buttonsLayout, 0);


    /////////////////////////////////////////////////////// Comments block
    this->commenter = new CommenterBlock(this->pController,
                                         this->globalObj,
                                         this);
    connect(commenter, SIGNAL(commentSent(QString)),
            this, SLOT(sendComment(QString)));
    connect(commenter, SIGNAL(commentUpdated(QString,QString)),
            this, SLOT(updateComment(QString,QString)));
    connect(commenter, SIGNAL(allCommentsRequested()),
            this, SLOT(getAllComments()));
    rightColumnLayout->addWidget(commenter,     0);

    rightColumnFrame->setLayout(rightColumnLayout);

    // Set the initial likes, comments and shares (4 most recent)
    this->setLikes(activity->object()->getLastLikesList(),
                   likesCount);
    this->commenter->setComments(activity->object()->getLastCommentsList(),
                                 commentsCount);
    this->setShares(activity->object()->getLastSharesList(),
                    sharesCount);


    mainLayout = new QHBoxLayout();
    mainLayout->setContentsMargins(0, 0, 0, 0);   // Minimal margins
    mainLayout->addWidget(leftColumnFrame,   2);  // stretch 2/13
    mainLayout->addWidget(rightColumnFrame, 11);  // stretch 11/13

    mainLayout->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    this->setLayout(mainLayout);


    this->minMaxHeight = 400; // Initialize, but it'll be set before it's used anyway

    // Set read status
    this->setPostUnreadStatus();

    // Set timestamps; Doing this here, after CommenterBlock has been initialized too
    this->setFuzzyTimestamps();


    qDebug() << "Post created" << this->postId;
}



Post::~Post()
{
    qDebug() << "Post destroyed" << this->postId;
}




void Post::setMinMaxHeight(int newMinMaxHeight)
{
    // Substract some pixels to account for the row of buttons, etc.
    this->minMaxHeight = qMax(newMinMaxHeight - 80,
                              50);
}



/*
 * Set the post contents, and trigger a vertical resize
 *
 */
void Post::setPostContents()
{
    //qDebug() << "Post::setPostContents()" << this->postID;
    QString postMediaHtml;

    // Embedded image
    int imageWidth;
    if (!this->postImageUrl.isEmpty())
    {
        QString imageCachedFilename = MiscHelpers::getCachedImageFilename(postImageUrl);

        if (QFile::exists(imageCachedFilename))
        {
            // If the image is wider than the post space, make it smaller
            imageWidth = qMin(MiscHelpers::getImageWidth(imageCachedFilename),
                              this->postWidth);

            this->postImageIsAnimated = MiscHelpers::isImageAnimated(imageCachedFilename);

            QString belowMessage;
            if (postImageIsAnimated)
            {
                belowMessage = tr("Image is animated. Click on it to play.");
            }
            else
            {
                belowMessage = this->seeFullImageString;
            }

            postMediaHtml = MiscHelpers::mediaHtmlBase(this->postType,
                                                       imageCachedFilename,
                                                       this->seeFullImageString,
                                                       belowMessage,
                                                       imageWidth);
        }
        else // use placeholder image while it loads...
        {
            const QString imageLoadingString = tr("Loading image...");
            postMediaHtml = "<br>"
                            "<div align=center "
                            "title=\"" + imageLoadingString + "\">"
                            "<img src=\":/images/image-loading.png\"  />"
                            "<br><br><b>" + imageLoadingString + "</b>"
                            "</div>"
                            "<hr><br>";
        }
    }

    // Embedded audio
    if (!this->postAudioUrl.isEmpty())
    {
        postMediaHtml = MiscHelpers::mediaHtmlBase(this->postType,
                                                   ":/images/attached-audio.png",
                                                   this->downloadAttachmentString,
                                                   tr("Attached Audio"));
    }


    // Embedded video
    if (!this->postVideoUrl.isEmpty())
    {
        postMediaHtml = MiscHelpers::mediaHtmlBase(this->postType,
                                                   ":/images/attached-video.png",
                                                   this->downloadAttachmentString,
                                                   tr("Attached Video"));
    }


    // Attached file
    if (!this->postFileUrl.isEmpty())
    {
        postMediaHtml = MiscHelpers::mediaHtmlBase(this->postType,
                                                   ":/images/attached-file.png",
                                                   this->downloadAttachmentString,
                                                   tr("Attached file")
                                                   + ": " + postFileMimeType);
    }



    // Text contents
    QStringList postTextImageList = MiscHelpers::htmlWithReplacedImages(postOriginalText,
                                                                        this->postWidth);
    QString postTextContents = postTextImageList.takeAt(0);

    // Add the text content of the post
    postText->setHtml(postMediaHtml
                    + postTextContents);


    this->setPostHeight();
}


void Post::onResizeOrShow()
{
    this->postWidth = qMax(postText->width() - 80,   // FIXME: use viewport?
                           50);                      // minimum width of 50

    this->setPostContents();
}



/*
 * Set the height of the post, based on the contents
 *
 */
void Post::setPostHeight()
{
    int height = qMax(postText->document()->size().toSize().height() + 12, // +12px error margin
                      60); // Minimum height of 60

    height = qMin(height, this->minMaxHeight); // Don't allow a post to be too tall

    // Don't force a specific height if the post is standalone
    if (!standalone)
    {
        postText->setMinimumHeight(height);
        postText->setMaximumHeight(height);
    }
}


void Post::resetResizesCount()
{
    this->resizesCount = 0;
}


/*
 * Download post image, and img-tag-based images parsed from the contents
 *
 */
void Post::getPendingImages()
{
    if (!pendingImagesList.isEmpty())
    {
        foreach (QString imageUrl, pendingImagesList)
        {
            pController->enqueueImageForDownload(imageUrl);
        }

        connect(pController, SIGNAL(imageStored(QString)),
                this, SLOT(redrawImages(QString)));
    }
}




/*
 * Return the likes/favorites URL for this post
 *
 */
QString Post::likesURL()
{
    return this->postLikesUrl;
}


/*
 * Update the tooltip in "%NUMBER likes" with the names
 * of the people who liked the post
 *
 */
void Post::setLikes(QVariantList likesList, int likesCount)
{
    if (likesList.size() > 0)
    {
        QString peopleString; // FIXME: repeated in Comment(), move to method
        foreach (QVariant likesMap, likesList)
        {
            QString name = likesMap.toMap().value("displayName").toString();
            if (name.isEmpty())
            {
                name = ASPerson::cleanupId(likesMap.toMap()
                                                   .value("id")
                                                   .toString()); // fallback
            }
            peopleString.append(name + ", ");
        }

        // Remove last comma and add "like this"
        peopleString.remove(-2, 2); // Last 2 characters

        QString likesString;
        if (likesList.size() == 1)
        {
            likesString = tr("%1 likes this",
                             "One person").arg(peopleString);
        }
        else
        {
            likesString = tr("%1 like this",
                             "More than one person").arg(peopleString);
        }
        likesString.prepend("<b></b>"); // Turn the tooltip into rich text

        this->postLikesCountLabel->setToolTip(likesString);
    }

    // TMP/FIXME this can set the number to lower than initially set
    // if called with the "initial" up-to-4 likes list that comes with the post
    if (likesCount != -1)
    {
        this->setLikesLabel(likesCount); // use the passed likesCount parameter
    }
    else
    {
        this->setLikesLabel(likesList.size()); // show size of actual names list
    }
}


/*
 * Update the "NUMBER likes" label in left side
 *
 */
void Post::setLikesLabel(int likesCount)
{
    if (likesCount != 0)
    {
        if (likesCount == 1)
        {
            postLikesCountLabel->setText(QString::fromUtf8("\342\231\245 ") // heart symbol
                                  + tr("1 like"));
        }
        else
        {
            postLikesCountLabel->setText(QString::fromUtf8("\342\231\245 ") // heart symbol
                                  + tr("%1 likes").arg(likesCount));
        }
        postLikesCountLabel->show();
    }
    else
    {
        postLikesCountLabel->clear();
        postLikesCountLabel->hide();
    }
}


/*
 * Return the comments URL for this post
 *
 */
QString Post::commentsURL()
{
    return this->postCommentsUrl;
}


/*
 * Ask the Commenter to set new comments
 *
 */
void Post::setComments(QVariantList commentsList)
{
    int commentCount = commentsList.size();

    this->commenter->setComments(commentsList, commentCount);

    // update number of comments in left side counter
    this->setCommentsLabel(commentCount);

    if (commentCount > 0)
    {
        rightColumnLayout->setStretchFactor(commenter, 3);
        /*if (standalone)
        {
            commenter->setMinimumHeight(commenter->sizeHint().height());
            qDebug() << "Post:setComments(); setting commenter minimum height:"
                     << commenter->sizeHint().height();
        }*/
    }
}


/*
 * Update the "NUMBER comments" label in left side
 *
 */
void Post::setCommentsLabel(int commentsCount)
{
    if (commentsCount != 0)
    {
        if (commentsCount == 1)
        {
            postCommentsCountLabel->setText(QString::fromUtf8("\342\234\215 ") // writing hand
                                            + tr("1 comment"));
        }
        else
        {
            postCommentsCountLabel->setText(QString::fromUtf8("\342\234\215 ") // writing hand
                                            + tr("%1 comments").arg(commentsCount));
        }
        postCommentsCountLabel->show();
    }
    else
    {
        postCommentsCountLabel->clear();
        postCommentsCountLabel->hide();
    }
}


/*
 * Return the shares URL for this post,
 * list of people who reshared it
 *
 */
QString Post::sharesURL()
{
    return this->postSharesUrl;
}


/*
 * Update the tooltip for "%NUMBER shares" with names
 *
 */
void Post::setShares(QVariantList sharesList, int sharesCount)
{
    if (sharesList.size() > 0)
    {
        QString peopleString;
        foreach (QVariant sharesMap, sharesList)
        {
            peopleString = sharesMap.toMap().value("displayName").toString() + ", "
                           + peopleString;
        }

        // Remove last comma and add "shared this"
        peopleString.remove(-2, 2); // Last 2 characters

        QString sharesString;
        if (sharesList.size() == 1)
        {
            sharesString = tr("%1 shared this", "%1 = One person name").arg(peopleString);
        }
        else
        {
            sharesString = tr("%1 shared this", "%1 = Names for more than one person").arg(peopleString);
        }
        sharesString.prepend("<b></b>");  // So that the label is rich text, wordwrapped

        this->postResharesCountLabel->setToolTip(sharesString);
    }

    // TMP/FIXME this can set the number to lower than initially set
    // if called with the "initial" up-to-4 shares list that comes with the post
    if (sharesCount != -1)
    {
        this->setSharesLabel(sharesCount); // use the passed sharesCount parameter
    }
    else
    {
        this->setSharesLabel(sharesList.size()); // show size of actual names list
    }
}


void Post::setSharesLabel(int resharesCount)
{
    if (resharesCount != 0)
    {
        if (resharesCount == 1)
        {
            postResharesCountLabel->setText(QString::fromUtf8("\342\231\273 ") // recycle symbol
                                      + tr("Shared once"));
        }
        else
        {
            postResharesCountLabel->setText(QString::fromUtf8("\342\231\273 ") // recycle symbol
                                      + tr("Shared %1 times").arg(resharesCount));
        }

        postResharesCountLabel->show();
    }
    else
    {
        postResharesCountLabel->clear();
        postResharesCountLabel->hide();
    }
}



/*
 * Set or unset the visual hint indicating if the post is unread
 *
 */
void Post::setPostUnreadStatus()
{
    if (!QColor::isValidColor(unreadPostColor))
    {
        unreadPostColor = "palette(highlight)";
    }

    // CSS for horizontal gradient from configured color to transparent
    QString css = QString("QFrame#RightFrame "
                          "{ background-color: "
                          "  qlineargradient(spread:pad, "
                          "  x1:0, y1:0, x2:1, y2:0,  "
                          "  stop:0 rgba(0, 0, 0, 0), "
                          "  stop:1 %1); "
                          "}").arg(unreadPostColor);

    if (this->postIsUnread)
    {
        postCreatedAtLabel->setHighlighted(true);

        if (!css.isEmpty())
        {
            rightColumnFrame->setStyleSheet(css);
        }
        else // Unused at the moment, but could be an option to avoid gradients
        {
            this->setAutoFillBackground(true);
            this->setBackgroundRole(QPalette::Mid);
        }
    }
    else
    {
        postCreatedAtLabel->setHighlighted(false);

        rightColumnFrame->setStyleSheet("");

        this->setAutoFillBackground(true);
        this->setBackgroundRole(QPalette::Window);
    }

    // Avoid possible later flickering of the "new" effect
    this->postCreatedAtLabel->update();
    this->update();
}


void Post::setPostAsNew()
{
    this->postIsUnread = true;
    setPostUnreadStatus();
}


void Post::setPostAsRead()
{
    if (postIsUnread)
    {
        this->postIsUnread = false;

        // Inform the Timeline()
        if (this->highlightType == NoHighlight)
        {
            emit postRead(false);
        }
        else
        {
            emit postRead(true); // Say it's marked as read, and was highlighted
        }


        setPostUnreadStatus();
    }
}

int Post::getHighlightType()
{
    return this->highlightType;
}


void Post::setFuzzyTimestamps()
{
    QString fuzzyTimestamps = Timestamp::fuzzyTime(postCreatedAtString);

    if (!postUpdatedAtString.isEmpty())
    {
        fuzzyTimestamps.append("\n"
                               + tr("Edited: %1")
                                 .arg(Timestamp::fuzzyTime(postUpdatedAtString)));
    }

    this->postCreatedAtLabel->setText(fuzzyTimestamps);

    // Update "share time" on share info tooltip too!
    if (!sharedByTooltipString.isEmpty())
    {
        QString shareTooltip = sharedByTooltipString;
        shareTooltip.append(QString("<br>(%1)")
                            .arg(Timestamp::fuzzyTime(shareTime)));
        shareTooltip.append(sharedByGeneratorString);
        this->postIsSharedLabel->setToolTip(shareTooltip);
    }

    commenter->updateFuzzyTimestamps();
}



////////////////////////////////////////////////////////////////////////////
////////////////////////////////// SLOTS ///////////////////////////////////
////////////////////////////////////////////////////////////////////////////


/*
 * Like (favorite) a post
 *
 */
void Post::likePost(bool like)
{
    qDebug() << "Post::likePost()" << (like ? "like" : "unlike");

    this->pController->likePost(this->postId, this->postType, like);

    this->fixLikeButton(like ? "true" : "false");

    connect(pController, SIGNAL(likeSet()),
            this, SLOT(getAllLikes()));
}

/*
 * Set the right labels and tooltips to the like button, depending on its state
 *
 */
void Post::fixLikeButton(QString state)
{
    if (state == "true")
    {
        likeButton->setToolTip("<b></b>"
                               + tr("You like this"));
        likeButton->setText(tr("Unlike"));
        likeButton->setChecked(true);
    }
    else
    {
        likeButton->setToolTip("<b></b>"
                               + tr("Like this post"));
        likeButton->setText(tr("Like"));
        likeButton->setChecked(false);
    }
}


void Post::getAllLikes()
{
    disconnect(pController, SIGNAL(likeSet()),
               this, SLOT(getAllLikes()));

    this->pController->getPostLikes(this->postLikesUrl);
}


/*
 * Make the commenter widget visible, so user can type the comment
 *
 * If some text was selected, insert it as quoted text
 *
 */
void Post::commentOnPost()
{
    qDebug() << "Commenting on post" << this->postTitle << this->postId;

    QString initialText;

    QString selectedText = this->postText->textCursor().selectedText();
    if (!selectedText.isEmpty())
    {
        // Selected text has literal < and >, so convert to HTML entities
        selectedText.replace("<", "&lt;");
        selectedText.replace(">", "&gt;");
        initialText = MiscHelpers::quotedText(this->postAuthorName,
                                              selectedText);
    }
    this->commenter->setFullMode(initialText);

    emit commentingOnPost(this->commenter);
}


/*
 * The actual sending of the comment to the Pump controller
 *
 */
void Post::sendComment(QString commentText)
{
    qDebug() << "About to publish this comment:" << commentText;

    this->pController->addComment(MiscHelpers::cleanupHtml(commentText),
                                  this->postId, this->postType);
}

void Post::updateComment(QString commentId, QString commentText)
{
    this->pController->updateComment(commentId,
                                     MiscHelpers::cleanupHtml(commentText));
}


void Post::getAllComments()
{
    this->pController->getPostComments(this->postCommentsUrl);
}


/*
 * Set all comments received from signal, when post is a separate window,
 * and not handled by Timeline()
 *
 */
void Post::setAllComments(QVariantList commentsList, QString originatingPostUrl)
{
    QString originatingPostCleanUrl = originatingPostUrl.split("?").at(0);
    if (this->commentsURL() == originatingPostCleanUrl)
    {
        this->setComments(commentsList);
    }
}


/*
 * Re-share a post
 *
 */
void Post::sharePost()
{
    int confirmation = QMessageBox::question(this, tr("Share post?"),
                                             tr("Do you want to share %1's post?")
                                              .arg(this->postAuthorNameLabel->text()),
                                             tr("&Yes, share it"), tr("&No"),
                                             "", 1, 1);

    if (confirmation == 0)
    {
        qDebug() << "Sharing this post:" << this->postId;
        this->pController->sharePost(this->postId, this->postType);
    }
    else
    {
        qDebug() << "Confirmation canceled, not sharing";
    }
}

void Post::unsharePost()
{
    int confirmation = QMessageBox::question(this, tr("Unshare post?"),
                                             tr("Do you want to unshare %1's post?")
                                              .arg(this->postAuthorNameLabel->text()),
                                             tr("&Yes, unshare it"), tr("&No"),
                                             "", 1, 1);

    if (confirmation == 0)
    {
        qDebug() << "Unsharing this post:" << this->postId;
        this->pController->unsharePost(this->postId, this->postType);

        this->setDisabled(true); // Disable the widget, to let user know it's been unshared
    }
    else
    {
        qDebug() << "Confirmation canceled, will not unshare";
    }
}


/*
 * Set the Publisher in editing mode with this post's contents
 *
 */
void Post::editPost()
{
    this->globalObj->editPost(this->postId,
                              this->postTitle,
                              this->postOriginalText);
    if (standalone)
    {
        this->close();
    }
}



/*
 * Delete a post
 *
 */
void Post::deletePost()
{
    int confirmation = QMessageBox::question(this, tr("WARNING: Delete post?"),
                                             tr("Are you sure you want to delete this post?"),
                                             tr("&Yes, delete it"), tr("&No"), "", 1, 1);

    if (confirmation == 0)
    {
        qDebug() << "Deleting post";
        this->pController->deletePost(this->postId, this->postType);

        this->setDisabled(true); // disable... maybe hide?
    }
    else
    {
        qDebug() << "Confirmation cancelled, not deleting the post";
    }
}


void Post::joinGroup()
{
    qDebug() << "Joining group " << this->postId << "via Post() button";
    this->pController->joinGroup(this->postId);

    this->joinLeaveButton->setDisabled(true); // FIXME; make it turn into "Leave", etc.
}



void Post::openClickedURL(QUrl url)
{
    qDebug() << "Anchor URL clicked:" << url.toString();

    if (url.scheme() == "image") // Use our own viewer, or maybe a configured external viewer
    {
        qDebug() << "Opening this image in our own viewer...";
        QString suggestedFilename = MiscHelpers::getSuggestedFilename(this->postAuthorId,
                                                                      this->postType,
                                                                      this->postTitle,
                                                                      this->postAttachmentPureUrl);
        ImageViewer *viewer = new ImageViewer(url.toString(),
                                              this->postTitle,
                                              suggestedFilename,
                                              this->postImageIsAnimated);
        viewer->show();
    }
    else if (url.scheme() == "attachment")
    {
        this->downloadWidget->downloadAttachment();
    }
    else
    {
        qDebug() << "Opening this link in browser";
        QDesktopServices::openUrl(url);
    }
}



void Post::showHighlightedUrl(QString url)
{
    if (!url.isEmpty())
    {
        if (url.startsWith("image:/")) // Own image:/ URL
        {
            this->pController->showTransientMessage(this->seeFullImageString);
        }
        else if (url.startsWith("attachment:/"))
        {
            this->pController->showTransientMessage(this->downloadAttachmentString);
        }
        else  // Normal URL
        {
            this->pController->showTransientMessage(url);
            qDebug() << "Highlighted url in post:" << url;
        }
    }
    else
    {
        this->pController->showTransientMessage("");
    }
}




void Post::openPostInBrowser()
{
    QDesktopServices::openUrl(this->postUrl);
}

void Post::copyPostUrlToClipboard()
{
    QApplication::clipboard()->setText(this->postUrl);
}

void Post::openParentPost()
{
    qDebug() << "Opening parent post...";

    // Create a fake activity for the parent post
    QVariantMap fakeActivityMap;
    fakeActivityMap.insert("object", this->postParentMap);
    fakeActivityMap.insert("actor",  this->postParentMap.value("author").toMap());

    ASActivity *originalPostActivity = new ASActivity(fakeActivityMap, this);

    Post *parentPost = new Post(originalPostActivity,
                                false, // Not highlighted
                                true,  // Post is standalone
                                pController,
                                globalObj,
                                this->parentWidget()); // Pass parent widget (Timeline or
                                                       // another Post) instead of
                                                       // 'this', so it won't be killed by reloads
    parentPost->show();
    connect(pController, SIGNAL(commentsReceived(QVariantList,QString)),
            parentPost, SLOT(setAllComments(QVariantList,QString)));
    parentPost->getAllComments();
}


/*
 * Normalize text colors; Used from menu when a post
 * has white text with white background, or similar
 *
 */
void Post::normalizeTextFormat()
{
    postText->selectAll();

    // Set default text color for all text
    postText->setTextColor(qApp->palette().windowText().color());
    postText->setTextBackgroundColor(QColor(Qt::transparent));

    // Take care of background colors
    QTextBlockFormat blockFormat = postText->textCursor().blockFormat();
    blockFormat.setBackground(QBrush());
    postText->textCursor().setBlockFormat(blockFormat);

    // Select 'none'
    postText->moveCursor(QTextCursor::Start);
    postText->textCursor().select(QTextCursor::WordUnderCursor);
}



/*
 * Trigger a resizeEvent, which will call
 * setPostContents() and setPostHeight()
 *
 */
void Post::triggerResize()
{
    this->resize(this->width() - 1,
                 this->height() - 1);
}



/*
 * Redraw post contents after receiving downloaded images
 *
 */
void Post::redrawImages(QString imageUrl)
{
    if (pendingImagesList.contains(imageUrl))
    {
        this->pendingImagesList.removeAll(imageUrl);
        if (pendingImagesList.isEmpty()) // If there are no more for this post, disconnect
        {
            disconnect(pController, SIGNAL(imageStored(QString)),
                       this, SLOT(redrawImages(QString)));

            // Trigger resizeEvent(), with setPostContents(), etc.
            triggerResize();
        }
    }
}



////////////////////////////////////////////////////////////////////////////
//////////////////////////////// PROTECTED /////////////////////////////////
////////////////////////////////////////////////////////////////////////////



void Post::resizeEvent(QResizeEvent *event)
{
    qDebug() << "Post::resizeEvent()" << event->size();

    if (this->resizesCount > 10) // Don't resize more than 10 times in a row
    {
        qDebug() << "Too many resizes, too fast; ignoring!";
        event->ignore();

        return;
    }


    this->onResizeOrShow();


    // Standalone mode doesn't need this kind of protection, and is not re-set by Timeline()
    if (!standalone)
    {
        ++this->resizesCount;
    }

    event->accept();
}



/*
 * On mouse click in any part of the post, set it as read
 *
 */
void Post::mousePressEvent(QMouseEvent *event)
{
    setPostAsRead();

    event->accept();
}

void Post::keyPressEvent(QKeyEvent *event)
{
    if (event->key() == Qt::Key_Escape)
    {
        event->accept();

        if (standalone)
        {
            this->close();
        }
    }
    else
    {
        event->ignore();
    }
}


/*
 * Ensure we change back the highlighted URL message when the mouse leaves the post
 *
 */
void Post::leaveEvent(QEvent *event)
{
    this->pController->showTransientMessage("");

    event->accept();
}

/*
 * closeEvent, needed when posts are opened in separate window
 *
 */
void Post::closeEvent(QCloseEvent *event)
{
    qDebug() << "Post::closeEvent()";
    if (standalone)
    {
        QSettings settings;
        settings.setValue("Post/postSize", this->size());
    }

    if (this->commenter->isFullMode())
    {
        // Ask composer to cancel post, which asks the user, unless empty
        this->commenter->getComposer()->cancelPost();
    }

    event->ignore(); // Event will be ignored anyway


    // Check again; if it's still full, it means user canceled
    if (!this->commenter->isFullMode()) // If not, accepted, so kill the post
    {
        this->hide();
        this->deleteLater();
    }
}

/*
 * Needed when a post is shown as a standalone window,
 * to ensure images are properly resized
 *
 */
void Post::showEvent(QShowEvent *event)
{
    if (standalone)
    {
        this->onResizeOrShow();
    }

    event->accept();
}
