/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "accountdialog.h"

AccountDialog::AccountDialog(PumpController *pumpController, QWidget *parent) : QWidget(parent)
{
    this->setWindowTitle(tr("Account Configuration") + " - Dianara");
    this->setWindowIcon(QIcon::fromTheme("dialog-password"));
    this->setWindowFlags(Qt::Dialog);
    this->setWindowModality(Qt::ApplicationModal);
    this->setMinimumSize(640, 360);

    this->pController = pumpController;

    connect(pController, SIGNAL(openingAuthorizeURL(QUrl)),
            this, SLOT(showAuthorizationURL(QUrl)));
    connect(pController, SIGNAL(authorizationStatusChanged(bool)),
            this, SLOT(showAuthorizationStatus(bool)));

    QFont helpFont;
    helpFont.setPointSize(helpFont.pointSize() - 1);

    this->helpMessage1Label = new QLabel(tr("First, enter your Webfinger ID, your pump.io address.") + "\n"
                                         + tr("Your address looks like username@pumpserver.org, and "
                                              "you can find it in your profile, in the web interface.") + "\n"
                                         + tr("If your profile is at https://pump.example/yourname, then "
                                              "your address is yourname@pump.example"));
    helpMessage1Label->setWordWrap(true);
    helpMessage1Label->setFont(helpFont);

    userIDIconLabel = new QLabel();
    userIDIconLabel->setPixmap(QIcon::fromTheme("preferences-desktop-user").pixmap(32,32));
    userIDLabel = new QLabel(tr("Your Pump.io address:"));
    userIDLineEdit = new QLineEdit();
    userIDLineEdit->setPlaceholderText(tr("Webfinger ID, like username@pumpserver.org"));
    userIDLineEdit->setToolTip(tr("Your address, as username@server"));

    getVerifierButton = new QPushButton(QIcon::fromTheme("object-unlocked"),
                                        tr("Get &Verifier Code"));
    getVerifierButton->setToolTip(tr("After clicking this button, a web browser will "
                                     "open, requesting authorization for Dianara"));
    connect(getVerifierButton, SIGNAL(clicked()),
            this, SLOT(askForToken()));


    this->helpMessage2Label = new QLabel(tr("Once you have authorized Dianara from your Pump server "
                                            "web interface, you'll receive a code called VERIFIER.\n"
                                            "Copy it and paste it into the field below.",
                                            // Comment for the translators
                                            "Don't translate the VERIFIER word!"));
    helpMessage2Label->setWordWrap(true);
    helpMessage2Label->setFont(helpFont);


    verifierIconLabel = new QLabel();
    verifierIconLabel->setPixmap(QIcon::fromTheme("dialog-password").pixmap(32,32));
    verifierLabel = new QLabel(tr("Verifier code:"));
    verifierLineEdit = new QLineEdit();
    verifierLineEdit->setPlaceholderText(tr("Enter the verifier code provided by your Pump server here"));
    verifierLineEdit->setToolTip(tr("Paste the verifier here"));


    authorizeApplicationButton = new QPushButton(QIcon::fromTheme("security-high"),
                                                 tr("&Authorize Application"));
    connect(authorizeApplicationButton, SIGNAL(clicked()),
            this, SLOT(setVerifierCode()));



    // To notify invalid ID or empty verifier code
    // Cleared when typing in any of the two fields
    errorsLabel = new QLabel();
    errorsLabel->setOpenExternalLinks(true);
    connect(userIDLineEdit, SIGNAL(textChanged(QString)),
            errorsLabel, SLOT(clear()));
    connect(verifierLineEdit, SIGNAL(textChanged(QString)),
            errorsLabel, SLOT(clear()));


    QFont authorizationStatusFont;
    authorizationStatusFont.setPointSize(authorizationStatusFont.pointSize() + 2);
    authorizationStatusFont.setBold(true);

    authorizationStatusLabel = new QLabel();
    authorizationStatusLabel->setFont(authorizationStatusFont);


    saveButton = new QPushButton(QIcon::fromTheme("document-save",
                                                  QIcon(":/images/button-save.png")),
                                 tr("&Save Details"));
    connect(saveButton, SIGNAL(clicked()),
            this, SLOT(saveDetails()));
    cancelButton = new QPushButton(QIcon::fromTheme("dialog-cancel",
                                                    QIcon(":/images/button-cancel.png")),
                                   tr("&Cancel"));
    connect(cancelButton, SIGNAL(clicked()),
            this, SLOT(hide()));


    // Layout
    this->mainLayout = new QVBoxLayout();
    mainLayout->addWidget(helpMessage1Label, 3);
    mainLayout->addStretch(2);

    this->idLayout = new QHBoxLayout();
    idLayout->addWidget(userIDIconLabel);
    idLayout->addSpacing(4);
    idLayout->addWidget(userIDLabel);
    idLayout->addWidget(userIDLineEdit);
    idLayout->addWidget(getVerifierButton);
    mainLayout->addLayout(idLayout, 3);
    mainLayout->addStretch(3);
    mainLayout->addSpacing(12);

    mainLayout->addWidget(helpMessage2Label, 3);
    mainLayout->addStretch(2);

    this->verifierLayout = new QHBoxLayout();
    verifierLayout->addWidget(verifierIconLabel);
    verifierLayout->addSpacing(4);
    verifierLayout->addWidget(verifierLabel);
    verifierLayout->addWidget(verifierLineEdit);
    verifierLayout->addWidget(authorizeApplicationButton);
    mainLayout->addLayout(verifierLayout, 3);

    mainLayout->addWidget(errorsLabel,              1, Qt::AlignCenter);
    mainLayout->addSpacing(4);
    mainLayout->addWidget(authorizationStatusLabel, 1, Qt::AlignCenter);
    mainLayout->addStretch(2);
    mainLayout->addSpacing(16);

    this->buttonsLayout = new QHBoxLayout();
    buttonsLayout->setAlignment(Qt::AlignRight);
    buttonsLayout->addWidget(saveButton);
    buttonsLayout->addWidget(cancelButton);
    mainLayout->addLayout(buttonsLayout, 1);

    this->setLayout(mainLayout);


    QSettings settings;  // Load saved User ID
    userIDLineEdit->setText(settings.value("userID", "").toString());

    this->showAuthorizationStatus(settings.value("isApplicationAuthorized", false).toBool());


    // Disable verifier input field and button initially
    // They will be used after requesting a token
    verifierLineEdit->setDisabled(true);
    authorizeApplicationButton->setDisabled(true);


    qDebug() << "Account dialog created";
}


AccountDialog::~AccountDialog()
{
    qDebug() << "Account dialog destroyed";
}



//////////////////////////////////////////////////////////////////////
///////////////////////////////// SLOTS //////////////////////////////
//////////////////////////////////////////////////////////////////////



void AccountDialog::askForToken()
{
    QSettings settings;
    settings.setValue("isApplicationAuthorized", false); // Until steps are completed

    // Don't fail if there're spaces before or after the ID, they're only human ;)
    userIDLineEdit->setText(userIDLineEdit->text().trimmed());

    // Match username@server.tld AND check for only 1 @ sign
    if (QRegExp("\\S+@\\S+\\.\\S+").exactMatch(this->userIDLineEdit->text())
     && userIDLineEdit->text().count("@") == 1)
    {
        // Clear verifier field and re-enable it and the button
        this->verifierLineEdit->clear();
        this->verifierLineEdit->setEnabled(true);
        this->authorizeApplicationButton->setEnabled(true);


        // Show message about the web browser that will be started
        this->errorsLabel->setText("[ "
                                   + tr("A web browser will start now, "
                                        "where you can get the verifier code")
                                   + " ]");

        this->pController->setNewUserId(userIDLineEdit->text());
        this->pController->getToken();
    }
    else // userID does not match user@hostname.domain
    {
        this->errorsLabel->setText("[ " + tr("Your Pump address is invalid") + " ]");
        qDebug() << "userID is invalid!";
    }
}



void AccountDialog::setVerifierCode()
{
    qDebug() << "AccountDialog::setVerifierCode()"
             << this->verifierLineEdit->text();

    if (!this->verifierLineEdit->text().trimmed().isEmpty())
    {
        this->pController->authorizeApplication(this->verifierLineEdit->text().trimmed());
    }
    else
    {
        this->errorsLabel->setText("[ " + tr("Verifier code is empty") + " ]");
    }
}




void AccountDialog::showAuthorizationStatus(bool authorized)
{
    if (authorized)
    {
        this->authorizationStatusLabel->setText(QString::fromUtf8("\342\234\224 ") // Check mark
                                                + tr("Dianara is authorized to "
                                                     "access your data"));
    }
    else
    {
        this->authorizationStatusLabel->clear();
    }
}


/*
 * Show the authorization URL in a label,
 * in case the browser doesn't open automatically
 *
 */
void AccountDialog::showAuthorizationURL(QUrl url)
{
    QString message = tr("If the browser doesn't open automatically, "
                         "copy this address manually")
                      + ":<br><a href=\""
                      + url.toString()
                      + "\">"
                      + url.toString()
                      + "</a>";
    this->errorsLabel->setText(message);
}





/*
 * Save the new userID and inform other parts of the program about it
 *
 */
void AccountDialog::saveDetails()
{
    QString newUserID = this->userIDLineEdit->text().trimmed();
    if (newUserID.isEmpty() || !newUserID.contains("@"))
    {
        return;  // If no user ID, or with no "@", ignore // FIXME
    }

    QSettings settings;
    settings.setValue("userID", newUserID);

    emit userIDChanged(userIDLineEdit->text());


    this->errorsLabel->clear(); // Clear previous error messages, if any

    this->hide(); // close() would end program if main window was hidden
}
