/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "profileeditor.h"


ProfileEditor::ProfileEditor(PumpController *pumpController,
                             QWidget *parent) : QWidget(parent)
{
    this->setWindowTitle("Dianara - " + tr("Profile Editor"));
    this->setWindowIcon(QIcon::fromTheme("user-properties"));
    this->setWindowFlags(Qt::Window);
    this->setWindowModality(Qt::NonModal);
    this->setMinimumSize(500, 350);

    this->pController = pumpController;

    avatarLabel = new QLabel();
    avatarLabel->setPixmap(QIcon::fromTheme("user-properties").pixmap(64, 64));

    changeAvatarButton = new QPushButton(QIcon::fromTheme("folder-image-people"),
                                         tr("Change &Avatar..."));
    connect(changeAvatarButton, SIGNAL(clicked()),
            this, SLOT(findAvatarFile()));

    this->avatarChanged = false;


    fullNameLineEdit = new QLineEdit();
    fullNameLineEdit->setToolTip(tr("This is your visible name"));
    hometownLineEdit = new QLineEdit();
    bioTextEdit = new QTextEdit();
    bioTextEdit->setTabChangesFocus(true);


    saveButton = new QPushButton(QIcon::fromTheme("document-save"),
                                 tr("&Save Profile"));
    connect(saveButton, SIGNAL(clicked()), this, SLOT(saveProfile()));

    cancelButton = new QPushButton(QIcon::fromTheme("dialog-cancel"),
                                   tr("&Cancel"));
    connect(cancelButton, SIGNAL(clicked()), this, SLOT(hide()));



    this->bottomLayout = new QHBoxLayout();
    bottomLayout->setAlignment(Qt::AlignRight);
    bottomLayout->addWidget(saveButton);
    bottomLayout->addWidget(cancelButton);


    this->mainLayout = new QFormLayout();
    mainLayout->setSpacing(8);
    mainLayout->addRow(tr("Avatar"),     avatarLabel);
    mainLayout->addWidget(changeAvatarButton);
    mainLayout->addRow(tr("Full &Name"), fullNameLineEdit);
    mainLayout->addRow(tr("&Hometown"),  hometownLineEdit);
    mainLayout->addRow(tr("&Bio"),       bioTextEdit);

    mainLayout->addRow(bottomLayout);

    this->setLayout(mainLayout);


    qDebug() << "ProfileEditor created";
}



ProfileEditor::~ProfileEditor()
{
    qDebug() << "ProfileEditor destroyed";
}



/*
 * Fill the fields from received info
 *
 */
void ProfileEditor::setProfileData(QString avatarURL, QString fullName, QString hometown, QString bio)
{
    this->currentAvatarURL = avatarURL;

    QString avatarFilename = MiscHelpers::getCachedAvatarFilename(this->currentAvatarURL);
    if (QFile::exists(avatarFilename))
    {
        this->avatarLabel->setPixmap(QPixmap(avatarFilename).scaled(64, 64,
                                                                    Qt::KeepAspectRatio,
                                                                    Qt::SmoothTransformation));
    }

    this->fullNameLineEdit->setText(fullName);
    this->hometownLineEdit->setText(hometown);
    this->bioTextEdit->setText(bio);
}



/****************************************************************************/
/******************************** SLOTS *************************************/
/****************************************************************************/



void ProfileEditor::findAvatarFile()
{
    newAvatarFilename = QFileDialog::getOpenFileName(this,
                                                     tr("Select avatar image"),
                                                     QDir::homePath(),
                                                     tr("Image files")
                                                     + "(*.png *.jpg *.jpeg *.gif);;"
                                                     + tr("All files") + " (*)");

    if (!newAvatarFilename.isEmpty())
    {
        qDebug() << "Selected" << newAvatarFilename << "as new avatar for upload";

        // FIXME: in the future, check file size and image size
        // and scale the pixmap to something sane before uploading.
        QPixmap avatarPixmap = QPixmap(newAvatarFilename);

        this->newAvatarContentType = MiscHelpers::getImageContentType(newAvatarFilename);
        if (!avatarPixmap.isNull() && !newAvatarContentType.isEmpty())
        {
            this->avatarLabel->setPixmap(QPixmap(newAvatarFilename).scaled(64, 64,
                                                                           Qt::KeepAspectRatio,
                                                                           Qt::SmoothTransformation));
            this->avatarChanged = true;
        }
        else
        {
            QMessageBox::warning(this,
                                 tr("Invalid image"),
                                 tr("The selected image is not valid."));
            qDebug() << "Invalid avatar file selected";
        }
    }
}



void ProfileEditor::saveProfile()
{
    if (avatarChanged)
    {
        connect(pController, SIGNAL(avatarUploaded(QString)),
                this, SLOT(sendProfileData(QString)));

        this->pController->uploadFile(this->newAvatarFilename,
                                      this->newAvatarContentType,
                                      PumpController::UploadAvatarRequest);
    }
    else
    {
        this->sendProfileData(); // without a new image ID
    }
}



void ProfileEditor::sendProfileData(QString newImageUrl)
{
    if (avatarChanged)
    {
        disconnect(pController, SIGNAL(avatarUploaded(QString)),
                   this, SLOT(sendProfileData(QString)));

        this->avatarChanged = false; // For next time the dialog is shown
    }

    this->pController->updateUserProfile(newImageUrl,
                                         this->fullNameLineEdit->text().trimmed(),
                                         this->hometownLineEdit->text().trimmed(),
                                         this->bioTextEdit->toPlainText().trimmed());


    this->hide(); // close() would end the program if main window is hidden
}
