/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "post.h"

Post::Post(PumpController *pumpController,
           QMap<QString,QString> postData,
           QVariantList lastLikesVariantList,
           QVariantList lastCommentsVariantList,
           QVariantList lastSharesVariantList,
           QWidget *parent)  :  QFrame(parent)
{
    this->pController = pumpController;

    this->setSizePolicy(QSizePolicy::MinimumExpanding,
                        QSizePolicy::Minimum);


    leftColumnLayout = new QVBoxLayout();


    this->postID = postData.value("id");
    this->postType = postData.value("postType");
    this->postURL = postData.value("url");
    this->postAuthorID = postData.value("authorId");
    this->postAuthorName = postData.value("author");
    if (postAuthorName.isEmpty()) // In case name is empty, use webfinger ID
    {
        postAuthorName = this->postAuthorID;
    }
    this->postAuthorURL = postData.value("authorUrl");

    // Save post title for openClickedURL(), etc
    this->postTitle = postData.value("title");


    postIsUnread = false;
    if (postData.value("postIsNew") == "true")
    {
        this->postIsUnread = true;
    }

    QFont detailsFont;
    detailsFont.setPointSize(detailsFont.pointSize() - 2); // FIXME: check size first


    /////////////////////////////////////////////////// Left column, post Meta info

    // Is the post a reshare? Indicate who shared it
    postIsSharedLabel = new QLabel();
    if (postData.value("postIsShared") == "true")
    {
        QString sharedBy = postData.value("sharedBy");
        if (sharedBy.length() > 16)   // very TMP optimization of LOOONG names / FIXME
        {
            sharedBy.replace("@", "@ ");
            sharedBy.replace(".", ". ");
        }
        postIsSharedLabel->setText(QString::fromUtf8("\342\231\272  ") // Recycling symbol
                                   + tr("Via %1").arg(sharedBy));
        postIsSharedLabel->setWordWrap(true);
        postIsSharedLabel->setAlignment(Qt::AlignCenter);
        postIsSharedLabel->setFont(detailsFont);
        postIsSharedLabel->setForegroundRole(QPalette::Text);
        postIsSharedLabel->setBackgroundRole(QPalette::Base);
        postIsSharedLabel->setAutoFillBackground(true);
        postIsSharedLabel->setFrameStyle(QFrame::Panel | QFrame::Raised);
        postIsSharedLabel->setToolTip(postData.value("sharedById"));
        leftColumnLayout->addWidget(postIsSharedLabel);
    }
    leftColumnLayout->addSpacing(2);


    QString authorTooltipInfo = "<b>" + postAuthorID + "</b>";
    authorTooltipInfo.append("<br>");
    if (!postData.value("authorHometown").isEmpty())
    {
        authorTooltipInfo.append("<br>");
        authorTooltipInfo.append("<b>" + tr("Hometown") + "</b>: " + postData.value("authorHometown"));
        authorTooltipInfo.append("<br>");
    }
    if (!postData.value("authorBio").isEmpty())
    {
        authorTooltipInfo.append("<br>");
        authorTooltipInfo.append(postData.value("authorBio"));
    }
    authorTooltipInfo.replace("\n", "<br>"); // HTML newlines

    if (this->postAuthorID == pController->currentUserID())
    {
        postIsOwn = true;
        qDebug() << "post is our own!";

        this->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    }
    else
    {
        postIsOwn = false;
        this->setFrameStyle(QFrame::Box | QFrame::Raised);
    }

    // Author avatar
    postAuthorAvatarButton = new QPushButton();
    postAuthorAvatarButton->setSizePolicy(QSizePolicy::Maximum,
                                          QSizePolicy::Maximum);
    postAuthorAvatarButton->setFlat(true);
    postAuthorAvatarButton->setIconSize(QSize(48,48));
    postAuthorAvatarButton->setToolTip(authorTooltipInfo);

    // Get "following" status before creating avatar menu
    this->postAuthorFollowed = this->pController->userInFollowing(this->postAuthorID);
    this->createAvatarMenu();
    postAuthorAvatarButton->setMenu(avatarMenu);

    this->postAuthorAvatarUrl = postData.value("authorAvatar");

    // Get local file name for avatar, which is stored in base64 hash form
    QString avatarFilename = MiscHelpers::getCachedAvatarFilename(postAuthorAvatarUrl);

    if (QFile::exists(avatarFilename) && !postAuthorAvatarUrl.isEmpty())
    {
        // Load avatar if already cached
        postAuthorAvatarButton->setIcon(QIcon(QPixmap(avatarFilename).scaled(48, 48,
                                                                  Qt::KeepAspectRatio,
                                                                  Qt::SmoothTransformation)));
    }
    else
    {
        // Placeholder image
        postAuthorAvatarButton->setIcon(QIcon::fromTheme("user-identity",
                                                         QIcon(":/images/no-avatar.png")));
        qDebug() << "Using placeholder, downloading real avatar now";

        // Download avatar for next time
        connect(pController, SIGNAL(avatarStored(QString,QString)),
                this, SLOT(redrawAvatar(QString,QString)));
        pController->getAvatar(postAuthorAvatarUrl);
    }
    leftColumnLayout->addWidget(postAuthorAvatarButton, 0, Qt::AlignLeft);
    leftColumnLayout->addSpacing(2);


    QFont authorFont;
    authorFont.setBold(true);
    authorFont.setUnderline(true);
    authorFont.setPointSize(authorFont.pointSize()-1);
    if (postIsOwn) // Another visual hint when the post is our own
    {
        authorFont.setItalic(true);
    }

    // Author name
    QString authorName = this->postAuthorName;
    if (authorName.length() > 16)   // very TMP optimization of LOOONG names / FIXME
    {
        authorName.replace("@", "@ ");
        authorName.replace(".", ". ");
    }
    postAuthorNameLabel = new QLabel(authorName);
    postAuthorNameLabel->setTextFormat(Qt::PlainText);
    postAuthorNameLabel->setWordWrap(true);
    postAuthorNameLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postAuthorNameLabel->setFont(authorFont);
    postAuthorNameLabel->setToolTip(authorTooltipInfo);

    leftColumnLayout->addWidget(postAuthorNameLabel);
    leftColumnLayout->addSpacing(2);


    // Post creation time
    QString createdAt = postData["createdAt"];
    // and update time
    QString updatedAt = postData["updatedAt"];
    if (updatedAt == createdAt)
    {
        updatedAt.clear();
    }
    // Format is "Combined date and time in UTC", like "2012-02-07T01:32:02Z"
    // as specified in ISO 8601 http://en.wikipedia.org/wiki/ISO_8601
    QString exactPostTime = Timestamp::localTimeDate(createdAt);
    createdAt = Timestamp::fuzzyTime(createdAt);
    if (!updatedAt.isEmpty())
    {
        createdAt.append("\n");
        createdAt.append(tr("Edited: %1").arg(Timestamp::fuzzyTime(updatedAt)));
    }

    QString generator = postData.value("generator");

    postCreatedAtLabel = new QLabel(createdAt);
    if (generator.isEmpty())
    {
        postCreatedAtLabel->setToolTip(tr("Posted on %1","1=Date")
                                  .arg(exactPostTime));
    }
    else
    {
        postCreatedAtLabel->setToolTip(tr("Posted on %1\n"
                                     "using %2",        "1=Date of post, 2=Program used for posting")
                                  .arg(exactPostTime).arg(generator));
    }
    postCreatedAtLabel->setTextFormat(Qt::PlainText);
    postCreatedAtLabel->setWordWrap(true);
    postCreatedAtLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postCreatedAtLabel->setFont(detailsFont);
    leftColumnLayout->addWidget(postCreatedAtLabel);
    leftColumnLayout->addSpacing(4);


    postToLabel = new QLabel(tr("To") + ": " + postData.value("to"));
    postToLabel->setWordWrap(true);
    postToLabel->setFont(detailsFont);
    postToLabel->setOpenExternalLinks(true);
    if (!postData["to"].isEmpty())
    {
        leftColumnLayout->addWidget(postToLabel);
    }

    postCCLabel = new QLabel(tr("CC") + ": " + postData.value("cc"));
    postCCLabel->setWordWrap(true);
    postCCLabel->setFont(detailsFont);
    postCCLabel->setOpenExternalLinks(true);
    if (!postData["cc"].isEmpty())
    {
        leftColumnLayout->addWidget(postCCLabel);
    }

    leftColumnLayout->addSpacing(4);


    this->postLikesURL = postData.value("likesURL");
    this->postCommentsURL = postData.value("commentsURL");


    // Set this QLabels with parent=this, since we're gonna hide them right away
    // They'll be reparented to the layout, but now having a parent before that
    // would cause visual glitches
    postLikesCountLabel = new QLabel(this);
    postLikesCountLabel->setWordWrap(true);
    postLikesCountLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postLikesCountLabel->setFont(detailsFont);
    leftColumnLayout->addWidget(postLikesCountLabel);
    postLikesCountLabel->hide();
    int likesCount = postData.value("likesCount").toInt();
    this->setLikesLabel(likesCount);


    postCommentsCountLabel = new QLabel(this);
    postCommentsCountLabel->setWordWrap(true);
    postCommentsCountLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postCommentsCountLabel->setFont(detailsFont);
    leftColumnLayout->addWidget(postCommentsCountLabel);
    postCommentsCountLabel->hide();
    int commentsCount = postData.value("commentsCount").toInt();
    this->setCommentsLabel(commentsCount);


    postResharesCountLabel = new QLabel(this);
    postResharesCountLabel->setWordWrap(true);
    postResharesCountLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postResharesCountLabel->setFont(detailsFont);
    leftColumnLayout->addWidget(postResharesCountLabel);
    postResharesCountLabel->hide();
    int sharesCount = postData.value("resharesCount").toInt();
    this->setSharesLabel(sharesCount);


    // Try to use all remaining space, aligning
    // all previous widgets nicely at the top
    // leftColumnLayout->setAlignment(Qt::AlignTop) caused glitches
    leftColumnLayout->addStretch(1);



    //////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////// Right column, content

    rightColumnLayout = new QVBoxLayout();
    rightColumnLayout->setAlignment(Qt::AlignRight | Qt::AlignTop);

    ///////////////////////////////////// Title
    this->postTitleLabel = new QLabel();
    postTitleLabel->setWordWrap(true);
    QFont postTitleFont;
    postTitleFont.setBold(true);
    postTitleFont.setPointSize(postTitleFont.pointSize() + 1);
    postTitleLabel->setFont(postTitleFont);
    postTitleLabel->setText(postTitle);


    ///////////////////////////////////// Post text
    postText = new QTextBrowser();
    postText->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postText->setSizePolicy(QSizePolicy::MinimumExpanding,
                            QSizePolicy::MinimumExpanding);

    postText->setWordWrapMode(QTextOption::WrapAtWordBoundaryOrAnywhere);
    postText->setOpenLinks(false); // don't open links, manage in openClickedURL()
    postText->setReadOnly(true); // it's default with QTextBrowser, but still...
    connect(postText, SIGNAL(anchorClicked(QUrl)),
            this, SLOT(openClickedURL(QUrl)));
    connect(postText, SIGNAL(highlighted(QUrl)),
            this, SLOT(showHighlightedURL(QUrl)));

    // Add a label in the bottom right corner, to show where links go
    highlightedUrlLabel = new QLabel();
    highlightedUrlLabel->setAlignment(Qt::AlignLeft);
    highlightedUrlLabel->setFont(detailsFont);
    highlightedUrlLabel->setForegroundRole(QPalette::ToolTipText); // use Tooltip colors
    highlightedUrlLabel->setBackgroundRole(QPalette::ToolTipBase);
    highlightedUrlLabel->setAutoFillBackground(true);
    highlightedUrlLabel->setWordWrap(true);
    highlightedUrlLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
    highlightedUrlLabel->hide(); // Hidden initially

    postTextLayout = new QVBoxLayout();
    postTextLayout->setAlignment(Qt::AlignRight | Qt::AlignBottom);
    postTextLayout->addWidget(highlightedUrlLabel);
    postText->setLayout(postTextLayout);

    QFont buttonsFont;
    buttonsFont.setPointSize(buttonsFont.pointSize() - 1);

    // Like, comment, share buttons
    likeButton = new QPushButton(QIcon::fromTheme("emblem-favorite"), "like");
    likeButton->setCheckable(true);
    this->fixLikeButton(postData.value("postIsLiked"));
    likeButton->setFlat(true);
    likeButton->setFont(buttonsFont);
    connect(likeButton, SIGNAL(clicked(bool)),
            this, SLOT(likePost(bool)));

    commentButton = new QPushButton(QIcon::fromTheme("mail-reply-sender"), tr("Comment"));
    commentButton->setToolTip(tr("Comment on this post.") + "<br>"
                              + tr("If you select some text, it will be quoted."));
    commentButton->setFlat(true);
    commentButton->setFont(buttonsFont);
    connect(commentButton, SIGNAL(clicked()),
            this, SLOT(commentOnPost()));

    // Note: Gwenview includes 'document-share'
    shareButton = new QPushButton(QIcon::fromTheme("mail-forward"),
                                  tr("Share"));
    shareButton->setToolTip(tr("Share this post"));
    shareButton->setFlat(true);
    shareButton->setFont(buttonsFont);
    connect(shareButton, SIGNAL(clicked()),
            this, SLOT(sharePost()));



    // Add like, comment, share and, if post is our own, edit and delete buttons
    buttonsLayout = new QHBoxLayout();
    buttonsLayout->setAlignment(Qt::AlignHCenter | Qt::AlignBottom);
    buttonsLayout->setContentsMargins(0, 0, 0, 0);
    buttonsLayout->setMargin(0);
    buttonsLayout->setSpacing(0);
    buttonsLayout->addWidget(likeButton,    0, Qt::AlignLeft);
    if (postType != "comment") // Don't add these buttons if it's a comment
    {                          // (can happen in the Favorites timeline)
        buttonsLayout->addWidget(commentButton, 0, Qt::AlignLeft);
        buttonsLayout->addWidget(shareButton,   0, Qt::AlignLeft);
    }
    buttonsLayout->addStretch(1); // so the (optional) Edit and Delete buttons get separated

    if (postIsOwn)
    {
        editButton = new QPushButton(QIcon::fromTheme("document-edit"), tr("Edit"));
        editButton->setToolTip(tr("Edit this post"));
        editButton->setFlat(true);
        editButton->setFont(buttonsFont);
        connect(editButton, SIGNAL(clicked()),
                this, SLOT(editPost()));
        buttonsLayout->addWidget(editButton, 0, Qt::AlignRight);

        deleteButton = new QPushButton(QIcon::fromTheme("edit-delete"), tr("Delete"));
        deleteButton->setToolTip(tr("Delete this post"));
        deleteButton->setFlat(true);
        deleteButton->setFont(buttonsFont);
        connect(deleteButton, SIGNAL(clicked()),
                this, SLOT(deletePost()));

        buttonsLayout->addWidget(deleteButton, 0, Qt::AlignRight);
    }


    /******************************************************************/


    // Get URL of post image, if it's "image" type of post
    if (this->postType == "image")
    {
        postImageURL = postData.value("image");

        this->enqueueImageForDownload(postImageURL);
    }

    this->postOriginalText = postData.value("text"); // Save it for later...
    QStringList postTextImageList = MiscHelpers::htmlWithReplacedImages(postOriginalText,
                                                                        this->postText->width());
    postTextImageList.removeFirst(); // First is the HTML itself

    // If the image list is not empty, get them (unless they're cached already)
    if (!postTextImageList.isEmpty())
    {
        qDebug() << "Post has" << postTextImageList.size() << "images included...";

        foreach (QString imageUrl, postTextImageList)
        {
            this->enqueueImageForDownload(imageUrl);
        }
    }


    // Post resizing takes place in resizeEvent()
    // which will also call setPostContents()



    /////////////////////////////////////////////////////// Comments block
    this->commenter = new CommenterBlock(this->pController);
    connect(commenter, SIGNAL(commentSent(QString)),
            this, SLOT(sendComment(QString)));
    connect(commenter, SIGNAL(allCommentsRequested()),
            this, SLOT(getAllComments()));


    if (!this->postTitle.isEmpty())
    {
        rightColumnLayout->addWidget(postTitleLabel);
    }
    rightColumnLayout->addWidget(postText,      2);
    rightColumnLayout->addLayout(buttonsLayout, 0);
    rightColumnLayout->addWidget(commenter,     1);


    // Set the initial likes, comments and shares (4 most recent)
    this->setLikes(lastLikesVariantList, likesCount);
    this->commenter->setComments(lastCommentsVariantList);
    this->setShares(lastSharesVariantList, sharesCount);


    mainLayout = new QHBoxLayout();
    mainLayout->addLayout(leftColumnLayout,  2); // stretch 2/11
    mainLayout->addLayout(rightColumnLayout, 9); // stretch 9/11

    mainLayout->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    this->setLayout(mainLayout);


    // Set read status
    this->setPostUnreadStatus();

    qDebug() << "Post created" << this->postURL;
}



Post::~Post()
{
    qDebug() << "Post destroyed" << this->postURL;
}


/*
 * Create the menu shown when clicking the post author's avatar
 *
 */
void Post::createAvatarMenu()
{
    this->avatarMenu = new QMenu();

    this->avatarMenuIdAction = new QAction(QIcon::fromTheme("user-identity"),
                                           postAuthorID, this);
    avatarMenuIdAction->setDisabled(true);
    avatarMenu->addAction(avatarMenuIdAction);
    avatarMenu->addSeparator();

    avatarMenu->addAction(QIcon::fromTheme("internet-web-browser"),
                          tr("Open %1's profile in web browser")
                          .arg(this->postAuthorName),
                          this,
                          SLOT(openAuthorProfileInBrowser()));

    this->avatarMenuFollowAction = new QAction("*follow/unfollow*", this);
    if (!postIsOwn) // Only add "follow/unfollow" option, if the post is not ours
    {
        avatarMenu->addAction(avatarMenuFollowAction);
        this->setFollowUnfollow();
    }

    avatarMenu->addSeparator();

    avatarMenu->addAction(QIcon::fromTheme("internet-web-browser"),
                          tr("Open post in web browser"),
                          this,
                          SLOT(openPostInBrowser()));
    avatarMenu->addAction(QIcon::fromTheme("edit-copy"),
                          tr("Copy post link to clipboard"),
                          this,
                          SLOT(copyPostUrlToClipboard()));
}


/*
 * Set the icon and text of the follow/unfollow option of the avatar menu
 * according to whether we're following that user or not
 *
 */
void Post::setFollowUnfollow()
{
    if (this->postAuthorFollowed)
    {
        this->avatarMenuFollowAction->setIcon(QIcon::fromTheme("list-remove-user"));
        this->avatarMenuFollowAction->setText(tr("Stop following"));
        connect(avatarMenuFollowAction, SIGNAL(triggered()),
                this, SLOT(unfollowUser()));
        disconnect(avatarMenuFollowAction, SIGNAL(triggered()),
                   this, SLOT(followUser()));
        qDebug() << "post author followed, connecting to UNFOLLOW()" << this->postAuthorID;
    }
    else
    {
        this->avatarMenuFollowAction->setIcon(QIcon::fromTheme("list-add-user"));
        this->avatarMenuFollowAction->setText(tr("Follow"));
        connect(avatarMenuFollowAction, SIGNAL(triggered()),
                this, SLOT(followUser()));
        disconnect(avatarMenuFollowAction, SIGNAL(triggered()),
                   this, SLOT(unfollowUser()));
        qDebug() << "post author not followed, connecting to FOLLOW()" << this->postAuthorID;
    }
}



/*
 * Set the post contents, and trigger a vertical resize
 *
 */
void Post::setPostContents()
{
    //qDebug() << "Post::setPostContents()" << this->postID;

    QString postImage;
    int imageWidth;
    if (!this->postImageURL.isEmpty())
    {
        QString imageCachedFilename = MiscHelpers::getCachedImageFilename(postImageURL);

        imageWidth = MiscHelpers::getImageWidth(imageCachedFilename);
        // if the image is wider than the post space, make it smaller
        if (imageWidth > this->postWidth)
        {
            imageWidth = this->postWidth;
        }

        if (QFile::exists(imageCachedFilename))
        {
            postImage = "<br>"
                        "<a href=\"image:/" + imageCachedFilename + "\">"
                        "<img src=\"" + imageCachedFilename + "\" "
                        "width=\"" + QString("%1").arg(imageWidth) + "\" />"
                        "</a>"
                        "<br><br>";
        }
        else // use placeholder image while it loads...
        {
            postImage = "<br>"
                        "<div title=\"" + tr("Image is loading...") + "\">"
                        "<img src=\":/images/image-loading.png\"  />"
                        "</div>"
                        "<br><br>";
        }
    }


    QStringList postTextImageList = MiscHelpers::htmlWithReplacedImages(postOriginalText,
                                                                        this->postWidth);
    QString postTextContents = postTextImageList.takeAt(0);

    // Add the text content of the post
    postText->setHtml(postImage
                      + postTextContents);


    this->setPostHeight();
}



/*
 * Set the height of the post, based on the contents
 *
 */
void Post::setPostHeight()
{
    int height = postText->document()->size().toSize().height() + 12; // +12px error margin

    if (height < 80)
    {
        height = 80;
    }
    // FIXME: this will be based on a percentage of window height, not hardcoded!
    if (height > 450)  // Don't allow a post to have a minimum height > 450 px
    {
        height = 450;  // Scrollbars might appear
    }

    postText->setMinimumHeight(height);
    postText->setMaximumHeight(height);
}





/*
 * Add the URL of an image to the queue of pending-download
 *
 * connect() signal/slot if necessary
 *
 */
void Post::enqueueImageForDownload(QString url)
{
    if (QFile::exists(MiscHelpers::getCachedImageFilename(url))
     || pendingImagesList.contains(url))
    {
        qDebug() << "Post::enqueueImageForDownload(), "
                    "Using cached post image, or requested image is pending download...";
    }
    else
    {
        qDebug() << "Post::enqueueImageForDownload(), "
                    "post image not cached, downloading" << url;

        // FIXME? multi-connect?
        connect(pController, SIGNAL(imageStored(QString)),
                this, SLOT(redrawImages(QString)));

        this->pendingImagesList.append(url);
        pController->getImage(url);
    }
}



/*
 * Return the likes/favorites URL for this post
 *
 */
QString Post::likesURL()
{
    return this->postLikesURL;
}


/*
 * Update the tooltip in "%NUMBER likes" with the names
 * of the people who liked the post
 *
 */
void Post::setLikes(QVariantList likesList, int likesCount)
{
    if (likesList.size() > 0)
    {
        QString peopleString;
        foreach (QVariant likesMap, likesList)
        {
            peopleString = likesMap.toMap().value("displayName").toString() + ", "
                    + peopleString;
        }

        // Remove last comma and add "like this"
        peopleString.remove(-2, 2); // Last 2 characters

        QString likesString;
        if (likesList.size() == 1)
        {
            likesString = tr("%1 likes this", "One person").arg(peopleString);
        }
        else
        {
            likesString = tr("%1 like this", "More than one person").arg(peopleString);
        }
        likesString.append("<a></a>"); // Turn the label into rich text

        this->postLikesCountLabel->setToolTip(likesString);
    }

    // TMP/FIXME this can set the number to lower than initially set
    // if called with the "initial" up-to-4 likes list that comes with the post
    if (likesCount != -1)
    {
        this->setLikesLabel(likesCount); // use the passed likesCount parameter
    }
    else
    {
        this->setLikesLabel(likesList.size()); // show size of actual names list
    }
}


/*
 * Update the "NUMBER likes" label in left side
 *
 */
void Post::setLikesLabel(int likesCount)
{
    if (likesCount != 0)
    {
        if (likesCount == 1)
        {
            postLikesCountLabel->setText(QString::fromUtf8("\342\231\245 ") // heart symbol
                                  + tr("1 like"));
        }
        else
        {
            postLikesCountLabel->setText(QString::fromUtf8("\342\231\245 ") // heart symbol
                                  + tr("%1 likes").arg(likesCount));
        }
        postLikesCountLabel->show();
    }
    else
    {
        postLikesCountLabel->clear();
        postLikesCountLabel->hide();
    }
}


/*
 * Return the comments URL for this post
 *
 */
QString Post::commentsURL()
{
    return this->postCommentsURL;
}


/*
 * Ask the Commenter to set new comments
 *
 */
void Post::setComments(QVariantList commentsList)
{
    this->commenter->setComments(commentsList);

    // update number of comments in left side counter
    this->setCommentsLabel(commentsList.size());
}

/*
 * Update the "NUMBER comments" label in left side
 *
 */
void Post::setCommentsLabel(int commentsCount)
{
    if (commentsCount != 0)
    {
        if (commentsCount == 1)
        {
            postCommentsCountLabel->setText(QString::fromUtf8("\342\234\215 ") // writing hand
                                      + tr("1 comment"));
        }
        else
        {
            postCommentsCountLabel->setText(QString::fromUtf8("\342\234\215 ") // writing hand
                                      + tr("%1 comments").arg(commentsCount));
        }
        postCommentsCountLabel->show();
    }
    else
    {
        postCommentsCountLabel->clear();
        postCommentsCountLabel->hide();
    }
}


/*
 * Return the shares URL for this post,
 * list of people who reshared it
 *
 */
QString Post::sharesURL()
{
    return this->postSharesURL;
}


/*
 * Update the tooltip for "%NUMBER shares" with names
 *
 */
void Post::setShares(QVariantList sharesList, int sharesCount)
{
    if (sharesList.size() > 0)
    {
        QString peopleString;
        foreach (QVariant sharesMap, sharesList)
        {
            peopleString = sharesMap.toMap().value("displayName").toString() + ", "
                    + peopleString;
        }

        // Remove last comma and add "shared this"
        peopleString.remove(-2, 2); // Last 2 characters

        QString sharesString;
        if (sharesList.size() == 1)
        {
            sharesString = tr("%1 shared this", "%1 = One person name").arg(peopleString);
        }
        else
        {
            sharesString = tr("%1 shared this", "%1 = Names for more than one person").arg(peopleString);
        }
        sharesString.append("<a></a>");  // So that the label is rich text

        this->postResharesCountLabel->setToolTip(sharesString);
    }

    // TMP/FIXME this can set the number to lower than initially set
    // if called with the "initial" up-to-4 shares list that comes with the post
    if (sharesCount != -1)
    {
        this->setSharesLabel(sharesCount); // use the passed sharesCount parameter
    }
    else
    {
        this->setSharesLabel(sharesList.size()); // show size of actual names list
    }
}


void Post::setSharesLabel(int resharesCount)
{
    if (resharesCount != 0)
    {
        if (resharesCount == 1)
        {
            postResharesCountLabel->setText(QString::fromUtf8("\342\231\273 ") // recycle symbol
                                      + tr("Shared once"));
        }
        else
        {
            postResharesCountLabel->setText(QString::fromUtf8("\342\231\273 ") // recycle symbol
                                      + tr("Shared %1 times").arg(resharesCount));
        }

        postResharesCountLabel->show();
    }
    else
    {
        postResharesCountLabel->clear();
        postResharesCountLabel->hide();
    }
}



/*
 * Set or unset the visual hint indicating if the post is unread
 *
 */
void Post::setPostUnreadStatus()
{
    if (this->postIsUnread)
    {
        postAuthorNameLabel->setAutoFillBackground(true);
        postAuthorNameLabel->setForegroundRole(QPalette::HighlightedText);
        postAuthorNameLabel->setBackgroundRole(QPalette::Highlight);

        postCreatedAtLabel->setAutoFillBackground(true);
        postCreatedAtLabel->setForegroundRole(QPalette::HighlightedText);
        postCreatedAtLabel->setBackgroundRole(QPalette::Highlight);
    }
    else
    {
        postAuthorNameLabel->setAutoFillBackground(false);
        postAuthorNameLabel->setForegroundRole(QPalette::Foreground);
        postAuthorNameLabel->setBackgroundRole(QPalette::Background);

        postCreatedAtLabel->setAutoFillBackground(false);
        postCreatedAtLabel->setForegroundRole(QPalette::Foreground);
        postCreatedAtLabel->setBackgroundRole(QPalette::Background);
    }
}



void Post::resizeEvent(QResizeEvent *event)
{
    this->postWidth = postText->width() - 40; // -40 px to account for a possible scrollbar /FIXME
    if (postWidth < 50)
    {
        postWidth = 50;  // minimum width!
    }
    setPostContents();

    event->accept();
}



/*
 * On mouse click in any part of the post, set it as read
 *
 */
void Post::mousePressEvent(QMouseEvent *event)
{
    if (postIsUnread)
    {
        this->postIsUnread = false;
        emit postRead(); // Inform the Timeline()

        setPostUnreadStatus();
    }

    event->accept();
}


/*
 * Ensure we hide the highlighted URL label when the mouse leaves the post
 *
 */
void Post::leaveEvent(QEvent *event)
{
    this->highlightedUrlLabel->clear();
    this->highlightedUrlLabel->hide();

    event->accept();
}



////////////////////////////////////////////////////////////////////////////
////////////////////////////////// SLOTS ///////////////////////////////////
////////////////////////////////////////////////////////////////////////////


/*
 * Like (favorite) a post
 *
 */
void Post::likePost(bool like)
{
    qDebug() << "Post::likePost()" << (like ? "like" : "unlike");

    this->pController->likePost(this->postID, this->postType, like);

    this->fixLikeButton(like ? "true" : "false");

    connect(pController, SIGNAL(likeSet()),
            this, SLOT(getAllLikes()));
}

/*
 * Set the right labels and tooltips to the like button, depending on its state
 *
 */
void Post::fixLikeButton(QString state)
{
    if (state == "true")
    {
        likeButton->setToolTip(tr("You like this"));
        likeButton->setText(tr("Unlike"));
        likeButton->setChecked(true);
    }
    else
    {
        likeButton->setToolTip(tr("Like this post"));
        likeButton->setText(tr("Like"));
        likeButton->setChecked(false);
    }
}


void Post::getAllLikes()
{
    disconnect(pController, SIGNAL(likeSet()),
               this, SLOT(getAllLikes()));

    this->pController->getPostLikes(this->postLikesURL);
}


/*
 * Make the commenter widget visible, so user can type the comment
 *
 */
void Post::commentOnPost()
{
    qDebug() << "Commenting on this post";

    QString initialText;

    QString selectedText = this->postText->textCursor().selectedText();
    if (!selectedText.isEmpty())
    {
        initialText = MiscHelpers::quotedText(this->postAuthorName,
                                              selectedText);
    }
    this->commenter->setFullMode(initialText);

    emit commentingOnPost(this->commenter);
}


/*
 * The actual sending of the comment to the Pump controller
 *
 */
void Post::sendComment(QString comment)
{
    qDebug() << "About to publish this comment:" << comment;

    this->pController->addComment(MiscHelpers::cleanupHtml(comment),
                                  this->postID, this->postType);
}


void Post::getAllComments()
{
    this->pController->getPostComments(this->postCommentsURL);
}


/*
 * Re-share a post
 *
 */
void Post::sharePost()
{
    int confirmation = QMessageBox::question(this, tr("Share post?"),
                                             tr("Do you want to share %1's post?").arg(this->postAuthorNameLabel->text()),
                                             tr("&Yes, share it"), tr("&No"),
                                             "", 1, 1);

    if (confirmation == 0)
    {
        qDebug() << "Sharing this post";
        this->pController->sharePost(this->postID, this->postType);
    }
    else
    {
        qDebug() << "Confirmation cancelled, not sharing";
    }
}

/*
 * Set the Publisher in editing mode with this post's contents
 *
 */
void Post::editPost()
{
    emit postEditRequested(this->postID,
                           this->postOriginalText,
                           this->postTitle);
}



/*
 * Delete a post
 *
 */
void Post::deletePost()
{
    int confirmation = QMessageBox::question(this, tr("WARNING: Delete post?"),
                                             tr("Are you sure you want to delete this post?"),
                                             tr("&Yes, delete it"), tr("&No"), "", 1, 1);

    if (confirmation == 0)
    {
        qDebug() << "Deleting post";
        this->pController->deletePost(this->postID, this->postType);

        this->setDisabled(true); // disable... maybe hide?
    }
    else
    {
        qDebug() << "Confirmation cancelled, not deleting the post";
    }
}



void Post::openClickedURL(QUrl url)
{
    qDebug() << "Anchor URL clicked:" << url.toString();

    if (url.scheme() == "image") // Use our own viewer, or maybe a configured external viewer
    {
        qDebug() << "Opening this image in our own viewer...";
        ImageViewer *viewer = new ImageViewer(url.toString(), this->postTitle);
        viewer->show();
    }
    else
    {
        qDebug() << "Opening this link in browser";
        QDesktopServices::openUrl(url);
    }
}



void Post::showHighlightedURL(QUrl url)
{
    QString urlString = url.toString();
    if (!urlString.isEmpty())
    {
        if (urlString.startsWith("image://")) // Own image:// URL
        {
            this->highlightedUrlLabel->setText(tr("Click the image to see it in full size"));
        }
        else  // Normal URL
        {
            this->highlightedUrlLabel->setText(tr("Link to: %1").arg(urlString));
            qDebug() << "Highlighted url:" << urlString;
        }
        this->highlightedUrlLabel->show();
    }
    else
    {
        this->highlightedUrlLabel->clear();
        this->highlightedUrlLabel->hide();
    }
}



void Post::openAuthorProfileInBrowser()
{
    QDesktopServices::openUrl(this->postAuthorURL);
}


void Post::openPostInBrowser()
{
    QDesktopServices::openUrl(this->postURL);
}

void Post::copyPostUrlToClipboard()
{
    QApplication::clipboard()->setText(this->postURL);
}




void Post::followUser()
{
    this->pController->followContact(this->postAuthorID);

    postAuthorFollowed = true; // FIXME: should wait until confirmed by server...
    this->setFollowUnfollow();
}


void Post::unfollowUser()
{
    int confirmation = QMessageBox::question(this, tr("Stop following?"),
                                             tr("Are you sure you want to stop following %1?")
                                             .arg(this->postAuthorID),
                                             tr("&Yes, stop following"), tr("&No"), "", 1, 1);

    if (confirmation == 0)
    {
        this->pController->unfollowContact(this->postAuthorID);

        postAuthorFollowed = false; // FIXME: should wait until confirmed by server...
        this->setFollowUnfollow();
    }
}



/*
 * Trigger a resizeEvent, which will call
 * setPostContents() and setPostHeight()
 *
 */
void Post::triggerResize()
{
    this->resize(this->width() -1,
                 this->height() -1);
}



/*
 * Redraw post author's avatar after receiving it
 *
 */
void Post::redrawAvatar(QString avatarUrl, QString avatarFilename)
{
    if (avatarUrl == this->postAuthorAvatarUrl)
    {
        postAuthorAvatarButton->setIcon(QIcon(QPixmap(avatarFilename).scaled(48, 48,
                                                                  Qt::KeepAspectRatio,
                                                                  Qt::SmoothTransformation)));

        disconnect(pController, SIGNAL(avatarStored(QString,QString)),
                   this, SLOT(redrawAvatar(QString,QString)));
    }
}


/*
 * Redraw post contents after receiving downloaded images
 *
 */
void Post::redrawImages(QString imageUrl)
{
    if (pendingImagesList.contains(imageUrl))
    {
        this->pendingImagesList.removeAll(imageUrl);
        if (pendingImagesList.isEmpty()) // If there are no more, disconnect
        {
            disconnect(pController, SIGNAL(imageStored(QString)),
                       this, SLOT(redrawImages(QString)));
        }

        // Trigger resizeEvent(), with setPostContents(), etc.
        triggerResize();
    }
}
