/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "listsmanager.h"

ListsManager::ListsManager(PumpController *pumpController,
                           QWidget *parent) : QWidget(parent)
{
    this->pController = pumpController;


    // To show the current lists
    this->listsTreeWidget = new QTreeWidget();
    listsTreeWidget->setColumnCount(2);
    listsTreeWidget->setHeaderLabels(QStringList() << tr("Name") << tr("Members"));
    listsTreeWidget->setAlternatingRowColors(true);


    this->deleteListButton = new QPushButton(QIcon::fromTheme("edit-table-delete-row"),
                                             tr("Delete Selected List"));
    connect(deleteListButton, SIGNAL(clicked()),
            this, SLOT(deleteList()));


    // Groupbox for the "create new list" stuff
    this->newListGroupbox = new QGroupBox(tr("Add New List"));

    this->newListNameLineEdit = new QLineEdit();
    newListNameLineEdit->setPlaceholderText(tr("Type a name for the new list..."));
    connect(newListNameLineEdit, SIGNAL(returnPressed()),
            this, SLOT(createList()));

    this->newListDescTextEdit = new QTextEdit();
    newListDescTextEdit->setToolTip(tr("Type an optional description here"));


    this->createListButton = new QPushButton(QIcon::fromTheme("edit-table-insert-row-above"),
                                             tr("Create List"));
    connect(createListButton, SIGNAL(clicked()),
            this, SLOT(createList()));

    this->groupboxLeftLayout = new QVBoxLayout();
    groupboxLeftLayout->addWidget(newListNameLineEdit);
    groupboxLeftLayout->addWidget(newListDescTextEdit);

    this->groupboxMainLayout = new QHBoxLayout();
    groupboxMainLayout->addLayout(groupboxLeftLayout);
    groupboxMainLayout->addWidget(createListButton, 0, Qt::AlignTop);
    newListGroupbox->setLayout(groupboxMainLayout);


    this->mainLayout = new QVBoxLayout();
    mainLayout->addWidget(listsTreeWidget);
    mainLayout->addWidget(deleteListButton, 0, Qt::AlignRight);
    mainLayout->addStretch(1);
    mainLayout->addWidget(newListGroupbox);
    this->setLayout(mainLayout);

    qDebug() << "ListsManager created";
}


ListsManager::~ListsManager()
{
    qDebug() << "ListsManager destroyed";
}



void ListsManager::setListsList(QVariantList listsList)
{
    qDebug() << "Setting person lists contents";

    this->listsTreeWidget->clear();

    QString listName;
    QString listMembersCount;
    QVariant listId;

    foreach (QVariant list, listsList)
    {
        QVariantMap listMap = list.toMap();
        listName = listMap.value("displayName").toString();
        listMembersCount = listMap.value("members").toMap().value("totalItems").toString();
        listId = listMap.value("id");
        qDebug() << "list ID:" << listId.toString();

        QTreeWidgetItem *listItem = new QTreeWidgetItem();
        listItem->setText(0, listName);
        listItem->setText(1, listMembersCount);
        listItem->setData(0, Qt::UserRole, listId);

        // Basic enumeration of first 4 members, ID only
        QVariantList membersList = listMap.value("members").toMap().value("items").toList();
        foreach (QVariant memberVariant, membersList)
        {
            QString memberId = memberVariant.toMap().value("id").toString();
            if (memberId.startsWith("acct:"))
            {
                memberId.remove(0, 5);
            }

            QTreeWidgetItem *childItem = new QTreeWidgetItem();
            childItem->setText(0, memberId);

            listItem->addChild(childItem);
        }


        this->listsTreeWidget->addTopLevelItem(listItem);
    }


    this->listsTreeWidget->resizeColumnToContents(0);
}



/****************************************************************************/
/********************************* SLOTS ************************************/
/****************************************************************************/



void ListsManager::createList()
{
    QString listName = this->newListNameLineEdit->text().trimmed();
    QString listDescription = this->newListDescTextEdit->toPlainText().trimmed();
    listDescription.replace("\n", "<br>");

    if (!listName.isEmpty())
    {
        qDebug() << "Creating list:" << listName;
        this->pController->createPersonList(listName, listDescription);

        this->newListNameLineEdit->clear();
        this->newListDescTextEdit->clear();
    }
    else
    {
        qDebug() << "Error: List name is empty!";
    }
}


void ListsManager::deleteList()
{
    int confirmation = QMessageBox::question(this, tr("WARNING: Delete list?"),
                                             tr("Are you sure you want to delete this list?"),
                                             tr("&Yes, delete it"), tr("&No"), "", 1, 1);

    if (confirmation == 0)
    {
        QString listId = this->listsTreeWidget->currentItem()->data(0, Qt::UserRole).toString();

        this->pController->deletePersonList(listId);
    }
    else
    {
        qDebug() << "Confirmation canceled, not deleting the list";
    }
}
