/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "commenter.h"

Commenter::Commenter(PumpController *pumpController, QWidget *parent) : QWidget(parent)
{
    this->pController = pumpController;


    this->showAllCommentsLinkLabel = new QLabel("<a href=\"showall://\">"
                                                + tr("Show All Comments")
                                                + "</a>");
    showAllCommentsLinkLabel->setContextMenuPolicy(Qt::NoContextMenu);
    QFont showAllFont;
    showAllFont.setPointSize(showAllFont.pointSize() - 3);
    showAllCommentsLinkLabel->setFont(showAllFont);
    connect(showAllCommentsLinkLabel, SIGNAL(linkActivated(QString)),
            this, SLOT(requestAllComments()));


    this->commentsTextBrowser = new QTextBrowser();
    commentsTextBrowser->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
    commentsTextBrowser->setWordWrapMode(QTextOption::WrapAtWordBoundaryOrAnywhere);
    commentsTextBrowser->setOpenExternalLinks(true);
    QFont commentsFont;
    commentsFont.setPointSize(commentsFont.pointSize() - 1); // 1 point less than default
    commentsTextBrowser->setFont(commentsFont);
    commentsTextBrowser->setSizePolicy(QSizePolicy::Minimum,
                                       QSizePolicy::Maximum);

    // Hide these until setComments() is called, if there are any comments
    showAllCommentsLinkLabel->hide();
    commentsTextBrowser->hide();




    this->commentComposer = new Composer(false); // forPublisher = false
    this->commentComposer->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    this->commentComposer->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    this->commentComposer->setSizePolicy(QSizePolicy::Minimum,
                                         QSizePolicy::Minimum);

    connect(commentComposer, SIGNAL(editingFinished()),  this, SLOT(sendComment()));
    connect(commentComposer, SIGNAL(editingCancelled()), this, SLOT(setMinimumMode()));


    this->commentButton = new QPushButton(QIcon::fromTheme("mail-message-new"),
                                          tr("&Comment"));
    commentButton->setSizePolicy(QSizePolicy::Maximum,
                                 QSizePolicy::Maximum);
    connect(commentButton, SIGNAL(clicked()), this, SLOT(sendComment()));

    this->cancelButton = new QPushButton(QIcon::fromTheme("dialog-cancel"),
                                         tr("C&ancel"));
    connect(cancelButton, SIGNAL(clicked()), commentComposer, SLOT(cancelPost()));


    mainLayout = new QGridLayout();
    mainLayout->addWidget(showAllCommentsLinkLabel, 0, 0, 1, 2, Qt::AlignRight | Qt::AlignTop);
    mainLayout->addWidget(commentsTextBrowser,      1, 0, 1, 2);
    mainLayout->addWidget(commentComposer,          2, 0, 2, 1);
    mainLayout->addWidget(commentButton,            2, 1, 1, 1);
    mainLayout->addWidget(cancelButton,             3, 1, 1, 1);
    this->setLayout(mainLayout);


    this->setMinimumMode();

    qDebug() << "Commenter created";
}


Commenter::~Commenter()
{
    qDebug() << "Commenter destroyed";
}



void Commenter::setComments(QVariantList commentsList)
{
    if (commentsList.size() > 0)
    {
        QString commentsString;
        foreach (QVariant commentMap, commentsList)
        {
            QString commentAuthorAvatarURL = commentMap.toMap().value("author").toMap().value("image").toMap().value("url").toString();
            QString commentAuthorAvatarFile = MiscHelpers::getCachedAvatarFilename(commentAuthorAvatarURL);
            if (!QFile::exists(commentAuthorAvatarFile))
            {
                this->pController->getAvatar(commentAuthorAvatarURL);
            }

            QString commentAuthorName = commentMap.toMap().value("author").toMap().value("displayName").toString();
            QString commentAuthorID = commentMap.toMap().value("author").toMap().value("id").toString();
            commentAuthorID.remove(0, 5); // remove "acct:" at the beginning
            QString commentContent = commentMap.toMap().value("content").toString();

            QString timestamp = commentMap.toMap().value("published").toString();
            QString commentExactTime = Timestamp::localTimeDate(timestamp);
            QString commentFuzzyTime = Timestamp::fuzzyTime(timestamp);

            commentsString = "<table>"
                             "<tr>"
                             "<td><span title=\"" + commentAuthorID + "\">"
                             "<img src=\"" + commentAuthorAvatarFile + "\" "
                             "height=\"32\" align=\"left\"></img>"
                             "</span></td>"
                             "<td><span title=\"" + commentAuthorID + "\">"
                             "<b>" + commentAuthorName + "</b></span> "
                             "- <span title=\"" + commentExactTime + "\">"
                             "<i>" + commentFuzzyTime + "</i>"
                             "</span></td>"
                             "</tr>"
                             "<tr>"
                             "<td></td>"  // Empty cell below avatar
                             "<td>" + commentContent + "</td>"
                             "</tr>"
                             "</table>"
                             "<hr>"
                             + commentsString; // previous comments added later,
                                               // since the list comes in reverse
        }
        this->commentsTextBrowser->setHtml(commentsString);

        this->showAllCommentsLinkLabel->show();
        this->commentsTextBrowser->show();

        // Move scrollbar to the bottom
        this->commentsTextBrowser->verticalScrollBar()->triggerAction(QScrollBar::SliderToMaximum);
    }
}



/*******************************************************************/
/****************************** SLOTS ******************************/
/*******************************************************************/


void Commenter::setMinimumMode()
{
    this->commentComposer->hide();
    this->commentButton->hide();
    this->cancelButton->hide();

    // Clear formatting options like bold or italic
    this->commentComposer->setCurrentCharFormat(QTextCharFormat());

    // Unblock updates to timeline, since we're done commenting
    this->pController->setUpdatesToTimelineBlocked(false);
}



void Commenter::setFullMode()
{
    this->commentComposer->show();
    this->commentButton->show();
    this->cancelButton->show();

    this->commentComposer->setFocus();

    // Block updates to timeline while we're commenting...
    this->pController->setUpdatesToTimelineBlocked(true);
}



void Commenter::requestAllComments()
{
    emit allCommentsRequested();

    disconnect(pController, SIGNAL(commentPosted()),
               this, SLOT(requestAllComments()));
}




void Commenter::sendComment()
{
    qDebug() << "Commenter character count:" << commentComposer->textCursor().document()->characterCount();

    // If there's some text in the comment, send it
    if (commentComposer->textCursor().document()->characterCount() > 1)
    {
        emit commentSent(commentComposer->toHtml());

        this->commentComposer->erase();

        // Done posting the comment, hide
        setMinimumMode();

        connect(pController, SIGNAL(commentPosted()),
                this, SLOT(requestAllComments()));
    }
    else
    {
        qDebug() << "Can't post, comment is empty";
    }
}
