/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "timeline.h"

TimeLine::TimeLine(int timelineType,
                   PumpController *pumpController,
                   QWidget *parent) :  QWidget(parent)
{
    this->timelineType = timelineType;

    this->pController = pumpController;

    mainLayout = new QVBoxLayout();
    mainLayout->setAlignment(Qt::AlignTop);


    QMap<QString,QString> demoPostData;
    demoPostData.insert("author",         "Demo User");
    demoPostData.insert("authorId",       "demo@somepump.example");
    demoPostData.insert("authorBio",      "I am not a real user");
    demoPostData.insert("authorHometown", "Demoville");
    demoPostData.insert("generator",      "Dianara");


    // Demo post content depends on timeline type
    switch (this->timelineType)
    {
    case TimelineTypeMain:
        demoPostData.insert("text",   "<h2><b>Welcome to Dianara</b></h2><br>"

                                      "Dianara is a <a href=\"http://pump.io\">pump.io</a> client.<br><br>"

                                      "After you have set up your account in <b>Settings > Account Details</b>,<br>"
                                      "you can press <b>F5</b> or <i>Update timelines</i> in the menu.<br><br>"

                                      "Upon startup, timeline should update automatically.<br><br><br>"

                                      "<a href=\"http://jancoding.wordpress.com/dianara\">Dianara's blog</a><br>"
                                      "<br>");
        break;


    case TimelineTypeDirect:
        demoPostData.insert("text",   "<h2><b>Direct Messages Timeline</b></h2><br>"

                                      "Here, you'll see posts specifically directed to you."

                                      "<br>");
        break;


    case TimelineTypeActivity:
        demoPostData.insert("text",   "<h2><b>Activity Timeline</b></h2><br>"

                                      "You'll see your own posts here."

                                      "<br>");
        break;


    case TimelineTypeFavorites:
        demoPostData.insert("text",   "<h2><b>Favorites Timeline</b></h2><br>"

                                      "Posts you've liked."

                                      "<br>");
        break;



    default:
        demoPostData.insert("text", "<h2>Empty timeline</h2>");

    }

    demoPostData.insert("createdAt", "2012-01-09T00:00:00Z");


    splitter = new QSplitter(Qt::Vertical, this);
    splitter->setChildrenCollapsible(false);
    mainLayout->addWidget(splitter);


    this->timelineOffset = 0;

    firstPageButton = new QPushButton(QIcon::fromTheme("go-first"), tr("&First Page"));
    connect(firstPageButton, SIGNAL(clicked()),
            this, SLOT(goToFirstPage()));

    currentPageLabel = new QLabel();
    currentPageLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
    currentPageLabel->setSizePolicy(QSizePolicy::Maximum,
                                    QSizePolicy::Maximum);

    previousPageButton = new QPushButton(QIcon::fromTheme("go-previous"), tr("&Previous Page"));
    connect(previousPageButton, SIGNAL(clicked()),
            this, SLOT(goToPreviousPage()));
    nextPageButton = new QPushButton(QIcon::fromTheme("go-next"), tr("&Next Page"));
    connect(nextPageButton, SIGNAL(clicked()),
            this, SLOT(goToNextPage()));


    bottomLayout = new QHBoxLayout();
    bottomLayout->addWidget(firstPageButton);
    bottomLayout->addSpacing(12);
    bottomLayout->addWidget(currentPageLabel);
    bottomLayout->addSpacing(12);
    bottomLayout->addWidget(previousPageButton);
    bottomLayout->addWidget(nextPageButton);
    mainLayout->addLayout(bottomLayout);


    // Add the default "demo" post
    splitter->addWidget(new Post(pController, demoPostData,
                                 QVariantList(), QVariantList(), QVariantList(),
                                 this));


    this->setLayout(mainLayout);


    // FIXME: take care of this for the different timelines
    QSettings settings;
    this->previousNewestPostID = settings.value("previousNewestPostID", "").toString();

    qDebug() << "TimeLine created";
}


TimeLine::~TimeLine()
{
    QSettings settings;
    settings.setValue("previousNewestPostID", this->previousNewestPostID);

    qDebug() << "TimeLine destroyed";
}



/*
 * Remove all widgets (Post *) from the timeline
 *
 */
void TimeLine::clearTimeLineContents()
{
    QList<QWidget*> previousPostList;

    // Build list of widgets contained in the layout
    for (int counter = 0; counter < splitter->count(); ++counter)
    {
        //qDebug() << "Adding for deletion:" << splitter->widget(counter);
        previousPostList.append(splitter->widget(counter));
    }

    qDebug() << "List to delete:" << previousPostList;


    foreach (QWidget *oldPost, previousPostList)
    {
        qDebug() << "will deleteLater:" << oldPost;
        oldPost->deleteLater(); // delete should be safe too, but still...
    }

    this->postsInTimeline.clear();

    /*
     * This is to avoid the splitter getting bigger and bigger every time the user
     * updates the timeline (F5, etc)
     *
     */
    this->splitter->setMinimumHeight(0);
    this->splitter->resize(1,1);
}



void TimeLine::requestTimelinePage()
{
    switch (this->timelineType)
    {
    case TimelineTypeMain:
        this->pController->getMainTimeline(this->timelineOffset);
        break;

    case TimelineTypeDirect:
        this->pController->getDirectTimeline(this->timelineOffset);
        break;

    case TimelineTypeActivity:
        this->pController->getActivityTimeline(this->timelineOffset);
        break;

    case TimelineTypeFavorites:
        this->pController->getFavoritesTimeline(this->timelineOffset);
        break;
    }

}


/*
 *  Update the label at the bottom of the page, indicating current "page"
 *
 */
void TimeLine::updateCurrentPageNumber()
{
    QString currentPageString;
    currentPageString = QString("%1").arg((this->timelineOffset / this->postsPerPage) + 1);

    this->currentPageLabel->setText(currentPageString);
}



/*********************************************************/
/*********************************************************/
/************************ SLOTS **************************/
/*********************************************************/
/*********************************************************/



void TimeLine::setTimeLineContents(QVariantList postList, int postsPerPage,
                                   QString previousLink, QString nextLink)
{
    qDebug() << "TimeLine::setTimeLineContents()";

    // Remove all previous posts in timeline
    qDebug() << "Removing previous posts from timeline";
    this->clearTimeLineContents();


    this->postsPerPage = postsPerPage;

    int newPostCount = 0;
    bool allNewPostsCounted = false;
    QString newestPostID; // Here we'll store the postID for the first (newest) post in the timeline
                          // With it, we can know how many new posts (if any) we receive next time

    // Fill timeline with new contents
    foreach (QVariant singlePost, postList)
    {
        if (singlePost.type() == QVariant::Map)
        {
            this->previousPageLink = previousLink;   // Useless at the moment
            this->nextPageLink = nextLink;

            QVariantMap singlePostMap = singlePost.toMap();
            QVariantMap objectMap;
            if (this->timelineType != TimelineTypeFavorites)  // VERY TMP!! Favorites-tmp-stuff
            {
                objectMap = singlePostMap["object"].toMap();
            }
            else
            {
                // Since "favorites" is a collection of objects directly, not activities
                objectMap = singlePostMap;
            }


            QString itemVerb = singlePostMap["verb"].toString();

            QString postIsDeleted = objectMap["deleted"].toString();


            QString postID;
            QString postURL;
            QString postType;
            QString postAuthor;
            QString postAuthorAvatar;
            QString postAuthorId;
            QString postAuthorHometown;
            QString postAuthorBio;
            QString postGenerator;
            QString postCreatedAt;
            QString postTo;
            QString postCC;
            QString postTitle;
            QString postImage;
            QString postText;
            QString postIsNsfw;
            QString postLikesCount;
            QString postCommentsCount;
            QString postResharesCount;

            QString postIsLiked;
            QString postIsShared;
            QString postSharedBy;


            postID = objectMap["id"].toString();
            if (newestPostID.isEmpty()) // only first time, for newest post
            {
                if (this->timelineOffset == 0)
                {
                    newestPostID = postID;
                }
                else
                {
                    newestPostID = this->previousNewestPostID;
                    allNewPostsCounted = true;
                }
            }

            if (!allNewPostsCounted)
            {
                if (postID == this->previousNewestPostID)
                {
                    allNewPostsCounted = true;
                }
                else
                {
                    ++newPostCount;
                }
            }

            postURL = objectMap["url"].toString();

            postType = objectMap["objectType"].toString();

            postIsLiked = objectMap["liked"].toString();

            QVariantMap authorMap;
            if (itemVerb == "share")
            {
                postIsShared = "true";

                // Get original author data
                authorMap = objectMap["author"].toMap();

                postSharedBy = singlePostMap["actor"].toMap().value("displayName").toString();
            }
            else
            {
                postIsShared = "false";

                // Not a share, so no original author, just get the actor
                authorMap = singlePostMap["actor"].toMap();
            }

            //// VERY TMP!! Favorites-tmp-stuff
            if (this->timelineType==TimelineTypeFavorites)
            {
                authorMap = objectMap["author"].toMap();
            }

            postAuthor = authorMap["displayName"].toString();
            postAuthorAvatar = authorMap["image"].toMap().value("url").toString();
            postAuthorId = authorMap["id"].toString();
            // For Author ID, remove the first 5 characters from the field, if they are "acct:"
            // (users have acct: in from of the ID, but services like OFG don't)
            if (postAuthorId.startsWith("acct:"))
            {
                postAuthorId.remove(0,5);
            }

            postAuthorBio = authorMap["summary"].toString();
            postAuthorHometown = authorMap["location"].toMap().value("displayName").toString();


            postCreatedAt = objectMap["published"].toString();
            postGenerator = singlePostMap["generator"].toMap().value("displayName").toString();


            // Audience
            foreach (QVariant postToVariant, singlePostMap.value("to").toList())
            {
                if (postToVariant.toMap().value("id").toString()
                  == "http://activityschema.org/collection/public")
                {
                    postTo += tr("Public") + ", ";
                }
                else
                {
                    postTo += postToVariant.toMap().value("displayName").toString() + ", ";
                }
            }
            postTo.remove(-2, 2); // remove last comma and space

            // CC
            foreach (QVariant postCCVariant, singlePostMap.value("cc").toList())
            {
                if (postCCVariant.toMap().value("id").toString()
                  == "http://activityschema.org/collection/public")
                {
                    postCC += tr("Public") + ", ";
                }
                else
                {
                    postCC += postCCVariant.toMap().value("displayName").toString() + ", ";
                }
            }
            postCC.remove(-2, 2);




            ///
            /// End of "meta"; Start of content
            ///

            if (postType == "image")
            {
                postTitle = objectMap["displayName"].toString();


                // See if there's a proxy URL for the image first (for private images)
                postImage = objectMap["fullImage"].toMap().value("pump_io").toMap().value("proxyURL").toString();
                qDebug() << "Trying Proxyed fullImage";

                // if that's empty, try the "small" version
                if (postImage.isEmpty())
                {
                    qDebug() << "Trying Proxyed thumbnail image";
                    postImage = objectMap["image"].toMap().value("pump_io").toMap().value("proxyURL").toString();
                }

                // If that's also empty, use regular fullImage->url field
                if (postImage.isEmpty())
                {
                    qDebug() << "Trying direct fullImage";
                    postImage = objectMap["fullImage"].toMap().value("url").toString();
                }

                // And if that is ALSO empty, use regular image->url
                if (postImage.isEmpty())
                {
                    qDebug() << "Trying direct thumbnail image";
                    postImage = objectMap["image"].toMap().value("url").toString();
                }
                qDebug() << "postImage:" << postImage;
            }


            postText = objectMap["content"].toString();

            postLikesCount = objectMap["likes"].toMap().value("totalItems").toString();
            postCommentsCount = objectMap["replies"].toMap().value("totalItems").toString();
            postResharesCount = objectMap["shares"].toMap().value("totalItems").toString();

            // get last likes, comments and shares list here, pass to the Post()
            QVariantList lastLikesList = objectMap["likes"].toMap().value("items").toList();
            QVariantList lastCommentsList = objectMap["replies"].toMap().value("items").toList();
            QVariantList lastSharesList = objectMap["shares"].toMap().value("items").toList();

            // Get URL for likes; first, proxyURL if it exists
            QString postLikesURL = objectMap["likes"].toMap().value("pump_io").toMap().value("proxyURL").toString();
            // If still empty, get regular URL (that means the post is in the same server we are)
            if (postLikesURL.isEmpty())
            {
                postLikesURL = objectMap["likes"].toMap().value("url").toString();
            }

            // Get URL for comments; first, proxyURL if it exists
            QString postCommentsURL = objectMap["replies"].toMap().value("pump_io").toMap().value("proxyURL").toString();
            if (postCommentsURL.isEmpty()) // If still empty, get regular URL
            {
                postCommentsURL = objectMap["replies"].toMap().value("url").toString();
            }



            QMap<QString,QString> postData;
            postData.insert("id",                 postID);
            postData.insert("url",                postURL);
            postData.insert("postType",           postType);
            postData.insert("author",             postAuthor);
            postData.insert("authorAvatar",       postAuthorAvatar);
            postData.insert("authorId",           postAuthorId);
            postData.insert("authorBio",          postAuthorBio);
            postData.insert("authorHometown",     postAuthorHometown);
            postData.insert("generator",          postGenerator);
            postData.insert("createdAt",          postCreatedAt);

            postData.insert("to",                 postTo);
            postData.insert("cc",                 postCC);

            postData.insert("title",              postTitle);
            postData.insert("image",              postImage);
            postData.insert("text",               postText);
            postData.insert("nsfw",               postIsNsfw);
            postData.insert("likesCount",         postLikesCount);
            postData.insert("likesURL",           postLikesURL);
            postData.insert("commentsCount",      postCommentsCount);
            postData.insert("commentsURL",        postCommentsURL);
            postData.insert("resharesCount",      postResharesCount);

            postData.insert("postIsLiked",        postIsLiked);
            postData.insert("postIsShared",       postIsShared);
            postData.insert("sharedBy",           postSharedBy);



            if (postIsDeleted.isEmpty())  // if post was NOT deleted
            {
                Post *newPost = new Post(pController,
                                         postData,
                                         lastLikesList,
                                         lastCommentsList,
                                         lastSharesList,
                                         this);
                this->postsInTimeline.append(newPost);

                this->splitter->addWidget(newPost);

                this->splitter->setMinimumHeight(this->splitter->height()
                                                 + newPost->sizeHint().height());
            }
            else
            {
                // if there's a "deleted" key, ignore this post
                qDebug() << "This post was deleted on" << postIsDeleted << " / Not adding.";
            }


            /*
            this->splitter->setStretchFactor(splitter->count()-1,
                                             splitter->widget(splitter->count()-1)->height()/200);
            */
        }
        else  // singlePost.type() is not a QVariant::Map
        {
            qDebug() << "Expected a Map, got something else";
            qDebug() << postList;
        }
    }

    emit scrollToTop(); // ask mainWindow to scroll the QScrollArea containing the timeline to the top

    this->previousNewestPostID = newestPostID;

    qDebug() << "New posts:" << newPostCount << "; Newest post ID:" << previousNewestPostID;

    this->updateCurrentPageNumber();

    emit timelineRendered(newPostCount);

    qDebug() << "setTimeLineContents() /END";
}





/*
 * Add the full list of likes to a post
 *
 */
void TimeLine::setLikesInPost(QVariantList likesList, QString originatingPostURL)
{
    qDebug() << "TimeLine::setLikesInPost()";

    QString originatingPostCleanURL = originatingPostURL.split("?").at(0);
    qDebug() << "Originating post URL:" << originatingPostCleanURL;


    // Look for the originating Post() object
    qDebug() << "Looking for the originating Post() object";
    foreach (Post *post, postsInTimeline)
    {
        qDebug() << "Checking if" << post->likesURL() << "==" << originatingPostCleanURL;

        if (post->likesURL() == originatingPostCleanURL)
        {
            qDebug() << "Found originating Post; setting likes on it...";
            post->setLikes(likesList);

            break;
        }
    }
}


/*
 * Add the full list of comments to a post
 *
 */
void TimeLine::setCommentsInPost(QVariantList commentsList, QString originatingPostURL)
{
    qDebug() << "TimeLine::setCommentsInPost()";

    QString originatingPostCleanURL = originatingPostURL.split("?").at(0);
    qDebug() << "Originating post URL:" << originatingPostCleanURL;


    // Look for the originating Post() object
    qDebug() << "Looking for the originating Post() object";
    foreach (Post *post, postsInTimeline)
    {
        qDebug() << "Checking if" << post->commentsURL() << "==" << originatingPostCleanURL;

        if (post->commentsURL() == originatingPostCleanURL)
        {
            qDebug() << "Found originating Post; setting comments on it...";
            post->setComments(commentsList);

            break;
        }
    }
}





void TimeLine::goToFirstPage()
{
    qDebug() << "TimeLine::goToFirstPage()";

    this->timelineOffset = 0;

    this->requestTimelinePage();
}



void TimeLine::goToPreviousPage()
{
    qDebug() << "TimeLine::goToPreviousPage()";

    this->timelineOffset -= this->postsPerPage;
    if (timelineOffset < 0)
    {
        timelineOffset = 0;
    }

    this->requestTimelinePage();
}



void TimeLine::goToNextPage()
{
    qDebug() << "TimeLine::goToNextPage()";

    this->timelineOffset += this->postsPerPage;

    this->requestTimelinePage();
}
