/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "contactlist.h"

ContactList::ContactList(PumpController *pumpController,
                         QWidget *parent) : QWidget(parent)
{
    this->pController = pumpController;

    mainLayout = new QVBoxLayout();
    mainLayout->setAlignment(Qt::AlignTop);


    topLayout = new QHBoxLayout();
    addressLineEdit = new QLineEdit();
    addressLineEdit->setPlaceholderText(tr("Enter a Webfinger address, like username@pumpserver.org"));

    followButton = new QPushButton(QIcon::fromTheme("list-add-user"), tr("&Follow"));
    connect(followButton, SIGNAL(clicked()),
            this, SLOT(followContact()));
    topLayout->addWidget(addressLineEdit);
    topLayout->addWidget(followButton);

    mainLayout->addLayout(topLayout);


    // Widget for list of 'following'
    this->followingWidget = new QWidget();

    followingLayout = new QVBoxLayout();
    followingLayout->setAlignment(Qt::AlignTop);
    QMap<QString,QString> demoFollowingData;
    demoFollowingData.insert("name", "Demo Contact");
    demoFollowingData.insert("id",   "democontact@pumpserver.org");
    demoFollowingData.insert("avatar", "");
    demoFollowingData.insert("hometown", "Some city");
    demoFollowingData.insert("follower", "false");
    demoFollowingData.insert("following", "true");
    followingLayout->addWidget(new ContactCard(this->pController, demoFollowingData, this));
    followingWidget->setLayout(followingLayout);

    followingScrollArea = new QScrollArea();
    followingScrollArea->setWidget(followingWidget);
    followingScrollArea->setWidgetResizable(true);


    // Widget for list of 'followers'
    this->followersWidget = new QWidget();

    followersLayout = new QVBoxLayout();
    followersLayout->setAlignment(Qt::AlignTop);
    QMap<QString,QString> demoFollowerData;
    demoFollowerData.insert("name", "Demo Follower");
    demoFollowerData.insert("id",   "demofollower@pumpserver.org");
    demoFollowerData.insert("avatar", "");
    demoFollowerData.insert("hometown", "Demo Town");
    demoFollowerData.insert("follower", "true");
    demoFollowerData.insert("following", "false");
    followersLayout->addWidget(new ContactCard(this->pController, demoFollowerData, this));
    followersWidget->setLayout(followersLayout);

    followersScrollArea = new QScrollArea();
    followersScrollArea->setWidget(followersWidget);
    followersScrollArea->setWidgetResizable(true);


    this->tabWidget = new QTabWidget();
    tabWidget->setDocumentMode(true); // To have less margins, with so many tabs inside tabs..

    tabWidget->addTab(followingScrollArea,
                      QIcon::fromTheme("meeting-participant"),
                      tr("Followin&g"));
    tabWidget->addTab(followersScrollArea,
                      QIcon::fromTheme("meeting-observer"),
                      tr("Follo&wers"));

    mainLayout->addWidget(tabWidget);
    this->setLayout(mainLayout);

    qDebug() << "Contact list created";
}


ContactList::~ContactList()
{
    qDebug() << "Contact list destroyed";
}



void ContactList::clearFollowingListContents()
{
    QLayoutItem *child;

    while ((child = followingLayout->takeAt(0)) != 0)
    {
        child->widget()->deleteLater(); // Delete the widget (contactCard) itself
        delete child;
    }    
}

void ContactList::clearFollowersListContents()
{
    QLayoutItem *child;

    while ((child = followersLayout->takeAt(0)) != 0)
    {
        child->widget()->deleteLater();
        delete child;
    }
}





/*****************************************************************************/
/*********************************** SLOTS ***********************************/
/*****************************************************************************/




void ContactList::setContactListContents(QString listType, QVariantList contactList)
{
    qDebug() << "Setting contact list contents";

    if (listType == "following")
    {
        this->clearFollowingListContents();
    }
    else
    {
        this->clearFollowersListContents();
    }


    QMap<QString,QString> contactData;

    foreach (QVariant contact, contactList)
    {
        contactData.insert("id", contact.toMap().value("id").toString().remove(0,5)); // remove "acct:"
        contactData.insert("name", contact.toMap().value("displayName").toString());
        contactData.insert("avatar", contact.toMap().value("image").toMap().value("url").toString());
        contactData.insert("hometown", contact.toMap().value("location").toMap().value("displayName").toString());
        contactData.insert("bio", contact.toMap().value("summary").toString());

        contactData.insert("following", contact.toMap().value("pump_io").toMap().value("followed").toString());

        if (listType == "following")
        {
            this->followingLayout->addWidget(new ContactCard(this->pController,
                                                            contactData,
                                                            this));
        }
        else // == "followers"
        {
            contactData.insert("follower", "true");

            this->followersLayout->addWidget(new ContactCard(this->pController,
                                                            contactData,
                                                            this));
        }
    }

}




/*
 * Add the address entered by the user to the /following list.
 *
 * This should probably support adding webfinger addresses
 * in the form https://host/username too.
 *
 */
void ContactList::followContact()
{
    QString address = this->addressLineEdit->text().trimmed();

    qDebug() << "ContactList::followContact()" << address;

    if (address.contains("@")) // FIXME, check correctly
    {
        this->pController->followContact(address); // this will trigger a reload of the contact list

        this->addressLineEdit->clear();
    }
    else
    {
        qDebug() << "Invalid webfinger address!";
    }

}
