/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "pumpcontroller.h"

PumpController::PumpController(QObject *parent) :  QObject(parent)
{
    this->userAgentString = "Dianara/0.4-dev";


    qoauth = new QOAuth::Interface();
    qoauth->setRequestTimeout(10000); // 10 sec timeout


    QSettings settings;
    this->clientID = settings.value("clientID", "").toString();
    this->clientSecret = settings.value("clientSecret", "").toString();
    qoauth->setConsumerKey(clientID.toLocal8Bit());
    qoauth->setConsumerSecret(clientSecret.toLocal8Bit());


    this->isApplicationAuthorized = settings.value("isApplicationAuthorized", false).toBool();

    if (isApplicationAuthorized)
    {
        qDebug() << "Dianara is already authorized for user ID:" << this->userID;

        this->token = settings.value("token", "").toString().toLocal8Bit();
        this->tokenSecret = settings.value("tokenSecret", "").toString().toLocal8Bit();

        qDebug() << "Using token" << token;
        qDebug() << "And token secret" << tokenSecret;
    }
    emit this->authorizationStatusChanged(isApplicationAuthorized);




    connect(&nam, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(requestFinished(QNetworkReply*)));

    // FIXME: setting this up for now, to at least have debug messages just in case
    connect(&nam, SIGNAL(sslErrors(QNetworkReply*,QList<QSslError>)),
            this, SLOT(sslErrorsHandler(QNetworkReply*,QList<QSslError>)));

    this->initialDataStep = 0;

    initialDataTimer = new QTimer(this);
    initialDataTimer->setSingleShot(false); // Triggered constantly until stopped
    connect(initialDataTimer, SIGNAL(timeout()),
            this, SLOT(getInitialData()));


    qDebug() << "PumpController created";



}


PumpController::~PumpController()
{
    qDebug() << "PumpController destroyed";
}


/*
 * Set new user ID (user@domain.tld) and clear OAuth-related tokens/secrets
 *
 *
 */
void PumpController::setNewUserID(QString userID)
{
    this->userID = userID;
    QStringList splittedUserID = this->userID.split("@");
    this->userName = splittedUserID.at(0); // get username, before @
    this->serverURL = splittedUserID.at(1); // get URL, after @

    qDebug() << "Server URL to connect:" << serverURL << "; username:" << userName;

    this->clientID.clear();
    this->clientSecret.clear();
    this->token.clear();
    this->tokenSecret.clear();

    this->isApplicationAuthorized = false;
    emit this->authorizationStatusChanged(isApplicationAuthorized);
}




/*
 * Get "pumpserver.org" and "user" from "user@pumpserver.org", set OAuth token from Account dlg
 *
 */
void PumpController::setUserCredentials(QString userID)
{
    this->initialDataTimer->stop(); // Just in case it was running before


    this->userID = userID;
    QStringList splittedUserID = this->userID.split("@");
    this->userName = splittedUserID.at(0);
    this->serverURL = splittedUserID.at(1);
    qDebug() << "New userID is:" << this->userID;


    this->getUserProfile(this->userID);


    // This will call getContactList() and getTimeLine(), etc.
    this->initialDataStep = 0;
    this->initialDataTimer->start(100);  // start immediately
}



/*
 * Get any user's profile (not only our own)
 *
 * GET https://pumpserver.example/api/user/username
 *
 */
void PumpController::getUserProfile(QString userID)
{
    QStringList splittedUserID = userID.split("@");

    QString url = "https://" + splittedUserID.at(1) + "/api/user/" + splittedUserID.at(0);

    QByteArray authorizationHeader = qoauth->createParametersString(url,
                                                                 QOAuth::GET,
                                                                 this->token,
                                                                 this->tokenSecret,
                                                                 QOAuth::HMAC_SHA1,
                                                                 QOAuth::ParamMap(),
                                                                 QOAuth::ParseForHeaderArguments);
    qDebug() << authorizationHeader;

    QNetworkRequest userProfileRequest;
    userProfileRequest.setUrl(QUrl(url));
    userProfileRequest.setRawHeader("Authorization", authorizationHeader);
    userProfileRequest.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
    userProfileRequest.setRawHeader("User-Agent",    userAgentString);

    userProfileRequest.setAttribute(QNetworkRequest::User,
                                    QVariant(UserProfileRequest));

    nam.get(userProfileRequest);

    qDebug() << "Requested:" << userProfileRequest.url();
}




void PumpController::getAvatar(QString avatarURL)
{
    qDebug() << "Getting avatar";


    QNetworkRequest avatarRequest(QUrl((const QString)avatarURL));
    avatarRequest.setRawHeader("User-Agent", userAgentString);
    avatarRequest.setAttribute(QNetworkRequest::User,
                               QVariant(AvatarRequest));

    nam.get(avatarRequest);
}



void PumpController::getImage(QString imageURL)
{
    qDebug() << "Getting image";


    QNetworkRequest imageRequest(QUrl((const QString)imageURL));
    imageRequest.setRawHeader("User-Agent", userAgentString);
    imageRequest.setAttribute(QNetworkRequest::User,
                              QVariant(ImageRequest));

    nam.get(imageRequest);
    qDebug() << "imageRequest sent";
}




/*
 * GET https://pumpserver.example/api/......
 *
 */
void PumpController::getContactList()
{
    emit currentJobChanged(tr("Getting contact list..."));
    qDebug() << "Getting contact list";


    QString url = "https://" + this->serverURL + "/api/user/" +this->userName + "/following";
                                                                                // also /followers?

    QByteArray authorizationHeader = qoauth->createParametersString(url,
                                                                 QOAuth::GET,
                                                                 this->token,
                                                                 this->tokenSecret,
                                                                 QOAuth::HMAC_SHA1,
                                                                 QOAuth::ParamMap(),
                                                                 QOAuth::ParseForHeaderArguments);

    QNetworkRequest contactListRequest;
    contactListRequest.setUrl(QUrl(url));
    contactListRequest.setRawHeader("Authorization", authorizationHeader);
    contactListRequest.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
    contactListRequest.setRawHeader("User-Agent",    userAgentString);

    contactListRequest.setAttribute(QNetworkRequest::User,
                                   QVariant(ContactListRequest));
    nam.get(contactListRequest);
}




/*
 * GET https://pumpserver.example/api/username/inbox
 *
 */
void PumpController::getTimeline()
{
    emit currentJobChanged(tr("Getting timeline..."));
    qDebug() << "Getting timeline";


    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/inbox/major";

    QByteArray authorizationHeader = qoauth->createParametersString(url,
                                                                 QOAuth::GET,
                                                                 this->token,
                                                                 this->tokenSecret,
                                                                 QOAuth::HMAC_SHA1,
                                                                 QOAuth::ParamMap(),
                                                                 QOAuth::ParseForHeaderArguments);

    QNetworkRequest timelineRequest;
    timelineRequest.setUrl(QUrl(url));
    timelineRequest.setRawHeader("Authorization", authorizationHeader);
    timelineRequest.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
    timelineRequest.setRawHeader("User-Agent",    userAgentString);

    timelineRequest.setAttribute(QNetworkRequest::User,
                                 QVariant(TimelineRequest));

    nam.get(timelineRequest);
}



/*
 * Get one specific post
 *
 * GET https://pumpserver.example/api/posts/.....
 *
 */
void PumpController::getPost(int id)
{
    qDebug() << "Getting post" << id;


    QNetworkRequest postRequest(QUrl("https://" + this->serverURL
                                     + "/api/posts/" + QString("%1").arg(id) ));
    postRequest.setRawHeader("User-Agent", userAgentString);
    postRequest.setAttribute(QNetworkRequest::User,
                             QVariant(PostRequest));

    nam.get(postRequest);
}



/***************************************************************************/
/***************************************************************************/
/*********************************** SLOTS *********************************/
/***************************************************************************/
/***************************************************************************/



void PumpController::requestFinished(QNetworkReply *reply)
{
    int httpCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
    int requestType = reply->request().attribute(QNetworkRequest::User).toInt();

    qDebug() << "Request finished. HTTP code:" << httpCode;
    qDebug() << "Size:" << reply->size() << "bytes; URL:" << reply->url();

    qDebug() << "isFinished()?" << reply->isFinished();
    qDebug() << "Request type:" << requestType;

    switch (httpCode)
    {
    case 503:
        emit showNotification("Service Unavailable (503)");
        qDebug() << "HTTP 503: Service Unavailable.";
        qDebug() << "Data:  " << reply->readAll();
        return;

    case 500:
        emit showNotification("Internal server error (500)");
        qDebug() << "HTTP 500: Internal server error.";
        qDebug() << "Data:  " << reply->readAll();
        return;


    case 404:
        emit showNotification("Not Found (404)");
        qDebug() << "HTTP 404: Not Found.";
        qDebug() << "Data:  " << reply->readAll();
        return;

    case 403:
        emit showNotification("Forbidden (403)");
        qDebug() << "HTTP 403: Forbidden.";
        qDebug() << "Data:  " << reply->readAll();
        return;


    case 400:
        emit showNotification("Bad Request (400)");
        qDebug() << "HTTP 400: Bad Request.";
        qDebug() << "Data:  " << reply->readAll();
        return;


    case 200:
        qDebug() << "HTTP 200: OK!";
    }


    // Read all received data
    QByteArray packetData = reply->readAll();

    // Prepare the JSON parser
    QJson::Parser jsonParser;
    bool jsonParsedOK = false;
    QVariantMap jsonData;
    QVariantList jsonDataList;


    switch (requestType)
    {

    case ClientRegistrationRequest:
        qDebug() << "Client Registration was requested";

        jsonData = jsonParser.parse(packetData, &jsonParsedOK).toMap();
        qDebug() << jsonData;

        if (jsonParsedOK && jsonData.size() > 0)
        {
            this->clientID = jsonData["client_id"].toString();
            this->clientSecret = jsonData["client_secret"].toString();

            // FIXME: error control, etc.
            // check if jsonData.keys().contains("client_id") !!

            QSettings settings;
            settings.setValue("clientID",     this->clientID);
            settings.setValue("clientSecret", this->clientSecret);


            this->getToken();
        }


        break;



    case UserProfileRequest:
        qDebug() << "A user profile was requested";

        jsonData = jsonParser.parse(packetData, &jsonParsedOK).toMap();
        qDebug() << jsonData;

        if (jsonParsedOK && jsonData.size() > 0)
        {
            QString profId = jsonData["profile"].toMap()["id"].toString();
            if (profId == "acct:" + this->userID)
            {
                qDebug() << "Received OWN profile";

                QString profDisplayName = jsonData["profile"].toMap()["displayName"].toString();
                QString profImageUrl = jsonData["profile"].toMap()["image"].toMap()["url"].toString();

                emit profileReceived(profDisplayName, profImageUrl);
            }
        }

        break;




    ////////////////////////////////////////////////// If a timeline was requested
    case TimelineRequest:
        qDebug() << "A Timeline was requested";

        jsonData = jsonParser.parse(packetData, &jsonParsedOK).toMap();
        qDebug() << "JSON data size (items):" << jsonData.size();
        qDebug() << jsonData.keys();

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";
            emit currentJobChanged(tr("Timeline received."));

            jsonDataList = jsonData["items"].toList();
            qDebug() << "Number of items in inbox timeline:" << jsonDataList.size();

            emit timeLineReceived(jsonDataList);
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << packetData; // JSON directly
        }
        break;



    ///////////////////////////////////////// If a single post was requested, by ID
    case PostRequest:
        qDebug() << "A single post was requested";

        break;




    case ContactListRequest:
        qDebug() << "The contact list was requested";

        jsonData = jsonParser.parse(packetData, &jsonParsedOK).toMap();
        qDebug() << "JSON data size (items):" << jsonData.size();

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";

            QVariant contactsVariant = jsonData.value("items");
            if (contactsVariant.type() == QVariant::List)
            {
                qDebug() << "Parsed a List, listing contacts...";
                emit currentJobChanged(tr("Contact list received."));
                emit contactListReceived(contactsVariant.toList());
            }
            else
            {
                qDebug() << "Expected a list of contacts, received something else:";
                qDebug() << jsonData;
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << packetData; // JSON directly
        }

        break;



    case AvatarRequest:
        qDebug() << "Received AVATAR data, from " << reply->url();
        // FIXME: take care of possible redirections

        //this->avatarData.append(packetData);
        if (reply->isFinished())
        {
            qDebug() << "Avatar received 100%";
            emit avatarPictureReceived(packetData, reply->url());
        }
        else
        {
            qDebug() << "Avatar not complete yet";
        }
        break;


    case ImageRequest:
        qDebug() << "Received IMAGE data, from " << reply->url();
        //this->imageData.append(packetData);
        if (reply->isFinished())
        {
            qDebug() << "Image received 100%";
            emit imageReceived(packetData, reply->url());
        }
        else
        {
            qDebug() << "Image not complete yet";
        }
        break;


    }
    // end switch (requestType)
}




void PumpController::sslErrorsHandler(QNetworkReply *reply, QList<QSslError> errorList)
{
    qDebug() << "\n\n==== SSL errors!! ====";
    qDebug() << "At:" << reply->url();
    qDebug() << "Error list:" << errorList << "\n\n";

}




void PumpController::getToken()
{
    // If we do not have client_id or client_secret, do dynamic client registration
    if (this->clientID.isEmpty() || this->clientSecret.isEmpty())
    {
        qDebug() << "PumpController::getToken()";
        qDebug() << "We do not have client_id/client_secret yet; doing Dynamic Client Registration";

        // POST to https://hotpump.net/api/client/register, f.e.
        QNetworkRequest postRequest(QUrl("https://" + this->serverURL + "/api/client/register"));


        postRequest.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
        postRequest.setRawHeader("User-Agent", userAgentString);
        postRequest.setAttribute(QNetworkRequest::User,
                                 QVariant(ClientRegistrationRequest));

        QByteArray data("{"
                        " \"type\": \"client_associate\",  "
                        " \"application_type\": \"native\", "
                        " \"application_name\": \"Dianara\" "
                        "}");

        qDebug() << "About to POST:" << data;


        nam.post(postRequest, data); // upon receiving data (id+secret), will execute getToken() again
    }
    else
    {
        qDebug() << "Using saved client_id and client_secret:" << this->clientID << this->clientSecret;

        // OAuth stuff.....
        // 1. obtaining an unauthorized Request Token from the Service Provider,
        // 2. asking the User to authorize the Request Token,
        // 3. exchanging the Request Token for the Access Token

        qDebug() << "Doing OAuth token stuff...";


        qoauth->setConsumerKey(this->clientID.toLocal8Bit());
        qoauth->setConsumerSecret(this->clientSecret.toLocal8Bit());


        QString requestTokenURL = "https://" + this->serverURL + "/oauth/request_token";
        qDebug() << "GET: " << requestTokenURL << "with" << qoauth->consumerKey() << qoauth->consumerSecret();


        QOAuth::ParamMap oAuthParams;
        oAuthParams.insert("oauth_callback", "oob");


        QOAuth::ParamMap reply = qoauth->requestToken(requestTokenURL,
                                                      QOAuth::GET,
                                                      QOAuth::HMAC_SHA1,   // or PLAINTEXT?
                                                      oAuthParams);


        if (qoauth->error() == QOAuth::NoError)
        {
            qDebug() << "requestToken OK:" << reply.keys();

            token = reply.value(QOAuth::tokenParameterName());
            tokenSecret = reply.value(QOAuth::tokenSecretParameterName());

            qDebug() << "Token:" << token;
            qDebug() << "Token Secret:" << tokenSecret;

            QUrl oAuthAuthorizeURL("https://" + this->serverURL + "/oauth/authorize");
            oAuthAuthorizeURL.addQueryItem("oauth_token", token);

            QDesktopServices::openUrl(oAuthAuthorizeURL);

            // Now, user should enter VERIFIER in AccountDialog to authorize the program
        }
        else
        {
            qDebug() << "QOAuth error" << qoauth->error() << "!";

            qDebug() << reply.keys();
        }

    }
}



void PumpController::authorizeApplication(QString verifierCode)
{
    qDebug() << "Verifier code entered by user:" << verifierCode;

    QOAuth::ParamMap moreParams;
    moreParams.insert("oauth_verifier", verifierCode.toUtf8()); // verifier as QByteArray

    QString requestAuthorizationURL = "https://" + this->serverURL + "/oauth/access_token";

    QOAuth::ParamMap reply = qoauth->accessToken(requestAuthorizationURL,
                                                 QOAuth::GET,
                                                 token,
                                                 tokenSecret,
                                                 QOAuth::HMAC_SHA1,
                                                 moreParams);

    if (qoauth->error() == QOAuth::NoError) // Woooohooo!!
    {
        qDebug() << "Got authorized token; Dianara is authorized to access the account";
        token = reply.value(QOAuth::tokenParameterName());
        tokenSecret = reply.value(QOAuth::tokenSecretParameterName());

        this->isApplicationAuthorized = true;

        QSettings settings;
        settings.setValue("isApplicationAuthorized", true);
        settings.setValue("token",       this->token);
        settings.setValue("tokenSecret", this->tokenSecret);

        qDebug() << "Token:" << token;
        qDebug() << "TokenSecret:" << tokenSecret;

        emit this->authorizationStatusChanged(isApplicationAuthorized);
    }
    else
    {
        qDebug() << "OAuth error while authorizing application" << qoauth->error();
    }

}




/*
 * Called by a QTimer, get initial data (contacts, timeline...) one step at a time
 *
 */
void PumpController::getInitialData()
{
    qDebug() << "PumpController::getInitialData() step" << initialDataStep;

    initialDataTimer->setInterval(5000);  // Every 5 sec


    switch (this->initialDataStep)
    {
    case 0:
        this->getContactList();
        break;

    case 1:
        this->getTimeline();
        break;


    default:
        emit currentJobChanged(tr("Ready.")); // FIXME
        initialDataTimer->stop();

        qDebug() << "All initial data loaded ------";
        qDebug() << "------------------------------";
    }

    ++initialDataStep;
}





/*
 * Send a post ("note") to the server
 *
 */
void PumpController::post(QString postText)
{
    qDebug() << "PumpController::post()";


    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QByteArray authorizationHeader = qoauth->createParametersString(url,
                                                                 QOAuth::POST,
                                                                 this->token,
                                                                 this->tokenSecret,
                                                                 QOAuth::HMAC_SHA1,
                                                                 QOAuth::ParamMap(),
                                                                 QOAuth::ParseForHeaderArguments);
    qDebug() << authorizationHeader;

    QNetworkRequest postRequest;
    postRequest.setUrl(QUrl(url));
    postRequest.setRawHeader("Authorization", authorizationHeader);
    postRequest.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
    postRequest.setRawHeader("User-Agent",    userAgentString);


    postRequest.setAttribute(QNetworkRequest::User,
                             QVariant(PublishPostRequest));


    QByteArray data = QString("{"
                              "\"verb\": \"post\","
                              "\"object\": {"
                              "              \"objectType\": \"note\","
                              "              \"content\": \"%1\" "
                              "            }"
                              "}").arg(postText).toLocal8Bit();


    qDebug() << "About to POST:" << postRequest.rawHeader("Authorization") << "\n" << data;

    nam.post(postRequest, data);
}




void PumpController::likePost(QString postID)
{
    qDebug() << "Liking post" << postID;
}



void PumpController::setCurrentTimeline(QString link)
{
    qDebug() << "PumpController::setCurrentTimeline()" << link;
}
