/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "mainwindow.h"


MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent)
{
    this->setWindowTitle("Dianara");
    this->setWindowIcon(QIcon(":/icon/64x64/dianara.png"));
    this->setMinimumSize(400, 400);

    firstRun = true;
    prepareDataDirectory();

    reallyQuitProgram = false;



    // Network control
    pumpController = new PumpController(/*userID, userAuth?*/);


    // GUI
    centralWidget = new QWidget();
    mainLayout = new QHBoxLayout();


    // Left side
    leftLayout = new QVBoxLayout();

    avatarIconLabel = new QLabel();
    avatarIconLabel->setPixmap(QPixmap(":/images/no-avatar.png")
                             .scaled(64, 64,
                                     Qt::KeepAspectRatio,
                                     Qt::SmoothTransformation));
    leftLayout->addWidget(avatarIconLabel);
    leftLayout->addSpacing(8);

    fullNameLabel = new QLabel("-Your-Name-");
    leftLayout->addWidget(fullNameLabel); // FIXME: below avatar, temporarily
    leftLayout->addSpacing(16);

    leftPanel = new QToolBox();
    aspectListLabel = new QLabel("- <a href=\"list1\">List 1</a><br /><br />"
                                 "- <a href=\"list2\">List 2</a><br /><br />"
                                 "- <a href=\"list3\">List 3</a>");
    aspectListLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    aspectListLabel->setContextMenuPolicy(Qt::NoContextMenu);
    connect(aspectListLabel, SIGNAL(linkActivated(QString)),
            pumpController, SLOT(setCurrentTimeline(QString)));
    leftPanel->addItem(aspectListLabel,
                       QIcon::fromTheme("view-list-text"),
                       "&Lists");
    QLabel *tmpTagsLabel = new QLabel("- <a href=\"tag1\">#tag1</a><br /><br />"
                                      "- <a href=\"tag2\">#tag2</a><br /><br />"
                                      "- <a href=\"tag3\">#tag3</a><br /><br />");
    tmpTagsLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    tmpTagsLabel->setContextMenuPolicy(Qt::NoContextMenu);
    connect(tmpTagsLabel, SIGNAL(linkActivated(QString)),
            pumpController, SLOT(setCurrentTimeline(QString)));
    leftPanel->addItem(tmpTagsLabel,
                       QIcon::fromTheme("mail-tagged"),
                       "#&Tags");
    QLabel *tmpMoreLabel = new QLabel("- Activity\n\n"
                                      "- @Mentions\n\n"
                                      "");
    tmpMoreLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    leftPanel->addItem(tmpMoreLabel,
                       QIcon::fromTheme("kde"),
                       "More...");

    leftLayout->addWidget(leftPanel);

    // Right side
    rightLayout = new QVBoxLayout();

    QSettings tmpSettings; // FIXME
    this->publisherType = tmpSettings.value("publisherType", 0).toInt(); // FIXME
    publisher = new Publisher(publisherType);


    timeLine = new TimeLine(pumpController, this);
    timeLineScrollArea = new QScrollArea();
    timeLineScrollArea->setWidget(timeLine);  // Make timeline scrollable
    timeLineScrollArea->setWidgetResizable(true); // Seems to work in opposite way


    messageList = new MessageList(this);
    messageListScrollArea = new QScrollArea();
    messageListScrollArea->setWidget(messageList);
    messageListScrollArea->setWidgetResizable(true);


    contactList = new ContactList(pumpController, this);
    contactListScrollArea = new QScrollArea();
    contactListScrollArea->setWidget(contactList); // Make contact list scrollable
    contactListScrollArea->setWidgetResizable(true);

    tabWidget = new QTabWidget();
    tabWidget->addTab(timeLineScrollArea,
                      QIcon::fromTheme("kde"),
                      tr("Time&line"));
    tabWidget->addTab(messageListScrollArea,
                      QIcon::fromTheme("mail-message"),
                      tr("&Messages"));
    tabWidget->addTab(contactListScrollArea,
                      QIcon::fromTheme("system-users"),
                      tr("&Contacts"));

    rightLayout->addWidget(publisher, 1); // stretch 1/10
    rightLayout->addWidget(tabWidget, 9); // stretch 9/10



    mainLayout->addLayout(leftLayout, 1); // stretch=1/10
    mainLayout->addLayout(rightLayout, 9); // stretch=9/10, take +space than left panel
    centralWidget->setLayout(mainLayout);

    this->setCentralWidget(centralWidget);


    ////////////////// Load configuration from disk
    loadSettings();


    // FreeDesktop.org notifications handler
    fdNotifier = new FDNotifications();
    fdNotifier->setNotificationType(showNotifications); // valid since loadSettings()
                                                        // was just called
    connect(fdNotifier, SIGNAL(showFallbackNotification(QString)),
            this, SLOT(showTrayFallbackMessage(QString)));


    updateTimer = new QTimer(this);
    updateTimer->setInterval(this->updateInterval * 1000 * 60); // min > msec
    connect(updateTimer, SIGNAL(timeout()), this, SLOT(updateTimelines()));
    updateTimer->start();


    accountDialog = new AccountDialog(pumpController);
    connect(accountDialog, SIGNAL(userIDChanged(QString)),
            this, SLOT(updateUserID(QString)));

    configDialog = new ConfigDialog(this->updateInterval,
                                    this->tabsPosition,
                                    this->tabsMovable,
                                    this->publisherType,
                                    this->showNotifications);
    connect(configDialog, SIGNAL(configurationChanged(int,int,bool,int,int)),
            this, SLOT(updateSettings(int,int,bool,int,int)));



    //////////////////////////////// Connections for PumpController
    //// Incoming    
    connect(pumpController, SIGNAL(profileReceived(QString,QString)),
            this, SLOT(updateProfileData(QString,QString)));
    connect(pumpController, SIGNAL(avatarPictureReceived(QByteArray,QUrl)),
            this, SLOT(storeAvatar(QByteArray,QUrl)));
    connect(pumpController, SIGNAL(imageReceived(QByteArray,QUrl)),
            this, SLOT(storeImage(QByteArray,QUrl)));
    connect(pumpController, SIGNAL(aspectListReceived(QVariantList)),
            this, SLOT(updateAspectList(QVariantList)));
    connect(pumpController, SIGNAL(timeLineReceived(QVariantList)),
            timeLine, SLOT(setTimeLineContents(QVariantList)));
    connect(pumpController, SIGNAL(contactListReceived(QVariantList)),
            contactList, SLOT(setContactListContents(QVariantList)));
    connect(pumpController, SIGNAL(showNotification(QString)),
            fdNotifier, SLOT(showMessage(QString)));

    // Update statusBar message from pumpController's infos
    connect(pumpController, SIGNAL(currentJobChanged(QString)),
            this, SLOT(setStatusBarMessage(QString)));

    //// Outgoing
    connect(publisher, SIGNAL(sharePost(QString)),
            pumpController, SLOT(post(QString)));



    // Don't set PumpController in motion if no User ID is set!
    if (!userID.isEmpty())
    {
        pumpController->setUserCredentials(this->userID);
        // getUserProfile() will be called from setUserCredentials()
    }


    // Add menus
    createMenus();

    // Add the system tray icon
    createTrayIcon();


    // tmp statusBar stuff
    this->statusBar()->showMessage("Initializing... (this is work in progress)");




    qDebug() << "MainWindow created";
}



MainWindow::~MainWindow()
{
    qDebug() << "MainWindow destroyed";
}




void MainWindow::closeEvent(QCloseEvent *event)
{
    qDebug() << "MainWindow::closeEvent()";


    if (reallyQuitProgram)
    {
        event->accept(); // really close, if called from Quit menu
        qDebug() << "Quit called from menu, shutting down program";
    }
    else
    {
        this->hide();     // Hide window, app accessible via tray icon
        qDebug() << "Tried to close main window, so hiding to tray";
        event->ignore();  // ignore the closeEvent
    }
}




/*
 * Prepare the data directory. Create if necessary
 *
 */
void MainWindow::prepareDataDirectory()
{
    #if QT_VERSION < QT_VERSION_CHECK(5, 0, 0)
    dataDirectory = QDesktopServices::storageLocation(QDesktopServices::DataLocation);
    #else
    dataDirectory = QStandardPaths::standardLocations(QStandardPaths::DataLocation).first();
    #endif
    qDebug() << "Data directory:" << this->dataDirectory;

    QDir dataDir;
    if (!dataDir.exists(dataDirectory))
    {
        qDebug() << "Creating data directory";
        if (dataDir.mkpath(dataDirectory))
        {
            qDebug() << "Data directory created";
        }
        else
        {
            qDebug() << "Error creating data directory!";
        }
    }

    if (!dataDir.exists(dataDirectory + "/images"))
    {
        qDebug() << "Creating images directory";
        if (dataDir.mkpath(dataDirectory + "/images"))
        {
            qDebug() << "Images directory created";
        }
        else
        {
            qDebug() << "Error creating images directory!";
        }
    }

    if (!dataDir.exists(dataDirectory + "/avatars"))
    {
        qDebug() << "Creating avatars directory";
        if (dataDir.mkpath(dataDirectory + "/avatars"))
        {
            qDebug() << "Avatars directory created";
        }
        else
        {
            qDebug() << "Error creating avatars directory!";
        }
    }

}



/*
 * Populate the menus
 *
 */
void MainWindow::createMenus()
{
    sessionMenu = new QMenu(tr("&Session"));

    sessionUpdateTimelines = new QAction(QIcon::fromTheme("view-refresh"),
                                         tr("&Update Timelines"),
                                         this);
    sessionUpdateTimelines->setShortcut(QKeySequence::Refresh);
    connect(sessionUpdateTimelines, SIGNAL(triggered()),
            this, SLOT(updateTimelines()));
    sessionMenu->addAction(sessionUpdateTimelines);


    sessionStartConversation = new QAction(QIcon::fromTheme("document-edit"),
                                          tr("&Post a Note"),
                                          this);
    sessionStartConversation->setShortcut(QKeySequence("Ctrl+N"));
    connect(sessionStartConversation, SIGNAL(triggered()),
            publisher, SLOT(setFullMode()));
    connect(sessionStartConversation, SIGNAL(triggered()),
            this, SLOT(show()));  // show window, in case it's hidden and
                                  // Start Conversation is used from tray menu
    sessionMenu->addAction(sessionStartConversation);


    sessionQuit = new QAction(QIcon::fromTheme("application-exit"),
                              tr("&Quit"), this);
    sessionQuit->setShortcut(QKeySequence::Quit);
    connect(sessionQuit, SIGNAL(triggered()),
            this, SLOT(quitProgram()));
    sessionMenu->addAction(sessionQuit);

    this->menuBar()->addMenu(sessionMenu);



    settingsMenu = new QMenu(tr("S&ettings"));

    settingsAccount = new QAction(QIcon::fromTheme("user-properties"),
                                  tr("&Account Details"), this);
    settingsAccount->setShortcut(QKeySequence("Ctrl+A"));
    connect(settingsAccount, SIGNAL(triggered()),
            accountDialog, SLOT(show()));
    settingsMenu->addAction(settingsAccount);

    settingsConfigure = new QAction(QIcon::fromTheme("configure"),
                                    tr("&Configure Dianara"), this);
    connect(settingsConfigure, SIGNAL(triggered()),
            configDialog, SLOT(show()));
    settingsMenu->addAction(settingsConfigure);
    this->menuBar()->addMenu(settingsMenu);
    this->menuBar()->addSeparator();



    helpMenu = new QMenu(tr("&Help"));

    helpVisitWebsite = new QAction(QIcon::fromTheme("internet-web-browser"),
                            tr("Visit &Website"), this);
    connect(helpVisitWebsite, SIGNAL(triggered()),
            this, SLOT(visitWebSite()));
    helpMenu->addAction(helpVisitWebsite);

    helpAbout = new QAction(QIcon::fromTheme("system-help"),
                            tr("About &Dianara"), this);
    connect(helpAbout, SIGNAL(triggered()),
            this, SLOT(aboutDianara()));
    helpMenu->addAction(helpAbout);
    this->menuBar()->addMenu(helpMenu);



    // Context menu for the tray icon
    trayContextMenu = new QMenu("Tray Context Menu");
    trayContextMenu->addAction(QIcon(":/icon/64x64/dianara.png"),
                               tr("&Show Window"),
                               this, SLOT(show()));
    trayContextMenu->addSeparator();
    trayContextMenu->addAction(sessionUpdateTimelines);
    trayContextMenu->addAction(sessionStartConversation);
    trayContextMenu->addAction(settingsConfigure);
    trayContextMenu->addAction(helpAbout);
    trayContextMenu->addSeparator();
    trayContextMenu->addAction(sessionQuit);

    // FIXME: if mainwindow is hidden, program quits
    // after closing Configure or About window (now partially fixed)


    qDebug() << "Menus created";
}



/*
 * Create an icon in the system tray, define its contextual menu, etc.
 *
 */
void MainWindow::createTrayIcon()
{
    trayIcon = new QSystemTrayIcon(QIcon::fromTheme("dianara",
                                                    QIcon(":/icon/64x64/dianara.png")), this);
    if (!userID.isEmpty())
    {
        trayIcon->setToolTip("Dianara - " + this->userID);
    }
    else
    {
        trayIcon->setToolTip("Dianara - [ " + tr("Your Pump.io account is not configured") + " ]");
    }


    // Catch clicks on icon
    connect(trayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
            this, SLOT(trayControl(QSystemTrayIcon::ActivationReason)));


    // Set contextual menu for the icon
    trayIcon->setContextMenu(this->trayContextMenu);


    trayIcon->show();

    qDebug() << "Tray icon created";
}



/*
 * Load general program settings and state: size, position...
 *
 */
void MainWindow::loadSettings()
{
    QSettings settings;

    this->setWindowTitle("Dianara - " + settings.value("userID",
                                                       tr("[ Configure your account ]")).toString());

    this->resize(settings.value("windowSize", QSize(640, 450)).toSize());

    this->updateInterval = settings.value("updateInterval", 5).toInt();

    this->tabsPosition = settings.value("tabsPosition", QTabWidget::North).toInt();
    tabWidget->setTabPosition((QTabWidget::TabPosition)tabsPosition);

    this->tabsMovable = settings.value("tabsMovable", true).toBool();
    tabWidget->setMovable(tabsMovable);

    // publisherType 0 by default, buttons below
    this->publisherType = settings.value("publisherType", 0).toInt();

    this->showNotifications = settings.value("showNotifications", 0).toInt();

    firstRun = settings.value("firstRun", true).toBool();
    if (firstRun)
    {
        qDebug() << "This is the first run";
    }
    else
    {
        this->move(settings.value("windowPosition").toPoint());
        userID = settings.value("userID", "").toString();
        userAuth = settings.value("userAuth", "").toByteArray();
    }


    qDebug() << "Settings loaded";
}

/*
 * Save general program settings and state: size, position...
 *
 */
void MainWindow::saveSettings()
{
    QSettings settings;

    settings.setValue("firstRun", false);

    settings.setValue("windowSize", this->size());
    settings.setValue("windowPosition", this->pos());

    settings.setValue("updateInterval", this->updateInterval);

    settings.setValue("tabsPosition", this->tabsPosition);
    settings.setValue("tabsMovable", this->tabsMovable);

    settings.setValue("publisherType", this->publisherType);

    settings.setValue("showNotifications", this->showNotifications);

    qDebug() << "Settings saved";
}


///////////// SLOTS


/*
 * Update UserID string from signal emitted in AccountDialog
 *
 */
void MainWindow::updateUserID(QString newUserID)
{
    this->userID = newUserID;

    this->setWindowTitle("Dianara - " + userID);
    this->trayIcon->setToolTip("Dianara - " + userID);

    this->pumpController->setUserCredentials(userID);

    // Remove current user's name and avatar
    this->fullNameLabel->setText("--");
    avatarIconLabel->setPixmap(QPixmap(":/images/no-avatar.png")
                             .scaled(64, 64,
                                     Qt::KeepAspectRatio,
                                     Qt::SmoothTransformation));


    qDebug() << "UserID updated from AccountDialog:" << userID;
}



/*
 * Update settings changed from ConfigDialog()
 *
 */
void MainWindow::updateSettings(int newUpdateInterval, int newTabsPosition,
                                bool newTabsMovable, int newPublisherType,
                                int newShowNotifications)
{
    this->updateInterval = newUpdateInterval;
    this->updateTimer->setInterval(updateInterval * 1000 * 60);

    this->tabsPosition = newTabsPosition;
    this->tabWidget->setTabPosition((QTabWidget::TabPosition)tabsPosition);

    this->tabsMovable = newTabsMovable;
    this->tabWidget->setMovable(tabsMovable);

    this->publisherType = newPublisherType;

    this->showNotifications = newShowNotifications;
    fdNotifier->setNotificationType(showNotifications); // FIXME: Safe?

    qDebug() << "updateInterval updated:" << updateInterval << updateInterval*60000;
    qDebug() << "tabsPosition updated:" << tabsPosition << tabWidget->tabPosition();
    qDebug() << "tabsMovable updated:" << tabsMovable;
    qDebug() << "publisherType updated:" << publisherType;
    qDebug() << "Notifications updated:" << showNotifications;
}




/*
 * Control interaction with the system tray icon
 *
 */
void MainWindow::trayControl(QSystemTrayIcon::ActivationReason reason)
{
    qDebug() << "Tray icon activation reason:" << reason;

    if (reason == QSystemTrayIcon::Context)
    {
        qDebug() << "Tray icon context menu here";
    }
    else
    {
         // Hide or show the main window
        this->setVisible(this->isHidden()); // this->setShown() deprecated on Qt5
                                            // replace with setVisible()
        qDebug() << "trayControl() - MainWindow hidden?" << this->isHidden();
    }
}



/*
 * If FreeDesktop.org notifications are not available,
 * fall back to Qt's balloon ones
 *
 */
void MainWindow::showTrayFallbackMessage(QString message)
{
    // if FreeDesktop.org notifications are not available,
    // fall back to Qt's balloon ones
    this->trayIcon->showMessage(tr("Dianara notification"),
                                message,
                                QSystemTrayIcon::Information,
                                4000); // 4 secs
}


void MainWindow::updateProfileData(QString fullName, QString avatarURL)
{
    this->fullNameLabel->setText(fullName);
    qDebug() << "Updated profile data from server:" << fullName;
    this->avatarURL = avatarURL;
    qDebug() << "Own avatar URL:" << avatarURL;


    // Get local file name, which is stored in base64 hash form
    QString avatarFilename = MiscHelpers::getCachedAvatarFilename(avatarURL);

    if (QFile::exists(avatarFilename))
    {
        // Load avatar if already cached
        this->avatarIconLabel->setPixmap(QPixmap(avatarFilename)
                                                .scaled(64, 64,
                                                        Qt::KeepAspectRatio,
                                                        Qt::SmoothTransformation));
        qDebug() << "Using cached avatar for user";
    }
    else
    {
        pumpController->getAvatar(avatarURL);
    }

}




void MainWindow::updateAspectList(QVariantList aspectList)
{
    QString aspectsString;

    foreach (QVariant aspect, aspectList)
    {
        if (aspect.type() == QVariant::Map)
        {
            QVariantMap aspectMap = aspect.toMap();

            QString aspectName = aspectMap.value("aspect").toMap().value("name").toString();
            QString aspectId = aspectMap.value("aspect").toMap().value("id").toString();

            aspectsString.append(QString("- <a href=\"aspect:/%2\"><b>%1</b> (%2)</a><br /><br />")
                                .arg(aspectName)
                                .arg(aspectId));
        }
        else
        {
            qDebug() << "Expected a Map, got something else";
        }
    }

    this->aspectListLabel->setText(aspectsString);

    qDebug() << "Updated aspect list from server:" << aspectsString;
}



void MainWindow::updateTimelines()
{
    pumpController->getTimeline();
    // received timeline data will come in a SIGNAL()


    if (showNotifications != 2) // If notifications are enabled (FD.org or Qt's)
    {
        fdNotifier->showMessage("Updating <u>timelines</u>.\n"
                                "Temporary <b>dummy</b> notification!\n"
                                "<a href=\"http://jancoding.wordpress.com/dianara\">Go to Dianara's website</a>");
    }

    qDebug() << "Updated timelines by menu or after:" << this->updateInterval << "min";
}




/*
 * Store avatars on disk
 *
 */
void MainWindow::storeAvatar(QByteArray avatarData, QUrl avatarURL)
{
    QString avatarString = avatarURL.toString();
    QString fileName = avatarString.toUtf8().toBase64();


    QString fileExtension = avatarString.remove(QRegExp(".*\\."));
    fileName.append("." + fileExtension);

    qDebug() << "Saving avatar to disk: " << fileName;

    QFile avatarFile(dataDirectory + "/avatars/" + fileName);
    avatarFile.open(QFile::WriteOnly);
    avatarFile.write(avatarData);
    avatarFile.close();

    if (avatarURL == this->avatarURL)
    {
        this->avatarIconLabel->setPixmap(QPixmap(dataDirectory + "/avatars/" + fileName)
                                         .scaled(64, 64,
                                                 Qt::KeepAspectRatio,
                                                 Qt::SmoothTransformation));
    }        

    qDebug() << "avatarData size:" << avatarData.size();
}





/*
 * Store images on disk
 *
 */
void MainWindow::storeImage(QByteArray imageData, QUrl imageURL)
{
    QString imageString = imageURL.toString();

    QString fileName = imageString.toUtf8().toBase64();
    QString fileExtension = imageString.remove(QRegExp(".*\\."));
    fileName.append("." + fileExtension);

    qDebug() << "Saving image to disk: " << fileName;

    QFile imageFile(dataDirectory + "/images/" + fileName);
    imageFile.open(QFile::WriteOnly);
    imageFile.write(imageData);
    imageFile.close();

    qDebug() << "imageData size:" << imageData.size();
}



/*
 * Update status bar message, from pumpController's signals
 *
 */
void MainWindow::setStatusBarMessage(QString message)
{
    //this->setStatusTip(message);
    this->statusBar()->showMessage(message, 0);
}





/*
 * Open website in browser
 *
 */
void MainWindow::visitWebSite()
{
    qDebug() << "Opening website in browser";
    QDesktopServices::openUrl(QUrl("http://jancoding.wordpress.com/dianara"));
}




/*
 * About... message
 *
 */
void MainWindow::aboutDianara()
{
    QMessageBox::about(this, tr("About Dianara"),
                       "<b>Dianara v0.4</b><br>"
                       "Copyright 2012-2013  JanKusanagi<br>"
                       "<a href=\"http://dianara.nongnu.org\">"
                       "http://dianara.nongnu.org</a><br>"
                       "<a href=\"http://jancoding.wordpress.com/dianara\">"
                       "http://jancoding.wordpress.com/dianara</a><br><br><br>"
                       + tr("Dianara is a pump.io social networking client.<br><br>"
                            "It is work in progress. Right now only the "
                            "interface and some basic capabilities: posting text, "
                            "showing last 20 posts and contact listing."
                            "<br><br>"));

    if (this->isHidden()) // FIXME: ugly workaround to avoid closing the program
    {                     // after closing About dialog, if mainWindow hidden
        this->show();
        this->hide();     // This hack might be causing problems under LXDE
        qDebug() << "MainWindow was hidden, showing and hiding again";
    }
}



/*
 * Close the program. Needed to quit correctly from context menu
 *
 */
void MainWindow::quitProgram()
{
    // Add more needed shutdown stuff here

    if (this->isHidden())
    {
        this->show();
    }

    saveSettings();

    reallyQuitProgram = true;

    qApp->closeAllWindows();

    qDebug() << "All windows closed, bye!";
}
