/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "diasporacontroller.h"


DiasporaController::DiasporaController(QObject *parent) : QObject(parent)
{
    this->userAgentString = "Dianara/0.4-dev";

    connect(&nam, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(requestFinished(QNetworkReply*)));

    // FIXME: setting this up for now, to at least have debug messages just in case
    connect(&nam, SIGNAL(sslErrors(QNetworkReply*,QList<QSslError>)),
            this, SLOT(sslErrorsHandler(QNetworkReply*,QList<QSslError>)));

    this->initialDataStep = 0;

    initialDataTimer = new QTimer(this);
    initialDataTimer->setSingleShot(false); // Triggered constantly until stopped
    connect(initialDataTimer, SIGNAL(timeout()),
            this, SLOT(getInitialData()));


    qDebug() << "DiasporaController created";
}


DiasporaController::~DiasporaController()
{
    qDebug() << "DiasporaController destroyed";
}



/*
 * Get "pod.org" and "user" from "user@pod.org", set OAuth token from Account dlg
 *
 */
void DiasporaController::setUserCredentials(QString userID, QString userAuth)
{
    this->initialDataTimer->stop(); // Just in case it was running before


    this->userID = userID;
    this->userAuth = userAuth;
    qDebug() << "New userID/auth are:" << this->userID << this->userAuth;

    QStringList splittedUserID = userID.split("@");
    qDebug() << "Splitted UserID:" << splittedUserID;

    this->userName = splittedUserID.at(0); // get username, before @
    this->serverURL = splittedUserID.at(1); // get URL, after @
    qDebug() << "Server URL to connect:" << serverURL << "; username:" << userName;

    //this->getUserProfile(this->userID);


    // This will call getAspectList(), getContactList() and getTimeLine();
    this->initialDataStep = 0;
    this->initialDataTimer->start(100);  // start immediately



    // TMP test, FIXME
    //this->getAvatar(user-avatar-url);
}


void DiasporaController::getUserProfile(QString userID)
{
    QStringList splittedUserID = userID.split("@");

    // GET https://joindiaspora.com/api/v0/users/username || DEPRECATED

    /*
    QNetworkRequest request(QUrl("https://" + splittedUserID.at(1)
                               + "/u/" + splittedUserID.at(0)
                               + ".json"));


    nam.get(request);
    qDebug() << "Requested:" << request.url();
    */
}



void DiasporaController::getAvatar(QString avatarURL)
{
    qDebug() << "Getting avatar";


    QNetworkRequest avatarRequest(QUrl((const QString)avatarURL));
    avatarRequest.setRawHeader("User-Agent", userAgentString);
    avatarRequest.setAttribute(QNetworkRequest::User,
                               QVariant(AvatarRequest));

    nam.get(avatarRequest);
}



void DiasporaController::getImage(QString imageURL)
{
    qDebug() << "Getting image";


    QNetworkRequest imageRequest(QUrl((const QString)imageURL));
    imageRequest.setRawHeader("User-Agent", userAgentString);
    imageRequest.setAttribute(QNetworkRequest::User,
                              QVariant(ImageRequest));

    nam.get(imageRequest);
    qDebug() << "imageRequest sent";
}


/*
 * GET https://diasporapod.org/fapi/v0/aspects.json
 *
 * *TEMPORARILY*, this method will only work with Pistos fork of Diaspora* code
 *
 */
void DiasporaController::getAspectList()
{
    emit currentJobChanged(tr("Getting list of aspects..."));

    qDebug() << "Getting list of aspects";
    qDebug() << "\n=== This only works with Pistos fork of Diaspora! ===\n";


    QNetworkRequest aspectListRequest(QUrl("https://" + this->serverURL
                                     + "/fapi/v0/aspects.json?token="
                                     + this->userAuth));

    aspectListRequest.setRawHeader("User-Agent", userAgentString);
    aspectListRequest.setAttribute(QNetworkRequest::User,
                                   QVariant(AspectListRequest));
    nam.get(aspectListRequest);
}




/*
 * GET https://diasporapod.org/fapi/v0/contacts.json
 *
 * *TEMPORARILY*, this method will only work with Pistos fork of Diaspora* code
 *
 */
void DiasporaController::getContactList()
{
    emit currentJobChanged(tr("Getting contact list..."));

    qDebug() << "Getting contact list";
    qDebug() << "\n=== This only works with Pistos fork of Diaspora! ===\n";


    QNetworkRequest contactListRequest(QUrl("https://" + this->serverURL
                                     + "/fapi/v0/contacts.json?token="
                                     + this->userAuth));

    contactListRequest.setRawHeader("User-Agent", userAgentString);
    contactListRequest.setAttribute(QNetworkRequest::User,
                                   QVariant(ContactListRequest));
    nam.get(contactListRequest);
}




/*
 * GET https://diasporapod.org/u/username.json
 *
 */
void DiasporaController::getTimeline()
{
    emit currentJobChanged(tr("Getting timeline..."));
    qDebug() << "Getting timeline";


    QNetworkRequest timelineRequest(QUrl("https://" + this->serverURL
                                     + "/u/" + this->userName
                                     + ".json"));


    // for tests: get a tag's timeline
    //QNetworkRequest timelineRequest(QUrl("https://" + this->serverURL
    //                                 + "/tags/dianara.json"));

    timelineRequest.setRawHeader("User-Agent", userAgentString);
    timelineRequest.setAttribute(QNetworkRequest::User,
                                 QVariant(TimelineRequest));
    nam.get(timelineRequest);
}



/*
 * Get one specific post
 *
 * GET https://diasporapod.org/posts/id.json
 *
 */
void DiasporaController::getPost(int id)
{
    qDebug() << "Getting post" << id;


    QNetworkRequest postRequest(QUrl("https://" + this->serverURL
                                     + "/posts/" + QString("%1").arg(id)
                                     + ".json"));
    postRequest.setRawHeader("User-Agent", userAgentString);
    postRequest.setAttribute(QNetworkRequest::User,
                             QVariant(PostRequest));

    nam.get(postRequest);
}



/***************************************************************************/
/*********************************** SLOTS *********************************/
/***************************************************************************/




void DiasporaController::requestFinished(QNetworkReply *reply)
{
    int httpCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
    int requestType = reply->request().attribute(QNetworkRequest::User).toInt();

    qDebug() << "Request finished. HTTP code:" << httpCode;
    qDebug() << "Size:" << reply->size() << "bytes; URL:" << reply->url();

    qDebug() << "isFinished()?" << reply->isFinished();
    qDebug() << "Request type:" << requestType;

    switch (httpCode)
    {
    case 503:
        emit showNotification("Service Unavailable (503)");
        qDebug() << "HTTP 503: Service Unavailable.";
        return;

    case 500:
        emit showNotification("Internal server error (500)");
        qDebug() << "HTTP 500: Internal server error.";
        return;

    case 404:
        emit showNotification("Not Found (404)");
        qDebug() << "HTTP 404: Not Found.";
        return;

    case 403:
        emit showNotification("Forbidden (403)");
        qDebug() << "HTTP 403: Forbidden.";
        return;

    case 200:
        qDebug() << "HTTP 200: OK!";
    }


    // Read all received data
    QByteArray packetData = reply->readAll();

    // Prepare the JSON parser
    QJson::Parser jsonParser;
    bool jsonParsedOK = false;    
    QVariantMap jsonData;

    QVariantList postsVariant; // TMP/FIXME

    switch (requestType)
    {

    ////////////////////////////////////////////////// If a timeline was requested
    case TimelineRequest:
        qDebug() << "A Timeline was requested";

        //jsonData = jsonParser.parse(packetData, &jsonParsedOK).toMap();
        postsVariant = jsonParser.parse(packetData, &jsonParsedOK).toList();
        qDebug() << "JSON data size (items):" << postsVariant.size();

        if (jsonParsedOK && postsVariant.size() > 0)
        {
            qDebug() << "JSON parsed OK";

            /*QVariant postsVariant = jsonData.value("posts");
            if (postsVariant.type() == QVariant::List)
            {
                qDebug() << "Parsed a List, rendering timeline...";
                emit currentJobChanged(tr("Timeline received."));
                emit timeLineReceived(postsVariant.toList());
            }
            else
            {
                qDebug() << "Expected a timeline, received something else:";
                qDebug() << postsVariant;
            }
            */
            emit currentJobChanged(tr("Timeline received."));
            emit timeLineReceived(postsVariant);
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << packetData; // JSON directly
        }
        break;



    ///////////////////////////////////////// If a single post was requested, by ID
    case PostRequest:
        qDebug() << "A single post was requested";

        break;


    case AspectListRequest:
        qDebug() << "The list of aspects was requested";

        jsonData = jsonParser.parse(packetData, &jsonParsedOK).toMap();
        qDebug() << "JSON data size (items):" << jsonData.size();

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";

            QVariant aspectsVariant = jsonData.value("aspects");
            if (aspectsVariant.type() == QVariant::List)
            {
                qDebug() << "Parsed a List, listing aspects...";
                emit currentJobChanged(tr("Aspect list received."));
                emit aspectListReceived(aspectsVariant.toList());
            }
            else
            {
                qDebug() << "Expected a list of aspects, received something else:";
                qDebug() << jsonData;
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << packetData; // JSON directly
        }
        break;


    case ContactListRequest:
        qDebug() << "The contact list was requested";

        jsonData = jsonParser.parse(packetData, &jsonParsedOK).toMap();
        qDebug() << "JSON data size (items):" << jsonData.size();

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";

            QVariant contactsVariant = jsonData.value("contacts");
            if (contactsVariant.type() == QVariant::List)
            {
                qDebug() << "Parsed a List, listing contacts...";
                emit currentJobChanged(tr("Contact list received."));
                emit contactListReceived(contactsVariant.toList());
            }
            else
            {
                qDebug() << "Expected a list of contacts, received something else:";
                qDebug() << jsonData;
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << packetData; // JSON directly
        }

        break;



    case AvatarRequest:
        qDebug() << "Received AVATAR data, from " << reply->url();
        // FIXME: take care of possible redirections

        //this->avatarData.append(packetData);
        if (reply->isFinished())
        {
            qDebug() << "Avatar received 100%";
            emit avatarPictureReceived(packetData, reply->url());
        }
        else
        {
            qDebug() << "Avatar not complete yet";
        }
        break;


    case ImageRequest:
        qDebug() << "Received IMAGE data, from " << reply->url();
        //this->imageData.append(packetData);
        if (reply->isFinished())
        {
            qDebug() << "Image received 100%";
            emit imageReceived(packetData, reply->url());
        }
        else
        {
            qDebug() << "Image not complete yet";
        }
        break;


    }
    // end switch (requestType)
}




void DiasporaController::sslErrorsHandler(QNetworkReply *reply, QList<QSslError> errorList)
{
    qDebug() << "\n\n==== SSL errors!! ====";
    qDebug() << "At:" << reply->url();
    qDebug() << "Error list:" << errorList << "\n\n";

}



/*
        // left from the old /api/v0/ methods...

        if (jsonData.contains("diaspora_id"))
        {
            qDebug() << "Received a user's profile.";
            qDebug() << "JSON Parsed OK. Raw data:" << jsonData << "\n\n";
            qDebug() << "Extracted:";
            qDebug() << "UserID:" << jsonData.value("diaspora_id").toString();
            qDebug() << "Full name:" << jsonData.value("first_name").toString()
                        + " " + jsonData.value("last_name").toString();
            emit profileReceived(jsonData.value("first_name").toString()
                                 + " " + jsonData.value("last_name").toString());
            ownAvatarURL = jsonData.value("image_url").toString();
            qDebug() << "Avatar URL:"  << ownAvatarURL;

            this->getOwnAvatar();
        }
*/


/*
 * Called by a QTimer, get initial data (aspects, contacts) one step at a time
 *
 */
void DiasporaController::getInitialData()
{
    qDebug() << "DiasporaController::getInitialData() step" << initialDataStep;

    initialDataTimer->setInterval(8000);  // Every 8 sec (5 is API throttle limit)


    switch (this->initialDataStep)
    {
    case 0:
        this->getAspectList();
        break;

    case 1:
        this->getContactList();
        break;

    case 2:
        this->getTimeline();
        break;


    default:
        emit currentJobChanged(tr("Ready.")); // FIXME
        initialDataTimer->stop();

        qDebug() << "All initial data loaded";
    }

    ++initialDataStep;
}





/*
 * Send a post to the server
 *
 * *TEMPORARILY*, this method will only work with Pistos fork of Diaspora* code
 *
 */
void DiasporaController::post(QString postText)
{
    qDebug() << "DiasporaController::post()";
    qDebug() << "\n=== This only works with Pistos fork of Diaspora! ===\n";

    QNetworkRequest postRequest(QUrl("https://" + this->serverURL
                                     + "/fapi/v0/posts.json"));

    postRequest.setRawHeader("User-Agent", userAgentString);
    postRequest.setAttribute(QNetworkRequest::User,
                             QVariant(PublishPostRequest));

    QUrl data;

#if QT_VERSION < QT_VERSION_CHECK(5, 0, 0)
    data.addQueryItem("text", postText);
    data.addQueryItem("token", this->userAuth);

    qDebug() << "About to POST:" << data.encodedQuery();

    nam.post(postRequest, data.encodedQuery());
#else
    QUrlQuery query;
    query.addQueryItem("text", postText);
    query.addQueryItem("token", this->userAuth);
    data.setQuery(query);

    qDebug() << "About to POST:" << data.query();

    nam.post(postRequest, data.query().toLocal8Bit());
#endif
}




void DiasporaController::setCurrentTimeline(QString link)
{
    qDebug() << "DiasporaController::setCurrentTimeline()" << link;
}
