/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <limits.h>
#include <assert.h>
#include "tbox.h"
#include "nsplit.h"

static void	print_usage( const char * );

/*
** This function processes the command line arguments.
*/
int get_args( int argc, char * const argv[], const char **input_filename, const char **output_filename, unsigned short *split_cnt )
{
	char	*ptr, msg[100];
	int	ch;
	extern char	*optarg;
	unsigned long	split;

	assert( argv != (char * const *)0 );
	assert( input_filename != (const char **)0 );
	assert( output_filename != (const char **)0 );
	assert( split_cnt != (unsigned short *)0 );

#ifndef NDEBUG
	/*
	** Initialize debug to off.
	*/
	Debug = 0;
#endif

	*input_filename = (const char *)0;
	*output_filename = (const char *)0;
	*split_cnt = (unsigned short)0;

	while ( ( ch = getopt( argc, argv, "di:n:o:" ) ) != EOF ) {
		switch ( ch ) {
		case 'd':
#ifdef NDEBUG
			(void) fputs( argv[ 0 ], stderr );
			(void) fputs( " not compiled for debug.\n", stderr );
			return -1;
#else
			Debug = 1;
#endif
			break;
		case 'i':
			*input_filename = optarg;
			break;
		case 'n':
			split = strtoul( optarg, &ptr, 10 );
			if ( ptr == (char *)0 || *ptr != (char)0 ) {
				(void) snprintf( msg, sizeof( msg ), "Failed to convert [%.24s] from ASCII to positive integer.", optarg );
				UNIX_ERROR( msg );
				print_usage( argv[ 0 ] );
				return -1;
			}

			if ( split > USHRT_MAX ) {
				(void) snprintf( msg, sizeof( msg ), "Split count %lu exceeds maximum value %hu", split, USHRT_MAX );
				FPUT_SRC_CODE( stderr );
				(void) fputs( msg, stderr );
				print_usage( argv[ 0 ] );
				return -1;
			}

			*split_cnt = split;

			break;
		case 'o':
			*output_filename = optarg;
			break;
		default:
			print_usage( argv[ 0 ] );
			return -1;
		}
	}

	if ( *input_filename == (const char *)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Must specify input file name.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *output_filename == (const char *)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Must specify output file name.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *split_cnt == (unsigned short)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Must specify number of ways to split input file.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( Debug ) {
		(void) fprintf( stderr, "Input File Name: [%s]\n", *input_filename );
		(void) fprintf( stderr, "Output File Name: [%s]\n", *output_filename );
		(void) fprintf( stderr, "Split count: [%hu]\n", *split_cnt );
	}

	return 0;
}

static void print_usage( const char *exec_name )
{
	(void) fputs( "usage: ", stderr );
	(void) fputs( exec_name, stderr );
	(void) fputs( " [-d]", stderr );
	(void) fputs( " -i file", stderr );
	(void) fputs( " -n n", stderr );
	(void) fputs( " -o file", stderr );
	(void) fputc( '\n', stderr );
	(void) fputs( "\t-d -> debug\n", stderr );
	(void) fputs( "\t-i -> input file name\n", stderr );
	(void) fputs( "\t-n -> split count\n", stderr );
	(void) fputs( "\t-o -> output file name containing %d format\n", stderr );
}
