/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "sexpr.h"
#include "dfile.h"
#include "where.h"
#include "_where.h"


/*
** This function is part of the interpreter for comparing data values.
*/

#define NUMERIC_CONSTANT( x )	( ( DATUM_TYPE( x ) == Constant && DATUM_CONST_TYPE( x ) == Real ) )

static int compare_numeric_datum( int *, datum_t *, datum_t * );
static int convert_to_numeric( double *, const char * );
static int assign_numeric( double *, datum_t * );
static void assign_string( vchar_t *, datum_t * );

int _where_compare_condition( int *ret_result, multi_datum_t *multi_datum )
{
	datum_t	*lhs, *rhs;
	vchar_t lhs_str, rhs_str;

	assert( ret_result != (int *)0 );
	assert( multi_datum != (multi_datum_t *)0 );

	DEBUG_FUNC_START;

	/*
	** Decide if comparison is to be done as ASCII strings or
	** double floating point.
	*/

	assert( multi_datum->datum_cnt == 2UL );

	lhs = &multi_datum->datum[ 0 ];
	assert( lhs != (datum_t *)0 );

	rhs = &multi_datum->datum[ 1 ];
	assert( rhs != (datum_t *)0 );

	if ( NUMERIC_CONSTANT( lhs ) || NUMERIC_CONSTANT( rhs ) ) {
		if ( compare_numeric_datum( ret_result, lhs, rhs ) == -1 ) {
			RETURN_INT( WHERE_INVLDNBR );
		}
	} else {
		assign_string( &lhs_str, lhs );
		assign_string( &rhs_str, rhs );
		*ret_result = strnncmp( lhs_str.value, lhs_str.length, rhs_str.value, rhs_str.length );

		if ( Debug ) {
			(void) fprintf( stderr, "compared string values [%*.*s], [%*.*s]\n", lhs_str.length, lhs_str.length, lhs_str.value, rhs_str.length, rhs_str.length, rhs_str.value );
		}
	}

	if ( Debug ) {
		(void) fprintf( stderr, "result = %d\n", *ret_result );
	}

	RETURN_INT( WHERE_NOERR );
}

static int compare_numeric_datum( int *ret_result, datum_t *lhs_datum, datum_t *rhs_datum )
{
	double	lhs, rhs;

	if ( assign_numeric( &lhs, lhs_datum ) == -1 ) {
		return -1;
	}

	if ( assign_numeric( &rhs, rhs_datum ) == -1 ) {
		return -1;
	}

	if ( lhs < rhs ) {
		*ret_result = -1;
	} else {
		if ( lhs > rhs ) {
			*ret_result = 1;
		} else {
			*ret_result = 0;
		}
	}

	if ( Debug ) {
		(void) fprintf( stderr, "compared numeric values %g, %g\n", lhs, rhs );
	}

	return 0;
}

static int assign_numeric( double *result, datum_t *datum )
{
	int	ret;
	char	nbr_str[24];
	size_t	len;

	if ( DATUM_TYPE( datum ) == Constant ) {
		if ( DATUM_CONST_TYPE( datum ) == Real ) {
			*result = DATUM_CONST_REAL( datum );
			ret = 0;
		} else {
			ret = convert_to_numeric( result, DATUM_CONST_LITERAL( datum ).value );
		}
	} else {
		assert( DATUM_TYPE( datum ) == Variable );
		if ( DATUM_VAR_BIND_LEN( datum ) != (size_t *)0 ) {
			len = *DATUM_VAR_BIND_LEN( datum );
			if ( len >= sizeof( nbr_str )  ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "Length of number [", stderr );
				(void) fput_uint( len, stderr );
				(void) fputs( "] exceeded max length [", stderr );
				(void) fput_uint( sizeof( nbr_str ) - 1, stderr );
				(void) fputs( "].\n", stderr );
				return -1;
			}
			(void) memcpy( (void *)nbr_str, DATUM_VAR_BIND_STR( datum ), len );
			nbr_str[ len ] = (char)0;
			ret = convert_to_numeric( result, nbr_str );
		} else {
			ret = convert_to_numeric( result, DATUM_VAR_BIND_STR( datum ) );
		}
	}

	return ret;
}

static int convert_to_numeric( double *result, const char *str )
{
	const char	*end;

	*result = strtod( str, (char **)&end );

	if ( str == end || *end != (char)0 ) {
		return -1;
	}

	return 0;
}

static void assign_string( vchar_t *result, datum_t *datum )
{
	if ( DATUM_TYPE( datum ) == Constant ) {
		assert( DATUM_CONST_TYPE( datum ) != Real );
		*result = DATUM_CONST_LITERAL( datum );
		return;
	}

	assert( DATUM_TYPE( datum ) == Variable );
	result->value = DATUM_VAR_BIND_STR( datum );
	if ( DATUM_VAR_BIND_LEN( datum ) == (size_t *)0 ) {
		result->length = strlen( DATUM_VAR_BIND_STR( datum ) );
	} else {
		result->length = *DATUM_VAR_BIND_LEN( datum );
	}
}
