/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <glob.h>
#include <assert.h>
#include "tbox.h"

static int	glob_err( const char *, int );

/*
** This function calls glob() with a file name expression to get
** an array of pointers that point to actual file names that match
** the expression.
*/
int glob_fname( const char ***ret_fname, const char *fname_exp )
{
	glob_t	glob_area;
	int	status, ret;
	unsigned short	ndx;
	const char	**ptr;

	assert( ret_fname != (const char ***)0 );
	assert( fname_exp != (const char *)0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s( [%s] )\n", __func__, fname_exp );
	}

	DEBUG_FUNC_START;

	/*
	** Initialize return values.
	*/
	*ret_fname = (const char **)0;

	status = glob( fname_exp, 0, glob_err, &glob_area );

	switch ( status ) {
	case GLOB_NOSPACE:
		/*
		** Out of memory.
		*/
		FPUT_SRC_CODE( stderr );
		(void) fputs( "glob() failed to allocate needed memory.\n", stderr );
		ret = -1;
		break;

	case GLOB_ABORTED:
		FPUT_SRC_CODE( stderr );
		(void) fputs( "glob() aborted on file pattern [", stderr );
		(void) fputs( fname_exp, stderr );
		(void) fputs( "].\n", stderr );
		ret = -1;
		break;

	case GLOB_NOSYS:
		FPUT_SRC_CODE( stderr );
		(void) fputs( "glob() encountered an unsupported function on file pattern [", stderr );
		(void) fputs( fname_exp, stderr );
		(void) fputs( "].\n", stderr );
		ret = -1;
		break;

	case GLOB_NOMATCH:
		FPUT_SRC_CODE( stderr );
		(void) fputs( "glob() did not find a match for file pattern [", stderr );
		(void) fputs( fname_exp, stderr );
		(void) fputs( "].\n", stderr );
		ret = -1;
		break;

	case 0:
		/*
		** glob() was successful.
		** Set return variable.
		*/
		*ret_fname = (const char **)glob_area.gl_pathv;
		ret = 0;
		break;

	default:
		FPUT_SRC_CODE( stderr );
		(void) fputs( "glob() returned unknown status code (", stderr );
		(void) fput_int( status, stderr );
		(void) fputs( ").\n", stderr );
		ret = -1;
	}

	if ( Debug ) {
		if ( status == 0 ) {
			ndx = (unsigned short)1;
			ptr = *ret_fname;
			while ( *ptr != (const char *)0 ) {
				(void) fprintf( stderr, "%hu. [%s]\n", ndx, *ptr );
				++ndx;
				++ptr;
			}
		}
	}

	RETURN_INT( ret );
}

static int glob_err( const char *gpath, int gerrno )
{
	int	save_errno;
	char	msg[300];

	save_errno = errno;
	errno = gerrno;

	(void) strcpy( msg, "glob() warning on [" );
	(void) strncat( msg, gpath, (size_t)200 );
	(void) strcat( msg, "]" );

	UNIX_ERROR( msg );

	errno = save_errno;

	return 0;
}
